/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2026 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - MassXpert, model polymer chemistries and simulate mass spectrometric data;
 * - MineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes


/////////////////////// Local includes
#include "FragmentOligomerTableViewModel.hpp"
#include <MsXpS/libXpertMassCore/Oligomer.hpp>


namespace MsXpS
{

namespace MassXpert
{


FragmentOligomerTableViewModel::FragmentOligomerTableViewModel(
  libXpertMassCore::OligomerCollection *oligomers_p, QObject *parent)
  : QAbstractTableModel(parent)
{
  if(!oligomers_p)
    qFatal("Fatal error at %s@%d. Aborting.", __FILE__, __LINE__);

  mp_oligomers = oligomers_p;

  if(!parent)
    qFatal("Fatal error at %s@%d. Aborting.", __FILE__, __LINE__);

  mp_parentDlg = static_cast<FragmentationDlg *>(parent);

  // Port to Qt5
  // reset();
}


FragmentOligomerTableViewModel::~FragmentOligomerTableViewModel()
{
}


const libXpertMassCore::OligomerCollection *
FragmentOligomerTableViewModel::oligomerList()
{
  return mp_oligomers;
}


FragmentationDlg *
FragmentOligomerTableViewModel::parentDlg()
{
  return mp_parentDlg;
}


void
FragmentOligomerTableViewModel::setTableView(
  FragmentOligomerTableView *tableView)
{
  if(!tableView)
    qFatal("Fatal error at %s@%d. Aborting.", __FILE__, __LINE__);

  mp_tableView = tableView;
}


int
FragmentOligomerTableViewModel::rowCount(const QModelIndex &parent) const
{
  Q_UNUSED(parent);

  return mp_oligomers->size();
}


int
FragmentOligomerTableViewModel::columnCount(const QModelIndex &parent) const
{
  Q_UNUSED(parent);

  return FRAGMENT_OLIGO_TOTAL_COLUMNS;
}


QVariant
FragmentOligomerTableViewModel::headerData(int section,
                                           Qt::Orientation orientation,
                                           int role) const
{
  if(role != Qt::DisplayRole)
    return QVariant();

  if(orientation == Qt::Vertical)
    {
      // Return the row number.
      QString valueString;
      valueString.setNum(section);
    }
  else if(orientation == Qt::Horizontal)
    {
      // Return the header of the column.
      switch(section)
        {
          case FRAGMENT_OLIGO_PATTERN_COLUMN:
            return tr("Frag. Pattern");
          case FRAGMENT_OLIGO_NAME_COLUMN:
            return tr("Name");
          case FRAGMENT_OLIGO_COORDINATES_COLUMN:
            return tr("Coords");
          case FRAGMENT_OLIGO_MONO_COLUMN:
            return tr("Mono");
          case FRAGMENT_OLIGO_AVG_COLUMN:
            return tr("Avg");
          case FRAGMENT_OLIGO_CHARGE_COLUMN:
            return tr("Charge");
          case FRAGMENT_OLIGO_FORMULA_COLUMN:
            return tr("Formula");
          case FRAGMENT_OLIGO_MODIF_COLUMN:
            return tr("Modif?");
          default:
            qFatal("Fatal error at %s@%d. Aborting.", __FILE__, __LINE__);
        }
    }
  // Should never get here.
  return QVariant();
}


QVariant
FragmentOligomerTableViewModel::data(const QModelIndex &index, int role) const
{
  if(!index.isValid())
    return QVariant();

  if(role == Qt::TextAlignmentRole)
    {
      return int(Qt::AlignRight | Qt::AlignVCenter);
    }
  else if(role == Qt::DisplayRole)
    {
      int row    = index.row();
      int column = index.column();

      // Let's get the data for the right column and the right
      // row. Let's find the row first, so that we get to the proper
      // oligomer.

      // libXpertMassCore::OligomerSPtr oligomer_sp = mp_oligomers->at(row);

      libXpertMassCore::OligomerSPtr oligomer_sp =
        mp_oligomers->getOligomersCstRef().at(row);

      // Now see what's the column that is asked for. Prepare a
      // string that we'll feed with the right data before returning
      // it as a QVariant.

      QString valueString;

      if(column == FRAGMENT_OLIGO_PATTERN_COLUMN)
        {
          valueString = oligomer_sp->getDescription();
        }
      else if(column == FRAGMENT_OLIGO_NAME_COLUMN)
        {
          valueString = oligomer_sp->getName();
        }
      else if(column == FRAGMENT_OLIGO_COORDINATES_COLUMN)
        {
          foreach(const libXpertMassCore::IndexRange *item,
                  oligomer_sp->getCalcOptionsCstRef()
                    .getIndexRangeCollectionCstRef()
                    .getRangesCstRef())
            valueString +=
              QString("[%1-%2]").arg(item->m_start + 1).arg(item->m_stop + 1);
        }
      else if(column == FRAGMENT_OLIGO_MONO_COLUMN)
        {
          valueString = QString("%1\n").arg(
            oligomer_sp->getMass(libXpertMassCore::Enums::MassType::MONO),
            0,
            'f',
            libXpertMassCore::OLIGOMER_DEC_PLACES);
        }
      else if(column == FRAGMENT_OLIGO_AVG_COLUMN)
        {
          valueString = QString("%1\n").arg(
            oligomer_sp->getMass(libXpertMassCore::Enums::MassType::AVG),
            0,
            'f',
            libXpertMassCore::OLIGOMER_DEC_PLACES);
        }
      else if(column == FRAGMENT_OLIGO_CHARGE_COLUMN)
        {
          valueString.setNum(oligomer_sp->getIonizerCstRef().charge());
        }
      else if(column == FRAGMENT_OLIGO_FORMULA_COLUMN)
        {
          // Do not use the oligomer_sp->elementalComposition() function
          // because it cannot account for all the fragmentation specifics.
          // This is why Oligomer::m_formula is for: while fragmenting,
          // store in that Formula all the fragmentation specifics that
          // translate into a Formula.
          valueString = oligomer_sp->getFormulaCstRef().getActionFormula();
        }
      else if(column == FRAGMENT_OLIGO_MODIF_COLUMN)
        {
          // If the user changed the sequence by removing monomers, looking
          // into the polymer might encounter a monomer index that is greater
          // than the actual size of the polymer. Thus look into
          // oligomer_sp->m_isModified only.
          if(oligomer_sp->isModified(false /*do not look deep in polymer*/))
            valueString = "True";
          else
            valueString = "False";
        }
      else
        {
          qFatal("Fatal error at %s@%d. Aborting.", __FILE__, __LINE__);
        }

      return valueString;
    }
  // End of
  // else if(role == Qt::DisplayRole)

  return QVariant();
}

int
FragmentOligomerTableViewModel::addOligomers(
  const libXpertMassCore::OligomerCollection &oligomers)
{
  // We receive a list of oligomers which we are asked to add to the
  // list of oligomers that we actually do not own, since they are
  // owned by the FragmentationDlg instance that is the "owner" of this
  // model. But since we want to add the oligomers to the list of
  // oligomers belonging to the FragmentationDlg instance in such a way
  // that the tableView takes them into account, we have to do that
  // addition work here.

  // Note that we are acutally *transferring* all the oligomers from
  // the oligomerList to this model's mp_oligomers.

  std::size_t initial_size = mp_oligomers->getOligomersCstRef().size();

  // qDebug() << __FILE__ << __LINE__
  //          << "model oligomers:" << oligomerCount;

  // We have to let know the model that we are going to modify the
  // list of oligomers and thus the number of rows in the table
  // view. We will append the oligomers to the preexisting ones,
  // which means we are going to have our first appended oligomer at
  // index = oligomerCount.

  beginInsertRows(QModelIndex(),
                  initial_size,
                  (initial_size + oligomers.getOligomersCstRef().size() - 1));

  for(const libXpertMassCore::OligomerSPtr &oligomer_sp :
      oligomers.getOligomersCstRef())
    mp_oligomers->getOligomersRef().push_back(oligomer_sp);

#if 0
//  old version
  for(int iter = 0; iter < oligomersToAdd; ++iter)
    {
      // We need to insert the oligomers as FragmentOligomer's because only this
      // class has the right elementalComposition() function to return the
      // proper m_formula as computed by the Fragmenter class.
      FragmentOligomerSPtr oligomer_sp =
        std::dynamic_pointer_cast<FragmentOligomer>(oligomerList.at(iter));

      // qDebug() << "Adding to the table view model the oligomer:"
      //<< oligomer_sp->elementalComposition()
      //<< "with charge:" << oligomer_sp->charge();

      mp_oligomers->append(oligomer_sp);

      ++addedOligomerCount;
    }
#endif

  endInsertRows();

  return mp_oligomers->size() - initial_size;
}


int
FragmentOligomerTableViewModel::removeOligomers(std::size_t start_index,
                                                std::size_t stop_index)
{
  // We are asked to remove the oligomers [start_index--stop_index].

  // We are asked to remove the oligomers from the list of oligomers
  // that we actually do not own, since they are owned by the
  // FragmentationDlg instance that is the "owner" of this model. But
  // since we want to remove the oligomers from the list of
  // oligomers belonging to the FragmentationDlg instance in such a way
  // that the tableView takes them into account, we have to do that
  // removal work here.

  std::size_t initial_size = mp_oligomers->getOligomersCstRef().size();

  // qDebug() << "The oligomers container has size:" << initial_size;

  if(!initial_size)
    return 0;

  if(start_index >= initial_size)
    qFatal() << "Programming error. Out-of-bounds error.";
  if(stop_index >= initial_size)
    qFatal() << "Programming error. Out-of-bounds error.";

  beginRemoveRows(QModelIndex(), start_index, stop_index);

#if 0
// Old version
  int iter = lastIdx;
  while(iter >= firstIdx)
    {
      libXpertMassCore::OligomerSPtr oligomer_sp = mp_oligomers->takeAt(iter);
      oligomer_sp.reset();
      ++removedOligomerCount;
      --iter;
    }
#endif

  std::vector<libXpertMassCore::OligomerSPtr>::iterator the_begin_iterator =
    mp_oligomers->getOligomersRef().begin() + start_index;
  std::vector<libXpertMassCore::OligomerSPtr>::iterator the_end_iterator =
    mp_oligomers->getOligomersRef().begin() + stop_index + 1;

  mp_oligomers->getOligomersRef().erase(the_begin_iterator, the_end_iterator);
  endRemoveRows();

  return initial_size - mp_oligomers->size();
}


} // namespace MassXpert

} // namespace MsXpS
