<?php

namespace Edispatcher\Entity;


class Metrics {


  private static $instance;
  private static $NAMESPACE="edispatcher";
  private $registry;
  private $prometheus_url = null;

  private $redis = null;
  private $pipe  = null;

  private $hasMetrics = false;

  public function __construct() {

    global $__METRICS;

    if (is_array($__METRICS))  {

        if ($__METRICS["mode"] == "redis" ) {
          \Prometheus\Storage\Redis::setDefaultOptions(
              [
                  'host' => $__METRICS["redis"]["host"],
                  'port' => $__METRICS["redis"]["port"],
                  'timeout' => 0.1, // in seconds
                  'read_timeout' => 10, // in seconds
                  'persistent_connections' => defined("REDIS_PERSISTENT") && REDIS_PERSISTENT
              ]
          );

          $this->redis = new \Redis();
          if (defined("REDIS_PERSISTENT") && REDIS_PERSISTENT) {
              if (!$this->redis->pconnect($__METRICS["redis"]["host"],$__METRICS["redis"]["port"])) {
                  $this->redis  = null;
              };
          } else {
              if (!$this->redis->connect($__METRICS["redis"]["host"],$__METRICS["redis"]["port"])) {
                  $this->redis  = null;
              };
          }

          if ($this->redis && !$this->redis->select(7)) {
              $this->redis  = null;
          }

          // prometheus
          $this->prometheus_url = $__METRICS["prometheus"];

          $this->registry = \Prometheus\CollectorRegistry::getDefault();

          $this->metrics=[];

          $this->metrics["educonnect_access"]     = $this->registry->registerCounter(self::$NAMESPACE, 'educonnect_access', 'Access Educonnect', ['uaj','type']);

          $this->metrics["etab_portail"]          = $this->registry->registerCounter(self::$NAMESPACE, 'etab_portail', 'portail', ['uaj','profil']);
          $this->metrics["etab_federation"]       = $this->registry->registerCounter(self::$NAMESPACE, 'etab_federation', 'fédération', ['uaj','profil']);

          $this->metrics["newUser"]               = $this->registry->registerCounter(self::$NAMESPACE, 'new_user', 'New User', ['profil']);
          $this->metrics["visit"]                 = $this->registry->registerCounter(self::$NAMESPACE, 'visit', 'Nb visite', ['profil']);
          $this->metrics["visitGlobalCounter"]    = $this->registry->registerCounter(self::$NAMESPACE, 'visit_global_counter', 'Nb visite global', ['host']);
          $this->metrics["dispatching"]           = $this->registry->registerCounter(self::$NAMESPACE, 'dispatching', 'Nombre de dispatching ( appel a /ng/hub )', ['profil']);

          $this->metrics["arenaFromCache"]        = $this->registry->registerGauge(self::$NAMESPACE, 'arena_from_cache', 'Load arena from cache', ['zone']);
          $this->metrics["arenaFromWebService"]   = $this->registry->registerGauge(self::$NAMESPACE, 'arena_from_ws', 'Load arena from WebService', ['zone']);
          $this->metrics["arenaToCache"]          = $this->registry->registerGauge(self::$NAMESPACE, 'arena_to_cache', 'Store arena to cache', ['zone']);
          $this->metrics["arenaCacheObsolete"]    = $this->registry->registerGauge(self::$NAMESPACE, 'arena_cache_obsolete', 'Nombre de cache obsolete');
          $this->metrics["arenaCacheAveragDuration"] = $this->registry->registerGauge(self::$NAMESPACE, 'arena_cache_averageduration', 'Durée moyenne du cache en secondes');

          $this->metrics["NbComptes"]             = $this->registry->registerGauge(self::$NAMESPACE, 'nb_comptes', 'Nombre de comptes total', ['profil']);
          $this->metrics["NbAccessLastMinuteByProfil"]    = $this->registry->registerGauge(self::$NAMESPACE, 'nb_access_byprofil_since_lastminute', 'Nombre d\'accès dernière minute', ['profil']);
          $this->metrics["NbAccessLastMinuteByHost"]      = $this->registry->registerGauge(self::$NAMESPACE, 'nb_access_byhost_since_lastminute', 'Nombre d\'accès dernière minute', ['host']);
          $this->metrics["NbAccessLastMinuteByEtabType"]  = $this->registry->registerGauge(self::$NAMESPACE, 'nb_access_byetabtype_since_lastminute', 'Nombre d\'accès dernière minute', ['etabtype']);

          $this->metrics["uajAccess"]             = $this->registry->registerCounter(self::$NAMESPACE, 'uaj_access', 'Accès établissements', ['profil','uaj','type']);
          $this->metrics["appAccess"]             = $this->registry->registerCounter(self::$NAMESPACE, 'app_access2', 'Accès aux applications', ['appid','urlid','profil','uaj','type']);
          $this->metrics["visitDelayInSeconds"]   = $this->registry->registerGauge(self::$NAMESPACE, 'visit_delay_seconds', 'Nombre de secondes écoulées entre 2 connexions', ['profil','uaj','type']);

          $this->metrics["accountsAvailable"]     = $this->registry->registerGauge(self::$NAMESPACE, 'accounts_available_gauge', 'Nombre de comptes disponibles', ['profil','uaj','type']);
          $this->metrics["accountsUsedByHour"]    = $this->registry->registerGauge(self::$NAMESPACE, 'accounts_used_h_gauge', 'Nombre de comptes utilisés (par heure)', ['profil','uaj','type']);
          $this->metrics["accountsUsedByDay"]     = $this->registry->registerGauge(self::$NAMESPACE, 'accounts_used_d_gauge', 'Nombre de comptes utilisés (par jour)', ['profil','uaj','type']);
          $this->metrics["accountsUsedByMonth"]   = $this->registry->registerGauge(self::$NAMESPACE, 'accounts_used_m_gauge', 'Nombre de comptes utilisés (par mois)', ['profil','uaj','type']);
          $this->metrics["accountsUsedByYear"]    = $this->registry->registerGauge(self::$NAMESPACE, 'accounts_used_y_gauge', 'Nombre de comptes utilisés (par an)', ['profil','uaj','type']);

          $this->hasMetrics=true;
        }
    }

  }

  public function getRegistry() {
    return $this->registry;
  }

  private function log($name,$callback) {
    if (!$this->hasMetrics) return false;

    try {
      $callback($this->metrics[$name]);
      return true;
    } catch(\Exception $e) {
      return false;
    }

  }

  public function getRedis() {
      return $this->redis;
  }

  public function getRedisPipe() {
    if (!$this->redis) return null;

    if (!$this->pipe) {
        $this->pipe = $this->redis->multi(\Redis::PIPELINE);
    }
    return $this->pipe;
  }

  public function manageFavoris($appid,$urlid,$intid,$profil,$bFavoris) {
    $pipe = $this->getRedisPipe();
    if (!$pipe) return null;

    if ($bFavoris) {
      $pipe->sAdd("favurl_".$urlid,intval($intid));
      $pipe->sAdd("favapp_".$appid,intval($intid));
    } else {
      $pipe->sRemove("favurl_".$urlid,intval($intid));
      $pipe->sRemove("favapp_".$appid,intval($intid));
    }
  }

  public function ping() {
      if (!$this->redis) return 0;
      return $this->redis->ping();
  }

  public function favUrl($id) {
    if (!$this->redis) return 0;
    return $this->redis->scard("favurl_".$id);
  }

  public function favApp($id) {
    if (!$this->redis) return 0;
    return $this->redis->scard("favapp_".$id);
  }

  /* ==== GrafabaTask & Slack =============================*/

  public function grafanaTask($task) {
    if (!$this->redis) return 0;
    $this->redis->rpush("grafana_tasks",$task);
    return true;
  }

  public function  addGrafanaWorker($uid,$channel,$key,$task){
     if (!$this->redis) return false;
     $this->redis->rpush("grafana_worker","$uid;$channel;$key;$task");
     return $this->grafanaTask($task);

  }
  public function  getGrafanaWorker(){
     if (!$this->redis) return false;
     return $this->redis->lpop("grafana_worker");
  }

  public function health(){

    $health = [];

    $data = $this->accountsByProfil();
    $nbAccountsByProfil=[];$totalAccounts = 0;
    foreach ($data as $field) {
        $nbAccountsByProfil[$field->metric->profil] = $field->value[1] ;
        $totalAccounts += $field->value[1];
    }

    $totalByHour = 0;
    $data = $this->accountsUsedByProfil("h");
    foreach ($data as $field) {
        $totalByHour += $field->value[1] ;
    }

    $totalByDay = 0;
    $data = $this->accountsUsedByProfil("d");
    foreach ($data as $field) {$totalByDay += $field->value[1] ;}

    $totalByMonth = 0;
    $data = $this->accountsUsedByProfil("m");
    foreach ($data as $field) {$totalByMonth += $field->value[1] ;}

    $health["accounts"] = number_format($totalAccounts, 0, ',', ' ') ;
    $health["hour"] = number_format($totalByHour, 0, ',', ' ') ;
    $health["day"]  = number_format($totalByDay, 0, ',', ' ') ;
    $health["day_percent"] = floor(($totalByDay/$totalAccounts)*100);
    $health["month"]  = number_format($totalByMonth, 0, ',', ' ') ;
    $health["month_percent"] = floor(($totalByMonth/$totalAccounts)*100);

    // ARENA
    $data=$this->prometheusApi("edispatcher_arena_cache_averageduration/3600");
    $health["arena_duration"] =  floor(floatval($data->data->result[0]->value[1]));

    $data=$this->prometheusApi("edispatcher_arena_cache_obsolete");
    $health["arena_obsolete"] =  floor(floatval($data->data->result[0]->value[1]));

    $data=$this->prometheusApi("avg(avg_over_time(edispatcher_arena_from_cache[5m]))");
    $health["arena_cache"] =  floor($data->data->result[0]->value[1]);

    $health["arena_time"] =  floor($data->data->result[0]->value[0]);
    $health["arena_status"] = "good";
    if ($health["arena_cache"]>1000) $health["arena_status"] = "warning";
    if ($health["arena_obsolete"]>1) $health["arena_status"] = "warning";
    if ($health["arena_duration"]>max(__CACHE_DURATION - 1,1)) $health["arena_status"] = "warning";
    if ($health["arena_duration"]>=__CACHE_DURATION) $health["arena_status"] = "danger";

    // == Seveurs  ========================================================================================
    $web_servers=["hubole-master.*","hubole-slave.*"];
    $health=array_merge($this->getSystemMetrics("web",$web_servers),$health);

    $sso_servers=["sso-master.*","sso-slave.*"];
    $health=array_merge($this->getSystemMetrics("sso",$sso_servers),$health);
    // =================================================================================================================


    return $health;

  }

  public function getAvailableServers($job) {
    $data = $this->prometheusApi("sum(haproxy_server_up{job='$job'})/count(haproxy_server_up{job='$job'})*100");
    return floor($data->data->result[0]->value[1]);
  }

  public function getSystemMetrics($tag,$servers) {
      $health = [];
      // CPU
      $total = 0; $time=0;
      foreach($servers as $node) {
          $data = $this->prometheusApi("100*(1-avg+by(instance)(irate(node_cpu{instance=~'$node',mode='idle'}[5m])))");
          $field = $data->data->result[0];
          $total += $field->value[1];
          $time=max($field->value[0],$time);
      }
      $health[$tag."_cpu"] =floor(($total/count($servers)));

      //MEM
      $total = 0;
      foreach($servers as $node) {
          $data = $this->prometheusApi("(node_memory_MemFree{instance=~'$node'}/node_memory_MemTotal{instance=~'$node'})*100");
          $field = $data->data->result[0];
          $total += $field->value[1];
          $time=max($field->value[0],$time);
      }
      $health[$tag."_mem"] =floor(($total/count($servers)));

      // SWAP
      $total = 0;
      foreach($servers as $node) {
          $data = $this->prometheusApi("(node_memory_SwapTotal{instance=~'$node'}-node_memory_SwapFree{instance=~'$node'})");
          $field = $data->data->result[0];
          $total += $field->value[1];
      }
      $val = floor($total/count($servers));
      $health[$tag."_swap"] = $val<10240?0:number_format(floor($val/(1024*1024)), 0, ',', ' ');


      $health[$tag."_time"] = floor($time);
      $health[$tag."_status"] = "good";
      if ($health[$tag."_cpu"] > 50 || $health[$tag."_mem"]<10 ) $health[$tag."_status"] = "warning";
      if ($health[$tag."_cpu"] > 50 && $health[$tag."_mem"]<10 ) $health[$tag."_status"] = "danger";

      return $health;
  }

  // =============================================================================

  public function saveFavoris() {
    $pipe = $this->getRedisPipe();
    if (!$pipe) return null;

    $pipe->exec();
  }

  public function incMetrics($name,$labels=[]) {
      $this->log($name, function($metric) use($labels) { $metric->inc($labels);});
  }


  public function activity($intid,$profil) {
    if (!$this->hasMetrics) return ;
    if (!$this->redis)      return;

    $this->redis->sAdd( "time_".date('mdy:H'), intval($intid));
    $this->redis->sAdd( "time_".date('mdy')  , intval($intid));
    $this->redis->sAdd( "time_".date('my')  , intval($intid));
    $this->redis->sAdd( "time_".date('y')  , intval($intid));

    $this->redis->sAdd( $profil  , intval($intid));
  }

  public function accountsAvailable($count,$profil,$uaj,$type) {
    $this->log(__FUNCTION__, function($metric)  use($count,$profil,$uaj,$type) {
      $metric->set($count,[$profil,$uaj,$type]);
    });
  }

  public function accountsUsed($by,$count,$profil,$uaj,$type) {
    $this->log("accountsUsedBy".$by, function($metric)  use($count,$profil,$uaj,$type) {
      $metric->set($count,[$profil,$uaj,$type]);
    });
  }

  public function appAccess($appid,$urlid,$profil,$uaj) {
    $this->log(__FUNCTION__, function($metric)  use($appid,$urlid,$profil,$uaj) {
      $etab=\R::findOne( 'etab', ' uaj = ? ', [$uaj] );
      if ($etab) {
        $metric->inc([$appid,$urlid,$profil,$uaj,$etab->type]);
      }

    } );
  }

  public function uajAccess($profil,$uaj) {
      $this->log(__FUNCTION__, function($metric)  use($uaj,$profil) {
          $etab=\R::findOne( 'etab', ' uaj = ? ', [$uaj] );
          if ($etab) {
            $metric->inc([$profil,$uaj,$etab->type]);
          }
      } );
  }

  public function visitUaj($uaj,$intid) {
    if ($this->redis) {$this->redis->sAdd( $uaj  , intval($intid));}
  }

  public function visitDelayInSeconds($profil,$uaj,$value,$intid) {
    $this->log(__FUNCTION__, function($metric)  use($profil,$uaj,$value,$intid) {
      $etab=\R::findOne( 'etab', ' uaj = ? ', [$uaj] );
      $type="na"; 
      if ($etab) {
        $metric->set($value,[$profil,$uaj,$etab->type]);
        $type=$etab->type;
      } else {
        try {
          $etab = \R::dispense( 'etab' );
          $etab->uaj  = $uaj;
          $etab->type = "na";
          $etab->nom  = $uaj;
          \R::store($etab);
        }  catch( \Exception $e) {}
      }
      $this->redis->sAdd( $uaj  , intval($intid));
      $this->redis->sAdd( $type  , intval($intid));
    } );
  }

  public function appStatsApp($id) {
    $exclude = "";
    $obj = $this->prometheusApi("sum(edispatcher_app_access2{appid='". $id . "'".$exclude."})");
    if (! $obj ) return -1;
    return $obj->data->result[0]->value[1];
  }

  public function urlStatsApp($id) {
    $exclude = "";
    $obj = $this->prometheusApi("sum(edispatcher_app_access2{urlid='". $id . "'".$exclude."})");
    if (! $obj ) return -1;
    return $obj->data->result[0]->value[1];
  }

  public function appByProfil($id,$queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{appid='".$id."'".$exclude."})+by+(profil)",$queryParams);
    return $obj->data->result;
  }

  // ====App ===================================================================================================================

  public function appsByProfilApp($queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi("sum(edispatcher_app_access2{appid!=''".$exclude."})+by+(profil,appid)",$queryParams);
    return $obj->data->result;
  }

  public function appsByProfil($queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi("sum(edispatcher_app_access2{appid!=''".$exclude."})+by+(profil)",$queryParams);
    return $obj->data->result;
  }

  public function appByUaj($id,$queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{appid='".$id."'".$exclude."})+by+(uaj)",$queryParams);
    return $obj->data->result;
  }

  public function apps($queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{appid!=''".$exclude."})+by+(appid)",$queryParams);
    return $obj->data->result;
  }

  public function urls($queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{urlid!=''".$exclude."})+by+(urlid)",$queryParams);
    return $obj->data->result;
  }

  public function url($id,$queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{urlid='".$id."'".$exclude."})+by+(urlid)",$queryParams);
    if (! $obj ) return 0;
    return $obj->data->result[0]->value[1];
  }

  // ===========================================================================================================================

  // Accounts ===================================================================================================================
  public function accountsByProfil($queryParams=null) {
    $obj = $this->prometheusApi("sum(edispatcher_accounts_available_gauge{type!=''".$this->getExclude($queryParams)."})+by+(profil)",$queryParams);
    return $obj->data->result;
  }

  public function accountsUsedByProfil($by,$queryParams=null) {
    $obj = $this->prometheusApi("sum(edispatcher_accounts_used_".$by."_gauge{type!=''".$this->getExclude($queryParams)."})+by+(profil)",$queryParams);
    return $obj->data->result;
  }

  public function accountsUajByProfil($uaj,$queryParams) {
    $obj = $this->prometheusApi("sum(edispatcher_accounts_available_gauge{uaj='".$uaj."'".$this->getExclude($queryParams)."})+by+(profil)",$queryParams);
    return $obj->data->result;
  }

  public function accountsUsedUajByProfil($uaj,$by,$queryParams) {
    $obj = $this->prometheusApi("sum(edispatcher_accounts_used_".$by."_gauge{uaj='".$uaj."'".$this->getExclude($queryParams)."})+by+(profil)",$queryParams);
    return $obj->data->result;
  }

  public function accountsUsedProfilByUaj($profil,$by,$queryParams) {
    $obj = $this->prometheusApi("sum(edispatcher_accounts_used_".$by."_gauge{profil='".$profil."'".$this->getExclude($queryParams)."})+by+(uaj)",$queryParams);
    return $obj->data->result;
  }

  public function accountsDelayByProfil($queryParams) {
    $obj = $this->prometheusApi("avg(edispatcher_visit_delay_seconds{uaj!=''".$this->getExclude($queryParams)."})+by+(profil)/(3600*24)",$queryParams);
    return $obj->data->result;
  }

  // UAJ =======================================================================================================

  public function uajByApp($uaj,$queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{uaj='".$uaj."'".$exclude."})+by+(appid)",$queryParams);
    return $obj->data->result;
  }

  public function uajLastActivity($uaj,$delayInSecondes=300) {

    $params= [];
    $params["time"] = time();

    $obj = $this->prometheusApi("sum(edispatcher_uaj_access{uaj='".$uaj."'})",$params);
    if (!$obj) return 0;

    $c1 = $obj->data->result[0]->value[1];

    $params["time"] =  time() - $delayInSecondes;
    $obj = $this->prometheusApi("sum(edispatcher_uaj_access{uaj='".$uaj."'})",$params);
    $c2 = $obj->data->result[0]->value[1];


    return $c1-$c2;
  }

  public function uajsLastActivity($delayInMinutes) {
    $params= [];
    $obj = $this->prometheusApi("sum(delta(edispatcher_uaj_access[".$delayInMinutes."m]))+by+(uaj)",$params);
    return $obj->data->result;
  }

  public function uajByUrl($uaj,$queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{uaj='".$uaj."'".$exclude."})+by+(urlid)",$queryParams);
    return $obj->data->result;
  }

  public function uajByProfil($uaj,$queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{uaj='".$uaj."'".$exclude."})+by+(profil)",$queryParams);
    return $obj->data->result;
  }

  public function uajsByProfil($queryParams) {
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi("sum(edispatcher_app_access2{appid!=''".$exclude."})+by+(profil,uaj)",$queryParams);
    return $obj->data->result;
  }

  public function urlByProfil($id,$queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{urlid='".$id."'".$exclude."})+by+(profil)",$queryParams);
    return $obj->data->result;
  }

  public function urlsByProfil($queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi("sum(edispatcher_app_access2{appid!=''".$exclude."})+by+(profil,urlid)",$queryParams);
    return $obj->data->result;
  }

  public function urlByUaj($id,$queryParams) {
    $agr = "sum";
    $exclude = $this->getExclude($queryParams);
    $obj = $this->prometheusApi($agr."(edispatcher_app_access2{urlid='".$id."'".$exclude."})+by+(uaj)",$queryParams);
    return $obj->data->result;
  }

  private function getExclude($queryParams) {

    $toExclude=[];

    // Profil
    if (isset($queryParams['exclude_profil'])) {
      foreach (explode(",",$queryParams['exclude_profil']) as $v) {
        if (!$v) continue;
        $toExclude[]="profil!='".$v."'";
      }
    }

    // type étab
    if (isset($queryParams['exclude_type'])) {
      foreach (explode(",",$queryParams['exclude_type']) as $v) {
        if (!$v) continue;
        $toExclude[]="type!='".$v."'";
      }
    }

    // type étab
    if (isset($queryParams['with_type'])) {
      $x=[];
      foreach (explode(",",$queryParams['with_type']) as $v) {
        if (!$v) continue;
        $x[]=$v;
      }

      $toExclude[]="type=~'".implode("|",$x)."'";
    }

    if (count($toExclude)==0) return "";

    return ",".implode(",",$toExclude);


  }



  private function prometheusApi($query,$queryParams = null) {

    if (!$this->prometheus_url) return null;

    $type="query";
    $params="";
    if ($queryParams && isset($queryParams['start']) && isset($queryParams['end'])) {
      $type="query_range";
      $params="&start=".$queryParams['start']."&end=".$queryParams['end']."&step=".$queryParams['step'];
    }

    if ($queryParams && isset($queryParams['time'])) {
      $params="&time=".$queryParams['time'];
    }


    try {
      $url =  $this->prometheus_url . "/api/v1/".$type."?query=" . $query.$params;
      $json = file_get_contents($url);
      $obj = json_decode($json);
      return $obj;
    } catch(\Exception $e) {
      echo($e);
      return null;
    }

    return null;
  }



  /* =========== ARENA METRICS ======== */
  public function arenaFromCache($zone,$time) {
     $this->log(__FUNCTION__, function($metric)  use($zone,$time) { $metric->set($time*1000,[$zone]); } );
  }

  public function arenaFromWebService($zone,$time) {
    $this->log(__FUNCTION__, function($metric)  use($zone,$time) { $metric->set($time*1000,[$zone]); } );
  }

  public function arenaToCache($zone,$time) {
     $this->log(__FUNCTION__, function($metric)  use($zone,$time) { $metric->set($time*1000,[$zone]); } );
  }

  public function arenaCacheObsolete($count) {
     $this->log(__FUNCTION__, function($metric) use($count) { $metric->set($count); } );
  }

  public function arenaCacheAveragDuration($avg) {
     $this->log(__FUNCTION__, function($metric) use($avg) { $metric->set($avg); } );
  }
  /* =================================== */

  public function NbComptes($profil,$count) {
     $this->log(__FUNCTION__, function($metric)  use($profil,$count) { $metric->set($count,[$profil]); } );
  }

  public function NbAccessLastMinuteByProfil($profil,$count) {
     $this->log(__FUNCTION__, function($metric)  use($profil,$count) { $metric->set($count,[$profil]); } );
  }

  public function NbAccessLastMinuteByHost($host,$count) {
     $this->log(__FUNCTION__, function($metric)  use($host,$count) { $metric->set($count,[$host]); } );
  }

  public function NbAccessLastMinuteByEtabType($etabtype,$count) {
    $this->log(__FUNCTION__, function($metric)  use($etabtype,$count) { $metric->set($count,[$etabtype]); } );
 }

  public function NbAccessLastMinute($profil,$count) {
     $this->log(__FUNCTION__, function($metric)  use($profil,$host,$count) { $metric->set($count,[$profil]); } );
  }

  public function dispatching($profil) {
     $this->log(__FUNCTION__, function($metric)  use($profil) { $metric->inc([$profil->getProfil()]); } );
  }

  public function newUser($profil) {
     $this->log(__FUNCTION__, function($metric)  use($profil) { $metric->inc([$profil->getProfil()]); } );
  }

  public function visit($profil) {
    $this->log(__FUNCTION__, function($metric)  use($profil) { $metric->inc([$profil->getProfil()]); } );
    $this->log("visitGlobalCounter", function($metric)               { $metric->inc([getConfigHostname()]); } );
  }

  public static function instance() {
    if (!self::$instance) {
      self::$instance=new Metrics();

    }

    return self::$instance;
  }

}
