<?php

namespace App\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Ethercalc
 *
 * @ORM\Table(name="ethercalc")
 * @ORM\Entity(repositoryClass="App\Repository\EthercalcRepository")
 * 
 * @UniqueEntity(fields="title", message="Un pad existe déjà avec ce titre")
 */
class Ethercalc
{
    /**
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @ORM\Column(name="name", type="string", length=100, unique=true)
     * @Assert\Regex(
     *     pattern="/^[a-zA-Z0-9_-]+$/",
     *     match=true,
     *     message="Caractères spéciaux interdit. Caractères autorisés a-z A-Z 0-9 - _"
     * )
     */
    private $title;

    /**
     * @ORM\ManyToOne(targetEntity="User", inversedBy="ethercalcs")
     * @ORM\JoinColumn(nullable=true)
     */
    private $user;

    /**
     * @ORM\ManyToMany(targetEntity="Group", inversedBy="ethercalcs", cascade={"persist"})
     * @ORM\JoinTable(name="ethercalcgroupe",
     *   joinColumns={@ORM\JoinColumn(name="ethercalc", referencedColumnName="id")},
     *   inverseJoinColumns={@ORM\JoinColumn(name="groupe", referencedColumnName="id")}
     * )
     */
    protected $groups;

    /**
     * @ORM\ManyToMany(targetEntity="User", inversedBy="ethercalcshareds", cascade={"persist"})
     * @ORM\JoinTable(name="ethercalcuser",
     *   joinColumns={@ORM\JoinColumn(name="ethercalc", referencedColumnName="id")},
     *   inverseJoinColumns={@ORM\JoinColumn(name="user", referencedColumnName="id")}
     * )
     */
    protected $users;

    public function __construct()
    {
        $this->groups = new ArrayCollection();
        $this->users = new ArrayCollection();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getTitle(): ?string
    {
        return $this->title;
    }

    public function setTitle(string $title): self
    {
        $this->title = $title;

        return $this;
    }

    public function getUser(): ?User
    {
        return $this->user;
    }

    public function setUser(?User $user): self
    {
        $this->user = $user;

        return $this;
    }

    /**
     * @return Collection|Group[]
     */
    public function getGroups(): Collection
    {
        return $this->groups;
    }

    public function addGroup(Group $group): self
    {
        if (!$this->groups->contains($group)) {
            $this->groups[] = $group;
        }

        return $this;
    }

    public function removeGroup(Group $group): self
    {
        if ($this->groups->contains($group)) {
            $this->groups->removeElement($group);
        }

        return $this;
    }

    /**
     * @return Collection|User[]
     */
    public function getUsers(): Collection
    {
        return $this->users;
    }

    public function addUser(User $user): self
    {
        if (!$this->users->contains($user)) {
            $this->users[] = $user;
        }

        return $this;
    }

    public function removeUser(User $user): self
    {
        if ($this->users->contains($user)) {
            $this->users->removeElement($user);
        }

        return $this;
    }

}
