<?php

namespace App\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;


/**
 * User
 * 
 * @ORM\Entity(repositoryClass="App\Repository\UserRepository")
 * @ORM\Table(name="user",uniqueConstraints={@ORM\UniqueConstraint(name="username", columns={"username"})})
 * @UniqueEntity("username", message="Ce nom d'utilisateur existe dèja")
 */

class User implements UserInterface, \Serializable
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="username", type="string", length=255)
     */
    private $username;

    /**
     * @var string
     *
     * @ORM\Column(name="password", type="string", length=255)
     */
    private $password;

    /**
     * @var array
     *
     * @ORM\Column(name="roles", type="array", length=255)
     */
    private $roles = array();

    /**
     * @ORM\Column(name="salt", type="string", length=255)
     */
    private $salt = '';

    /**
     * @ORM\Column(type="string", length=250, nullable=true)
     */
    private $firstname;

    /**
     * @ORM\Column(type="string", length=250)
     */
    private $lastname;
    private $displayname;

    /**
      * @ORM\Column(type="string", length=200, nullable=true, options={"default" : 0})
     */
    private $avatar;

    /**
     * @ORM\Column(type="string", length=250)
     */
    private $email;

    /**
     * @ORM\Column(type="string", length=250, nullable=true)
     */
    private $apikey;

    /**
     * @ORM\Column(name="preference", type="array", nullable=true)
     */
    private $preference;

    /**
     * @ORM\ManyToMany(targetEntity="Group", inversedBy="users", cascade={"persist"})
     * @ORM\JoinTable(name="usergroupe",
     *   joinColumns={@ORM\JoinColumn(name="user", referencedColumnName="id")},
     *   inverseJoinColumns={@ORM\JoinColumn(name="groupe", referencedColumnName="id")}
     * )
     */
    private $groups;

    /**
     * @ORM\OneToMany(targetEntity="Activity", mappedBy="user", cascade={"persist"}, orphanRemoval=true)
     */
    private $activitys;

    /**
     * @ORM\OneToMany(targetEntity="Answer", mappedBy="user", cascade={"persist"}, orphanRemoval=true)
     */
    private $answers;

    /**
     * @ORM\OneToMany(targetEntity="Message", mappedBy="user", cascade={"persist"}, orphanRemoval=true)
     */
    private $messages;

    /**
    * @ORM\ManyToMany(targetEntity="Message", mappedBy="readers")
    */    
    protected $messagereaders;
    
    public function __construct()
    {
        $this->groups = new ArrayCollection();
        $this->activitys = new ArrayCollection();
        $this->answers = new ArrayCollection();
        $this->messages = new ArrayCollection();
        $this->messagereaders = new ArrayCollection();
    }

    public function getUsername(): ?string
    {
        return $this->username;
    }

    public function getSalt(): ?string
    {
        return $this->salt;
    }

    public function setPassword($password): self
    {
        if($password!=$this->password&&$password!=""&&!is_null($password)){
            $this->salt = uniqid(mt_rand(), true);
            $hash = "{SSHA}" . base64_encode(pack("H*", sha1($password . $this->salt)) . $this->salt);

            $this->password = $hash;
        }

        return $this;
    }

    public function getPassword(): ?string
    {
        return $this->password;
    }

    public function getRoles(): ?array
    {
        return $this->roles;
    }

    public function hasRole(string $role): ?bool
    {
        return in_array($role,$this->roles);
    }

    public function eraseCredentials()
    {
    }

    public function serialize()
    {
        return serialize(array(
            $this->id,
            $this->username,
            $this->password,
            $this->salt,
        ));
    }

    public function unserialize($serialized)
    {
        list (
            $this->id,
            $this->username,
            $this->password,
            $this->salt
        ) = unserialize($serialized, array('allowed_classes' => false));
    }

    public function getDisplayname()
    {
        return $this->firstname." ".$this->lastname;
    }

    public function setId(int $id): self
    {
        $this->id = $id;

        return $this;
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function setUsername(string $username): self
    {
        $this->username = $username;

        return $this;
    }


    public function setRoles(array $roles): self
    {
        $this->roles = $roles;

        return $this;
    }

    public function setSalt(string $salt): self
    {
        $this->salt = $salt;

        return $this;
    }

    public function getFirstname(): ?string
    {
        return $this->firstname;
    }

    public function setFirstname(?string $firstname): self
    {
        $this->firstname = $firstname;

        return $this;
    }

    public function getLastname(): ?string
    {
        return $this->lastname;
    }

    public function setLastname(?string $lastname): self
    {
        $this->lastname = $lastname;

        return $this;
    }

    public function getAvatar(): ?string
    {
        if($this->avatar)
            return $this->avatar;
        else
            return "noavatar.png";
    }

    public function setAvatar(?string $avatar): self
    {
        $this->avatar = $avatar;

        return $this;
    }

    public function getEmail(): ?string
    {
        return $this->email;
    }

    public function setEmail(string $email): self
    {
        $this->email = $email;

        return $this;
    }


    public function getApikey(): ?string
    {
        return $this->apikey;
    }

    public function setApikey(?string $apikey): self
    {
        $this->apikey = $apikey;

        return $this;
    }

    /**
     * @return Collection|Group[]
     */
    public function getGroups(): Collection
    {
        return $this->groups;
    }

    public function addGroup(Group $group): self
    {
        if (!$this->groups->contains($group)) {
            $this->groups[] = $group;
        }

        return $this;
    }

    public function removeGroup(Group $group): self
    {
        if ($this->groups->contains($group)) {
            $this->groups->removeElement($group);
        }

        return $this;
    }

    /**
     * @return Collection|Activity[]
     */
    public function getActivitys(): Collection
    {
        return $this->activitys;
    }

    public function addActivity(Activity $activity): self
    {
        if (!$this->activitys->contains($activity)) {
            $this->activitys[] = $activity;
            $activity->setUser($this);
        }

        return $this;
    }

    public function removeActivity(Activity $activity): self
    {
        if ($this->activitys->contains($activity)) {
            $this->activitys->removeElement($activity);
            // set the owning side to null (unless already changed)
            if ($activity->getUser() === $this) {
                $activity->setUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection|Answer[]
     */
    public function getAnswers(): Collection
    {
        return $this->answers;
    }

    public function addAnswer(Answer $answer): self
    {
        if (!$this->answers->contains($answer)) {
            $this->answers[] = $answer;
            $answer->setUser($this);
        }

        return $this;
    }

    public function removeAnswer(Answer $answer): self
    {
        if ($this->answers->contains($answer)) {
            $this->answers->removeElement($answer);
            // set the owning side to null (unless already changed)
            if ($answer->getUser() === $this) {
                $answer->setUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection|Message[]
     */
    public function getMessages(): Collection
    {
        return $this->messages;
    }

    public function addMessage(Message $message): self
    {
        if (!$this->messages->contains($message)) {
            $this->messages[] = $message;
            $message->setUser($this);
        }

        return $this;
    }

    public function removeMessage(Message $message): self
    {
        if ($this->messages->contains($message)) {
            $this->messages->removeElement($message);
            // set the owning side to null (unless already changed)
            if ($message->getUser() === $this) {
                $message->setUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection|Message[]
     */
    public function getMessagereaders(): Collection
    {
        return $this->messagereaders;
    }

    public function addMessagereader(Message $messagereader): self
    {
        if (!$this->messagereaders->contains($messagereader)) {
            $this->messagereaders[] = $messagereader;
            $messagereader->addReader($this);
        }

        return $this;
    }

    public function removeMessagereader(Message $messagereader): self
    {
        if ($this->messagereaders->contains($messagereader)) {
            $this->messagereaders->removeElement($messagereader);
            $messagereader->removeReader($this);
        }

        return $this;
    }

    public function getPreference(): ?array
    {
        return $this->preference;
    }

    public function getKeypreference($key,$id) {
        if(!is_array($this->preference)) return false;
        if(!array_key_exists($key,$this->preference)) return false;
        if(!is_array($this->preference[$key])) return false;
        if(!array_key_exists($id,$this->preference[$key])) return false;

        return $this->preference[$key][$id];
    }

    public function setPreference(?array $preference): self
    {
        $this->preference = $preference;

        return $this;
    }

}
