# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Gere les groupes de machine
#
###########################################################################
""" gere des groupes de machine definit par plage d'ips
    propose create_group, del_group, get_all_groups, is_group, get_group, test_ip_range
"""

from os.path import isfile

from amon.backend import get_interface, get_filter_zones
from amon.ipset import time_manager
from amon.ipset.config import ipset
from netaddr.core import AddrFormatError
from netaddr import iter_iprange

zones = get_filter_zones().keys()
group_file = ipset['group']


# format du template de gestion des groupes
# un groupe c'est un nom, deux ips, un statut 0:restrictions inactives,
#                                             1:restriction web,
#                                             2: restriction totale (sans horaires)
# un statut horaire: (pris en compte quand le statut vaut '1'
#                   0:"pas d'horaire",
#                   1:"avec horaire"
SYNTAX = "%(name)s#%(ip_from)s#%(ip_to)s#%(active)s#%(time)s#%(interface)s"

def create_group(zone, name, ip_from, ip_to, interface=None):
    """ cree un groupe de machine nivo fichier de configuration
        @name: nom du groupe
        @ip_from: ip de début plage
        @ip_to: ip de fin de plage
        @interface: interface associée à la plage d'ip
    """
    if not is_group(zone, name, True) and test_ip_range(ip_from, ip_to) and test_group_name(name):

        if interface is None:
            interface = get_interface(zone, ip_from)

        line = SYNTAX%{'name':name, 'ip_from':ip_from, 'ip_to':ip_to, 'active':'0', 'time':'0', 'interface':interface}
        add_lines(group_file%zone, line)
        return True
    raise Exception("Erreur: le groupe %s existe déjà."%name)

def is_group(zone, name, all=False):
    """ vérifie l'existence d'un groupe
        @name: nom du groupe
        @all: tester sur toute les zones
    """
    if not all:
        return name in get_all_groups(zone, 'name')
    else:
        for zone in zones:
            if name in name in get_all_groups(zone, 'name'):
                return True
        return False

def get_group_lines(zone):
    return [group for group in open(group_file%zone).read().splitlines() if group != '']

def get_all_groups(zone, attr=None):
    """ renvoie la liste des groupes, attr, renvoie la liste des valeur de l'attribut attr des groupes
        @attr: nom de l'attribut
    """
    if isfile(group_file%zone):
        if attr == 'name':
            return [group.split('#')[0] for group in get_group_lines(zone)]
        elif attr == 'ip_from':
            return [group.split('#')[1] for group in get_group_lines(zone)]
        elif attr == 'ip_to':
            return [group.split('#')[2] for group in get_group_lines(zone)]
        elif attr == 'interface':
            return [group.split('#')[5] for group in get_group_lines(zone)]
        elif attr == None:
            return [{'name':group.split('#')[0],
                     'ip_from':group.split('#')[1],
                     'ip_to':group.split('#')[2],
                     'active':group.split('#')[3],
                     'time':group.split('#')[4],
                     'interface':group.split('#')[5]}for group in get_group_lines(zone)]
    return []

def get_group(zone, name):
    """ renvoie les données concernant un groupe
        @name: nom du groupe
    """
    if is_group(zone, name):
        for group in get_all_groups(zone):
            if group['name'] == name:
                return group
    return None

def del_group(zone, name):
    """ supprime un groupe et ses références dans la gestion des horaires
        @name: nom du groupe
    """
    if is_group(zone, name):
        # suprpession des horaires associés au groupe
        time_manager.del_schedule(zone, name)
        del_only_group(zone, name)
        return True
    raise Exception("Erreur: le groupe %s n'existe pas."%name)

def del_only_group(zone, name):
    """ supprime un groupe sans toucher à sa référence horaire
    """
    if is_group(zone, name):
        line = SYNTAX%get_group(zone, name)
        del_lines(group_file%zone, line)
        return True
    raise Exception("Erreur: le groupe %s n'existe pas.")

def mod_group(zone, name, **kw):
    """ modifie un groupe
        @name: nom du groupe
        @kw: attributs à modifier nomdel'attribut=valeur
    """
    group = get_group(zone, name)
    for key, value in kw.items():
        if value is not None:
            group[key] = value
    del_only_group(zone, name)
    add_lines(group_file%zone, SYNTAX%group)
    return True

def active(zone, name):
    """ active un groupe
        @name: nom du groupe
    """
    mod_group(zone, name, active='1')
    return True

def unactive(zone, name):
    """ désactive un groupe
        @name: nom du groupe
    """
    mod_group(zone, name, active='0')
    return True

def set_total(zone,name):
    """
        active l'interdiction totale de navigation
        @name: nom du groupe
    """
    mod_group(zone, name, active='2')
    return True

def active_time(zone, name):
    """ active la gestion horaire pour un groupe
        @name: nom du groupe
    """
    mod_group(zone, name, time='1')
    return True

def unactive_time(zone, name):
    """ désactive la gestion horaire pour un groupe
        @name: nom du groupe
    """
    mod_group(zone, name, time='0')
    return True

def test_group_name(name):
    """ teste la validité du nom d'un groupe de machine
    """
    if ' ' in name:
        raise Exception("Erreur : erreur le nom de groupe de machine ne doit pas comprendre d'espace")
    return True

def test_ip_range(start, end):
    """ teste si le couple start, end représente bien une plage d'ips
        @start: ip de début de plage
        @end: ip de fin de plage
    """
    try:
        if len(list(iter_iprange(start, end))) == 0:
            raise Exception('Erreur : les ips ne définissent pas une plage valide.')
    except AddrFormatError:
        raise Exception('Erreur : les valeurs ne sont pas des IPs')
    return True

# fonction de gestion de fichiers texte #

def get_lines(filename):
    """ renvoit le contenu d'un fichier sous forme de tableau """
    if isfile(filename):
        return open(filename).read().splitlines()
    else: return []

def write_file(filename, lines):
    fp = open(filename, 'w')
    fp.writelines('\n'.join(lines))
    fp.write('\n')
    fp.close()

def test_line(line):
    """ remplace les caractères ' et " en plaçant des anti-slashs """
    line = line.replace('"', r"'")
    line = line.replace("'", r"'")
    return line

def add_lines(filename, lines):
    """ ajoute une ligne dans un fichier (sans doublon) """
    if type(lines) != list:
        lines = [lines]
    oldies = get_lines(filename)
    for line in lines:
        if line and line not in oldies:
            line = test_line(line)
            oldies.append(line)
    try:
        write_file(filename, oldies)
    except:
        from traceback import print_exc
        print_exc()
        return False
    return True

def del_lines(filename, lines):
    """ supprime des lignes dans un fichier """
    if type(lines) != list:
        lines = [lines]
    oldies = get_lines(filename)
    for line in lines:
        if line in oldies:
            oldies.remove(line)
    try:
        write_file(filename, oldies)
    except:
        return False
    return True
