# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Génère les règles ipset pour les groupes de machine
#
###########################################################################

from amon.ipset import group_manager
from amon.ipset.parser import Parser
from amon.backend import get_filter_zones
from pyeole.process import system_out
#ipset identique dans le conteneur et en dehors
#from creole.config import VIRTDISABLED
from amon.config import VIRTDISABLED

# nom du set de services utilise pour toutes les restrictions (avec tous les groupes de machine)
SERVICE_SET_NAME = 'group-web_services'

# syntaxe des règles ipset liées aux groupes à appliquer
# creation de groupe/ajout d'element/binding du groupe d'ip avec un groupe de service
GROUP = "ipset create group-%(name)s bitmap:ip range %(ip_from)s-%(ip_to)s"
GROUP_ADD = "ipset add group-%(name)s %(ip_from)s-%(ip_to)s"

#L'option n'existe plus dans ipset
#BIND = "ipset -B %s :default: -b %s"
GROUP_DELETE = "ipset -X %s"

# syntaxe des règles ipset des groupes de service
SERVICE = "ipset -N %s portmap --from 0 --to 65000"
SERVICE_ADD = "ipset -A %s 3128"

class RuleGenerator:
    """ crée les groupes de machine au niveau ipset """

    def __init__(self):
        """ parser: outil de parsing de la commande ipset --save
            existing_ipsets: ipset déjà créé
        """
        self.parser = Parser()
        code, out, err = system_out(['ipset', '--save'])
        if code:
            raise Exception("Erreur : erreur lors de l'éxécution ipset --save")
        self.existing_ipsets = self.get_existing_ipsets(out)
        if not VIRTDISABLED:
            self.containers = ['proxy', 'domaine']
            self.existing_ipsets_containers = {}
            for container in self.containers:
                code, out, err = system_out(['ipset', '--save'], container=container)
                if code:
                    raise Exception("Erreur : erreur lors de l'éxécution ipset --save dans le conteneur")
                self.existing_ipsets_containers[container] = self.get_existing_ipsets(out)

    def exec_master_proxy(self, cmd, err_msg, test_var=None):
        """ exécute les commandes sur le maître puis dans les conteneurs
        """
        if test_var is None or test_var not in self.existing_ipsets:
            code, out, err = system_out(cmd.split())
            if code:
                raise Exception("Erreur : %s : %s, %s"%(err_msg, out, err))
        if not VIRTDISABLED:
            for container in self.containers:
                if test_var is None or test_var not in self.existing_ipsets_containers[container]:
                    code, out, err = system_out(cmd.split(), container=container)
                    if code:
                        raise Exception("Erreur %s (%s) dans le conteneur : %s, %s"%(err_msg,cmd, out, err))

    def create_ip_sets(self):
        """ crée et supprime les groupes d'ips configurés depuis l'interface ead2
            niveau ipset
            lien entre le fichier de configuration édité depuis le group_manager et ipset
        """
        created_group_name = [SERVICE_SET_NAME]
        ## creation du set de service si ce n'est pas fait
        self.exec_master_proxy(SERVICE%SERVICE_SET_NAME, test_var=SERVICE_SET_NAME,
                err_msg="erreur lors de la création du set de service {0}".format(SERVICE_SET_NAME))
        self.exec_master_proxy(SERVICE_ADD%SERVICE_SET_NAME, test_var=SERVICE_SET_NAME,
                err_msg="erreur lors de l'ajout du port 3128 au set de service")
        ## creation des groupes de machine
        for zone in get_filter_zones():
            for group in group_manager.get_all_groups(zone):
                group_name = 'group-' + group['name']
                created_group_name.append(group_name)
                self.exec_master_proxy(GROUP%group, test_var=group_name,
                        err_msg="erreur lors de la création du set d'ip {0}".format(group_name))
                self.exec_master_proxy(GROUP_ADD%group, test_var=group_name,
                        err_msg="erreur lors de l'ajout d'ip dans le groupe {0}".format(group_name))

        ## suppression des groupes
        for group in self.existing_ipsets:
            if group not in created_group_name:
                code, out, err = system_out((GROUP_DELETE%group).split())
                if code:
                    raise Exception("Erreur : erreur lors de la suppression du set d'ip {0}".format(group))
        if not VIRTDISABLED:
            for container in self.containers:
                for group in self.existing_ipsets_containers[container]:
                    if group not in created_group_name:
                        system_out((GROUP_DELETE%group).split(), container=container)
                        if code:
                            raise Exception("Erreur : erreur lors de la suppression du set d'ip {0} dans le conteneur".format(group))

    def get_existing_ipsets(self, datas):
        """ parse le résultat de ipset --save
            et renvoie un dico décrivant les instructions effectuées
            :create: les groupes créés
            :add: membres ajoutés au groupe
                    (ne sert à rien dans l'utilisation faite actuellement d'ipset)
            :bind: lien entre les sets de machine et les sets de services(ports)
                    (ne sert à rien dans l'utilisation faite actuellement d'ipset)
        """
        parsed_lines = set()
        lines = datas.splitlines()
        for line in lines:
            line = line.split()
            parsed_line = self.parser.parse_new_ipset_line(line)
            if parsed_line is not None:
                parsed_lines.add(parsed_line['name'])
        return list(parsed_lines)


def get_ips_range(start, end):
    """ renvoie la liste des ips comprises dans la plage start-end
        @start: ip de début de plage
        @end: ip de fin de plage
    """
    group_manager.test_ip_range(start, end)
    start = start.split('.')
    end = end.split('.')
    begin_chain = '.'.join(start[:-1])
    ip_range = []
    for num in range(int(start[-1:][0]), (int(end[-1:][0])+1)):
        ip_range.append("%s.%d"%(begin_chain, num))
    return ip_range
