# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2013
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""Parsing des fichiers de données AAF
 - parse_aaf_eleves
 - parse_aaf_profs
 - parse_aaf_responsables
 - parse_aaf_etab_educnat
"""
from lxml import etree  # fast and furious xml library

from eoleaaf.util import modif_fields, MinimalSQLGenerator, pwgen, log, db, \
                 parse_multi, get_new_id

from eoleaaf.miscutil import eleve_id, eleve_uid_and_password, classes_for_one_eleve, \
                                groupe_eleve


# mode (complet ou mise a jour) pour l'importation
# le mode d'utilisation peut être fixé de l'extérieur du module
maj_mode = False

# ____________________________________________________________
# buffers (pour rendre plus rapide les database queries)
# ATTENTION : ce sont des variables globales
classes_buffer = {}
groupes_buffer = {}
eleves_responsable_buffer = {}


# ____________________________________________________________
def is_empty(dico, cle):
    """
    pas de clé ou valeur vide
    """
    if dico.has_key(cle) and dico[cle] != '':
        return False
    else:
        return True


def not_empty(dico, cle):
    """
    verifie que la cle existe
    et que la valeur associee n'est pas vide
    """
    return not is_empty(dico, cle)


# ____________________________________________________________
DEBUG_NUM = 100

# ____________________________________________________________
# global (and unique) sql generator object
sqlgen = MinimalSQLGenerator()


#_______________________________________________________________________________
# sql inserts

def insert_classes(classes_buffer, db, sqlgen):
    """traitement des classes enregistrées dans le buffer

    :param classes_buffer: l'ensemble des classes à insérer dans la table classe
    :param db: le db connecteur
    :param sqlgen: le generateur de queries sql
    """
    for desc, member in classes_buffer.items():
        classe = {'description': desc, 'member':'\n'.join(member)}
        if maj_mode:
            sql = sqlgen.where_from('classe', '*', 'description', classe['description'])
            db_classe = db.fetchone(sql)
            # dans le cas d'une mise a jour, la classe peut etre presente
            if db_classe == []:
                modif_fields(classe, modif_type="CREATE", tablename="classe",
                            fromxml=True, db=db, id_column='description')
        else:
            modif_fields(classe, modif_type="CREATE", tablename="classe",
                        fromxml=True, db=db, id_column='description')

def insert_groupes(groupes_buffer, db, sqlgen):
    """traitement des groupes enregistrés dans le buffer

    :param groupes_buffer: l'ensemble des classes à insérer dans la table classe
    :param db: le db connecteur
    :param sqlgen: le generateur de queries sql
    """
    # traitement des groupes enregistrées dans le buffer
    for desc, member in groupes_buffer.items():
        groupe = {'description':desc, 'member':'\n'.join(member)}
        if maj_mode:
            sql = sqlgen.where_from('groupe', '*', 'description', groupe['description'])
            db_groupe = db.fetchone(sql)
            # dans le cas d'une mise a jour, le groupe peut etre deja present
            if db_groupe == []:
                modif_fields(groupe, modif_type="CREATE", tablename="groupe",
                            fromxml=True, db=db, id_column='description')
        else:
            modif_fields(groupe, modif_type="CREATE", tablename="groupe",
                        fromxml=True, db=db, id_column='description')

# ____________________________________________________________
# Extraction AAF Eleves et Responsables
def parse_aaf_eleves(eleve_file):
    """parsing des élèves depuis AAF"""
    num = 0
    log.info("Lecture des élèves...")
    context = etree.iterparse(eleve_file, events=('end',), tag='addRequest')
    for event, televe in context:
        _type = televe.find('operationalAttributes/attr/value').text
        if _type != 'Eleve':
            log.info("%s pas un élève :(" % _type)
            continue
        #**Attention** identifier/id == ENTPersonJointure
        eleve = eleve_id(televe)
        eleve, uid = eleve_uid_and_password(eleve)
        for attr in televe.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            eleve[balise] = vals
        # ajout de l'eleve a la classe et au groupe
        classes_for_one_eleve(eleve, classes_buffer, uid)
        groupe_eleve(eleve, groupes_buffer, uid)
        # ____________________________________________________________
        # TODO: on avait évoqué la possibilité de pouvoir limiter la création
        # aux responsables "1" & "2" (resp.split('$')[1] in ['1', '2'])
        # ce qui revient à être dans la liste
        # de "ENTEleveAutoriteParentale" (cf. Scribe)
        if 'ENTElevePersRelEleve' not in eleve:
            #log.error("ENTElevePersRelEleve not present in {0}".format(eleve['id']))
            continue
        for resp in eleve['ENTElevePersRelEleve'].split('\n'):
            numresp = resp.split('$')[0]
            if numresp in eleves_responsable_buffer:
                if uid not in eleves_responsable_buffer[numresp]:
                    eleves_responsable_buffer[numresp].append(uid)
            else:
                eleves_responsable_buffer[numresp] = [uid]
        if maj_mode:
            # il faut aller voir dans la base si les champs n'existent pas
            columns = ','.join(eleve.keys())
            sql = sqlgen.where_from('eleve', columns, 'id', eleve['id'])
            db_eleve = db.fetchone(sql)
            if db_eleve == []:
                modif_fields(eleve, modif_type="DELTACREATE", tablename="eleve",
                            fromxml=True, db=db)
            else:
                # retrieves uid and userPassword from database
                eleve['uid'] = db_eleve['uid']
                eleve['userPassword'] = db_eleve['userPassword']
                eleve['ENTEleveEnseignements'] = db_eleve['ENTEleveEnseignements']
                for normalkeys in db_eleve.keys():
                    if db_eleve[normalkeys] == None:
                        db_eleve[normalkeys] = ''
                if db_eleve != eleve:
                    modif_fields(eleve, modif_type="DELTAUPDATE", tablename="eleve",
                                fromxml=True, db=db)
        else:
            # ____________________________________________________________
            # traitement de eleve
            modif_fields(eleve, modif_type="CREATE", tablename="eleve",
                        fromxml=True, db=db)
        num += 1
        if num % DEBUG_NUM == 0:
            log.debug("%d élèves lus..." % num)
        televe.clear()
        while televe.getprevious() is not None:
            del televe.getparent()[0]
    del context

    db.commit()
    log.info("TOTAL : %d élèves" % num)


def flush_classes_buffer():
    """
    insertion des classes dans la database
    """
    global classes_buffer
    log.info("Traitement des classes...")
    insert_classes(classes_buffer, db, sqlgen)
    db.commit()
    log.info("TOTAL : %d classes" % len(classes_buffer))
    classes_buffer = {}


def flush_groupes_buffer():
    """
    insertion des groupes dans la database
    """
    global groupes_buffer
    log.info("Traitement des groupes...")
    insert_groupes(groupes_buffer, db, sqlgen)
    db.commit()
    log.info("TOTAL : %d groupes" % len(groupes_buffer))
    groupes_buffer = {}


def parse_aaf_responsables(responsables_file):
    """Parsing XML des responsables depuis AAF
    :responsables_file: file name like ``*_PersRelEleve_*.xml``
    """
    # init
    num = 0
    log.info("Lecture des responsables...")
    # ____________________________________________________________
    # xml parsing
    context = etree.iterparse(responsables_file, events=('end',), tag='addRequest')
    # identifier/id == ENTPersonJointure/value
    #'ENTPersonNomPatro':'nom_patronymique', #XXX : différent de sn ?
    # ____________________________________________________________
    # dict construction by XML parsing
    responsable = dict()
    for event, tresp in context:
        _type = tresp.find('operationalAttributes/attr/value').text
        if _type != 'PersRelEleve':
            log.info("%s n'est pas un responsable :(" % _type)
            continue
        respid = tresp.find('identifier/id').text
        responsable['id'] = unicode(respid)
        for attr in tresp.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            responsable[balise] = vals

        # new calculated uid and password
        responsable['uid'] = get_new_id()
        responsable['userPassword'] = pwgen()
        # ____________________________________________________________
        # remplissage des élèves associé au responsable
        # à partir des inforamtions bufferisées
        if responsable['id'] in eleves_responsable_buffer:
            responsable['ENTAuxPersRelEleveEleve'] = '\n'.join(eleves_responsable_buffer[responsable['id']])
        else:
            # FIXME: should not append and if it is the case
            # the account should not be created
            responsable['ENTAuxPersRelEleveEleve'] = None
        # ____________________________________________________________
        # database insert
        if maj_mode:
            # il faut aller voir dans la base si les champs n'existent pas
            columns = ','.join(responsable.keys())
            sql = sqlgen.where_from('responsable', columns, 'id', responsable['id'])
            db_responsable = db.fetchone(sql)
            if db_responsable == []:
                modif_fields(responsable, modif_type="DELTACREATE", tablename="responsable",
                            fromxml=True, db=db)
            else:
                # retrieves uid and userPassword from database
                responsable['uid'] = db_responsable['uid']
                responsable['userPassword'] = db_responsable['userPassword']
                if db_responsable != responsable:
                    modif_fields(responsable, modif_type="DELTAUPDATE", tablename="responsable",
                                fromxml=True, db=db)
        else:
            # ____________________________________________________________
            # traitement du responsable
            modif_fields(responsable, modif_type="CREATE", tablename="responsable",
                         fromxml=True, db=db)
        # ____________________________________________________________
        num += 1
        if num % DEBUG_NUM == 0:
            log.debug("%d responsables lus..." % num)

        tresp.clear()
        while tresp.getprevious() is not None:
            del tresp.getparent()[0]
    db.commit()
    log.info("TOTAL : %d responsables" % num)


def parse_aaf_profs(profs_file):
    """ parsing des professeurs depuis AAF
    Extraction AAF Professeurs + Administratifs
    Fichier de type : ``ENT_$RNE_Complet_$DATE_PersEducNat_0000.xml``
    """
    num = 0
    log.info("Lecture des personnels...")
    context = etree.iterparse(profs_file, events=('end',), tag='addRequest')
    # ____________________________________________________________
    # dict construction by XML parsing
    for event, tprof in context:
        _type = tprof.find('operationalAttributes/attr/value').text
        if _type != 'PersEducNat':
            log.info("%s n'est pas un personnel " % _type)
            continue

        #identifier/id == ENTPersonJointure
        profid = tprof.find('identifier/id').text
        professeur = {'id':unicode(profid)}
        classes = []
        matieres = []
        options = []
        principal = []
        groupe = None
        for attr in tprof.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            professeur[balise] = vals
        # new calculated uid and password
        professeur['uid'] = get_new_id()
        professeur['userPassword'] = pwgen()

        #Règle de répartition des personnes
        #cf. Annexe 4 (Alimentation depuis le SI MEN) version 4.2
        if professeur['PersEducNatPresenceDevantEleves'] == 'O' or \
           '$DOC$' in professeur['ENTPersonFonctions']:
            # c'est un enseignant
            if maj_mode:
                # il faut aller voir dans la base si les champs n'existent pas
                columns = ','.join(professeur.keys())
                sql = sqlgen.where_from('enseignant', columns, 'id', professeur['id'])
                db_professeur = db.fetchone(sql)
                if db_professeur== []:
                    modif_fields(professeur, modif_type="DELTACREATE", tablename="enseignant",
                                fromxml=True, db=db)
                else:
                    # retrieves uid and userPassword from database
                    professeur['uid'] = db_professeur['uid']
                    professeur['userPassword'] = db_professeur['userPassword']
                    if db_professeur!= professeur:
                        modif_fields(professeur, modif_type="DELTAUPDATE", tablename="enseignant",
                                    fromxml=True, db=db)
            else:
                # ____________________________________________________________
                # traitement de l'enseignant
                modif_fields(professeur, modif_type="CREATE", tablename="enseignant",
                            fromxml=True, db=db)
            num += 1
        else:
            # c'est un administratif. enlevons lui ses champs inutiles
            unused_fields = ['ENTAuxEnsClassesPrincipal', 'ENTAuxEnsCategoDiscipline',
                    'ENTAuxEnsDisciplinesPoste', 'ENTAuxEnsGroupes',
                    'ENTAuxEnsGroupesMatieres']
            for field in unused_fields:
                if field in professeur:
                    del(professeur[field])
            # traitement de l'administratif
            if maj_mode:
                # il faut aller voir dans la base si les champs n'existent pas
                columns = ','.join(professeur.keys())
                sql = sqlgen.where_from('administratif', columns, 'id', professeur['id'])
                db_professeur = db.fetchone(sql)
                if db_professeur== []:
                    modif_fields(professeur, modif_type="DELTACREATE", tablename="administratif",
                                fromxml=True, db=db)
                else:
                    # retrieves uid and userPassword from database
                    professeur['uid'] = db_professeur['uid']
                    professeur['userPassword'] = db_professeur['userPassword']
                    if db_professeur!= professeur:
                        modif_fields(professeur, modif_type="DELTAUPDATE", tablename="administratif",
                                    fromxml=True, db=db)
            else:
                modif_fields(professeur, modif_type="CREATE", tablename="administratif",
                        fromxml=True, db=db)
            num += 1
        tprof.clear()
        while tprof.getprevious() is not None:
            del tprof.getparent()[0]
        if num % DEBUG_NUM == 0:
            log.debug("%d personnels lus..." % num)
    db.commit()
    log.info("TOTAL : %d personnels" % num)


# ____________________________________________________________
def parse_aaf_etab_educnat(etab_file):
    """Parsing XML des etablisseuments depuis AAF
    :etab_file: file name like ``*_EtabEducNat_*.xml``
    """
    # init
    num = 0
    log.info("Lecture des etablissements...")
    # ____________________________________________________________
    # xml parsing
    context = etree.iterparse(etab_file, events=('end',), tag='addRequest')
    # ____________________________________________________________
    # dict construction by XML parsing
    etablissement = dict()
    for event, tresp in context:
        _type = tresp.find('operationalAttributes/attr/value').text
        if _type != 'EtabEducNat':
            log.info("%s n'est pas un etablissement :(" % _type)
            continue
        etabid = tresp.find('identifier/id').text
        etablissement['id'] = unicode(etabid)
        for attr in tresp.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            etablissement[balise] = vals
        if etablissement['ENTStructureSIREN'] == '':
            log.error("Pas de SIREN pour l'établissement {0}".format(etablissement['ENTStructureUAI']))
            continue
        if maj_mode:
            # il faut aller voir dans la base si les champs n'existent pas
            columns = ','.join(etablissement.keys())
            sql = sqlgen.where_from('etablissement', columns, 'id', etablissement['id'])
            db_etab = db.fetchone(sql)
            if db_etab == []:
                modif_fields(etablissement, modif_type="DELTACREATE",
                        tablename="etablissement", fromxml=True, db=db)
            else:
                for normalkeys in db_etab.keys():
                    if db_etab[normalkeys] == None:
                        db_etab[normalkeys] = ''
                if db_etab != etablissement:
                    modif_fields(etablissement, modif_type="DELTAUPDATE",
                            tablename="etablissement", fromxml=True, db=db)
        else:
            # ____________________________________________________________
            # database insert
            modif_fields(etablissement, modif_type="CREATE", tablename="etablissement",
                        fromxml=True, db=db)
        # ____________________________________________________________
        num += 1
        if num % DEBUG_NUM == 0:
            log.debug("%d etablissements lus..." % num)

    db.commit()
    log.info("TOTAL : %d etablissements" % num)


# ____________________________________________________________
def reset_db():
    log.info('Suppression de toutes les entrées de la base de données')
    db.reset()
    db.commit()
