# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2013
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
parsing des delas provenant des XML AAF
"""
from lxml import etree # fast and furious xml library

from eoleaaf.aaf import classes_for_one_eleve, classes_buffer, groupes_buffer, \
                eleves_responsable_buffer, sqlgen
from eoleaaf.util import pwgen, log, modif_fields, db, parse_multi, \
        get_new_id, gen_user_login
from eoleaaf.miscutil import user_id, groupe_eleve

# ______________________________________________________________________________
# eleves

def modify_request_eleve(eleve_file):
    """delta correspondant à une modification d'élève
    """
    log.info("Deltas -- modification d'élèves...")
    context = etree.iterparse(eleve_file, events=('end',), tag='modifyRequest')
    for event, televe in context:
        _type = televe.find('operationalAttributes/attr/value').text
        if _type != 'Eleve':
            log.info("%s pas un élève :(" % _type)
            continue
        #**Attention** identifier/id == ENTPersonJointure
        eleve = user_id(televe)
        for attr in televe.findall('modifications/modification'):
            balise, vals = parse_multi(attr)
            eleve[balise] = vals
        # il faut aller voir dans la base si les champs n'existent pas
        sql = sqlgen.where_from('eleve', '*', 'id', eleve['id'])
        db_eleve = db.fetchone(sql)

        retrieved = True
        if db_eleve == []:
            log.warn("ATTENTION: l'eleve {0} sensé être mis a jour "
            "n'existe pas dans la base".format(eleve['id']))
            retrieved = False

        if 'userPassword' not in db_eleve:
            if retrieved:
                log.error("ATTENTION: l'eleve {0} aurait dû avoir un mot de "
                    "passe".format(eleve['id']))
            eleve['userPassword'] = pwgen()

        if 'uid' not in db_eleve:
            if retrieved:
                log.error("ATTENTION: l'eleve {0} aurait dû avoir un "
                    "uid".format(eleve['id']))
            uid = eleve['uid'] = get_new_id({}, sqlgen)
        else:
            # utilisé plus bas...
            uid = db_eleve['uid']

        if 'ENTPersonLogin' not in db_eleve:
            if retrieved:
                log.error("ATTENTION: l'eleve {0} aurait dû avoir un "
                    "login".format(eleve['id']))
            eleve['ENTPersonLogin'] = gen_user_login(eleve['sn'], eleve['givenName'], uid, 'eleve', eleve['id'], sqlgen, {})

        # ____________________________________________________________
        # traitement de la classe (le champ classe est obligatoire)
        # classe (format AAF : "numero$classe")
        classes_for_one_eleve(eleve, classes_buffer, uid)
        # ____________________________________________________________
        # traitement du groupe
        groupe_eleve(eleve, groupes_buffer, uid)

        # ____________________________________________________________
        if not retrieved:
            # conformement aux recommandations du SDET 4.2,
            # il faut inserer l'eleve quand mme (mme s'il n'existe pas
            # dans la database)
            try:
                modif_fields(eleve, sqlgen, modif_type="DELTACREATE", tablename="eleve",
                            fromxml=True, db=db)
            except Exception, msg:
                log.error("Erreur à l'ajout de l'élève %s : %s" % (eleve['id'], msg))
        else:
            # maj de l'élève retrouvé
            try:
                modif_fields(eleve, sqlgen, modif_type="DELTAUPDATE", tablename="eleve", fromxml=True, db=db)
            except Exception, msg:
                log.error("Erreur à la modification de l'élève %s : %s" % (eleve['id'], msg))

        televe.clear()
        while televe.getprevious() is not None:
            del televe.getparent()[0]

def add_request_eleve(eleve_file):
    """delta correspondant a un ajout d'eleve

    :param eleve_file: fichier XML delta eleve
    """
    log.info("Deltas -- ajout d'élèves...")
    context = etree.iterparse(eleve_file, events=('end',), tag='addRequest')
    for event, televe in context:
        _type = televe.find('operationalAttributes/attr/value').text
        if _type != 'Eleve':
            log.info("%s pas un élève :(" % _type)
            continue
        #**Attention** identifier/id == ENTPersonJointure
        eleve = user_id(televe)
        uid = eleve['uid'] = get_new_id({}, sqlgen)
        eleve['userPassword'] = pwgen()
        attrs = televe.findall('attributes/attr')
        if attrs == []:
            log.error("aucune information de type modification pour l'élève "
                      "avec l'id : {0}".format(eleve['id']))
            continue
        for attr in attrs:
            # FIXME: on fait quoi si une opération est différente des autres ???
            balise, vals = parse_multi(attr)
            eleve[balise] = vals

        eleve['ENTPersonLogin'] = gen_user_login(eleve['sn'], eleve['givenName'], uid, 'eleve', eleve['id'], sqlgen, {})

        for resp in eleve['ENTElevePersRelEleve'].split('\n'):
            numresp = resp.split('$')[0]
            if numresp in eleves_responsable_buffer:
                if uid not in eleves_responsable_buffer[numresp]:
                    eleves_responsable_buffer[numresp].append(uid)
            else:
                eleves_responsable_buffer[numresp] = [uid]
        # ____________________________________________________________
        # traitement de la classe (le champ classe est obligatoire)
        # classe (format AAF : "numero$classe")
        classes_for_one_eleve(eleve, classes_buffer, uid)
        # ____________________________________________________________
        # traitement du groupe
        groupe_eleve(eleve, groupes_buffer, uid)
        # ____________________________________________________________
        # traitement de eleve
        try:
            modif_fields(eleve, sqlgen, modif_type="DELTACREATE", tablename="eleve",
                        fromxml=True, db=db)
        except Exception, msg:
            log.error("Erreur à l'ajout de l'élève %s : %s" % (eleve['id'], msg))
        televe.clear()
        while televe.getprevious() is not None:
            del televe.getparent()[0]


def delete_request_eleve(eleve_file):
    """delta correspondant a la suppression d'un eleve

    :param eleve_file: fichier XML delta eleve
    """
    log.info("Deltas -- suppression d'élèves...")
    context = etree.iterparse(eleve_file, events=('end',), tag='deleteRequest')
    for event, televe in context:
        _type = televe.find('operationalAttributes/attr/value').text
        if _type != 'Eleve':
            log.info("%s pas un élève :(" % _type)
            continue
        #**Attention** identifier/id == ENTPersonJointure
        eleve = user_id(televe)
        # il faut aller voir dans la base si les champs n'existent pas
        sql = sqlgen.where_from('eleve', 'id', 'id', eleve['id'])
        db_eleve = db.fetchone(sql)
        if db_eleve == []:
            log.error("ATTENTION: l'eleve {0} sensé être supprimé "
            "n'existe pas dans la base".format(eleve['id']))
            continue
        # ____________________________________________________________
        # suppression effective de eleve
        try:
            modif_fields(eleve, sqlgen, modif_type="DELTADELETE", tablename="eleve", fromxml=True,
                         db=db)
        except Exception, msg:
            log.error("Erreur au moment de la suppression"
                    " de l'élève %s : %s" % (str(eleve['id']), str(msg) ))
        televe.clear()
        while televe.getprevious() is not None:
            del televe.getparent()[0]


def parse_delta_eleves(eleve_file):
    """parsing global des deltas pour les eleves

    :param eleve_file: fichier XML des eleves
    """
    log.info("Lecture des delta des élèves...")
    # <addRequest> <modifyRequest> <deleteRequest>
    log.info('ajout eleve')
    add_request_eleve(eleve_file)
    log.info('modification eleve')
    modify_request_eleve(eleve_file)
    log.info('suppression eleve')
    delete_request_eleve(eleve_file)
    db.commit(None)

# ______________________________________________________________________________
# responsables

def add_request_responsable(responsables_file):
    """
    Search 'addRequest' tags
    """
    log.info("Deltas -- modification de responsables...")
    context = etree.iterparse(responsables_file, events=('end',), tag='addRequest')
    # identifier/id == ENTPersonJointure/value
    #'ENTPersonNomPatro':'nom_patronymique', #XXX : différent de sn ?
    # ____________________________________________________________
    # dict construction by XML parsing
    responsable = dict()
    for event, tresp in context:
        _type = tresp.find('operationalAttributes/attr/value').text
        if _type != 'PersRelEleve':
            log.info("%s n'est pas un responsable :(" % _type)
            continue
        respid = tresp.find('identifier/id').text
        responsable['id'] = unicode(respid)
        for attr in tresp.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            responsable[balise] = vals

        # new calculated uid and password
        responsable['uid'] = get_new_id({}, sqlgen)
        responsable['userPassword'] = pwgen()
        responsable['ENTPersonLogin'] = gen_user_login(responsable['sn'], responsable['givenName'], responsable['uid'], 'responsable', responsable['id'], sqlgen, {})
        # ____________________________________________________________
        # remplissage des élèves associé au responsable
        # à partir des inforamtions bufferisées
        if responsable['id'] in eleves_responsable_buffer:
            responsable['ENTAuxPersRelEleveEleve'] = '\n'.join(eleves_responsable_buffer[responsable['id']])
        else:
            # FIXME: should not append and if it is the case
            # the account should not be created
            responsable['ENTAuxPersRelEleveEleve'] = None
        # ____________________________________________________________
        # database insert
        try:
            modif_fields(responsable, sqlgen, modif_type="DELTACREATE", tablename="responsable",
                         fromxml=True, db=db)
        except Exception, msg:
            log.error("Erreur au moment de l'ajout"
                      " du responsable %s : %s" % (responsable['id'], msg))
        # ____________________________________________________________

        tresp.clear()
        while tresp.getprevious() is not None:
            del tresp.getparent()[0]


def modify_request_responsable(responsables_file):
    """delta correspondant à une modification de responsable

    """
    log.info("Deltas -- modification de responsables...")
    context = etree.iterparse(responsables_file, events=('end',), tag='modifyRequest')
    # identifier/id == ENTPersonJointure/value
    #'ENTPersonNomPatro':'nom_patronymique', #XXX : différent de sn ?
    # ____________________________________________________________
    # dict construction by XML parsing
    responsable = dict()
    for event, tresp in context:
        _type = tresp.find('operationalAttributes/attr/value').text
        if _type != 'PersRelEleve':
            log.info("%s n'est pas un responsable :(" % _type)
            continue
        respid = tresp.find('identifier/id').text
        responsable['id'] = unicode(respid)
        for attr in tresp.findall('modifications/modification'):
            balise, vals = parse_multi(attr)
            responsable[balise] = vals
        # il faut aller voir si le responsable existe bien dans la base
        sql = sqlgen.where_from('responsable', '*', 'id', responsable['id'])
        db_resp = db.fetchone(sql)

        retrieved = True
        if db_resp == []:
            log.warn("ATTENTION: le responsable {0} sensé être mis a jour "
            "n'existe pas dans la base".format(responsable['id']))
            retrieved = False

        if 'userPassword' not in db_resp:
            responsable['userPassword'] = pwgen()

        if 'uid' not in db_resp:
            responsable['uid'] = get_new_id({}, sqlgen)

        if 'ENTPersonLogin' not in db_resp:
            if retrieved:
                log.error("ATTENTION: le responsable {0} aurait dû avoir un "
                    "login".format(responsable['id']))
            responsable['ENTPersonLogin'] = gen_user_login(responsable['sn'], responsable['givenName'], responsable['uid'], 'responsable', responsable['id'], sqlgen, {})

        if not retrieved:
            # création du responsable non retrouvé
            try:
                modif_fields(responsable, sqlgen, modif_type="DELTACREATE", tablename="responsable",
                             fromxml=True, db=db)
            except Exception, msg:
                log.error("Erreur au moment de la modification"
                          " du responsable %s : %s" % (responsable['id'], msg))
        else:
            # maj du responsable retrouvé
            try:
                modif_fields(responsable, sqlgen, modif_type="DELTAUPDATE", tablename="responsable",
                             fromxml=True, db=db)
            except Exception, msg:
                log.error("Erreur au moment de la modification"
                          " du responsable %s : %s" % (responsable['id'], msg))
        # ____________________________________________________________
        tresp.clear()
        while tresp.getprevious() is not None:
            del tresp.getparent()[0]


def delete_request_responsable(responsables_file):
    """delta correspondant a la suppression d'un responsable

    :param responsable_file: fichier XML delta responsable
    """
    log.info("Deltas -- suppression de responsables...")
    context = etree.iterparse(responsables_file, events=('end',), tag='deleteRequest')
    for event, tresponsable in context:
        _type = tresponsable.find('operationalAttributes/attr/value').text
        if _type != 'PersRelEleve':
            log.info("%s pas un responsable :(" % _type)
            continue
        #**Attention** identifier/id == ENTPersonJointure
        responsable = user_id(tresponsable)
        # il faut aller voir dans la base si les champs n'existent pas
        sql = sqlgen.where_from('responsable', 'id', 'id', responsable['id'])
        db_responsable = db.fetchone(sql)
        if db_responsable == []:
            log.error("ATTENTION: l'responsable {0} sensé être supprimé "
            "n'existe pas dans la base".format(responsable['id']))
            continue
        # ____________________________________________________________
        # suppression effective de responsable
        try:
            modif_fields(responsable, sqlgen, modif_type="DELTADELETE",
                        tablename="responsable", fromxml=True, db=db)
        except Exception, msg:
            log.error("Erreur au moment de la suppression"
                      " du responsable %s : %s" % (responsable['id'], msg))
        tresponsable.clear()
        while tresponsable.getprevious() is not None:
            del tresponsable.getparent()[0]

def parse_delta_responsables(responsables_file):
    """Parsing global XML des responsables depuis AAF

    :param responsables_file: XML file name like ``*_PersRelEleve_*.xml``
    """
    log.info("Lecture des responsables...")
    # <addRequest> <modifyRequest> <deleteRequest>
    log.info('ajout responsable')
    add_request_responsable(responsables_file)
    log.info('modification responsable')
    modify_request_responsable(responsables_file)
    log.info('suppression responsable')
    delete_request_responsable(responsables_file)
    db.commit(None)


def add_request_profs(profs_file):
    """delta correspondant à l'ajout de prof

    :param profs_file: fichier XML qui concerner uniquement les profs
    """
    log.info("Deltas -- ajout de profs...")
    context = etree.iterparse(profs_file, events=('end',), tag='addRequest')
    # ____________________________________________________________
    # dict construction by XML parsing
    for event, tprof in context:
        _type = tprof.find('operationalAttributes/attr/value').text
        if _type != 'PersEducNat':
            log.info("%s n'est pas un personnel " % _type)
            continue

        #identifier/id == ENTPersonJointure
        profid = tprof.find('identifier/id').text
        professeur = {'id':unicode(profid)}
        classes = []
        matieres = []
        options = []
        principal = []
        #groupe = None
        for attr in tprof.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            professeur[balise] = vals
        # new calculated uid and password
        professeur['uid'] = get_new_id({}, sqlgen)
        professeur['userPassword'] = pwgen()

        #Règle de répartition des personnes
        #cf. Annexe 4 (Alimentation depuis le SI MEN) version 4.2
        if professeur['PersEducNatPresenceDevantEleves'] == 'O' or \
           '$DOC$' in professeur['ENTPersonFonctions']:
            usertype = 'enseignant'
        else:
            usertype = 'administratif'
        professeur['ENTPersonLogin'] = gen_user_login(professeur['sn'], professeur['givenName'], professeur['uid'], usertype, professeur['id'], sqlgen, {})

        if usertype == 'enseignant':
            # c'est un enseignant
            try:
                modif_fields(professeur, sqlgen, modif_type="DELTACREATE", tablename="enseignant",
                            fromxml=True, db=db)
            except Exception, msg:
                log.info("Erreur sur l'enseignant <%s>, "
                "(explicit database msg : %s)" % (profid, msg))
        else:
            # c'est un administratif
            del(professeur['ENTAuxEnsCategoDiscipline'])
            if 'ENTAuxEnsDisciplinesPoste' in professeur:
                del(professeur['ENTAuxEnsDisciplinesPoste'])
            if 'ENTAuxEnsGroupes' in professeur:
                del(professeur['ENTAuxEnsGroupes'])
            try:
                modif_fields(professeur, sqlgen, modif_type="DELTACREATE",
                             tablename="administratif", fromxml=True, db=db)
            except Exception, msg:
                log.info("Erreur sur le personnel <%s>, "
                "(explicit database msg : %s)" % (profid, msg))


def modify_request_prof(profs_file):
    """delta correspondant à une modification de prof

    :param profs_file: fichier XML qui concerner uniquement les profs
    """
    log.info("Deltas -- modification de profs...")
    context = etree.iterparse(profs_file, events=('end',), tag='modifyRequest')
    # ____________________________________________________________
    # dict construction by XML parsing
    for event, tprof in context:
        _type = tprof.find('operationalAttributes/attr/value').text
        if _type != 'PersEducNat':
            log.info("%s n'est pas un personnel " % _type)
            continue

        #identifier/id == ENTPersonJointure
        administratif = True
        profid = tprof.find('identifier/id').text
        professeur = {'id':unicode(profid)}
        classes = []
        matieres = []
        options = []
        principal = []
        groupe = None
        for attr in tprof.findall('modifications/modification'):
            balise, vals = parse_multi(attr)
            professeur[balise] = vals

        #Règle de répartition des personnes
        #cf. Annexe 4 (Alimentation depuis le SI MEN) version 4.2
        if professeur['PersEducNatPresenceDevantEleves'] == 'O' or \
           '$DOC$' in professeur['ENTPersonFonctions']:
            # c'est un enseignant
            usertype = 'enseignant'
        else:
            usertype = 'administratif'

        # il faut aller voir si le personnel existe bien dans la base
        sql = sqlgen.where_from(usertype, '*', 'id', professeur['id'])
        db_ens = db.fetchone(sql)

        retrieved = True
        if db_ens == []:
            log.warn("ATTENTION: le personnel {1} {0} sensé être mis a jour "
            "n'existe pas dans la base".format(professeur['id'], usertype))
            retrieved = False

        if 'userPassword' not in db_ens:
            professeur['userPassword'] = pwgen()

        if 'uid' not in db_ens:
            professeur['uid'] = get_new_id({}, sqlgen)

        if 'ENTPersonLogin' not in db_ens:
            if retrieved:
                log.error("ATTENTION: le personnel {0} aurait dû avoir un login".format(usertype))
            professeur['ENTPersonLogin'] = gen_user_login(professeur['sn'], professeur['givenName'], professeur['uid'], usertype, professeur['id'], sqlgen, {})

        if usertype == 'enseignant':
            if not retrieved:
                # création de l'enseignant non retrouvé
                try:
                    modif_fields(professeur, sqlgen, modif_type="DELTACREATE",
                            tablename="enseignant", fromxml=True, db=db)
                except Exception, msg:
                    log.info("Erreur sur l'enseignant <%s>, "
                    "(explicit database msg : %s)" % ( profid, str(msg) ))
            else:
                # maj de l'enseignant retrouvé
                try:
                    modif_fields(professeur, sqlgen, modif_type="DELTAUPDATE",
                            tablename="enseignant", fromxml=True, db=db)
                except Exception, msg:
                    log.info("Erreur sur l'enseignant <%s>, "
                    "(explicit database msg : %s)" % ( profid, str(msg) ))
        else:
            # traitement des attributs propres aux personnels administratifs
            del(professeur['ENTAuxEnsCategoDiscipline'])
            if 'ENTAuxEnsDisciplinesPoste' in professeur:
                del(professeur['ENTAuxEnsDisciplinesPoste'])
            if 'ENTAuxEnsGroupes' in professeur:
                del(professeur['ENTAuxEnsGroupes'])
            if not retrieved:
                # création de l'administratif non retrouvé
                try:
                    modif_fields(professeur, sqlgen, modif_type="DELTACREATE",
                                 tablename="administratif", fromxml=True, db=db)
                except Exception, msg:
                    log.info("Erreur sur le personnel <%s>, "
                    "(explicit database msg : %s)" % (profid, msg))
            else:
                # maj de l'administratif retrouvé
                try:
                    modif_fields(professeur, sqlgen, modif_type="DELTAUPDATE",
                                 tablename="administratif", fromxml=True, db=db)
                except Exception, msg:
                    log.info("Erreur sur le personnel <%s>, "
                    "(explicit database msg : %s)" % (profid, msg))
        tprof.clear()
        while tprof.getprevious() is not None:
            del tprof.getparent()[0]


def delete_request_prof(profs_file):
    """delta correspondant à la suppression de prof

    :param profs_file: fichier XML qui concerne les profs ou les personnels adminstratifs
    """
    log.info("Deltas -- suppression de personnels...")
    context = etree.iterparse(profs_file, events=('end',), tag='deleteRequest')
    for event, tprof in context:
        _type = tprof.find('operationalAttributes/attr/value').text
        if _type != 'PersEducNat':
            log.info("%s n'est pas un personnel " % _type)
            continue
        #**Attention** identifier/id == ENTPersonJointure
        prof = user_id(tprof)
        # il faut aller voir dans les tables enseignant ou administratif
        # si l'entrée existe
        sql = sqlgen.where_from('enseignant', 'id', 'id', prof['id'])
        table = "enseignant"
        db_prof = db.fetchone(sql)
        if db_prof == []:
            sql = sqlgen.where_from('administratif', 'id', 'id', prof['id'])
            table = "administratif"
            db_admin = db.fetchone(sql)
            if db_admin == []:
                log.error("ATTENTION: le personnel {0} sensé être supprimé "
                "n'existe pas dans la base".format(prof['id']))
                continue
        # ____________________________________________________________
        # suppression effective du personnel
        try:
            modif_fields(prof, sqlgen, modif_type="DELTADELETE", tablename=table,
                         fromxml=True, db=db)
        except Exception:
            log.error("Erreur au moment de la suppression"
                      " du personnel %s " % prof['id'])
        tprof.clear()
        while tprof.getprevious() is not None:
            del tprof.getparent()[0]


def parse_delta_profs(profs_file):
    """ parsing des professeurs depuis AAF
    Extraction AAF Professeurs + Administratifs
    Fichier de type : ``ENT_$RNE_Complet_$DATE_PersEducNat_0000.xml``
    """
    log.info("Lecture des deltas pour les personnels...")
    # <addRequest> <modifyRequest> <deleteRequest>
    log.info('ajout professeur')
    add_request_profs(profs_file)
    log.info('modification professeur')
    modify_request_prof(profs_file)
    log.info('suppression professeur')
    delete_request_prof(profs_file)
    db.commit(None)

# ____________________________________________________________
# etablissements
def add_request_etab_educnat(etab_file):
    """
    Search 'addRequest' tags
    """
    context = etree.iterparse(etab_file, events=('end',), tag='addRequest')
    etablissement = dict()
    for event, tresp in context:
        _type = tresp.find('operationalAttributes/attr/value').text
        if _type != 'EtabEducNat':
            log.info("%s n'est pas un etablissement :(" % _type)
            continue
        etabid = tresp.find('identifier/id').text
        etablissement['id'] = unicode(etabid)
        for attr in tresp.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            etablissement[balise] = vals
        if etablissement['ENTStructureSIREN'] == '':
            log.error("Pas de SIREN pour l'établissement {0}".format(etablissement['ENTStructureUAI']))
            continue
        # database insert
        try:
            modif_fields(etablissement, sqlgen, modif_type="DELTACREATE", tablename="etablissement",
                        fromxml=True, db=db)
        except Exception:
            log.error("Erreur au moment de l'ajout"
                      " de l'établissement %s " % etablissement['id'])
        tresp.clear()
        while tresp.getprevious() is not None:
            del tresp.getparent()[0]

def modify_request_etab_educnat(etab_file):
    """
    Search 'modifyRequest' tags
    """
    context = etree.iterparse(etab_file, events=('end',), tag='modifyRequest')
    etablissement = dict()
    for event, tresp in context:
        _type = tresp.find('operationalAttributes/attr/value').text
        if _type != 'EtabEducNat':
            log.info("%s n'est pas un etablissement :(" % _type)
            continue
        etabid = tresp.find('identifier/id').text
        etablissement['id'] = unicode(etabid)
        for attr in tresp.findall('modifications/modification'):
            balise, vals = parse_multi(attr)
            etablissement[balise] = vals
        # il faut aller voir dans la base si l'etablissement n'existe pas
        sql = sqlgen.where_from('etablissement', '*', 'id', etablissement['id'])
        db_etab = db.fetchone(sql)
        if db_etab == []:
            log.error("ATTENTION: l'etablissement {0} sensé être mis a jour "
            "n'existe pas dans la base".format(etablissement['id']))
            db_etab = {}
            # conformement aux recommandations du SDET 4.2,
            # il faut inserer l'etablissement quand mme (mme s'il n'existe pas
            # dans la database)
            try:
                modif_fields(etablissement, sqlgen, modif_type="DELTACREATE",
                             tablename="etablissement", fromxml=True, db=db)
            except Exception:
                log.error("Erreur au moment de la modification"
                          " de l'établissement %s " % etablissement['id'])
            continue
        # database insert
        try:
            modif_fields(etablissement, sqlgen, modif_type="DELTAUPDATE",
                         tablename="etablissement", fromxml=True, db=db)
        except Exception:
            log.error("Erreur au moment de la modification"
                      " de l'établissement %s " % etablissement['id'])
        # ____________________________________________________________
        tresp.clear()
        while tresp.getprevious() is not None:
            del tresp.getparent()[0]


def parse_delta_etab_educnat(etab_file):
    """Parsing XML des responsables depuis AAF
    :responsables_file: file name like ``*_PersRelEleve_*.xml``
    """
    log.info("Lecture des etablissements...")
    log.info('ajout etablissement')
    add_request_etab_educnat(etab_file)
    log.info('modification etablissement')
    modify_request_etab_educnat(etab_file)
    db.commit(None)
