# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2013
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""Parsing des fichiers de données AAF
 - parse_aaf_eleves
 - parse_aaf_profs
 - parse_aaf_responsables
 - parse_aaf_etab_educnat
"""
from lxml import etree  # fast and furious xml library

from eoleaaf.config import aaf_type, ConfigError

from config import dbtype

if dbtype in ["mysql", "sqlite"]:
    from eoleaaf.util import MinimalSQLGenerator as DataBaseGenerator
elif dbtype == "mongodb":
    from eoleaaf.util import MongoDBGenerator as DataBaseGenerator


from eoleaaf.util import modif_fields, pwgen, log, db, \
                 parse_multi, get_new_id, gen_user_login
from eoleaaf.miscutil import user_id, classes_for_one_eleve, \
                                groupe_eleve


# mode (complet ou mise a jour) pour l'importation
# le mode d'utilisation peut être fixé de l'extérieur du module
maj_mode = False

# ____________________________________________________________
# buffers (pour rendre plus rapide les database queries)
# ATTENTION : ce sont des variables globales
classes_buffer = {}
groupes_buffer = {}
eleves_responsable_buffer = {}


# ____________________________________________________________
def is_empty(dico, cle):
    """
    pas de clé ou valeur vide
    """
    if dico.has_key(cle) and dico[cle] != '':
        return False
    else:
        return True


def not_empty(dico, cle):
    """
    verifie que la cle existe
    et que la valeur associee n'est pas vide
    """
    return not is_empty(dico, cle)


# ____________________________________________________________
DEBUG_NUM = 100

# ____________________________________________________________
# global (and unique) sql generator object
sqlgen = DataBaseGenerator()


#_______________________________________________________________________________
# sql inserts

def insert_classes(classes_buffer, db, sqlgen):
    """traitement des classes enregistrées dans le buffer

    :param classes_buffer: l'ensemble des classes à insérer dans la table classe
    :param db: le db connecteur
    :param sqlgen: le generateur de queries sql
    """
    for desc, member in classes_buffer.items():
        classe = {'description': desc, 'member':'\n'.join(member)}
        if maj_mode:
            sql = sqlgen.where_from('classe', '*', 'description', classe['description'])
            db_classe = db.fetchone(sql)
            # dans le cas d'une mise a jour, la classe peut etre presente
            if db_classe == []:
                modif_fields(classe, sqlgen, modif_type="CREATE", tablename="classe",
                            fromxml=True, db=db, id_column='description')
        else:
            modif_fields(classe, sqlgen, modif_type="CREATE", tablename="classe",
                        fromxml=True, db=db, id_column='description')

def insert_groupes(groupes_buffer, db, sqlgen):
    """traitement des groupes enregistrés dans le buffer

    :param groupes_buffer: l'ensemble des classes à insérer dans la table classe
    :param db: le db connecteur
    :param sqlgen: le generateur de queries sql
    """
    # traitement des groupes enregistrées dans le buffer
    for desc, member in groupes_buffer.items():
        groupe = {'description':desc, 'member':'\n'.join(member)}
        if maj_mode:
            sql = sqlgen.where_from('groupe', '*', 'description', groupe['description'])
            db_groupe = db.fetchone(sql)
            # dans le cas d'une mise a jour, le groupe peut etre deja present
            if db_groupe == []:
                modif_fields(groupe, sqlgen, modif_type="CREATE", tablename="groupe",
                            fromxml=True, db=db, id_column='description')
        else:
            modif_fields(groupe, sqlgen, modif_type="CREATE", tablename="groupe",
                        fromxml=True, db=db, id_column='description')


def _parse_etab_values(key, userdata, etab_buffer):
    def _parse_etab_value(value, etab_buffer):
        splitted = value.split('$')
        etab = splitted[0]
        name = splitted[-1]
        return {'etablissement': _mongo_fetch_value(etab, "etablissement", "ENTStructureJointure", etab_buffer),
                'name': name}
    if key in userdata:
        values = userdata[key]
        if isinstance(values, list):
            new_values = []
            for value in values:
                new_values.append(_parse_etab_value(value, etab_buffer))
        else:
            new_values = _parse_etab_value(values, etab_buffer)
        userdata[key] = new_values


def _parse_functions(values, etab_buffer):
    def _parse_function(value, etab_buffer):
        splitted = value.split('$')
        etab = splitted[0]
        name = splitted[2]
        discipline = splitted[4]
        return {'etablissement': _mongo_fetch_value(etab, "etablissement", "ENTStructureJointure", etab_buffer),
                'function': name,
                'discipline': discipline}
    if values == '':
        return values
    if isinstance(values, list):
        new_values = []
        for value in values:
            new_values.append(_parse_function(value, etab_buffer))
    else:
        new_values = _parse_function(values, etab_buffer)
    return new_values


def _parse_ens_class_subjects(values, etab_buffer, subject_buffer, name='class'):
    def _parse_ens_class_subject(value, etab_buffer, subject_buffer):
        splitted = value.split('$')
        etab = splitted[0]
        classe = splitted[1]
        subject = splitted[2]
        return {'etablissement': _mongo_fetch_value(etab, "etablissement", "ENTStructureJointure", etab_buffer),
                name: classe,
                'subject': _mongo_fetch_value(subject, "subject", "ENTMatJointure", subject_buffer)}
    if values == '':
        return values
    if isinstance(values, list):
        new_values = []
        for value in values:
            new_values.append(_parse_ens_class_subject(value, etab_buffer, subject_buffer))
    else:
        new_values = _parse_ens_class_subject(values, etab_buffer, subject_buffer)
    return new_values


def _parse_attrib_split(value, idx):
    if value == '':
        new_value = value
    elif isinstance(value, list):
        new_value = []
        for val in value:
            new_value.append(val.split('$')[idx])
    else:
        new_value = value.split('$')[idx]
    return new_value


def _parse_aaf_profadmin(usertype, userdata):

    if usertype == 'administratif':
        unused_fields = ['ENTAuxEnsClassesPrincipal', 'ENTAuxEnsCategoDiscipline',
            'ENTAuxEnsDisciplinesPoste', 'ENTAuxEnsGroupes',
           'ENTAuxEnsGroupesMatieres']
    else:
        unused_fields = []
    unused_fields.append('ENTPersonCategorieEnseignant')
    for field in unused_fields:
        if field in userdata:
            del(userdata[field])

    if dbtype == "mongodb":
        etab_buffer = {}
        subject_buffer = {}
        _replace_id_by_reference(userdata, "ENTPersonStructRattach", "etablissement", "ENTStructureJointure", etab_buffer)
        if usertype == 'enseignant':
            for key in ['ENTAuxEnsMatiereEnseignEtab', 'ENTAuxEnsClasses', 'ENTAuxEnsMEF', 'ENTAuxEnsClassesPrincipal', 'ENTAuxEnsGroupes']:
                _parse_etab_values(key, userdata, etab_buffer)
            if 'ENTAuxEnsDisciplinesPoste' in userdata:
                userdata['ENTAuxEnsDisciplinesPoste'] = _parse_attrib_split(userdata['ENTAuxEnsDisciplinesPoste'].split('$'), 0)
            if 'ENTAuxEnsCategoDiscipline' in userdata:
                userdata['ENTAuxEnsCategoDiscipline'] = _parse_attrib_split(userdata['ENTAuxEnsCategoDiscipline'], 1)
            if u"ENTAuxEnsClassesMatieres" in userdata:
                userdata[u"ENTAuxEnsClassesMatieres"] = _parse_ens_class_subjects(userdata[u"ENTAuxEnsClassesMatieres"], etab_buffer, subject_buffer)
            if u"ENTAuxEnsGroupesMatieres" in userdata:
                userdata[u"ENTAuxEnsGroupesMatieres"] = _parse_ens_class_subjects(userdata[u"ENTAuxEnsGroupesMatieres"], etab_buffer, subject_buffer, 'group')
        if 'ENTPersonFonctions' in userdata:
            userdata['ENTPersonFonctions'] = _parse_functions(userdata['ENTPersonFonctions'], etab_buffer)



def _mongo_fetch_value(value, tablename, filterby, tbuffer):
    if tbuffer is not None and value in tbuffer:
        new_value = tbuffer[value]
    else:
        fetch_result = db.fetchone(tablename, filterby, value)
        if fetch_result is not None:
            new_value = fetch_result[u'_id']
        else:
            new_value = value
        if tbuffer is not None:
            tbuffer[value] = new_value
    return new_value


def _mongo_fetch_values(values, tablename, filterby, tbuffer):
    if isinstance(values, list) and len(values) == 1:
        values = values[0]
    if values == '':
        return ''
    if isinstance(values, list):
        new_values = []
        for val in values:
            new_values.append(_mongo_fetch_value(val, tablename, filterby, tbuffer))
    else:
        new_values = _mongo_fetch_value(values, tablename, filterby, tbuffer)
    return new_values


def _replace_id_by_reference(userdata, key, tablename, filterby, tbuffer=None):
    if key in userdata:
        userdata[key] = _mongo_fetch_values(userdata[key], tablename, filterby, tbuffer)


def _parse_aaf_student(userdata, uid, classes_buffer, groupes_buffer, eleves_responsable_buffer):

    unused_fields = ['ENTPersonAdresse', 'ENTPersonCodePostal', 'ENTPersonVille', 'ENTPersonPays', 'ENTEleveAdresseRel']
    for field in unused_fields:
	if field in userdata:
	    del(userdata[field])

    # ajout de l'eleve a la classe et au groupe
    if not dbtype == "mongodb":
        if aaf_type == "samba4":
            cn = userdata['ENTPersonLogin']
            classes_for_one_eleve(userdata, classes_buffer, cn)
            groupe_eleve(userdata, groupes_buffer, cn)
        elif aaf_type== "openldap":
            classes_for_one_eleve(userdata, classes_buffer, uid)
            groupe_eleve(userdata, groupes_buffer, uid)
        else:
            raise ConfigError("wrong or unknown aaf_type configuration value: {}".format(aaf_type))
    else:
        if 'ENTEleveGroupes' in userdata:
            if userdata['ENTEleveGroupes'] == "":
                groups = []
            elif isinstance(userdata['ENTEleveGroupes'], list):
                groups = userdata['ENTEleveGroupes']
            else:
                groups = [userdata['ENTEleveGroupes']]
            userdata['ENTEleveGroupes'] = []
            for group in groups:
                group_name = group.split('$')[1]
                userdata['ENTEleveGroupes'].append(group_name)
        resp_buffer = {}
        for key in ('ENTEleveParents', 'ENTElevePere', 'ENTEleveMere', 'ENTEleveAutoriteParentale'):
            _replace_id_by_reference(userdata, key, "user", "ENTPersonJointure", resp_buffer)
        _replace_id_by_reference(userdata, u'ENTEleveCodeEnseignements', "subject", "ENTMatJointure")

        etab_buffer = {}
        _replace_id_by_reference(userdata, "ENTPersonStructRattach", "etablissement", "ENTStructureJointure", etab_buffer)
        _parse_etab_values('ENTEleveClasses', userdata, etab_buffer)
    if isinstance(userdata['ENTElevePersRelEleve'], list):
        resps = userdata['ENTElevePersRelEleve']
    else:
        resps = userdata['ENTElevePersRelEleve'].split('\n')

    for resp in resps:
        numresp = resp.split('$')[0]
        if numresp in eleves_responsable_buffer:
            if uid not in eleves_responsable_buffer[numresp]:
                eleves_responsable_buffer[numresp].append(uid)
        else:
            eleves_responsable_buffer[numresp] = [uid]

def _parse_aaf_resp(userdata, eleves_responsable_buffer):
    # ____________________________________________________________
    # remplissage des élèves associé au userdata
    # à partir des inforamtions bufferisées
    if userdata['id'] in eleves_responsable_buffer:
        userdata['ENTAuxPersRelEleveEleve'] = '\n'.join(eleves_responsable_buffer[userdata['id']])
    else:
        # FIXME: should not append and if it is the case
        # the account should not be created
        userdata['ENTAuxPersRelEleveEleve'] = None

# ____________________________________________________________
# Generic parsing function
def parse_aaf(type_, filename, aaftype):
    """parsing depuis AAF"""
    num = 0
    log.info("Lecture des {}s...".format(type_))
    login_buffer = []
    rnd_buffer = list()
    context = etree.iterparse(filename, events=('end',), tag='addRequest')
    for event, tdata in context:
        _type = tdata.find('operationalAttributes/attr/value').text
        if _type != aaftype:
            log.info("%s pas un {} :(".format(_type, type_))
            continue
        #**Attention** identifier/id == ENTPersonJointure
        userdata = user_id(tdata)
        for attr in tdata.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            userdata[balise] = vals

        if type_ == 'administratif':
            #Règle de répartition des personnes
            #cf. Annexe 4 (Alimentation depuis le SI MEN) version 4.2
            if userdata['PersEducNatPresenceDevantEleves'] == 'O' or \
                   '$DOC$' in userdata['ENTPersonFonctions'] or \
                   '$ENS$' in userdata['ENTPersonFonctions']:
                usertype = 'enseignant'
            else:
                usertype = 'administratif'
        else:
            usertype = type_

        if maj_mode:
            # cherche si l'utilisateur existe déjà
            sql = sqlgen.where_from(usertype, 'uid', 'id', userdata['id'])
            db_user = db.fetchone(sql)

        if not maj_mode or db_user == []:
            # new user
            uid = userdata['uid'] = get_new_id(rnd_buffer, sqlgen)
            userdata['ENTPersonLogin'] = gen_user_login(userdata['sn'], userdata['givenName'], uid, type_, userdata['id'], sqlgen, login_buffer)
            userdata['userPassword'] = pwgen()
        else:
            # existant user
            uid = userdata['uid'] = db_user['uid']

        if usertype == 'eleve':
            # TODO: on avait évoqué la possibilité de pouvoir limiter la création
            # aux responsables "1" & "2" (resp.split('$')[1] in ['1', '2'])
            # ce qui revient à être dans la liste
            # de "ENTEleveAutoriteParentale" (cf. Scribe)
            if 'ENTElevePersRelEleve' not in userdata:
                #log.error("ENTElevePersRelEleve not present in {0}".format(eleve['id']))
                continue
            _parse_aaf_student(userdata, uid, classes_buffer, groupes_buffer, eleves_responsable_buffer)
        elif type_ == 'responsable' and dbtype != 'mongodb':
            _parse_aaf_resp(userdata, eleves_responsable_buffer)
        elif type_ == 'administratif':
            _parse_aaf_profadmin(usertype, userdata)

        if maj_mode:
            if db_user == []:
                # new user in maj
                modif_fields(userdata, sqlgen, modif_type="DELTACREATE", tablename=usertype,
                            fromxml=True, db=db)
            else:
                # existant user in maj
                # il faut aller voir dans la base si les champs n'existent pas
                columns = ','.join(userdata.keys())
                sql = sqlgen.where_from(usertype, columns, 'id', userdata['id'])
                db_userdata = db.fetchone(sql)
                for normalkeys in db_userdata.keys():
                    if db_userdata[normalkeys] == None:
                        db_userdata[normalkeys] = ''
                if db_userdata != userdata:
                    modif_fields(userdata, sqlgen, modif_type="DELTAUPDATE", tablename=usertype,
                                fromxml=True, db=db)
        else:
            # ____________________________________________________________
            # traitement de l'utilisateur
            modif_fields(userdata, sqlgen, modif_type="CREATE", tablename=usertype,
                        fromxml=True, db=db)
        num += 1
        if num % DEBUG_NUM == 0:
            log.debug("{} {}s lus...".format(num, type_))
        tdata.clear()
        while tdata.getprevious() is not None:
            del tdata.getparent()[0]
    del context
    login_buffer = []

    db.commit("user")
    log.info("TOTAL : {} {}s".format(num, type_))

# ____________________________________________________________
# Extraction AAF Eleves et Responsables
def parse_aaf_eleves(eleve_file):
    """parsing des élèves depuis AAF"""
    parse_aaf('eleve', eleve_file, 'Eleve')

def flush_classes_buffer():
    """
    insertion des classes dans la database
    """
    global classes_buffer
    log.info("Traitement des classes...")
    insert_classes(classes_buffer, db, sqlgen)
    db.commit(None)
    log.info("TOTAL : %d classes" % len(classes_buffer))
    classes_buffer = {}


def flush_groupes_buffer():
    """
    insertion des groupes dans la database
    """
    global groupes_buffer
    log.info("Traitement des groupes...")
    insert_groupes(groupes_buffer, db, sqlgen)
    db.commit(None)
    log.info("TOTAL : %d groupes" % len(groupes_buffer))
    groupes_buffer = {}


def parse_aaf_responsables(responsables_file):
    """Parsing XML des responsables depuis AAF
    :responsables_file: file name like ``*_PersRelEleve_*.xml``
    """
    parse_aaf('responsable', responsables_file, 'PersRelEleve')


def parse_aaf_profs(profs_file):
    """ parsing des professeurs depuis AAF
    Extraction AAF Professeurs + Administratifs
    Fichier de type : ``ENT_$RNE_Complet_$DATE_PersEducNat_0000.xml``
    """
    #FIXME administratif ?
    parse_aaf('administratif', profs_file, 'PersEducNat')


# ____________________________________________________________
def parse_aaf_etab_educnat(etab_file):
    """Parsing XML des etablisseuments depuis AAF
    :etab_file: file name like ``*_EtabEducNat_*.xml``
    """
    # init
    num = 0
    log.info("Lecture des etablissements...")
    # ____________________________________________________________
    # xml parsing
    context = etree.iterparse(etab_file, events=('end',), tag='addRequest')
    # ____________________________________________________________
    # dict construction by XML parsing
    for event, tresp in context:
        _type = tresp.find('operationalAttributes/attr/value').text
        if _type != 'EtabEducNat':
            log.info("%s n'est pas un etablissement :(" % _type)
            continue
        etablissement = dict()
        etabid = tresp.find('identifier/id').text
        etablissement['id'] = unicode(etabid)
        for attr in tresp.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            etablissement[balise] = vals
        if etablissement['ENTStructureSIREN'] == '':
            log.error("Pas de SIREN pour l'établissement {0}".format(etablissement['ENTStructureUAI']))
            continue

	unused_fields = ['ENTStructureMailSI']
	for field in unused_fields:
	    if field in etablissement:
		del(etablissement[field])

        if isinstance(etablissement['ENTStructureGroupes'], list):
            new_value = []
            for value in etablissement['ENTStructureGroupes']:
                spl = value.split('$')
                new_value.append({'class': spl[-1], 'group': spl[0]})
            etablissement['ENTStructureGroupes'] = new_value
        if u"ENTEtablissementBassin" in etablissement and '$' in etablissement[u"ENTEtablissementBassin"]:
            etablissement[u"ENTEtablissementBassin"] = etablissement[u"ENTEtablissementBassin"].split('$')[1]
        if maj_mode:
            # il faut aller voir dans la base si les champs n'existent pas
            columns = ','.join(etablissement.keys())
            sql = sqlgen.where_from('etablissement', columns, 'id', etablissement['id'])
            db_etab = db.fetchone(sql)
            if db_etab == []:
                # new etab in maj mode
                modif_fields(etablissement, sqlgen, modif_type="DELTACREATE",
                        tablename="etablissement", fromxml=True, db=db)
            else:
                # existing establishment
                for normalkeys in db_etab.keys():
                    if db_etab[normalkeys] == None:
                        db_etab[normalkeys] = ''
                # id is declared as INTEGER in "etablissement" table
                db_etab['id'] = etablissement['id']
                if db_etab != etablissement:
                    modif_fields(etablissement, sqlgen, modif_type="DELTAUPDATE",
                            tablename="etablissement", fromxml=True, db=db)
        else:
            # ____________________________________________________________
            # database insert
            modif_fields(etablissement, sqlgen, modif_type="CREATE", tablename="etablissement",
                        fromxml=True, db=db)
        # ____________________________________________________________
            num += 1
            if num % DEBUG_NUM == 0:
                log.debug("%d etablissements lus..." % num)

    db.commit('etablissement')
    log.info("TOTAL : %d etablissements" % num)


def parse_aaf_subjects(subject_file):
    """Parsing XML des matières depuis AAF
    :etab_file: file name like ``*_MatiereEducNat_*.xml``
    """
    # init
    num = 0
    log.info("Lecture des matieres...")
    # ____________________________________________________________
    # xml parsing
    context = etree.iterparse(subject_file, events=('end',), tag='addRequest')
    # ____________________________________________________________
    # dict construction by XML parsing
    for event, tresp in context:
        subject = dict()
        _type = tresp.find('operationalAttributes/attr/value').text
        if _type != 'MatEducNat':
            log.info("%s n'est pas une matiere :(" % _type)
            continue
        subjectid = tresp.find('identifier/id').text
        subject['id'] = unicode(subjectid)
        for attr in tresp.findall('attributes/attr'):
            balise, vals = parse_multi(attr)
            subject[balise] = vals
        if maj_mode:
            raise NotImplementedError("le mode maj n'est pas gere pour les matieres")
            ## il faut aller voir dans la base si les champs n'existent pas
            #columns = ','.join(etablissement.keys())
            #sql = sqlgen.where_from('etablissement', columns, 'id', etablissement['id'])
            #db_etab = db.fetchone(sql)
            #if db_etab == []:
            #    modif_fields(etablissement, modif_type="DELTACREATE",
            #            tablename="etablissement", fromxml=True, db=db)
            #else:
            #    for normalkeys in db_etab.keys():
            #        if db_etab[normalkeys] == None:
            #            db_etab[normalkeys] = ''
            #    if db_etab != etablissement:
            #        modif_fields(etablissement, modif_type="DELTAUPDATE",
            #                tablename="etablissement", fromxml=True, db=db)
        else:
            # ____________________________________________________________
            # database insert
            modif_fields(subject, sqlgen, modif_type="CREATE", tablename="subject",
                        fromxml=True, db=db)
        # ____________________________________________________________
            num += 1
            if num % DEBUG_NUM == 0:
                log.debug("%d matières lues..." % num)

    db.commit('subject')
    log.info("TOTAL : %d matières" % num)


# ____________________________________________________________
def reset_db():
    log.info('Suppression de toutes les entrées de la base de données')
    db.reset()
    db.commit(None)
