#!/bin/bash
# -*- shell-script -*-

# shellchek disable=SC1091
. /usr/lib/eole/ihm.sh

#
# affecte une valeur à la variable TEMP_PASSWORD
#
function getValidPassword() {
    local prompt="${1:-Saisie du mot de passe}"
    local confirm="${2:-true}"
    local max_try=5
    local try=0
    local password
    local confirm_password

    while [ ${try} -le ${max_try} ]
    do
        if [ ${try} -eq ${max_try} ]
        then
            EchoRouge "Erreur de ${prompt,,}. Abandon."
            echo
            return 1
        fi

        read -s -p "${prompt} : " password
        result=$(validSambaPassword "$password")
        echo

        if [ ${result} -eq 0 ]
        then
            break
        else
            try=$((try+1))
            echo "Mot de passe invalide ($((max_try - try)) tentatives restantes)"
            echo
        fi
    done

    if [ "${confirm}" = 'false' ]
    then
        TEMP_PASSWORD="${password}"
        return 0
    fi

    try=0
    while [ ${try} -le ${max_try} ]
    do
        if [ ${try} -eq ${max_try} ]
        then
            EchoRouge "Erreur de confirmation du mot de passe. Abandon."
            echo
            return 1
        fi

        read -s -p "Confirmation du mot de passe : " confirm_password
        echo
        if [ "${confirm_password}" = "${password}" ]
        then
            TEMP_PASSWORD="${password}"
            return 0
        else
            try=$((try+1))
            echo "Les mots de passe ne correspondent pas ($((max_try - try)) tentatives restantes)"
            echo
        fi
    done
}

#
# genere une clef SSH si elle manque
#
function check_ssh_key()
{
    [ ! -d /root/.ssh ] && mkdir -p /root/.ssh

    if [ ! -f /root/.ssh/id_rsa ]
    then
        echo "Generation de la clef SSH pour les echanges entre DC"
        ssh-keygen -b 2048 -t rsa -f /root/.ssh/id_rsa -q -N ""
    fi
}

#
# Echange la clef SSH root DC1/DC2
#
function echange_ssh_key()
{
    if [ -n "${AD_DC_SYSVOL_REF}" ]
    then
        echo "Envoi de la clef SSH ${AD_HOST_NAME} vers le DC ${AD_DC_SYSVOL_REF}"
        ssh-copy-id "root@${AD_DC_SYSVOL_REF}"
        scp "root@${AD_DC_SYSVOL_REF}:/root/.ssh/id_rsa.pub" /tmp/id_rsa.pub
        cat /tmp/id_rsa.pub >> /root/.ssh/authorized_keys
    fi
}

#
# Echange la clef SSH root DC1/DC2
#
function ssh_on_dc()
{
    echo "ssh_on_dc ${AD_DC_SYSVOL_REF} : $*"
    # shellchek disable=SC2029
    ssh root@"${AD_DC_SYSVOL_REF}" "$@"
}


# Arrête tous les services liés à Samba
function stop_samba()
{
    echo "Redémarrage service"
    service samba-ad-dc stop &> /dev/null
    service smbd stop &> /dev/null
    service nmbd stop &> /dev/null
    service winbind stop &> /dev/null
    pgrep ^samba$ > /dev/null && killall samba -9
    pgrep ^smbd$ > /dev/null && killall smbd -9
    pgrep ^nmbd$ > /dev/null && killall nmbd -9
    pgrep ^winbind$ > /dev/null && killall winbind -9
    pgrep ^winbindd$ > /dev/null && killall winbindd -9
    return 0
}

# Le démarrage de Samba peut prendre
# quelques instants
# attendre que le démon ait complètement démarré
function wait_samba_start()
{
    echo Waiting Samba 127.0.0.1:445 to be alive
    for i in `seq 600`
    do
        if tcpcheck 1 127.0.0.1:445 |grep -q alive; then
            PORT445OK=ok
            break
        fi
    done
    if [ "$PORT445OK" != "ok" ]; then
        EchoRouge "Le service samba-ad-dc n'a pas démarré dans le temps imparti"
        exit 1
    fi
}

#
# AIM: Initialise the AD databases
#
function samba_init_ad()
{
    echo "Initialisation DC"

    check_ssh_key

    TEMP_PASSWORD=''
    if getValidPassword 'Création du mot de passe "Administrator" Active Directory'
    then
        AD_ADMIN_PASSWORD="${TEMP_PASSWORD}"
    else
        EchoRouge "Impossible d’initialiser le DC."
        exit 1
    fi

    # Updating hostname
    hostnamectl set-hostname "$(cat /etc/hostname)"

    kdestroy
    #[ -f /etc/samba/smb.conf ] && mv /etc/samba/smb.conf /etc/samba/smb.conf.eole

    # Initialisation of the AD
    stop_samba
    service samba-ad-dc restart

    # --use-rfc2307 : this argument adds POSIX attributes (UID/GID) to
    #                 the AD Schema. This will be necessary if you
    #                 intend to authenticate Linux, BSD, or OS X
    #                 clients (including the local machine) in
    #                 addition to Microsoft Windows.
    #
    # --use-xattrs=yes : this argument enables the use of unix
    #                    extended attributes (ACLs) for files hosted
    #                    on this server. If you intend not have file
    #                    shares on the domain controller, you can omit
    #                    this switch (but this is not
    #                    recommended). You should also ensure that any
    #                    filesystems that will host Samba shares are
    #                    mounted with support for ACLs.
    #

    mkdir -p "/var/lib/samba/private/tls"
    if [[ -z "${AD_DOMAIN_SID}" ]]
    then
        samba-tool domain provision --use-rfc2307 \
               --use-xattrs=yes \
               --dns-backend=SAMBA_INTERNAL \
               --realm="${AD_REALM^^}" \
               --domain="${AD_DOMAIN^^}" \
               --adminpass="${AD_ADMIN_PASSWORD}" \
               --server-role=dc \
               --host-ip="${AD_HOST_IP}" \
               --option="bind interfaces only=yes" \
               --option=interfaces="lo ${NOM_CARTE_NIC1}"
    else
        samba-tool domain provision --use-rfc2307 \
               --domain-sid="${AD_DOMAIN_SID}" \
               --use-xattrs=yes \
               --dns-backend=SAMBA_INTERNAL \
               --realm="${AD_REALM^^}" \
               --domain="${AD_DOMAIN^^}" \
               --adminpass="${AD_ADMIN_PASSWORD}" \
               --server-role=dc \
               --host-ip="${AD_HOST_IP}" \
               --option="bind interfaces only=yes" \
               --option=interfaces="lo ${NOM_CARTE_NIC1}"
    fi

    #[ ! -f /etc/samba/smb.conf ] && mv /etc/samba/smb.conf.eole /etc/samba/smb.conf

    if [[ "$?" -ne 0 ]]
    then
        echo "Impossible de initialiser l'annuaire Active Directory"
        exit 1
    fi

    # export keytab Administrator
    [[ -f "${AD_ADMIN_KEYTAB_FILE}" ]] && rm "${AD_ADMIN_KEYTAB_FILE}"
    samba-tool domain exportkeytab "${AD_ADMIN_KEYTAB_FILE}" --principal="${AD_ADMIN}@${AD_REALM^^}"
    if [[ "$?" -ne 0 ]]
    then
        echo "Impossible de générer le keytab ${AD_ADMIN}"
        exit 1
    fi

    echo "Redémarrage service"
    service samba-ad-dc restart
    wait_samba_start

    #echo "Nettoyage Cache"
    #net cache flush

    echo "Test connection kerberos/AD"
    sleep 5
    kinit "${AD_ADMIN}@${AD_REALM^^}" -k -t "${AD_ADMIN_KEYTAB_FILE}"
    if [[ "$?" -ne 0 ]]
    then
        echo "Connection kerberos/AD impossible"
        exit 1
    fi

    echo "Set Administrator password never expire"
    samba-tool user setexpiry "${AD_ADMIN}" --noexpiry
    if [[ "$?" -ne 0 ]]
    then
        echo "Impossible de déactiver l'expiration de mot de passe de l'Administrator"
        exit 1
    fi

    echo Waiting Samba 127.0.0.1:445 to be alive
    for i in `seq 600`
    do
        tcpcheck 1 127.0.0.1:445 |grep -q alive && PORT445OK=ok
    done
    if [ "$PORT445OK" != "ok" ]; then
        EchoRouge "Le service samba-ad-dc n'a pas démarré dans le temps imparti"
        exit 1
    fi

    echo "Ajout SeDiskOperatorPrivilege au groupe 'Domain Admins'"
    net rpc rights grant "${AD_DOMAIN^^}\Domain Admins" SeDiskOperatorPrivilege -U"${AD_ADMIN}%${AD_ADMIN_PASSWORD}"
    if [[ "$?" -ne 0 ]]
    then
        echo "Impossible d'attribuer le privilège SeDiskOperatorPrivilege au groupe Domain Admins"
        exit 1
    fi

    #TODO : create reverse zone pour PEDAGO, ADMIN,DMZ !
    #samba-tool dns zonecreate pedago.eole.lan 2.1.10.in-addr.arpa --username=${AD_ADMIN}
    #samba-tool dns zonecreate admin.eole.lan 1.1.10.in-addr.arpa --username=${AD_ADMIN}
    #samba-tool dns zonecreate dmz.eole.lan 3.1.10.in-addr.arpa --username=${AD_ADMIN}

    #samba-tool dns zonecreate dc1.eole.lan 0.168.192.in-addr.arpa --username=${AD_ADMIN}
    #samba-tool dns add dc1.eole.lan.tld 0.168.192.in-addr.arpa 17 PTR  dc1.eole.lan.tld --username=${AD_ADMIN}

    echo "Creation utilisateur 'admin'"
    TEMP_PASSWORD=''
    if getValidPassword 'Création du mot de passe "admin" Active Directory'
    then
        ADMIN_PASSWORD="${TEMP_PASSWORD}"
        unset TEMP_PASSWORD
    else
        EchoRouge 'Impossible de créer l’utilisateur "admin".'
        exit 1
    fi

    samba-tool user create admin "$ADMIN_PASSWORD"

    if [[ "$?" -ne 0 ]]
    then
        echo "Impossible de créer l'utilisateur admin"
        exit 1
    fi
    #mkdir -p "/var/lib/samba/usershares"
    samba_create_homes_dir
    samba_create_profiles_dir

    samba-tool group addmembers "Domain Admins" admin
    #samba-tool group addmembers "${AD_DOMAIN^^}\Administrators" admin
    if [[ "$?" -ne 0 ]]
    then
        echo "Impossible d'inscire admin dans le groupe Domain Admins"
        exit 1
    fi

    #CreoleGet check_passwd

    #manage_share_profiles
#
#    echo "Reset the ACLs on the local sysvol folder"
#    samba-tool ntacl sysvolreset

    touch "${AD_INSTANCE_LOCK_FILE}"
}

#
# AIM: Initialise the Additional DC
#
function samba_init_additional()
{
    echo "Initialisation DC Secondaire"

    check_ssh_key

    if [ -n "${AD_DC_SYSVOL_REF}" ]
    then
        if ! grep -q "${AD_DC_SYSVOL_REF}" /etc/resolv.conf
        then
            echo "nameserver ${AD_DC_SYSVOL_REF}" >>/etc/resolv.conf
        fi
    fi

    # Updating hostname
    hostnamectl set-hostname "$(cat /etc/hostname)"

    echange_ssh_key

    echo "Jonction au domaine"
    read  -p "Compte pour joindre le serveur au domaine [${AD_ADMIN}] : " AD_DELEGATION
    echo
    if [ -z "${AD_DELEGATION}" ]
    then
        AD_DELEGATION="${AD_ADMIN}"
    fi

    TEMP_PASSWORD=''
    if getValidPassword 'Mot de passe de jonction au domaine' false
    then
        AD_DELEGATION_PASSWORD="${TEMP_PASSWORD}"
        unset TEMP_PASSWORD
    else
        EchoRouge "Impossible de joindre le domaine."
        exit 1
    fi

    stop_samba
    service samba-ad-dc restart

    # Initialisation of the AD
    samba-tool domain join "${AD_REALM}" ${AD_SERVER_MODE}DC \
               -U"${AD_DELEGATION}%${AD_DELEGATION_PASSWORD}" \
               --realm="${AD_REALM}" \
               -W "${AD_DOMAIN}"

    if [[ "$?" -ne 0 ]]
    then
        echo "Impossible de joindre le DC à l'annuaire existant"
        exit 1
    fi

    if [[ -z "${AD_SERVER_MODE}" ]]; then
    # export keytab ${AD_ADMIN}
    [[ -f "${AD_ADMIN_KEYTAB_FILE}" ]] && rm "${AD_ADMIN_KEYTAB_FILE}"
    samba-tool domain exportkeytab "${AD_ADMIN_KEYTAB_FILE}" --principal="${AD_ADMIN}@${AD_REALM^^}"
    if [[ "$?" -ne 0 ]]
    then
        echo "Impossible de générer le keytab ${AD_ADMIN}"
        exit 1
    fi
    fi

    # workaround samba < 4.2.0 pour https://bugzilla.samba.org/show_bug.cgi?id=10928
    ssh_on_dc /usr/share/eole/sbin/createDnsRecord "${AD_HOST_NAME}" "${AD_HOST_IP}"

    mkdir -p /var/lib/samba/private
    # Apparemment, toujours nécessaire en 4.3
    # https://wiki.samba.org/index.php/Join_an_additional_Samba_DC_to_an_existing_Active_Directory#GID_mappings_of_built-in_groups
    ssh_on_dc tdbbackup -s .bak /var/lib/samba/private/idmap.ldb
    scp "root@${AD_DC_SYSVOL_REF}:/var/lib/samba/private/idmap.ldb.bak" /var/lib/samba/private/idmap.ldb
    chmod 600 /var/lib/samba/private/idmap.ldb

    echo "Redémarrage service"
    service samba-ad-dc restart
    wait_samba_start

    echo "Nettoyage Cache"
    #net cache flush

    echo "Execute Synchro Sysvol"
    JobSynchroSysvol

    # Samba 4.1 : replication of the SysVol share isn't implemented. If you make any changes on that share,
    # you have to keep them in sync on all your Domain Controllers. An example, how to achieve
    # this automatically, is provided in the SysVol Replication documentation.
    # https://wiki.samba.org/index.php/Rsync_based_SysVol_replication_workaround
    ## we need rsync to create the directory structure with extended attributes
    echo "Install SysVol Replication Workaround "
    if [ ! -f /etc/cron.d/sysvol-sync ]
    then
        cat >/etc/cron.d/sysvol-sync <<EOF
*/5 * * * * root  /usr/share/eole/sbin/JobSynchroSysvol >/dev/null
EOF
    fi

    samba_create_homes_dir
    samba_create_profiles_dir
    touch "${AD_INSTANCE_LOCK_FILE}"
}

#
# AIM: Initialise a Member Server
#
function samba_init_member()
{
    echo "Initialisation Server Membre"
    stop_samba
    # Attention : https://wiki.samba.org/index.php/Setup_a_Samba_AD_Member_Server

    echo "Jonction au domaine"
    read  -p "Compte pour joindre le serveur au domaine [${AD_ADMIN}] : " AD_DELEGATION
    echo
    if [ -z "${AD_DELEGATION}" ]
    then
        AD_DELEGATION="${AD_ADMIN}"
    fi

    TEMP_PASSWORD=''
    if getValidPassword 'Mot de passe de jonction au domaine' false
    then
        AD_DELEGATION_PASSWORD="${TEMP_PASSWORD}"
        unset TEMP_PASSWORD
    else
        EchoRouge "Impossible de joindre le domaine."
        exit 1
    fi

    net ads join -U"${AD_DELEGATION}%${AD_DELEGATION_PASSWORD}"

    echo "Redémarrage service"
    service nmbd restart
    service smbd restart
    service winbind restart

    echo "Nettoyage Cache"
    net cache flush

    samba_create_homes_dir
    samba_create_profiles_dir
    touch "${AD_INSTANCE_LOCK_FILE}"
}

function samba_create_homes_dir()
{
    if [ "${ACTIVER_AD_HOMES_SHARE}" == "oui" ]; then
        mkdir -p ${AD_HOME_SHARE_PATH}
        if [ "${AD_SERVER_ROLE}" == "controleur de domaine" ] && \
           [ "${AD_ADDITIONAL_DC=}" == "non" ]; then
            pdbedit -h "\\\\${AD_HOST_NAME}.${AD_REALM}\\admin" -D 'U:' admin > /dev/null
        fi
    else
        if [ "${AD_SERVER_ROLE}" == "controleur de domaine" ] && \
           [ "${AD_ADDITIONAL_DC=}" == "non" ] && \
           [ -n "${AD_HOMES_SHARE_HOST_NAME}" ]; then
            pdbedit -h "\\\\${AD_HOMES_SHARE_HOST_NAME}.${AD_REALM}\\admin" -D 'U:' admin > /dev/null
        fi
    fi
}

function samba_create_profiles_dir()
{
    if [ "${ACTIVER_AD_PROFILES_SHARE}" == "oui" ]; then
        mkdir -p ${AD_PROFILE_SHARE_PATH}
        setfacl -Rbk ${AD_PROFILE_SHARE_PATH}
        setfacl -m g:"${AD_REALM}/domain users":rwx ${AD_PROFILE_SHARE_PATH}
        if [ "${AD_SERVER_ROLE}" == "controleur de domaine" ] && \
           [ "${AD_ADDITIONAL_DC=}" == "non" ]; then
            pdbedit -p "\\\\${AD_HOST_NAME}.${AD_REALM}\\profiles\\admin" admin > /dev/null
        fi
    else
        if [ "${AD_SERVER_ROLE}" == "controleur de domaine" ] && \
           [ "${AD_ADDITIONAL_DC=}" == "non" ] && \
           [ -n "${AD_PROFILES_SHARE_HOST_NAME}" ]; then
            pdbedit -p "\\\\${AD_PROFILES_SHARE_HOST_NAME}.${AD_REALM}\\profiles\\admin" admin > /dev/null
        fi
    fi
}
#
# AIM: Initialise a standalone host
#
function samba_init_standalone()
{
    echo "Initialisation Standalone Server"

    #TODO

    touch "${AD_INSTANCE_LOCK_FILE}"
}

function samba_init_pre()
{
    mkdir -p /etc/samba/conf.d/
}

#function samba_init_post()
#{
#}

#
# AIM: Initialise AD
#
function samba_instance()
{
    samba_init_pre
    case "${AD_SERVER_ROLE}" in
        "controleur de domaine")
            if [ "${AD_ADDITIONAL_DC}" == 'non' ]
            then
                samba_init_ad
            else
                samba_init_additional
            fi
            ;;

        "membre")
            samba_init_member
            ;;

        *)
            EchoRouge "Server Role inconnu : '${AD_SERVER_ROLE}'"
            exit 1
            ;;
    esac
#    samba_init_post
}


#
# AIM: Initialise the AD databases
#
function samba_reconfigure()
{
    echo "25-manage-samba: Active Directory est initialisé, reload la configuration"
    smbcontrol all reload-config
}


# perm is a permission mask and can be specified in one of two forms:
#        a sequence of simple rights:
#                F - full access
#                M - modify access
#                RX - read and execute access
#                R - read-only access
#                W - write-only access
#        a comma-separated list in parentheses of specific rights:
#                D - delete
#                RC - read control
#                WDAC - write DAC
#                WO - write owner
#                S - synchronize
#                AS - access system security
#                MA - maximum allowed
#                GR - generic read
#                GW - generic write
#                GE - generic execute
#                GA - generic all
#                RD - read data/list directory
#                WD - write data/add file
#                AD - append data/add subdirectory
#                REA - read extended attributes
#                WEA - write extended attributes
#                X - execute/traverse
#                DC - delete child
#                RA - read attributes
#                WA - write attributes
#        inheritance rights may precede either form and are applied
#        only to directories:
#                (OI) - object inherit
#                (CI) - container inherit
#                (IO) - inherit only
#                (NP) - don't propagate inherit

#function get_uid_by_name
#{
#    declare -a SID_ET_NAME
#    # interrogation wbinfo ==> 3 valeurs. Convertion en tableau pour garder la premiere !
#    SID_ET_NAME=($(wbinfo -n "$1"))
#    [[ -z "${SID_ET_NAME[0]}" ]] && return 1
#    wbinfo -S "${SID_ET_NAME[0]}"
#    return ${?}
#}
#
#function manage_share_home()
#{
#    [ -n "${AD_HOME_SHARE_PATH}" ] || return 0
#    # protection !
#    [[ ${AD_HOME_SHARE_PATH} == /* ]] || AD_HOME_SHARE_PATH="/${AD_HOME_SHARE_PATH}"
#    [ ! -d "${AD_HOME_SHARE_PATH}" ] || return 0
#    echo "Creation of '${AD_HOME_SHARE_PATH}' shares"
#
#    mkdir -p "${AD_HOME_SHARE_PATH}"
#
#    AUTH_USERS_UID=$(get_uid_by_name "${AD_DOMAIN^^}\Domain Users")
#    DADMIN_UID=$(get_uid_by_name "${AD_DOMAIN^^}\Domain Admins")
#
#    setfacl -Rm "g:${AUTH_USERS_UID}:r-x,g:${DADMIN_UID}:rwx" "${AD_HOME_SHARE_PATH}"
#    if [[ ${?} -ne 0 ]]
#    then
#        echo "Oops no way to manage homes ACL !"
#        return 4
#    fi
#
#    # Creation of the "home" share NTACL (SDDL Format)"
#    HOME_ACL="O:LAG:S-1-22-2-0D:PAI"
#    HOME_ACL="${HOME_ACL}(A;OICI;0x001f01ff;;;LA)"
#    HOME_ACL="${HOME_ACL}(A;OICI;0x001f01ff;;;DA)"
#    HOME_ACL="${HOME_ACL}(A;OICIIO;0x001f01ff;;;CO)"
#    HOME_ACL="${HOME_ACL}(A;;0x001200a9;;;AU)"
#    HOME_ACL="${HOME_ACL}(A;OICI;0x001f01ff;;;SY)"
#
#    samba-tool ntacl set ${HOME_ACL} "${AD_HOME_SHARE_PATH}"
#    if [[ ${?} -ne 0 ]]
#    then
#        echo "Ooops ntacl set failed for ${AD_HOME_SHARE_PATH}"
#        return 5
#    fi
#}
#
#function manage_share_profiles()
#{
#    [ ! -d "/var/lib/samba/profiles" ] || return 0
#    echo "Creation partage 'profiles'"
#
#    mkdir -p "/var/lib/samba/profiles"

#    AUTH_USERS_SID=$(wbinfo -n "${AD_DOMAIN^^}\Domain Users")
#    AUTH_USERS_UID=$(wbinfo -S "${AUTH_USERS_SID}")
#    DADMIN_SID=$(wbinfo -n "${AD_DOMAIN^^}\Domain Admins")
#    DADMIN_UID=$(wbinfo -S "${DADMIN_SID}")
#
#    setfacl -Rm "g:${AUTH_USERS_UID}:r-x,g:${DADMIN_UID}:rwx" "${AD_HOME_SHARE_PATH}"
#    if [[ ${?} -ne 0 ]]
#    then
#        echo "Oops no way to manage homes ACL !"
#        return 4
#    fi
#
#    # Creation of the "profiles" share NTACL (SDDL Format)"
#    HOME_ACL="O:LAG:S-1-22-2-0D:PAI"
#    HOME_ACL="${HOME_ACL}(A;OICI;0x001f01ff;;;LA)"
#    HOME_ACL="${HOME_ACL}(A;OICI;0x001f01ff;;;DA)"
#    HOME_ACL="${HOME_ACL}(A;OICIIO;0x001f01ff;;;CO)"
#    HOME_ACL="${HOME_ACL}(A;;0x001200a9;;;AU)"
#    HOME_ACL="${HOME_ACL}(A;OICI;0x001f01ff;;;SY)"
#
#    samba-tool ntacl set ${HOME_ACL} "${AD_HOME_SHARE_PATH}"
#    if [[ ${?} -ne 0 ]]
#    then
#        echo "Ooops ntacl set failed for ${AD_HOME_SHARE_PATH}"
#        return 5
#    fi
#}
#
#function manage_acl
#{
#  user="$1"
#  directory="$2"
#
#  #RAPPEL sur un Win2012 (. = home de 'admin')
#  # . W2012\admin:(F)
#  #   W2012\admin:(OI)(CI)(IO)(F)
#  #   BUILTIN\Administrateurs:(F)
#  #   BUILTIN\Administrateurs:(OI)(CI)(IO)(F)
#  #   W2012\Administrateur:(I)(OI)(CI)(F)
#  #   BUILTIN\Administrateurs:(I)(OI)(CI)(F)
#  #   AUTORITE NT\Syst�me:(I)(OI)(CI)(F)
#  #
#  # .\essai.txt W2012\admin:(I)(F)
#  #             BUILTIN\Administrateurs:(I)(F)
#  #             W2012\Administrateur:(I)(F)
#  #             AUTORITE NT\Syst�me:(I)(F)
#
#
#  # l'Administrator
#  ADMINS_UID=$(get_uid_by_name "Administrator")
#
#  # Group ACL
#  setfacl -Rm g:"${ADMINS_UID}":rwx "${directory}"
#  [[ ${?} -ne 0 ]] && error "setfacl for groups ADMINS_UID failed on ${directory}"
#
#  # SYSTEM ACL
#  #SYSTEM_UID=$(get_uid_by_name "NT AUTHORITY\SYSTEM")
#  #setfacl -Rm u:"${SYSTEM_UID}":rwx "${directory}"
#  #[[ ${?} -ne 0 ]] && error "setfacl for users SYSTEM_UID failed on ${directory}"
#  #setfacl -Rm g:"${SYSTEM_UID}":rwx "${directory}"
#  #[[ ${?} -ne 0 ]] && error "setfacl for groups SYSTEM_UID failed on ${directory}"
#
#  # Domain Admins ACL
#  DADMINS_UID=$(get_uid_by_name "Domain Admins")
#  setfacl -Rm u:"${DADMINS_UID}":rwx "${directory}"
#  [[ ${?} -ne 0 ]] && error "setfacl for users DADMINS_UID failed on ${directory}"
#  setfacl -Rm g:"${DADMINS_UID}":rwx "${directory}"
#  [[ ${?} -ne 0 ]] && error "setfacl for groups DADMINS_UID failed on ${directory}"
#
#  # User ACL
#  USER_UID=$(get_uid_by_name "${user}")
#  setfacl -Rm u:"${USER_UID}":rwx "${directory}"
#  [[ ${?} -ne 0 ]] && error "setfacl for users USER_UID failed on ${directory}"
#  setfacl -Rm g:"${USER_UID}":rwx "${directory}"
#  [[ ${?} -ne 0 ]] && error "setfacl for groups USER_UID failed on ${directory}"
#
#  return 0
#
#}
#
#function create_home
#{
#  user=$1
#  [[ ${AD_HOME_SHARE_PATH} == /* ]] || AD_HOME_SHARE_PATH="/${AD_HOME_SHARE_PATH}"
#  home_directory_user="${AD_HOME_SHARE_PATH}/${AD_DOMAIN^^}/${user}"
#  home_mode="770"
#
#  mkdir -p "${home_directory_user}"
#  ret=${?}
#  if [[ ${ret} -eq 0 ]]
#  then
#    chmod "${home_mode}" "${home_directory_user}"
#    manage_acl "${user}" "${home_directory_user}"
#    return ${?}
#  else
#    return ${ret}
#  fi
#}
#
#function error
#{
#  echo "Error : ${1}"
#  exit "${2}"
#}
