#!/usr/bin/env python3

# general
import traceback
from os import path, makedirs
from io import StringIO
from configparser import ConfigParser
import ntpath

# samba
import ldb
from samba import get_debug_level
from samba.samdb import SamDB
from samba.ndr import ndr_unpack
from samba.auth import system_session
from samba.dcerpc import nbt, security
from samba.ntacls import dsacl2fsacl
from samba.net import Net
from samba.netcmd.gpo import (
    dc_url,
    smb_connection,
    get_gpo_dn,
    parse_unc,
    parse_gplink,
    encode_gplink,
    attr_default,
    get_gpo_info,
    gpo_flags_string,
    CommandError,
    GPOCommand
)


#from samba.samba3 import param as s3param
#from samba.samba3 import libsmb_samba_internal as libsmb
class EoleGPOCommand(GPOCommand):
    """Abstract Class to manage Eole GPO."""

    # to disable pylint error !
    displayname = None
    gpo_id = None
    samdb = None
    lp = None
    creds = None
    realm = None
    dc_hostname = None
    gpo_displayname = None
    url = None
    unc = None
    gpo_dn = None
    gpo_version = None
    dom_name = None
    service = None
    sharepath = None
    conn = None
    
    #     
    def debug(self, text):
        ''' Envoi le texte sur la sortie Stderr si l'option '-d 1' est passé en arguments à la commande '''
        if get_debug_level() > 0:
            self.errf.write(" * ")
            self.errf.write(text)
            self.errf.write("\n")
            
    def debug2(self, text):
        ''' Envoi le texte sur la sortie Stderr si l'option '-d 2' est passé en arguments à la commande '''
        if get_debug_level() > 1:
            self.errf.write("    * ")
            self.errf.write(text)
            self.errf.write("\n")
            
    # de gpo.py 4.11 avant commit    
    def smb_connection(self, dc_hostname, service, lp, creds, sign=False):
        # SMB connect to DC
        try:
            self.debug2 ("smb_connection '%s' %s " % (dc_hostname, service))
            conn = smb_connection(dc_hostname, service, lp=lp, creds=creds, sign=sign)
        except Exception:
            raise CommandError("Error connecting to '%s' using SMB" % dc_hostname)
        return conn

    # de gpo.py 4.11    
    def samdb_connect(self):
        '''make a ldap connection to the server'''
        try:
            self.samdb = SamDB(url=self.url,
                               session_info=system_session(),
                               credentials=self.creds, lp=self.lp)
        except Exception as e:
            raise CommandError("LDAP connection to %s failed " % self.url, e)

    def load_gpo_entry(self, gpo_name):
        """Get GPO information using get_gpo_info, with displayname
        
        Return :
                False si la GPO n'existe pas
                True si elle existe, et que les variables ont été injectées
                
        Inject :
                self.gpc_entry ='ldb result' 
                self.gpo_id = name
                self.gpo_displayname = displayName 
                self.unc = gPCFileSysPath
                self.gpo_dn = dn
                self.gpo_version = versionNumber
                self.dom_name = dommain extrait de UNC ( le realm )
                self.service = service extrait de UNC ('sysvol')
                self.sharepath = path extrait de UNC 
        
        """
        try:
            self.debug2 ("Get GPO '%s' " % gpo_name)
            gpc_entries = get_gpo_info(self.samdb, displayname=gpo_name)
            if gpc_entries.count == 0:
                self.debug2 ( "gpc_entry = None !")
                return False
            else: 
                self.gpc_entry = gpc_entries[0]
                #self.debug2 ( "gpc_entry = " + str(type( self.gpc_entry )))
                self.gpo_id = self.gpc_entry['name'][0]
                self.debug2 ("gpo name     : %s" % self.gpo_id)
                self.gpo_displayname = self.gpc_entry['displayName'][0]
                self.debug2 ("display name : %s" %  self.gpo_displayname )
                self.unc = str(self.gpc_entry['gPCFileSysPath'][0])
                self.debug2 ("path         : %s" % self.unc)
                self.gpo_dn = self.gpc_entry.dn
                self.debug2 ("dn           : %s" % self.gpo_dn)
                self.gpo_version = int(attr_default(self.gpc_entry, 'versionNumber', '0'))
                self.debug2 ("version      : %s" % self.gpo_version)
                self.debug2 ("flags        : %s" % gpo_flags_string(int(attr_default(self.gpc_entry, 'flags', '0'))))
            
                # verify UNC path
                try:
                    [self.dom_name, self.service, self.sharepath] = parse_unc(self.unc)
                except ValueError:
                    raise CommandError("Invalid GPO path (%s)" % self.unc)
                self.debug2 ("domain       : %s" % self.dom_name)
                self.debug2 ("service      : %s" % self.service)
                self.debug2 ("sharepath    : %s" % self.sharepath)
                return True
        except Exception:
            traceback.print_exc()
            raise CommandError("GPO '%s' does not exist" % gpo_name)
    
    def initialisation(self, gpo_name, H=None, sambaopts=None, credopts=None, versionopts=None):
        """initialisation :
        - Initialise les variables
        - Initialise la connection à samdb
        - Charge les informations de la GPO 
        
        Return :
                False si la GPO n'existe pas
                True si elle existe, et que les variables ont été injectées
                
        Inject :
                self.lp = load parm from smb.conf
                self.creds = credentials depuis la ligne de commande
                self.realm = déclarer dans smb.conf
                self.dc_hostname = hostname du DC a utiliser (depuis -H ldap:// )
                self.url = (depuis -H ldap:// )
        
        Inject depuis load_gpo_entry:        
                self.gpc_entry ='ldb result' 
                self.gpo_id = name
                self.gpo_displayname = displayName 
                self.unc = gPCFileSysPath
                self.gpo_dn = dn
                self.gpo_version = versionNumber
                self.dom_name = dommain extrait de UNC ( le realm )
                self.service = service extrait de UNC ('sysvol')
                self.sharepath = path extrait de UNC 
        
        """
        self.lp = sambaopts.get_loadparm()
        self.creds = credopts.get_credentials(self.lp, fallback_machine=True)
        
        self.debug2 ( "gpo_name     : " + gpo_name)
        self.debug2 ( "H            : " + str(H))
        self.debug2 ( "samaopts     : " + str(sambaopts))
        self.debug2 ( "credopts     : " + str(credopts))
        self.debug2 ( "versionopts  : " + str(versionopts))
        self.realm = self.lp.get('realm')
        self.debug2 ( "realm        : " + self.realm)
        
        # We need to know writable DC to setup SMB connection
        net = Net(creds=self.creds, lp=self.lp)
        if H and H.startswith('ldap://'):
            self.dc_hostname = H[7:]
            self.url = H
            flags = (nbt.NBT_SERVER_LDAP |
                     nbt.NBT_SERVER_DS |
                     nbt.NBT_SERVER_WRITABLE)
            cldap_ret = net.finddc(address=self.dc_hostname, flags=flags)
            self.debug2 ("url distante  : '%s'" % str(self.url ))
        else:
            flags = (nbt.NBT_SERVER_LDAP |
                     nbt.NBT_SERVER_DS |
                     nbt.NBT_SERVER_WRITABLE)
            cldap_ret = net.finddc(domain=self.realm, flags=flags)
            self.dc_hostname = cldap_ret.pdc_dns_name
            self.url = dc_url(self.lp, self.creds, dc=self.dc_hostname)
            self.debug2 ("url local    : '%s'" % str(self.url ))
        
#         if H and H.startswith('ldap://'):
#             self.dc_hostname = H[7:]
#             self.url = H
#             self.debug2 ("url distante : '%s'" % str(self.url ))
#         else:
#             # ********************************************************************************************
#             # ICI j'IMPOSE LE DC LOCAL
#             # ********************************************************************************************
#             self.dc_hostname = netcmd_dnsname(self.lp)
#             self.url = dc_url(self.lp, self.creds, dc=self.dc_hostname)
#             self.debug2 ("url local    : '%s'" % str(self.url ))
#         
        
        self.debug2 ("dc_hostname  : '%s'" % str(self.dc_hostname ))
        self.samdb_connect()
        
        return self.load_gpo_entry( gpo_name )
   

    def run(self):
        """Run the command. This should be overridden by all subclasses."""
        raise NotImplementedError(self.run)

    def runInTransaction(self):
        """runInTransaction. This should be overridden by all subclasses.
        
        Cette fonction s'execute dans le cotntext d'une transaction
        - si la fonction renvoi 0, alors la transaction est confirmée
        - si la fonction renvoi 1 ou une exception, alors la transaction est annulée
        
        """
        raise NotImplementedError(self.run)

    def connectAndRunInTransaction(self, displayname, H=None, sambaopts=None, credopts=None, versionopts=None):
        """ connect And call runInTransaction
        Cette fonction est appelée pour :
        - charger la GPO,
        - initier la connexion SmaDB,
        - ouvrir la connexion SMB
        - appeler runInTransaction dans le contexte d'une transaction
        - garantir 1 seul commit/rollback  
        """

        self.displayname = displayname
        if self.initialisation( displayname, H, sambaopts, credopts, versionopts ) is False:
            print ("GPO %s is unkown." % displayname)
            return 1
        
        self.samdb.transaction_start()
        try:
            self.conn = self.smb_connection(self.dc_hostname, self.service, lp=self.lp, creds=self.creds, sign=True)
            self.debug2( str( self.conn ))
            self.runInTransaction()
            self.samdb.transaction_commit()
            return 0
        except Exception as e:
            self.samdb.transaction_cancel()
            traceback.print_exc()
            raise e
        
    def set_ownership_and_mode(self, file_path, gpo, conn, samdb):
        """Set ownership and acl for rule file file_path in GPT to user connected
        through conn and samdb connection (copied from samba.netcmd.gpo).
        :param file_path: location of rule file in unc format
        :type file_path: str
        :param gpo: gpo ID
        :type gpo: str
        :param conn: connection to server with SMB protocol
        :type conn: smb.SMB
        :param samdb: connection to ldb
        :type samdb: samba.samdb.SamDB
        """
        # Get new security descriptor
        ds_sd_flags = ( security.SECINFO_OWNER |
                        security.SECINFO_GROUP |
                        security.SECINFO_DACL )
        msg = get_gpo_info(samdb, gpo=gpo, sd_flags=ds_sd_flags)[0]
        ds_sd_ndr = msg['nTSecurityDescriptor'][0]
        ds_sd = ndr_unpack(security.descriptor, ds_sd_ndr).as_sddl()
    
        # Create a file system security descriptor
        domain_sid = security.dom_sid(samdb.get_domain_sid())
        sddl = dsacl2fsacl(ds_sd, domain_sid)
        fs_sd = security.descriptor.from_sddl(sddl, domain_sid)
        # Set ACL
        sio = ( security.SECINFO_OWNER |
                security.SECINFO_GROUP |
                security.SECINFO_DACL |
                security.SECINFO_PROTECTED_DACL )
        conn.set_acl(file_path, fs_sd, sio)

    def gpc_update_extension(self, gpo_dn, gpc_entry, cse_info, samdb):
        """Update GPC with cse information from added policy.
        :param gpo_dn: Group Policy Object dn
        :type gpo_dn: str
        :param cse_info: information needed to declare extension in GPC
        :type cse_info: tuple -> ( str , str list )
        :param samdb: ldb connection
        :type samdb: SamDB
        """
        #print("gpo_dn="+str(gpo_dn) + " gpc_entry=" + str( gpc_entry ) + " cse_info=" + str(cse_info))
        if isinstance(cse_info[1], list):
            cse_guid_list = cse_info[1]
        elif isinstance(cse_info[1], str):
            cse_guid_list = [cse_info[1]]
        else:
            raise Exception('type extension unknown')
        field = 'gPC{}ExtensionNames'.format(cse_info[0])
        if not field in gpc_entry.keys():
            # First value in field
            new_state = '[{}]'.format(']['.join(sorted(cse_guid_list)))
        else:
            # Merge old values with new values in field; must be sorted and uniq.
            gpc_entries_list = str(gpc_entry[field]).strip('[]').split('][')
            new_state = sorted(set(gpc_entries_list + cse_guid_list))
            new_state = '[{}]'.format(']['.join(new_state))
        m = ldb.Message()
        m.dn = gpo_dn
        m['a05'] = ldb.MessageElement(new_state, ldb.FLAG_MOD_REPLACE, field)
        controls = ["permissive_modify:0"]
        samdb.modify(m, controls=controls)

    def update_gpt_version(self, gpt_path, version, conn):
        """Save new version value to GPT.
        :param gpt_path: location of GPO in GPT
        :type gpt_path: str
        :param version: version number
        :type version: str
        :param conn: connection to server through SMB
        :type conn: smb.SMB
        """
        self.debug2 ("update_gpt_version version=" + version)
        self.debug2 ("update_gpt_version conn=" + str(conn) )
        gpt_ini_path = gpt_path + '\\GPT.INI'
        self.debug2 ("update_gpt_version gpt_ini_path=" + gpt_ini_path)
        gpt_ini_content = conn.loadfile(gpt_ini_path)
        cp = ConfigParser()
        cp.readfp(StringIO(gpt_ini_content.decode('utf-8') if isinstance(gpt_ini_content, bytes) else gpt_ini_content))
        cp.set('General', 'Version', version)
        cp_content = StringIO()
        cp.write(cp_content)
        cp_content.seek(0)
        self.savecontent(gpt_ini_path, cp_content.read(), conn=conn)
    
    def savecontent(self, remote_path, content, conn=None):
        """Create file with content through SMB connection or locally.
        :param remote_path: location of file to create, either in unc format if
                            conn is provided, or as absolute path.
        :type remote_path: str
        :param content: content of file
        :type content: str
        :param conn: connection to server through SMB
        :type conn: smb.SMB
        """
        #print("savecontent " + remote_path)
        if conn is None:
            if not path.isdir(path.dirname(remote_path)):
                makedirs(path.dirname(remote_path))
            with open(remote_path, 'wb') as pol_file:
                pol_file.write(content)
        else:
            self.create_directory_hier(conn, ntpath.dirname(remote_path))
            conn.savefile(remote_path, content.encode('utf-8') if isinstance(content, str) else content)
        
    def create_directory_hier(self, conn, remotedir):
        """Create directory through SMB connection (copied from samba.netcmd.gpo).
        :param conn: connection to server with SMB protocol
        :type conn: smb.SMB
        :param remotedir: folder to create remotely
        :type remotedir: str
        """
        #print("create_directory_hier " + remotedir)
        elems = remotedir.replace('/', '\\').split('\\')
        path = ""
        for e in elems:
            path = path + '\\' + e
            if not conn.chkpath(path):
                conn.mkdir(path)

    def update_gpc_version(self, gpo_dn, version, samdb):
        """Save new version value to GPC.
        :param gpo_dn: DN identifying GPO in GPC
        :type gpo_dn: DN object
        :param version: version number
        :type version: str
        :param conn: connection to server through SMB
        :type conn: smb.SMB
        """
        m = ldb.Message()
        m.dn = gpo_dn
        m['a05'] = ldb.MessageElement(version, ldb.FLAG_MOD_REPLACE, "versionNumber")
        controls = ["permissive_modify:0"]
        samdb.modify(m, controls=controls)
        
    def doSetLink(self, container_dn ):

        self.debug2 ("doSetLink :" + str(container_dn))
        gpo_dn = str(get_gpo_dn(self.samdb, self.gpo_id))
        self.debug2 ("gpo dn :" + gpo_dn ) 
        
        # Check if valid Container DN
        try:
            containerObject = self.samdb.search(base=container_dn, scope=ldb.SCOPE_BASE,
                                    expression="(objectClass=*)",
                                    attrs=['gPLink'])[0]
        except Exception:
            raise CommandError("Container '%s' does not exist" % container_dn)

        # Update existing GPlinks or Add new one
        existing_gplink = False
        gplink_options = 0
        if 'gPLink' in containerObject:
            gplist = parse_gplink(str(containerObject['gPLink'][0]))
            existing_gplink = True
            found = False
            for g in gplist:
                gplink_options = g['options'] 
                if g['dn'].lower() == gpo_dn.lower():
                    g['options'] = gplink_options
                    found = True
                    break
            if found:
                self.debug ("GPO '%s' already linked to this container" % self.gpo_id)
                return False
            else:
                gplist.insert(0, {'dn': gpo_dn, 'options': gplink_options})
        else:
            gplist = []
            gplist.append({'dn': gpo_dn, 'options': gplink_options})

        gplink_str = encode_gplink(gplist)

        m = ldb.Message()
        m.dn = ldb.Dn(self.samdb, container_dn)

        if existing_gplink:
            m['new_value'] = ldb.MessageElement(gplink_str, ldb.FLAG_MOD_REPLACE, 'gPLink')
        else:
            m['new_value'] = ldb.MessageElement(gplink_str, ldb.FLAG_MOD_ADD, 'gPLink')

        try:
            self.samdb.modify(m)
        except Exception as e:
            raise CommandError("Error adding GPO Link to %s " % container_dn, e)
        return True

