# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
fonction associées à l'édition groupée
"""
import datetime
from os.path import isfile, join, isdir

from twisted.python import log

from creole.eosfunc import gen_random

from scribe import ldapconf
from scribe import linker
from scribe.eolegroup import Group
from scribe.eoletools import formate_date

from ead2.backend.actions import tools
from ead2.backend.actions.lib.widgets import form as F, main as M
from ead2.backend.actions.scribe.tool import scribe_tools, formencode
from ead2.backend.actions.scribe.tool.validate import test_password_length

def get_date():
    """
    renvoie la date courante
    """
    abc = datetime.date
    return "%d_%d" % (abc.today().day, abc.today().month)

def get_filename(rootname, user_rep):
    """
    calcule le nom du fichier des mots de passe
    """
    if not isfile(join(user_rep, rootname + '.csv')):
        return join(user_rep, rootname + '.csv')
    else:
        rootname = "%s_%%d.csv" % rootname
        indice = 0
        while True:
            if not isfile(join(user_rep, rootname % indice)):
                return join(user_rep, rootname % indice)
            indice += 1

def set_rights(filename, username):
    """ définit les acls sur le fichier filename pour l'utilisateur username
    """
    cmd = "setfacl -m u:%s:rw %s" % (username, filename)
    code, res = tools.command_statusoutput(cmd)
    if code:
        raise Exception, "Erreur d'application des droits sur le fichier : %s." % res
    return True

def copy_pwd_file(filename, username):
    """ copie le fichier filename dans le rep perso de username
    """
    user_rep = join('/home', username[0], username, 'perso')
    if isdir(user_rep) and isfile(filename):
        rootname = "mot-de-passe_%s" % get_date()
        user_file = get_filename(rootname, user_rep)
        code, res = tools.command_statusoutput("cp %s %s" % (filename,
                                                             user_file))
        if code:
            raise Exception, "Erreur de copie du fichier de mdp dans le répertoire perso de l'utilisateur."
        # application d'acls (on repasse un coup pour être sûr, normalement c'est déjà bon avec le cp)
        set_rights(user_file, username)
        return "La liste des nouveaux mot de passe se trouve dans votre répertoire perso : \\n              '%s'\\n" % user_file
    else:
        message = "Vous trouverez la liste des nouveaux mots de passe dans le répertoire %s." % filename
        message += "\\n NB: l'utilisateur %s n'a pas de répertoire personnel." % username
        return message

def write_pwd(login, pwd, rapport_file):
    """ ecrit un rapport à la modification groupée de mot de passe """
    try:
        rapport_file.write("%s;%s;\r\n" % (login, pwd))
    except:
        return False
    return True

def modify_password(login, rapport_file, passtype='name', passvalue=None,
                    force=False, connexion=None):
    """
        Modifie le mot de passe d'un utilisateur
        (utilisé pour le mode groupé)
        :param login: login de l'utilisateur
        :param rapport_file: fichier dans lequel écrire le mot de passe
        :param passtype: type de mot de passe à générer
        :param passvalue: mot de passe si passtype == 'main'
        :param force: indique si l'utilisateur devra changer son mot de passe
        :param connexion: connexion ldap à utiliser
    """
    log.msg(" * Modification du mot de passe de \"%s\"" % (login,))
    ldapuser = linker._user_factory(login, connexion.connexion)
    if passtype == 'name':
        prenom = ldapuser._get_attr(login, 'givenName')[0]
        nom = ldapuser._get_attr(login, 'sn')[0]
        pwd = '%s.%s' % (nom, prenom)
        pwd = pwd.replace(' ', '').lower()
        if force:
            ldapuser.c_mod_password(login, pwd, 1)
        else:
            ldapuser.c_mod_password(login, pwd, 0)
        write_pwd(login, pwd, rapport_file)
        ret = "Le mot de passe de %s est désormais '%s'.\\n" % (login, pwd)
    elif passtype == 'random':
        pwd = gen_random(8)
        if force:
            ldapuser.c_mod_password(login, pwd, 1)
        else:
            ldapuser.c_mod_password(login, pwd, 0)
        write_pwd(login, pwd, rapport_file)
        ret = "Le mot de passe de %s est désormais '%s'.\\n" % (login, pwd)
    elif passtype == 'date':
        date = ldapuser._get_attr(login, 'dateNaissance')
        if date == []:
            write_pwd(login, "error", rapport_file)
            return False, "Modification de mot de passe impossible pour %s,\\n sa date de naissance n'est pas renseignée.\\n" % login
        pwd = formate_date(date[0], sep='')
        if force:
            ldapuser.c_mod_password(login, pwd, 1)
        else:
            ldapuser.c_mod_password(login, pwd, 0)
        write_pwd(login, pwd, rapport_file)
        ret = "Le mot de passe de %s est désormais '%s'.\\n" % (login, pwd)
    elif passtype == 'main':
        pwd = passvalue
        # validation du mot de passe
        test_password_length(pwd)
        if force:
            ldapuser.c_mod_password(login, pwd, 1)
        else:
            ldapuser.c_mod_password(login, pwd, 0)
        write_pwd(login, pwd, rapport_file)
        ret = "Le mot de passe de %s est désormais '%s'.\\n" % (login, pwd)
    else:
        return False, "Erreur dans la modfication de mot de passe, type inconnu"
    """
    if force:
        ldapuser.password_must_change(login)
    """
    # retour global en cas de succès
    return True, ret

def get_none_users_btn():
    """ renvoie la description du bouton de déselection des utilisateurs de la liste """
    title = 'Aucun utilisateur'
    href = "javascript:formSelectNone('userlist')"
    return M.Bouton(href=href, libelle='Aucun', title=title, icone='')

def get_all_users_btn():
    """ renvoie la description du bouton de selection de
        tous les utilisateurs de la liste
    """
    href = "javascript:formSelectAll('userlist')"
    return M.Bouton(href=href,
                    icone='',
                    libelle='Tous')


#----------------- les formulaires ---------------------#

def get_toexec_links(actual_number, server_nb, name, role):
    """ renvoie les liens javascript à exécuter """
    if scribe_tools.is_admin(role):
        action_list = ('inscription', 'quota', 'domain', 'profil',
                       'password', 'shell', 'role')
    else:
        action_list = ('inscription', 'quota', 'domain', 'profil',
                       'password', 'shell')
    toexec = ''
    for indice, value in enumerate(action_list):
        if indice != actual_number:
            toexec += tools.make_js_link(server_nb, name,
                                     balise='div_%s_container' % value,
                                     confirm=True,
                                     btn='%d' % indice)
    return toexec

def get_inscription_form(server_nb, name, role):
    """
    renvoie la description du formulaire d'inscription à des groupes
    """
    group_icone = "/image/scribe/action_groupe_over.png"
    subscribe_to_group = F.Select(name='group_name')
    gr_types = ['Groupe', 'Option']
    for key in gr_types:
        group_names = Group().get_groups(key)
        group_names.sort()
        for grp in group_names:
            subscribe_to_group.add_option(grp, group=key)
    toexec = get_toexec_links(0, server_nb, name, role)
    texte = "Inscrire les utilisateurs sélectionnés au groupe :"
    form_names = ['inscription_edit', 'userlist']
    html_container = 'div_inscription_container'
    href = tools.make_form_link(server_nb, name, True, form_names,
                                balise=html_container)
    validate = M.Submit(href=href, title="Inscrire à ce groupe")
    return dict(texte=texte,
                group_icone=group_icone,
                subscribe_to_group=subscribe_to_group,
                toexec=toexec,
                validate=validate)

def get_quota_form(server_nb, name, role):
    """
    renvoie la description du formulaire de modification de quota
    """
    quota_icone = "/image/scribe/action_quota_over.png"
    quota_input = F.Input(name='quota',
                          libelle='Quota disque (en Mo)',
                          _type='number')
    toexec = get_toexec_links(1, server_nb, name, role)
    texte = "Attribuer un quota disque aux utilisateurs sélectionnés"
    form_names = ['quota_edit', 'userlist']
    html_container = 'div_quota_container'
    href = tools.make_form_link(server_nb, name, True, form_names,
                                balise=html_container)
    validate = M.Submit(href=href, title="Attribuer les quotas")
    return dict(texte=texte,
                quota_icone=quota_icone,
                quota_input=quota_input,
                toexec=toexec,
                validate=validate)

def get_domain_form(server_nb, name, role):
    """
    renvoie la description du formulaire
    de modification de domaine mail
    """
    domain_icone = "/image/scribe/action_liste_over.png"
    domain_select = F.Select(name='domain',
                             libelle='Domaine de messagerie')
    for domaine in ldapconf.MAIL_DOMAIN:
        domain_select.add_option(value=domaine,
                                 libelle="domaine %s" % domaine)
    toexec = get_toexec_links(2, server_nb, name, role)
    texte = "Changer le domaine mail des utilisateurs sélectionnés"
    form_names = ['domain_edit', 'userlist']
    html_container = "div_domain_container"
    href = tools.make_form_link(server_nb, name, True, form_names,
                                balise=html_container)
    validate = M.Submit(href=href, title="Changer le domaine mail")
    return dict(texte=texte,
                domain_icone=domain_icone,
                domain_select=domain_select,
                toexec=toexec,
                validate=validate)

def get_profil_form(server_nb, name, role):
    """ renvoie la description du formulaire d'édition de profil """
    profil_icone = "/image/scribe/action_profil_over.png"
    profil_select = F.Select(name='_sambaProfilePath',
                             libelle="Profil Windows")
    for indice, libelle in ldapconf.PROFILS:
        profil_select.add_option(indice, libelle=libelle)
    # profil obligatoire par défaut
    profil_select.default('2')
    toexec = get_toexec_links(3, server_nb, name, role)
    texte = "Changer le profil des utilisateurs sélectionnés"
    form_names = ['profil_edit', 'userlist']
    html_container = 'div_profil_container'
    href = tools.make_form_link(server_nb, name, True, form_names,
                                balise=html_container)
    validate = M.Submit(href=href, title="Changer le profil")
    return dict(texte=texte,
                profil_icone=profil_icone,
                profil_select=profil_select,
                toexec=toexec,
                validate=validate)

def get_shell_form(server_nb, name, role):
    """
    renvoie la description du formulaire de modification de shell
    """
    shell_icone = "/image/scribe/action_shell_over.png"
    shell_checkbox = F.Checkbox(name='shell',
                                libelle='Activer le shell',
                                inline=True,
                                checked=True)
    toexec = get_toexec_links(5, server_nb, name, role)
    texte = "Attribuer un shell aux utilisateurs sélectionnés"
    form_names = ['shell_edit', 'userlist']
    html_container = 'div_shell_container'
    href = tools.make_form_link(server_nb, name, True, form_names,
                                balise=html_container)
    validate = M.Submit(href=href, title="Changer le shell")
    return dict(texte=texte,
                shell_icone=shell_icone,
                shell_checkbox=shell_checkbox,
                toexec=toexec,
                validate=validate)

def get_password_form(server_nb, name, role):
    """ renvoie la description du formulaire d'édition de mot de passe """
    purge_btn = M.Bouton(href=tools.make_js_link(server_nb, name,
                                          purge=True,
                                          balise='div_password_container'),
                        libelle='Purger les fichiers de mot de passe',
                        icone='/image/suppr.gif',
                        _class='btn')
    pwd_icone = '/image/scribe/action_password_over.png'
    texte = 'Génération groupée de mot de passe'
    radio_date = F.Radio(name='password',
                         value='date',
                         libelle='Date de naissance (si renseignée)',
                         inline=True)
    radio_random = F.Radio(name='password',
                           value='random',
                           libelle='Mot de passe aléatoire',
                           inline=True)
    radio_name = F.Radio(name='password',
                         value='name',
                         libelle="De la forme 'nom.prenom'",
                         inline=True,
                         checked=True)
    radio_main = F.Radio(name='password',
                         value='main',
                         libelle="Le même mot de passe pour tous",
                         inline=True)
    input_password = F.Input(name='input_password', libelle="-->", inline=True)
    if scribe_tools.use_eclair():
        force_password = formencode.make_force_passwd_checkbox(checked=False)
    else:
        force_password = formencode.make_force_passwd_checkbox()

    toexec = get_toexec_links(4, server_nb, name, role)
    form_names = ['password_edit', 'userlist']
    html_container = 'div_password_container'
    href = tools.make_form_link(server_nb, name, True, form_names,
                                balise=html_container)
    validate = M.Submit(href=href, title="Générer de nouveau mot de passe")
    return dict(texte=texte,
                pwd_icone=pwd_icone,
                radio_date=radio_date,
                radio_random=radio_random,
                radio_name=radio_name,
                radio_main=radio_main,
                input_password=input_password,
                force_password=force_password,
                toexec=toexec,
                validate=validate,
                purge_btn=purge_btn)

