# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action scribe_user*
# Creation d'utilisateur
# Recherche puis modification d'utilisateur
# modfication de mot de passe
# Supression d'utilisateur
#
###########################################################################
"""Actions permettant de gérer le mot de passe utilisateur"""

from twisted.python import log
from scribe.eoleldap import Ldap
from scribe.linker import _user_factory
from scribe.eoletools import formate_date
from ead2.lib.error import MissingKey, MissingValue
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.backend.actions.lib.widgets import main as M, form as F, ajax
from ead2.backend.actions.scribe.tool import scribe_tools, formencode, validate
from ead2.backend.actions.scribe.tool.user import used_templates


class UserModPwd(Action):
    """
    Modification de son mot de passe
    par un utilisateur
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_user_password'
    libelle = "Mon mot de passe"
    category = None
    description = "Modification de mon mot de passe"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'user'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['mod_pwd'])

    def execute(self):
        """ exécute l'action et renvoie les données pour la mise en forme
            1 - renvoie le formulaire de modification de mot de passe
            2 - modifie le mot de passe
        """
        params, self.server_nb = tools.get_request(self.request)
        self.user = self.user_description['name']
        result = {}
        # params['user'] n'a plus d'utilité ici
        if self.form_result != {}:
            ## 1
            try:
                result['message'] = self._valid_form()
            except (MissingValue, MissingKey, Exception), mess:
                log.err("Erreur dans user.py 4: %s" % mess)
                log.err()
                result['message'] = mess.message
            self.form_result = {}

        ## 2
        #self.user = self.user_description['name']
        result.update(self._get_form('prof'))
        return self.send_all(result, template='scribe_users',
                             templates=used_templates)

    def _get_form(self, _type=None):
        """
        Renvoie la description du formulaire de modification de mot de passe
        """
        if _type == 'prof':
            ## connexion professeur ou prof admin
            ## -> on revérifie son authentification
            password = F.Input(name='old_password',
                               libelle="Ancien mot de passe",
                               inline=True, required=True)
            pwd_input = F.Input(name='password',
                                libelle="Nouveau mot de passe",
                                inline=True, required=True)
            re_pwd_input = F.Input(name='repassword',
                                libelle="Confirmation",
                                inline=True, required=True)

            return dict(icone="/image/scribe/action_pwd_over.png",
                        title="Modification de mot de passe pour %s" % self.user,
                        old_pwd=password,
                        pwd_input=pwd_input,
                        re_pwd_input=re_pwd_input,
                        validate=self._get_validate_btn(),)
        else:
            # changement forcé disponibles uniquement
            # pour les utilisateurs "Samba" (#683)
            disable_force = not _user_factory(self.user).has_samba
            checked = not scribe_tools.use_eclair()
            force_password = formencode.make_force_passwd_checkbox(disable_force, checked)
            radio_date = F.Radio(name='_type',
                                 value='birthdate',
                                 libelle='Depuis la date de naissance',
                                 inline=True,
                                 onchange="$('pwd_inputs_table').style.display='none'")
            radio_perso = F.Radio(name='_type',
                                  value='perso',
                                  libelle='Mot de passe personnalisé',
                                  inline=True,
                                  checked=True,
                                  onchange="$('pwd_inputs_table').style.display='block'")
            # attention : il faut mettre required=False
            # sinon, on ne peut pas valider avec la date de naissance !
            pwd_input = F.Input(name='password',
                                libelle="Nouveau mot de passe",
                                inline=True, required=False)
            re_pwd_input = F.Input(name='repassword',
                                libelle="Confirmation",
                                inline=True, required=False)

            return dict(title="Modification de mot de passe pour %s" % self.user,
                        pwd_input=pwd_input,
                        re_pwd_input=re_pwd_input,
                        radio_date=radio_date,
                        radio_perso=radio_perso,
                        force_password=force_password,
                        validate=self._get_validate_btn(),)


    def _get_validate_btn(self):
        """ renvoie la description du bouton valider """
        href = ajax.valid(self.server_nb,
                          self.name,
                          ['mod_pwd'],
                          container='mod_user_div_container',
                          user=self.user)
        title = "Modifier le mot de passe"
        return M.Submit(href=href, title=title)

    def _valid_form(self):
        """ valide la modification de mot de passe
        """
        # on deformate les données de formulaire
        if not self.form_result.has_key('mod_pwd'):
            raise MissingKey, "Erreur : Il manque des données pour la modification de mot de passe."
        resultat = tools.format_form_result(self.form_result['mod_pwd'])

        log.msg("dans valid_form")
        log.msg(resultat)
        #if resultat.has_key('user_name'): raise Exception, "Erreur: utilisateur inconnu (il manque des données)."
        user_name = self.user

        roles = self.user_description['role']
        prof = self.user_description['name']

        conn = Ldap()
        conn.connect()
        ldapuser = _user_factory(user_name, conn.connexion)

        # cas 1 : avec un radiobox pour le type de mdp : admin,
        #         ou  prof_admin modifie le mdp d'un de ses élèves
        if resultat.has_key('_type'):
            _type = resultat['_type']
            if _type == 'birthdate':
                # utilisation de la date de naissance
                date = ldapuser._get_attr(user_name, 'dateNaissance')
                if date == []:
                    raise Exception, "Erreur : la date de naissance de cet utilisateur (%s) n'est pas renseignée." % user_name
                new_pwd = formate_date(date[0], sep='')
                log.msg("pass de %s -> %s" % (user_name, new_pwd))
            else:
                # utilisation d'un mot de passe perso
                new_pwd = validate.test_confirm_passwd(resultat['password'],
                                                       resultat['repassword'])
            # on re-teste le cas du professeur responsbale
            if scribe_tools.is_prof_admin(roles) and \
                    not scribe_tools.is_eleve_from_prof(self.user, prof):
                raise Exception, "Erreur : action non autorisée."

        # cas 2 : cas d'un prof ou d'un prof admin qui modifie son mot de passe
        else:
            # si un prof change son mdp, il doit connaitre le précédent
            if not resultat.has_key('old_password'):
                raise MissingKey, "Erreur : Il manque des données pour la modification de votre mot de passe."
            if not ldapuser.auth(user_name, resultat['old_password']):
                raise MissingKey("Erreur d'authentification :\\n"
                                 + "vérifiez votre mot de passe.")
            new_pwd = validate.test_confirm_passwd(resultat['password'],
                                                   resultat['repassword'])
        """
        # application du changement de mot de passe
        ldapuser.c_mod_password(user_name, new_pwd)
        if resultat.has_key('force_password'):
            ldapuser.password_must_change(user_name)
        """
        # application du changement de mot de passe
        if resultat.has_key('force_password'):
            ldapuser.c_mod_password(user_name, new_pwd, 1)
        else:
            ldapuser.c_mod_password(user_name, new_pwd, 0)

        conn.close()
        return "Le mot de passe de %s a bien été modifié" % user_name


class UserModPwd2(UserModPwd):
    """
    Réutilisation de "scribe_user_password"
    pour "scribe_user_modpassword"
    """
    name = 'scribe_user_modpassword'
    libelle = "Mot de passe"
    category = None
    description = "Modification du mot de passe d'un utilisateur"

    def execute(self):
        """ exécute l'action et renvoie les données pour la mise en forme
            1 - récupère le nom du user dont le mot de passe doit être modifié
            2 - renvoie les données pour le formulaire de modification de mot de passe selon le role de l'appelant
                a - si prof modifie son propre mdp, on doit lui demander l'ancien
                b - si prof_admin modifie le mdp d'un élève, on doit vérifier les droits
                c - sinon c'est un admin qui demande
            3 - modifie le mot de passe
        """
        params, self.server_nb = tools.get_request(self.request)
        self.user = ''
        result = {}
        ## 1 -
        if params.has_key('user'):
            self.user = params['user'][0].strip()
        if self.form_result != {}:
            ## 3 -
            try:
                result['message'] = self._valid_form()
            except (MissingValue, MissingKey, Exception), mess:
                log.err("Erreur dans user.py 4: %s" % mess)
                log.err()
                result['message'] = mess.message
            self.form_result = {}

        # cas particulier : les professeurs responsables
        # sont limités à leurs élèves
        if scribe_tools.is_prof_admin(self.user_description['role']):
            prof = self.user_description['name']
            # => controle les droits
            if not scribe_tools.is_eleve_from_prof(self.user, prof) \
               and self.user != prof:
                # si on passe ici c'est qu'on essaye de truander
                log.msg('ALERT : tentative abusive de modification de mot de passe.')
                result.update(dict(message="Action non autorisée."))
                return self.send_all(result, template='scribe_users',
                                     templates=used_templates)
        result.update(self._get_form())
        return self.send_all(result, template='scribe_users',
                             templates=used_templates)
