# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# backend.py
#
# librairie pour la gestion des utilisateurs horus
#
###########################################################################

import ldap
from os.path import join, isfile, isdir
from os import makedirs, symlink, chmod, listdir
from creole import parsedico
from creole.config import containers_file
from pyeole.process import system_out
from pyeole.deprecation import deprecated
from fichier.acl import clear_acl, set_group_acl, copy_default_acl, \
set_group, set_owner
from fichier import models, quota, passwd

if isfile(containers_file):
    # chemins des containers
    dico_container = {}
    execfile(containers_file, {}, dico_container)
    if dico_container.has_key('container_path_fichier') and dico_container['container_path_fichier'] != "":
        CONTAINER_PATH_FICHIER = dico_container['container_path_fichier']
    else:
        CONTAINER_PATH_FICHIER = '/'
else:
    CONTAINER_PATH_FICHIER = '/'

dico = parsedico.parse_dico()
smb_serveur = dico["smb_netbios_name"]
branche = "ou=%s,ou=%s,ou=education,o=gouv,c=fr" % (dico["numero_etab"], dico["nom_academie"])
user_filters = "(objectclass=sambaSamAccount)(objectclass=inetOrgPerson)(!(description=Computer))"
group_filters = "(objectclass=sambaGroupMapping)(objectclass=posixGroup)"
station_filters = "(description=Computer)(objectclass=posixAccount)"
share_filters = "(objectClass=sambaFileShare)"

LDAPCONF = '/etc/eole/eoleldap.conf'
SKEL_REP = "/usr/share/horus/skel"
HOMEBASE = "/home"

ldap_server = ''
ldap_passwd = ''
if isfile(LDAPCONF):
    execfile(LDAPCONF)
else:
    raise Exception("Fichier %s non trouvé" % LDAPCONF)

#####################
## Primitives LDAP ##
#####################

def ldapsearch(filtre, attrlist=None):
    """
    recherche dans l'annuaire
    """
    query = ldap.open(ldap_server)
    #reload(backend_conf)
    query.simple_bind_s("cn=admin,o=gouv,c=fr", ldap_passwd)
    result = query.search_s("o=gouv,c=fr", ldap.SCOPE_SUBTREE, filtre, attrlist)
    query.unbind()
    return result


def ldapadd(dn, data):
    """
    ajout d'une entrée dans l'annuaire
    """
    query = ldap.open(ldap_server)
    #reload(backend_conf)
    query.simple_bind_s("cn=admin,o=gouv,c=fr", ldap_passwd)
    result = query.add_s(dn, data)
    query.unbind()
    return result


def ldapdelete(dn):
    """
    suppression d'une entrée de l'annuaire
    """
    query = ldap.open(ldap_server)
    #reload(backend_conf)
    query.simple_bind_s("cn=admin,o=gouv,c=fr", ldap_passwd)
    result = query.delete(dn)
    query.unbind()
    return result


def ldapmodify(dn, attribute, value):
    """
    modification d'une entrée de l'annuaire
    """
    if value == '':
        value = None
    query = ldap.open(ldap_server)
    #reload(backend_conf)
    query.simple_bind_s("cn=admin,o=gouv,c=fr", ldap_passwd)
    try:
        query.modify_s(dn, [(ldap.MOD_REPLACE, attribute, value)])
    except:
        try:
            query.modify_s(dn, [(ldap.MOD_ADD, attribute, value)])
        except:
            return False
    return True


def authenticate(user, password):
    """
    tentative de validation d'un couple login/mdp
    """
    if user == '' or password == '':
        return False
    query = ldap.open(ldap_server)
    try:
        query.simple_bind_s("uid=%s,ou=local,ou=personnels,ou=utilisateurs,%s" % (user, branche),
                            password)
        query.unbind()
        return True
    except ldap.INVALID_CREDENTIALS:
        return False


############################
## fonctions de recherche ##
############################

def is_user(user):
    """
    vérification de l'existance d'un utilisateur
    """
    result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), ['uid'])
    if result == []:
        return False
    else:
        return True


def is_system_user(user):
    """
    indique si le login proposé est déjà un utilisateur système
    """
    user = user.lower()
    passfile = open('/etc/passwd','r')
    passbuffer = passfile.readlines()
    passfile.close()
    for ligne in passbuffer:
        if user == ligne.split(':')[0]:
            return True
    return False


def get_gid(user):
    """
    gid de l'utilisateur
    """
    result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), ['gidNumber'])
    try:
        result = ldapsearch("(&%s(gidNumber=%s))" % (group_filters, result[0][1]['gidNumber'][0]), ['cn'])
        return result[0][1]['cn'][0]
    except:
        return ''


def get_user_sid(user):
    """
    SID de l'utilisateur
    """
    try:
        result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), ['sambaSID'])
        return result[0][1]['sambaSID'][0]
    except:
        return ''


def get_user_long_name(user):
    """
    Nom complet de l'utilisateur
    """
    try:
        result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), ['displayName'])
        return result[0][1]['displayName'][0]
    except:
        return ''


def get_attr(user, attr):
    """
    renvoi le contenu d'un attribut utilisateur
    """
    result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), [attr])
    try:
        return result[0][1][attr]
    except:
        return []


def get_users():
    """
    liste des utilisateurs
    """
    result = ldapsearch("(&%s)" % user_filters, ['uid'])
    users = []
    for user in result:
        users.append(user[1]['uid'][0])
    users.sort()
    return users


def get_user_groups(user):
    """
    liste des groupes d'un utilisateur
    """
    result = ldapsearch("(&%s(memberUid=%s))" % (group_filters, user), ['cn'])
    groups = []
    for group in result:
        groups.append(group[1]['cn'][0])
    groups.sort()
    return groups


def get_user_no_groups(user):
    """
    liste des groupes dans lesquels l'utilisateur est absent
    """
    result = ldapsearch("(&%s(!(memberUid=%s))(!(cn=DomainComputers)))" % (group_filters, user), ['cn'])
    groups = []
    for group in result:
        groups.append(group[1]['cn'][0])
    groups.sort()
    return groups


def get_user_data(user):
    """
    renvoit tout les attributs de l'utilisateur en vrac
    """
    return ldapsearch("(&%s(uid=%s))" % (user_filters, user))[0][1]


def get_active_users(active=True):
    """
    liste des utilisateurs actifs (True)
    ou inactifs (False)
    """
    result = ldapsearch("(&%s)" % (user_filters), ['uid', 'sambaAcctFlags'])
    users = []
    for user in result:
        if user[1]['sambaAcctFlags'][0].count('DU'):
            if not active:
                users.append(user[1]['uid'][0])
        elif active:
            users.append(user[1]['uid'][0])
    return users


def get_workstations():
    """
    liste des postes de travail
    """
    result = ldapsearch("(&%s)" % station_filters, ['uid'])
    stats = []
    for stat in result:
        # les postes se terminent par $
        stats.append(stat[1]['uid'][0].replace("$",""))
    stats.sort()
    return stats


def is_group(group):
    """ groupe existant ? """
    result = ldapsearch("(&%s(cn=%s))" % (group_filters, group), ['cn'])
    return result != []


def get_groups():
    """
    liste des groupes
    """
    result = ldapsearch("(&%s(!(cn=DomainComputers)))" % group_filters, ['cn'])
    groups = []
    for group in result:
        groups.append(group[1]['cn'][0])
    groups.sort()
    return groups


def get_members(group):
    """
    liste des membres d'un groupe
    """
    result = ldapsearch("(&%s(cn=%s))" % (group_filters, group), ['memberUid'])
    try:
        res = result[0][1]['memberUid']
        res.sort()
        return res
    except:
        return []


def get_group_shares(group):
    """
    liste des partages liés à un groupe
    """
    result = ldapsearch("(&%s(sambaShareGroup=%s))" % (share_filters, group), ['sambaShareName'])
    list_shares = []
    for share in result:
        list_shares.append(share[1]['sambaShareName'][0])
    return list_shares


def get_special_shares():
    """ récupération des partages spéciaux, i.e. : avec une lettre réservée """
    result = ldapsearch("(&%s(sambaShareDrive=*))" % share_filters, ['sambaShareName', 'sambaShareDrive'])
    special_shares = {}
    for share in result:
        if share[1]['sambaShareDrive'][0] != '':
            special_shares[share[1]['sambaShareName'][0]] = share[1]['sambaShareDrive'][0]
    return special_shares

def get_shares():
    """
    récupération des partages et des lettres de lecteur
    """
    result = ldapsearch("%s" % share_filters, ['sambaShareName', 'sambaShareDrive'])
    shares = {}
    for share in result:
        if share[1].has_key('sambaShareDrive') :
            shares[share[1]['sambaShareName'][0]] = share[1]['sambaShareDrive'][0]
        else:
            shares[share[1]['sambaShareName'][0]] = 'None'
    return shares

def get_group_sharedirs(group):
    """
    liste des répertoires partagés liés à un groupe
    """
    result = ldapsearch("(&%s(sambaShareGroup=%s))" % (share_filters, group), ['sambaShareName', 'sambaFilePath'])
    list_shares = []
    for share in result:
        list_shares.append([share[1]['sambaShareName'][0], share[1]['sambaFilePath'][0]])
    return list_shares

def get_share_template(sharename=None):
    """ renvoie le modele du partage sharename """
    result = ldapsearch("(&(sambaShareName=%s)%s)" % (sharename, share_filters), ['sambaShareName', 'sambaShareModel'])
    try:
        return result[0][1]['sambaShareModel'][0]
    except:
        return ''

def get_share_group(sharename):
    """ renvoie le groupe associé au partage sharename """
    result = ldapsearch("(&(sambaShareName=%s)%s)" % (sharename, share_filters), ['sambaShareGroup'])
    try:
        return result[0][1]['sambaShareGroup'][0]
    except:
        return ''

@deprecated
def get_share_templates():
    """
    retourne la liste des modèles de partage disponibles
    """
    return models.get_share_models()

def get_share_filepath(sharename):
    """ renvoie le chemin du partage sharename """
    result = ldapsearch("(&(sambaShareName=%s)%s)" % (sharename, share_filters), ['sambaFilePath'])
    try:
        return result[0][1]['sambaFilePath'][0]
    except:
        return ''

@deprecated
def get_quota(login):
    """
        renvoie le quota pour login
    """
    return quota.get_quota(login)


#######################
## fonctions d'ajout ##
#######################


def inscription(user, group):
    """
    inscription d'un utilisateur dans un groupe
    """
    cmd = ['/usr/sbin/smbldap-groupmod', '-m', user, group]
    ret = system_out(cmd, container='fichier')
    if ret[0] != 0:
        raise Exception(' '.join(ret[1:]))
    gen_ftpdir(user)
    return True


def add_user(login, group, groups=[], drive='U:', profile=False, shell=False):
    """
    ajout d'un utilisateur
    @param login  : login de l'utilisateur
    @param group  : groupe principal de l'utilisateur
    @param groups : autres groupes de l'utilisateur
    @param drive  : lettre de lecteur pour le montage du répertoire personnel
    @param shell : activation du shell
    """
    home = join(HOMEBASE, login)
    smbhome = "\\\\%s\\%s\\perso" % (smb_serveur, login)
    script = "%s.bat" % login
    acctflags = "[U]"

    if not profile:
        opt_profile = ""
    elif profile == "obligatoire":
        opt_profile = "\\\\%s\\netlogon\\profil" % smb_serveur
    else:
        opt_profile = "\\\\%s\\%s\\profil" % (smb_serveur, login)

    if shell:
        opt_shell = "/bin/bash"
    else:
        opt_shell = "/bin/false"

    if 'DomainUsers' not in groups :
        # bug ? and group != 'DomainUsers':
        # samba3 : tout le monde dans DomainUsers
        groups.append('DomainUsers')

    # gestion des groupes supplémentaires
    grouplist = ''
    for gro in groups:
        if grouplist == '':
            grouplist = '%s' % gro
        else:
            grouplist += ',%s' % gro

    #res = system("/usr/sbin/smbldap-useradd -a -A 1 -m -k '"+SKEL_REP+"' -d '"+home+"' -D '"+drive+"' -F '"+opt_profile+"' -C '"+smbhome+"' -E '"+script+"' -g '"+group+"' "+grouplist+" -H '"+acctflags+"' -s '"+opt_shell+"' -c '"+login+"' "+login)

    cmd = ['/usr/sbin/smbldap-useradd', '-o', 'ou=local,ou=personnels',
           '-a', '-A', '1', '-m', '-k', SKEL_REP,
           '-d', home, '-D', drive, '-F', opt_profile, '-C', smbhome,
           '-E', script, '-g', group, '-G', grouplist, '-H', acctflags,
           '-s', opt_shell, '-c', login, login]
    ret = system_out(cmd, container='fichier')
    if ret[0] != 0:
        raise Exception(' '.join(ret[1:]))
    gen_ftpdir(login)
    return True


def add_workstation(workstation):
    """
    ajoute une station de travail
    """
    # FIXME: ne devrait pas être utilisé !
    cmd = ['/usr/sbin/smbldap-useradd', '-w', workstation]
    ret = system_out(cmd, container='fichier')
    if ret[0] != 0:
        raise Exception(' '.join(ret[1:]))
    return True


def add_group(group):
    """
    ajout d'un groupe
    """
    cmd = ['/usr/sbin/smbldap-groupadd', '-a', group]
    ret = system_out(cmd, container='fichier')
    if ret[0] != 0:
        raise Exception(' '.join(ret[1:]))
    return True


def add_share(sharename, groupe, filepath=None, drive=None, sticky=False, model="standard", sync=True):
    """
    ajout d'un partage samba associé à un groupe donné
    """
    if filepath is None:
        filepath = "/home/workgroups/"+sharename
    if not is_group(groupe):
        return False
    # construction des données ldap
    dn = "cn=smb://%s/%s,ou=local,ou=partages,%s" % (smb_serveur, sharename, branche)
    data = [("objectClass","sambaFileShare"),
    ( "cn","smb://"+smb_serveur+"/"+sharename),
    ("sambaShareName", sharename),
    ("description", sharename),
    ("sambaShareGroup", groupe),
    ("sambaFilePath", filepath),
    ("sambaShareURI", "\\\\%s\\%s" % (smb_serveur, sharename)),
    ("sambaShareModel", model),
    ]
    if sticky:
        data.append(("sambaShareOptions", "sticky"))
    if drive is not None:
        if not drive.endswith(":"):
            drive += ':'
        if len(drive) != 2:
            return False
        data.append(("sambaShareDrive", drive))
    try:
        ldapadd(dn, data)
    except:
        return False
    # création du répertoire
    if not filepath.startswith('%'):
        if not isdir(filepath):
            makedirs(filepath)
        clear_acl(filepath)
        if sticky:
            set_group(filepath, groupe)
            # sticky bit sur le user (#2325)
            chmod(filepath, 01770)
        set_group_acl(filepath, groupe, 'rwx')
        copy_default_acl(filepath)
    if sync:
        synchronize()
    for user in get_members(groupe):
        gen_ftpdir(user)
    return True


###############################
## fonctions de modification ##
###############################

def mod_user(login, homedrive=None, enable=None, profile=None, gid=None, shell=None):
    """
    modification d'un utilisateur existant.
    """
    # les élément à ne pas modifier sont à None
    args = []
    if homedrive is not None:
        args.extend(['-D', homedrive])
    if profile is not None:
        if not profile:
            opt_profile = ""
        elif profile == "obligatoire":
            opt_profile = "\\\\%s\\netlogon\\profil" % smb_serveur
        else:
            opt_profile = "\\\\%s\\%s\\profil" % (smb_serveur, login)
        args.extend(['-F', opt_profile])
    if shell is not None:
        if shell:
            args.extend(['-s', '/bin/bash'])
        else:
            args.extend(['-s', '/bin/false'])
    if enable is not None:
        if enable:
            args.append("-J")
        else:
            args.append("-I")
    if gid is not None:
        args.extend(['-g', gid])
    if args != []:
        cmd = ['/usr/sbin/smbldap-usermod']
        cmd.extend(args)
        cmd.append(login)
        ret = system_out(cmd, container='fichier')
        if ret[0] != 0:
            raise Exception(' '.join(ret[1:]))
        if gid is not None:
            # le changement de groupe principal ne met pas à jour l'appartenance au groupe
            inscription(login, gid)
    return True


def mod_password(login, user_passwd):
    """ changement d'un mot de passe """
    return passwd.mod_password(login, user_passwd)

def password_must_change(login):
    """ changement d'un mot de passe """
    return passwd.password_must_change(login)

def set_active_users(users, active=True):
    """
    active/désactive les utilisateurs passés en paramètre
    """
    if type(users) != list:
        users = [users]
    if active:
        value = '[U]'
    else:
        value = '[DU]'
    for user in users:
        try:
            set_attr(user, 'sambaAcctFlags', value)
        except:
            return False
    return True


def set_attr(user, attr, content):
    """
    modifie un attribut utilisateur
    """
    try:
        result = ldapsearch("(&%s(uid=%s))" % (user_filters, user), [''])
        ldapmodify(result[0][0], attr, content)
        return True
    except:
        return False


def mod_share(share, drive=None, model=None):
    """
    modification d'un partage (lettre ou modèle samba)
    """
    result = ldapsearch("(&%s(sambaShareName=%s))" % (share_filters, share))
    if drive != None:
        drive = drive.upper()
        ldapmodify(result[0][0], 'sambaShareDrive', drive)
    if model != None:
        ldapmodify(result[0][0], 'sambaShareModel', model)
        synchronize()

@deprecated
def set_quota(login, user_quota):
    """
    définit le quota 'quota' pour l'utilisateur login
    """
    return quota.set_quota(login, user_quota)

##############################
## fonctions de suppression ##
##############################

def desinscription(user, group):
    """
    désinscription d'un utilisateur d'un groupe
    """
    cmd = ['/usr/sbin/smbldap-groupmod', '-x', user, group]
    ret = system_out(cmd, container='fichier')
    if ret[0] != 0:
        raise Exception(' '.join(ret[1:]))
    gen_ftpdir(user)
    return True


def del_user(user, remove_data=False):
    """
    suppression d'un utilisateur avec ou sans son répertoire personnel
    """
    cmd = ['/usr/sbin/smbldap-userdel']
    if remove_data:
        cmd.extend(['-r', user])
    else:
        cmd.append(user)
    ret = system_out(cmd, container='fichier')
    if ret[0] != 0:
        raise Exception(' '.join(ret[1:]))
    return True


def del_workstation(workstation):
    """
    suppression d'une station de travail
    """
    if not workstation.endswith('$'):
        workstation = workstation+'$'
    cmd = ['/usr/sbin/smbldap-userdel', workstation]
    ret = system_out(cmd, container='fichier')
    if ret[0] != 0:
        raise Exception(' '.join(ret[1:]))
    return True


def del_group(group):
    """
    suppression d'un groupe
    Attention il faut au préalable avoir fait les vérifications d'usage :
       - il reste des membres
       - il reste des partages liés
    """
    res2 = 1
    shares = get_group_shares(group)
    cmd = ['/usr/sbin/smbldap-groupdel', group]
    ret = system_out(cmd, container='fichier')
    if ret[0] != 0:
        raise Exception(' '.join(ret[1:]))
    # suppression des partages associés (1 partage => 1 groupe)
    if shares:
        for share in shares:
            res2 = del_share(share, sync=False, rmdir=False)
        synchronize()
    return res2


def del_share(share, sync=True, rmdir=False):
    """
    suppression d'un partage
    """
    result = ldapsearch("(&%s(sambaShareName=%s))" % (share_filters, share), ['sambaFilePath'])
    try:
        if rmdir:
            sharedir = result[0][1]['sambaFilePath'][0]
            system_out(['rm', '-rf', sharedir])
        ldapdelete(result[0][0])
    except:
        return False
    if sync:
        synchronize()
    return True


##########################
## gestion des fichiers ##
##########################
def use_esu():
    """
    utilisation d'Esu
    @return True/False
    """
    return dico['activer_esu'] == 'oui' and 'esuclnt.exe' in [ i.lower() for i in listdir('/home/esu/Base/') ]

def gen_ftpdir(user):
    """
    Gestion du répertoire special ".ftp"
    """
    homedir = get_attr(user, 'homeDirectory')[0].strip()
    ftpdir = join(homedir, '.ftp')
    system_out(['rm', '-rf', ftpdir])
    makedirs(ftpdir)
    set_owner(ftpdir, user)
    clear_acl(ftpdir)
    chmod(ftpdir, 0500)
    symlink(join(homedir, 'perso'), join(ftpdir, 'perso'))
    user_groups = get_user_groups(user)
    for group in user_groups:
        for share, sharedir in get_group_sharedirs(group):
            if share not in ['icones$', 'groupes']:
                symlink(sharedir, join(ftpdir, share))

def synchronize(restart=True):
    """
    ajout des partages dans le fichier smb.conf
    """
    shares = []
    result = ldapsearch(share_filters)
    for share in result:
        dico_share = {}
        dico_share['name'] = share[1]['sambaShareName'][0]
        dico_share['path'] = share[1]['sambaFilePath'][0]
        dico_share['group'] = share[1]['sambaShareGroup'][0]
        dico_share['desc'] = share[1]['description'][0]
        try:
            dico_share['model'] = share[1]['sambaShareModel'][0]
        except:
            pass
        shares.append(dico_share)
    models.synchronize(shares, restart)

