# -*- coding: UTF-8 -*-
#
# Module de gestion des blocages :
# utilisateurs(réseau + partage)/machines(réseau seulement)
# Applique la restriction réseau la plus haute
#
# Les informations sont stockées dans la RAM,
# donc ré-initialisé si le service controle-vnc est redémarré
#
#

from twisted.internet import reactor, defer
#from twisted.internet.task import LoopingCall
#from twisted.python import log
import sys
sys.path.append('/usr/share/eole/client')

import machines
from connexions import Connexions
from cliscribe import Cliscribe
import ldap_utils

# Liste des ips à ne pas bloquer (pas un client)
# Amon + Auth NTLM = connexion de l'utilisateur depuis le parefeu
# "net status session" => elevetemp2008  eleves        pf-amon      (10.121.11.1)
# from creole import parsedico
# IPS_NOUSER = [parsedico.parse_dico()['adresse_ip_gw']]


# Constantes
# modes de blocage réseau/partages
BLOCAGES = {'reseau' : { 0 : u'Aucun',
                         1 : u'Tout internet'},
            'partages' : { 0: u'Par défaut',
                           1: u'Par défaut + partage "devoirs" (Y:)',
                           2: u'Aucun lecteur',
                           3: u'Seulement le partage "devoirs" (Y:)'}
               }
# partmod = (mode, [liste de lettre à ne pas masquer]/None)
BLOCAGE_DEFAUT = {'netmod' : 0 , 'partmod' : (0, None)}
# commande à exécuter sur la station (voir servscribe_class.py => remote_fw)
NETCMD = {0: 'SETMODE::block;;allow',
          1: 'SETMODE::block;;block'}


# Item
class Bloc:
    """Classe représentant un item (user/computer) à bloquer
    gère le déblocage (lorsque la durée est écoulée)
    """
    def __init__(self, name):
        self.name = name.lower()
        self.netmod = BLOCAGE_DEFAUT['netmod']
        self.deferred = None

    def cancel(self):
        """Annule le deferred du déblocage
        """
        try: self.deferred.cancel()
        except: pass

    def debloc(self):
        """initialise au blocage par défaut
        """
        self.netmod = BLOCAGE_DEFAUT['netmod']
        self.cancel()

    def endbloc(self):
        """Appelé lorsque la durée est écoulée
        """
        self.debloc()
        self.callback(self.name, debloc=True)

    def set_blocage(self, netmod, duree, callback):
        """renseigne un item et son blocage,
        crée le callback de fin de blocage
        """
        self.netmod = int(netmod)
        self.callback = callback
        self.cancel()
        duree = int(duree)*60 # de minutes vers secondes
        self.deferred = reactor.callLater(duree, self.endbloc)

    def apply_net(self, ip):
        """Appliquer le blocage réseau de l'item
        """
        if not machines.isup(ip): return defer.succeed(True)
        else: return Cliscribe(ip).fw(cmd=NETCMD[self.netmod])

# Item user
class User(Bloc):
    """Classe représentant un utilisateur
    """
    def __init__(self, name):
        """Ajout de la notion de partage à l'utilisateur
        """
        Bloc.__init__(self, name)
        self.partmod = BLOCAGE_DEFAUT['partmod']
        self.ip_list = []

    def debloc(self):
        """initialisation du mode de partage
        """
        self.partmod = BLOCAGE_DEFAUT['partmod']
        Bloc.debloc(self)

    def set_blocage(self, netmod, partmod, duree, callback):
        self.partmod = (int(partmod[0]), partmod[1])
        Bloc.set_blocage(self, netmod, duree, callback)

    def get_ip_list(self):
        """renvoie la liste des ips où est connectées l'utilisateur
        et recharge self.ip_list, permet de savoir si l'utilisateur est connecté
        """
        if not Connexions().is_connecte(self.name):
            self.ip_list = []
        else:
            self.ip_list = Connexions().get_ip(self.name)
        return self.ip_list

    def apply_partage(self, ip, logon=None):
        """Applique le blocage du partage
        """
        if not machines.isup(ip): return defer.succeed(True)
        sid = ldap_utils.get_user_sid(self.name)
        return Cliscribe(ip).bloc(partmod=self.partmod, sid=sid, logon=logon)

class Users:
    """Class représentant les utilisateurs
    """
    def __init__(self):
        self.users = {}

    # récupération d'informations
    def get_user(self, name):
        """renvoie l'objet utilisateur correspondant à <name>
        """
        try: return self.users[name]
        except: return User(name)

    def get_net_mod(self, name):
        return self.get_user(name).netmod

    def get_part_mod(self, name):
        return self.get_user(name).partmod

    def get_blocage(self, name):
        return {'netmod': self.get_net_mod(name),
                'partmod': self.get_part_mod(name)}

    def bloc_user(self, name, netmod, partmod, duree, callback):
        if self.users.has_key(name):
            u = self.users[name]
        else: u = User(name)
        u.set_blocage(netmod, partmod, duree, callback)
        self.users[name] = u

    def debloc_user(self, name):
        user = self.users.pop(name, User(name))
        user.debloc()
        return user

# Item computer
class Computer(Bloc):
    """classe représentant une machine avec son type de blocage
    """
    def __init__(self, name):
        Bloc.__init__(self, name)

    def get_ip(self):
        self.ip = machines.get_ip(self.name)
        return self.ip

    def get_username(self):
        """renvoie le login connecté ou rien
        """
        self.get_ip()
        if not Connexions().has_session(self.ip): return
        return Connexions().get_user(self.ip)

    def apply_net(self):
        """Une machine a une ip, appel simple
        """
        return Bloc.apply_net(self, self.get_ip())

# Classes de regroupement
class Computers:
    """Classe représentant les machines
    """
    def __init__(self):
        self.computers = {}

    # récupération d'information
    def get_computer_by_ip(self, ip):
        """renvoie l'objet computer correspondant à son ip <ip>
        """
        name = machines.get_nom(ip)
        if not name: return
        return self.get_computer(name)

    def get_computer(self, name):
        """renvoie l'objet computer correspondant à son nom <name>
        """
        try: return self.computers[name]
        except: return Computer(name)

    def get_net_mod(self, name):
        """renvoie le type de blocage réseau pour la machine <name>
        """
        return self.get_computer(name).netmod

    def get_blocage(self, name):
        """renvoie un dictionnaire de blocage
        """
        return {'netmod': self.get_net_mod(name)}

    def bloc_computer(self, name, netmod, duree, callback):
        """Application du blocage
        """
        if self.computers.has_key(name):
            c = self.computers[name]
        else: c = Computer(name)
        c.set_blocage(netmod, duree, callback)
        self.computers[name] = c

    def debloc_computer(self, name):
        """Suppression du blocage
        """
        computer = self.computers.pop(name, Computer(name))
        computer.debloc()
        return computer

class Blocage:
    """Classe principale. Représente les utilisateurs et les machines
    lorsqu'un objet est bloqué, son blocage est renseigné et appliqué.
    * L'application choisi la restriction réseau la plus forte entre
    la machine et l'utilisateur connecté dessus
    """
    def __init__(self):
        self.users =  Users()
        self.computers = Computers()
#        a = LoopingCall(self.print_users)
#        a.start(10)
#
#    def print_users(self):
#        """Fonction de test
#        """
#        print self.users.users, self.computers.computers
#        for u  in  self.users.users.values():
#            print u.name, u.netmod, u.partmod, u.deferred
#        for c in self.computers.computers.values():
#            print c.name, c.netmod, c.deferred

    def get_bloc_list(self):
        """Renvoie la liste des blocages disponibles
        """
        return BLOCAGES

    # Machines
    def get_computer_mod(self, name):
        """renvoie le dictionnaire de blocage pour une machine
        """
        return self.computers.get_blocage(name)

    def get_computers_mod(self):
        """renvoie un dictionnaire de dictionnaire de blocage par machine
        { 'machine1': {'reseau': 0}, 'machine2': {'reseau': 1}, ... }
        """
        dico = dict()
        for name in machines.get_machines():
            dico[name] = self.get_computer_mod(name)
        return dico

    def bloc_computer(self, name, netmod, duree):
        """bloque une machine, et applique si la machine est démarrée
        netmod = 0 annule le blocage
        """
        self.computers.bloc_computer(name, netmod, duree, self.apply_from_machine)
        if netmod == 0:
            debloc = True
        else:
            debloc = None
        return self.apply_from_machine(name, debloc=debloc)

    def bloc_computers(self, computers):
        """bloque une liste de machines, durée en minutes
        (('machines1', 1, 15), ('machines2', 1, 15), ... )
        """
        for computer, netmod, duree in computers:
            self.bloc_computer(computer, netmod, duree)

    # Utilisateurs
    def get_user_mod(self, name):
        """dictionnaire blocage utilisateur
        {'partage':(2, None), 'reseau': 1}
        """
        return self.users.get_blocage(name)

    def get_users_mod(self, groupe, user):
        """dictionnaire de dictionnaire de blocage par utilisateur
        {'user1': {'reseau: 1, 'partage': (3, ['d','f'])}, 'user2': {...}, ...}
        """
        users = ldap_utils.get_membres_croise([groupe, 'eleves'], user=user)
        dico = dict()
        for name in users:
            dico[name] = self.get_user_mod(name)
        return dico

    def bloc_user(self, name, netmod, partmod, duree):
        """Bloque un utilisateur et applique le blocage sur la/les machine(s)
        où il est connecté
        """
        self.users.bloc_user(name, netmod, partmod, duree, self.apply_from_user)
        if netmod == 0 and partmod[0] == 0:
            debloc = True
        else:
            debloc = None
        return self.apply_from_user(name, debloc=debloc)

    def bloc_users(self, users):
        """Bloque une liste d'utilisateurs, durée en minutes
        (('user1', 1, (4, None), 15)
        """
        for user, netmod, partmod, duree in users:
            self.bloc_user(user, netmod, partmod, duree)

    # Application
    def apply_from_user(self, name, ip=None, logon=None, debloc=None):
        """Applique le blocage à un utilisateur,
        "ip" peut être renseigné lorsqu'utilisé dans gest_sessions.py
        pour appliquer le blocage à l'ouverture de session
        """
        # debloc=True si appelée en fin de blocage pour appliquer
        # le retour à l'état "Par défaut"
        if debloc: user = self.users.debloc_user(name)
        else: user = self.users.get_user(name)
        if ip: user.ip_list = [ip]
        # l'utilisateur n'est pas connecté
        elif not user.get_ip_list(): return
        # première machine
        computer = self.computers.get_computer_by_ip(user.ip_list[0])
        d = user.apply_partage(user.ip_list[0], logon)
        d.addCallback(lambda _, c=computer: self.apply_highest_net(user, c))
        # les suivantes
        for ip in user.ip_list[1:]:
            computer = self.computers.get_computer_by_ip(ip)
            # un callback s'exécute "plus tard" (t+1)
            # addCallback à l'instant "t" => ip=v1 et i=v1
            # à l'exécution du callback "plus tard" (t+1) => ip=vX mais i=v1
            d.addCallback(lambda _, i=ip: user.apply_partage(i))
            d.addCallback(lambda _, c=computer: self.apply_highest_net(user, c))
        return d

    def apply_highest_net(self, user, computer):
        """applique le blocage réseau le plus fort entre l'utilisateur
        et la machine sur laquelle il est connecté
        """
        if not user: return
        if not computer: return
        ip = computer.get_ip()
        if user.netmod > computer.netmod:
            return user.apply_net(ip)
        else:
            return computer.apply_net()

    def apply_from_machine(self, machine, debloc=None):
        """Applique le blocage à une machine
        appelé aussi comme callback (debloc=True)
        """
        if debloc: computer = self.computers.debloc_computer(machine)
        else: computer = self.computers.get_computer(machine)
        username = computer.get_username()
        if not username:
            return computer.apply_net()
        else:
            user = self.users.get_user(username)
            return self.apply_highest_net(user, computer)

