# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# connectes.py
#
# Librairie de gestion des connectés
#
###########################################################################

import sys, commands, gzip, os
sys.path.append('/usr/share/eole/client')

class Connexions:
    def __init__(self):
        self.__init__connexions()
        self.__init__osdict()
        self.__init__sessions()

    def __init__connexions(self):
        """Charge une liste contenant l'historique des connexions, le premier élément
        est la connexion la plus récente
        # ['admin', 'professeurs', 'vm-xp3-sp2', 'WinXP', '172.16.0.185']
        """
        fh = file('/var/log/samba/connexions.log')
        self.connexions = [ i.split()[6:] for i in fh.read().splitlines()]
        fh.close()
        self.connexions.reverse() # les dernières connexions au début
        if os.path.isfile('/var/log/samba/connexions.log.1.gz'):
            fh = gzip.GzipFile('/var/log/samba/connexions.log.1.gz')
            liste2 = [ i.split()[6:] for i in fh.read().splitlines()]
            fh.close()
            liste2.reverse()
            self.connexions.extend(liste2)
        self.connexions = tuple(self.connexions)

    def __init__osdict(self):
        """Charge un dictionnaire { <Ip> : <OsType> }
        """
        self.os_dict = {}
        for i in self.connexions:
            try: os, ip = i[3:5]
            except: continue
            if ip not in self.os_dict.keys():
                self.os_dict[ip] = os

    def __init__sessions(self):
        """Récupère la liste des sessions et fait le ménage
        pour renvoyer ((user1, grp1, machine1, os, ip), (user2, grp2, ...))
        """
        cmd = "net status sessions parseable"
        liste = []
        for i in commands.getoutput(cmd).splitlines():
            try: user, group, machine, ip = i.split('\\')[1:]
            except: continue
            try: os = self.os_dict[ip]
            except: os = ''
            liste.append((user, group, machine, os, ip))
        #Il arrive que 'net status sessions' renvoie plusieurs utilisateurs
        #sur la même machine...
        ips = [ i[4] for i in liste ]
        res = []
        for username, prigrp, nom_mach, os, ip in liste:
            if ips.count(ip) == 1: # Aucun doublon
                res.append((username, prigrp, nom_mach, os, ip))
            elif ip not in [ i[4] for i in res]: # 2 connectés sur la mm IP
                user = self.get_last_connecte(ip)
                # Personne ne s'est jamais connecté sur IP
                # ou c'est lui le dernier
                if not user or user == username:
                    res.append((username, prigrp, nom_mach, os, ip))
        self.sessions = tuple(res)

    def get_last_connecte(self, ip):
        """Renvoie le dernier username connecte sur ip (ou None)
        """
        for ligne in self.connexions:
            if ip == ligne[4]:return ligne[0]

    ## des informations depuis une IP ##
    def get_user_from_ip(self, ip):
        """renvoie le dernier <login> a s'être connecté sur <ip>
        à l'ouverture de session, le client dé-reconnecte
        au mauvais moment 'net status sessions' ne renvoie rien pour l'IP
        donc utilisation de l'historique
        """
        for username, prigrp, nom_mach, os, ip_adr in self.sessions:
            if ip_adr == ip: return True, (username, prigrp, nom_mach, os, ip)
        for i in self.connexions:
            username, prigrp, nom_mach, os, ip_adr = i[:5]
            if ip_adr == ip: return False, (username, prigrp, nom_mach, os, ip)
        raise Exception("Personne ne s'est connecte sur %s"%ip)

    def get_active_user(self, ip):
        user = self.get_user_from_ip(ip)
        if user[0]: return user[1][0]

    def get_user(self, ip):
        """retourne le username de l'utilisateur connecte sur "ip"
        """
        return self.get_user_from_ip(ip)[1][0]

    def has_session(self, ip):
        return ip in [ i[4] for i in self.sessions ]

    ## des informations depuis une LOGIN ##
    def get_ip_from_user(self, user):
        """renvoie l(es) element(s) dont USER==user de la liste get_connectes
        pid, username, prigrp, nom_mach, ip, os
        [('admin', 'professeurs', 'vm-xp1', 'WinXP', '172.16.0.157')]
        """
        res = []
        for username, prigrp, nom_mach, os, ip in self.sessions:
            if user == username: res.append((username, prigrp, nom_mach, os, ip))
        return res

    def get_ip(self, user):
        """retourne l'ip de la (des) station(s) ou est connecte <user>
        """
        res = []
        for i in self.get_ip_from_user(user):
            # n'est utilisé que dans blocage.py qui n'est pas compatible avec Win9x
            if i[3] != 'Win95': res.append(i[4])
        return res
