import re
import filecmp
import random
import tempfile
import shutil

from os import remove, close, write, chmod, chown, path
from pwd import getpwnam
from grp import getgrnam


class EolePassword(object):
    """ Master class for password handling in Eole

    This Class offer the common methods and attributes for all password
    manipulation classes.

    An EolePassword is represented by :
        - a username :attr:`self.user`
        - an old password :attr:`self.old_pass`
        - an new clear password :attr:`self.ncl_pass`
        - an new crypted password :attr:`self.ncr_pass`

    Do not use this class directly use the interface :class:`Password`

    """

    alpha = "abcdefghijklmnopqrstuvwxyz0123456789"

    def __init__(self, name):
        self.user = name
        self.old_pass = ""
        self.ncl_pass = ""
        self.ncr_pass = ""

    def gen_salt(self, length=8):
        """ Generate a salt of passed length

        The generated salt is composed of alphanumeric characters.
        It picks random character in an alphanumeric list

        :param length: The salt length (default 8)
        :type length: Integer
        """

        salt = []
        for elm in range(length):
            salt.append(random.choice(EolePassword.alpha))
        return "".join(salt)

    def gen_rand_pass(self):
        """ Generate a random password

        The generated a random password composed of alphanumeric and special
        characters.

        :return: Random password
        :rtype: string
        """
        passwd_chars = EolePassword.alpha  # + ';@,><-=+?'
        passwd_length = random.sample([10, 11, 12], 1).pop()
        return ''.join(random.sample(passwd_chars, passwd_length))

    def update_conf_file(self, fpath, pattern, endpat=None, mod=None, owner=None, bck=None):
        """ Replace a password on given file
        :param file: The file name
        :type file: string (full fpath)
        :param pattern: The pattern to match
        :type pattern: regexp pattern
        :param mod: Mod of file (as in chmod)
        :type mod:  string
        :param owner: owner of file (as in chown)
        :type owner:  string
        :param bck: Backup directory
        :type bck: string
        """

        try:
            close_chars = [';']
            conf_fd = open(fpath, 'r')
            tmpfd, tmpfpath = tempfile.mkstemp()
            pcl = ""

            if pattern[-1] == '"' or pattern[-1] == "'":
                pcl = pattern[-1]

            pat = ('(?<=%s).[\w,\W]\S*' % re.escape(pattern))
            for line in conf_fd.readlines():
                if line.startswith(pattern):
                    m = re.search(r'\W$', line)
                    if m is not None:
                        if m.group() in close_chars:
                            end = m.group()
                        else:
                            end = ""
                    if endpat:
                        end = endpat

                    line = line.rstrip()
                    line = re.sub(pat,
                                  '{0}{1}{2}\n'.format(self.ncl_pass,
                                                       pcl,
                                                       end),
                                  line)
                write(tmpfd, line.encode())
            conf_fd.close

            if not filecmp.cmp(fpath, tmpfpath):
                if bck:
                    fname = fpath.replace('/','_')
                    if fname.startswith('_'):
                        fname = fname[1:]
                    dest = path.join(bck, fname)
                else:
                    dest = fpath
                shutil.copy2(fpath, dest + ".bck")
                shutil.copy2(tmpfpath, fpath)
            if mod:
                try:
                    chmod(fpath, int(mod, 8))
                except:
                    msg = "Error changing mode for {0}".format(fpath)
                    raise Exception(msg)
            if owner:
                try:
                    owner = owner.split(':')
                    uid = getpwnam(owner[0]).pw_uid
                    if len(owner) == 1:
                        gid = 0
                    else:
                        gid = getgrnam(owner[1]).gr_gid

                    chown(fpath, uid, gid)
                except:
                    msg = "Error changing owner for {0}".format(fpath)
                    raise Exception(msg)

            close(tmpfd)
            remove(tmpfpath)
        except IOError as err:
            print(("WARN {0}: {1} {2}".format(err.errno,
                                             err.strerror,
                                             fpath)))
            return False
