#!/usr/bin/python

"""Display eole-debsums reports per group of containers.

"""

import os
import sys

from creole.client import CreoleClient
from pyeole.debsums import EoleDebsumsReport
from pyeole.debsums import NoEoleDebsumsReportError
from pyeole.debsums import EoleDebsumsReportParseError

from datetime import datetime

from glob import glob

REPORT_DIR = 'var/log/eole-debsums'
REPORT_FILE = 'report.log'

def print_pkgs(report):
    """Display the list of packages

    """
    for pkg in report:
        print("{0}:".format(pkg))
        for filename in report[pkg]:
            print(" " * 4 + filename)
        print('')


def main():
    """
    Aggregate reports of all containers and print it on stdout.
    """
    client = CreoleClient()
    for group in client.get_groups():
        group_infos = client.get_container(group)
        filename = os.path.join('/',
                                group_infos[u'path'].lstrip('/'),
                                REPORT_DIR.lstrip('/'),
                                REPORT_FILE.lstrip('/'))
        ignored_files_list = glob(os.path.join('/',
                                               group_infos[u'path'].lstrip('/'),
                                               REPORT_DIR.lstrip('/'),
                                               '*.ignored'))
        header = """Container: {0}""".format(group)
        print(header)
        print("=" * len(header))
        try:
            m_date = datetime.fromtimestamp( os.stat(filename).st_mtime )
            m_date_str = m_date.strftime('%Y-%m-%d %H:%M:%S')
            print("\nFilename: {0}".format(filename))
            print("Last update: {0}\n".format(m_date_str))
            report = EoleDebsumsReport(filename=filename)
            print_pkgs(report)

            for ignored_file in ignored_files_list:
                filename = os.path.basename(ignored_file)
                source = os.path.splitext(filename)[0]
                print("Ignored by {0}".format(source))
                print('-' * (len(source) + 11))
                ignored = EoleDebsumsReport(filename=ignored_file)
                print_pkgs(ignored)
                print('')

        except (OSError, NoEoleDebsumsReportError,
                EoleDebsumsReportParseError), err:
            print "\n{0}\n".format(err)

if __name__ == "__main__":
    sys.exit(main())
