# -*- coding: utf-8 -*-
from os.path import isfile as _isfile, dirname as _dirname, abspath as _abspath, basename as _basename
from copy import copy as _copy

import ipaddress as _ipaddress
from datetime import datetime as _datetime

from creole.loader import creole_loader as _creole_loader, config_save_values as _config_save_values
from tiramisu.option import DomainnameOption as _DomainnameOption

_CONFIG_FILE = '/var/lib/eole/config/eoledhcp.cfg'

_HERE = _dirname(_abspath(__file__))
_MODNAME = _basename(_HERE)
if _MODNAME == 'salt':
    #Only for test
    _HERE = _dirname(_HERE)
    _MODNAME = _basename(_HERE)
_EWTAPP = _basename(_dirname(_HERE))

# __________________________________________________________

_force_dirs = None
_force_eoleextradico = None
_force_configeol= None
_force_eoleextraconfig = None
_reload_config = True


def _check_activation(func):
    def wrapper(*args, **kwargs):
        cfg = _get_loader()
        if cfg.dhcpactivation.dhcp_activation.dhcp_activation_ead3 == "non":
            return { 'messages': [{'level': 'error', 'msg': "Veuillez d'abord activer l'action DHCP de l'EAD3."}] }
        else:
            kwargs["cfg"] = cfg
            return func(*args, **kwargs)
    return wrapper


def _get_loader(mandatory_permissive=True):
    return _creole_loader(load_extra=True, rw=True, owner=_MODNAME,
                         mandatory_permissive=mandatory_permissive, force_dirs=_force_dirs,
                         force_eoleextradico=_force_eoleextradico, force_configeol=_force_configeol,
                         force_eoleextraconfig=_force_eoleextraconfig, reload_config=_reload_config)


def _validate_iprange(ipaddr, startip, endip):
    """range ip calculation with _ipaddress
    """
    startip = _ipaddress.IPv4Address(unicode(startip))
    endip = _ipaddress.IPv4Address(unicode(endip))
    for intip in xrange(startip, endip + 1):
        iprange = _ipaddress.IPv4Address(intip)
        if ipaddr == iprange:
            raise ValueError("L'adresse IP {} ne doit pas être entre {} et {}"
                             "".format(str(ipaddr), str(startip), str(endip)))

def _is_in_ipnetwork(ipaddr, network, netmask):
    net = _ipaddress.ip_network(u'{}/{}'.format(network, netmask))
    if str(ipaddr) == network:
        raise ValueError("L'adresse IP {} est l'adresse du réseau".format(ipaddr))
    if ipaddr == net.broadcast_address:
        raise ValueError("L'adresse IP {} est l'adresse du broadcast".format(ipaddr))
    return ipaddr in net


def _validate_subnet(ipaddr, cfg):
    """range ip calculation with creole variables
    """
    try:
        ipaddr = _ipaddress.ip_address(unicode(ipaddr))
    except:
        raise ValueError("L'adresse IP n'est pas valide {}".format(str(ipaddr)))

    networkdhcp = list(cfg.creole.dhcp.adresse_network_dhcp.adresse_network_dhcp)
    netmaskdhcp = list(cfg.creole.dhcp.adresse_network_dhcp.adresse_netmask_dhcp)
    iphautes = list(cfg.creole.dhcp.adresse_network_dhcp.ip_haute_dhcp)
    ipbasses = list(cfg.creole.dhcp.adresse_network_dhcp.ip_basse_dhcp)
    tested_networks = []
    in_a_network = False
    for idx, network in enumerate(networkdhcp):
        netmask = netmaskdhcp[idx]
        netname = "{}_{}".format(network, netmask)
        if netname not in tested_networks:
            tested_networks.append(netname)
            if not _is_in_ipnetwork(ipaddr, network, netmask):
                continue
            in_a_network = True
        ipbasse = ipbasses[idx]
        iphaute = iphautes[idx]
        _validate_iprange(ipaddr, ipbasse, iphaute)
    if not in_a_network:
        raise ValueError("L'adresse IP {} n'est dans aucun réseau déclaré".format(str(ipaddr)))

def _validate_rangename(host_rangename, range_names, unknowns):
    for idx, name in enumerate(range_names):
        if name == host_rangename:
            if unknowns[idx] != 'oui':
                raise ValueError("La plage \"{}\" ne peut pas contenir d'IP réservée"
                                 "".format(name))
            break
    else:
        raise ValueError("Le nom {} ne correspond à aucune plage".format(host_rangename))


# __________________________________________________________

def _get_config(name):
    if not _isfile(_CONFIG_FILE):
        raise Exception("Configuration file not present")
    from ConfigParser import ConfigParser
    cfg = ConfigParser(allow_no_value=True)
    cfg.read(_CONFIG_FILE)

    if name == "SUBNETS":
        # c'est un tuple (variable multiple)
        return eval(cfg.get('eole', 'subnets'))
        #return eval(cfg.get('eole', 'eole.dhcp.adresse_network_dhcp'))
    elif name == "LEASES_FILE":
        return cfg.get('eole', 'dhcp_path')


@_check_activation
def get_leases(*args, **kwargs):
    """
    Send list of existing leases
    leases : machines connected to the network that have received a dynamic ip

    :return: dict of leases, for example::
                 {
                     "leases": [
                         {
                         "mac": "02:00:c0:a8:00:66",
                         "rangename": "pcxubuntu",
                         "address": "192.168.0.200"
                         },
                         {
                         "mac": "02:00:c0:a8:00:67",
                         "rangename": "pcxubuntu",
                         "address": "192.168.0.201"
                         },
                         {
                         "mac": "02:00:c0:a8:00:68",
                         "rangename": "pcxubuntu",
                         "address": "192.168.0.202"
                         }
                     ]
                 }

    """
    leases = {}
    dates = {}
    blocs = file(_get_config("LEASES_FILE")).read().split('}')
    now = _datetime.utcnow()
    for bloc in blocs:
        if 'lease ' not in bloc:
            continue
        lines = bloc.split('lease ')[-1].splitlines()
        address = lines[0].split()[0]
        if '  client-hostname ' in lines[-1]:
            name = lines[-1].split('"')[1]
            try:
                _DomainnameOption('test', '', name, type_='domainname', allow_ip=False, allow_without_dot=True)
            except ValueError:
                name = None
        else:
            name = None
        invalid_date = False
        date = None
        for line in lines:
            line = line.strip()
            if line.startswith('ends '):
                curdate = line.split(' ', 2)[-1].replace(';', '')
                date = _datetime.strptime(curdate, "%Y/%m/%d %H:%M:%S")
                if date < now:
                    invalid_date = True
                break
        if invalid_date:
            continue
        for line in lines:
            line = line.strip()
            if line.startswith('hardware ethernet '):
                mac = line.split('hardware ethernet ')[1][0:-1]
                if mac in leases:
                    if date and date > dates[mac]:
                        del leases[mac]
                    else:
                        break
                dates[mac] = date
                leases[mac] = dict(name=name, address=address, mac=mac, expiration=str(date))
                break
    return dict(leases=leases.values())
    #dhcp-lease-list --lease /var/lib/dhcp/dhcpd.leases


def _get_subnets(cfg):
    subnetlist = []
    adresse_network_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.adresse_network_dhcp)
    adresse_netmask_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.adresse_netmask_dhcp)
    nom_plage_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.nom_plage_dhcp)
    ip_haute_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.ip_haute_dhcp)
    ip_basse_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.ip_basse_dhcp)
    interdire_hotes_inconnus = list(cfg.creole.dhcp.adresse_network_dhcp.interdire_hotes_inconnus)
    for idx, network in enumerate(adresse_network_dhcp):
        netmask = adresse_netmask_dhcp[idx]
        dynrange = dict(name=nom_plage_dhcp[idx],
                        high=ip_haute_dhcp[idx],
                        low=ip_basse_dhcp[idx],
                        reservable=interdire_hotes_inconnus[idx] == 'oui')
        for subnet in subnetlist:
            if network == subnet['network'] and netmask == subnet['netmask']:
                subnet['dynamicRanges'].append(dynrange)
                break
        else:
            subnetlist.append(
                dict(network=network,
                     netmask=adresse_netmask_dhcp[idx],
                     dynamicRanges=[dynrange]
                    )
            )
    return {"subnets": subnetlist}


@_check_activation
def get_subnets(*args, **kwargs):
    """
    Send list of declared subnets
    exemple:
    # salt-call ead.dhcp_get_subnets
    :returns: a list network design like this:
              {"subnets": [{
                  "dynamicRanges": [
                      {
                          "rangename": "test1",
                          "high": "192.168.0.120",
                          "reservable": false,
                          "low": "192.168.0.110"
                      },
                      {
                          "rangename": "test2",
                          "high": "192.168.0.130",
                          "reservable": true,
                          "low": "192.168.0.125"
                      },
                      {
                          "rangename": "test3",
                          "high": "192.168.0.140",
                          "reservable": false,
                          "low": "192.168.0.135"
                      }
                  ],
                  "netmask": "255.255.255.0",
                  "network": "192.168.0.0"
              }]}
    """
    if kwargs["cfg"] is None:
        cfg = _get_loader()
    else:
        cfg = kwargs["cfg"]
    return _get_subnets(cfg)


#-------------------------------------
# reservation


def _get_reserved(cfg, reserved_id, rel):
    def _get_host_by_id(id_dh, idx):
        dico = dict(id=id_dh, hostname=hostname[idx], macaddress=macaddress[idx])
        if dyn[idx] == 'oui':
            dico['rangename'] = name[idx]
        else:
            dico['ip'] = ip[idx]
        return dico
    id_dhcp = list(cfg.dhcp.dhcp.id_dhcp.id_dhcp)
    hostname = list(cfg.dhcp.dhcp.id_dhcp.hostname)
    macaddress = list(cfg.dhcp.dhcp.id_dhcp.macaddress)
    dyn = list(cfg.dhcp.dhcp.id_dhcp.dyn)
    ip = list(cfg.dhcp.dhcp.id_dhcp.ip)
    name = list(cfg.dhcp.dhcp.id_dhcp.name)
    reserved = []
    messages = []
    if reserved_id is not None:
        try:
            idx = id_dhcp.index(reserved_id)
        except ValueError:
            messages.append({'level': 'error', 'msg': "L'ID de réservation {} n'existe pas".format(reserved_id),
                             'rel': rel})
        else:
            reserved.append(_get_host_by_id(reserved_id, idx))
    else:
        for idx, id_dh in enumerate(id_dhcp):
            reserved.append(_get_host_by_id(id_dh, idx))
    ret = {'reserved': reserved}
    if messages != []:
        ret['messages'] = messages
    return ret


@_check_activation
def get_reserved(reserved_id=None, *args, **kwargs):
    """
    Send list of reserved IP
    return list of tuple (id, machine name, IP, MAC Adress)
    """
    if kwargs["cfg"] is None:
        cfg = _get_loader()
    else:
        cfg = kwargs["cfg"]
    return _get_reserved(cfg, reserved_id, reserved_id)

@_check_activation
def upsert_reserved(reserved, hosts_range=None, prefix=None, save_when_error=False, del_reservations=False, *args, **kwargs):
    """
    Creates or modifies an existing DHCP bookings
    :param dict kwargs:  - with the 'id' key -> update
                         - without the 'id' key -> insert
                         {'id': 1, 'hostname': '..', 'ip': '..', 'macaddress': '..'}
    :return: True or False (FIXME : an error message has to be returned)
    """
    def _validate_macaddress(macaddresses, dyn, names, ips, hostnames, host, cfg):
        def _build_network():
            if not host['dyn']:
                try:
                    ipaddr = _ipaddress.ip_address(unicode(host['ip']))
                except:
                    raise ValueError("L'adresse IP {} n'est pas valide".format(str(host['ip'])))

            networkdhcp = list(cfg.creole.dhcp.adresse_network_dhcp.adresse_network_dhcp)
            netmaskdhcp = list(cfg.creole.dhcp.adresse_network_dhcp.adresse_netmask_dhcp)
            nomplagedhcp = list(cfg.creole.dhcp.adresse_network_dhcp.nom_plage_dhcp)
            current_net = {}
            rangenames = {}
            for idx, network in enumerate(networkdhcp):
                netmask = netmaskdhcp[idx]
                netname = "{}_{}".format(network, netmask)
                if current_net == {}:
                    if not host['dyn'] and _is_in_ipnetwork(ipaddr, network, netmask) or \
                            host['dyn'] and nomplagedhcp[idx] == host['rangename']:
                        current_net['name'] = netname
                        current_net['network'] = network
                        current_net['netmask'] = netmask
                rangenames.setdefault(netname, []).append(nomplagedhcp[idx])
            if current_net == {}:
                raise ValueError("L'adresse MAC {} n'est dans un aucun réseau déclaré"
                                 "".format(host['macaddress']))
            return rangenames[current_net['name']], current_net['network'], current_net['netmask']

        id_macaddresses = []
        for idx, macaddress in enumerate(macaddresses):
            if macaddress == host['macaddress']:
                id_macaddresses.append(idx)
        if len(id_macaddresses) != 0:
            # MAC address validation: a MAC address associated to:
            # - an IP address cannot be associated to an other IP address or dynamic pool in same network
            # - an IP address cannot be associated to a dynamic pool
            # - a dynamic pool cannot be associated to an other IP or another dynamic pool in same network
            rangenames, network, netmask = _build_network()
            for idx in id_macaddresses:
                if dyn[idx] == 'non':
                    if _is_in_ipnetwork(_ipaddress.ip_address(ips[idx]), network, netmask):
                        raise ValueError("L'hôte {} a déjà une adresse IP réservée dans le réseau {}/{}".format(host['macaddress'], network, netmask))
                else:
                    if names[idx] in rangenames:
                        raise ValueError("L'hôte {} ({}) est déjà dans la plage dynamique {} du réseau {}/{}".format(host.get('hostname'), host['macaddress'], names[idx], network, netmask))
                del hostnames[idx]
        # the hostname shall be defined if we modify and existing host
        host_hostname = host.get('hostname')
        if host_hostname is None and 'id' in host:
            raise ValueError("Le nom d'hôte de {} n'est pas défini".format(host['macaddress']))

        if host_hostname is not None:
            for hostname in hostnames:
                if hostname == host_hostname:
                    raise ValueError("Le nom d'hôte {} est déjà attribué".format(hostname))

    def _validate_ip(ips, host):
        if 'ip' not in host:
            raise ValueError("L'adresse IP de {} n'est pas définie".format(host['macaddress']))
        host_ip = host['ip']
        for ip in ips:
            if ip == host_ip:
                raise ValueError("L'adresse IP {} est déjà attribuée".format(ip))
        _validate_subnet(host_ip, cfg)

    def _validate_range_name(host):
        #Valid range name
        if host['dyn'] == True:
            host_rangename = host['rangename']
            _validate_rangename(host_rangename, list(cfg.creole.dhcp.adresse_network_dhcp.nom_plage_dhcp),
                                list(cfg.creole.dhcp.adresse_network_dhcp.interdire_hotes_inconnus)
                               )

    def _validate_hostname(host):
        if 'hostname' in host:
            try:
                _DomainnameOption('test', '', host['hostname'], type_='domainname', allow_ip=False, allow_without_dot=True)
            except ValueError:
                raise ValueError("Le nom d'hôte {} n'est pas valide".format(host['hostname']))

    def _make_ip(hosts_range, ips, mac):
        lall_intip = _copy(all_intip)
        for intip in lall_intip:
            current_ip = str(_ipaddress.IPv4Address(intip))
            all_intip.pop(0)
            try:
                if current_ip not in ips:
                    return current_ip
            except ValueError:
                pass
        raise ValueError('Pas d\'IP disponible dans la plage "{}-{}" pour {}'.format(hosts_range['low'],
                                                                             hosts_range['high'], mac))

    def _make_name(hosts_range):
        if set(['rangename']) != set(hosts_range.keys()):
            raise ValueError("Le dictionnaire du nom de la plage est invalide, il faut la clef 'rangename'")
        return hosts_range['rangename']

    def _make_hostname(prefix, hostnames):
        hostnames_with_prefix = [pref for pref in hostnames if pref is not None and pref.startswith(prefix)]
        idx = 0
        max_iteration = 10000
        iteration = 0
        new_name = None
        while new_name is None or new_name in hostnames_with_prefix:
            new_name = prefix + str(idx)
            iteration += 1
            idx += 1
            if iteration == max_iteration:
                raise ValueError('Impossible de calculer un nom de machine à partir du préfixe : '
                                 '"{}"'.format(prefix))
        return new_name

    def _add_new_host(host):
        if host['dyn'] is True and 'rangename' not in host:
            host['rangename'] = _make_name(hosts_range)
        hostnames = list(cfg.dhcp.dhcp.id_dhcp.hostname)
        _validate_macaddress(cfg.dhcp.dhcp.id_dhcp.macaddress,
                             cfg.dhcp.dhcp.id_dhcp.dyn,
                             cfg.dhcp.dhcp.id_dhcp.name,
                             cfg.dhcp.dhcp.id_dhcp.ip,
                             hostnames,
                             host, cfg)
        _validate_hostname(host)
        _validate_range_name(host)
        if host['dyn'] is False:
            _validate_ip(cfg.dhcp.dhcp.id_dhcp.ip, host)
        c_id = cfg.dhcp.dhcp.last_id + 1
        cfg.dhcp.dhcp.last_id = c_id
        id_dhcp.append(c_id)
        return c_id

    # retrieving the existing booking list in the cfg.dhcp.dhcp config variable
    messages = []
    try:
        _DomainnameOption('test', '', prefix, type_='hostname', allow_ip=False, allow_without_dot=True)
    except ValueError as err:
        messages.append({'level': 'error',
                         'msg': "Le préfixe n'est pas valide : \"{}\"".format(err),
                         'rel': {'source': prefix}})
    else:
        if kwargs["cfg"] is None:
            cfg = _get_loader()
        else:
            cfg = kwargs["cfg"]
        if del_reservations:
            _del_reserved(cfg, messages, None, True)
        if messages != []:
            pass
        elif not isinstance(reserved, list):
            messages.append({'level': 'error', 'msg': 'reserved parameter must be a list of dict',
                             'rel': reserved})
        else:
            # construct all_int_ip to configure unknown IP
            if hosts_range is not None and 'low' in hosts_range:
                if set(['low', 'high']) != set(hosts_range.keys()):
                    raise ValueError("Le dictionnaire des plages d'IP est invalide, "
                                     "il faut les clef 'low' et 'high'")
                startip = _ipaddress.ip_address(unicode(hosts_range['low']))
                endip = _ipaddress.ip_address(unicode(hosts_range['high']))
                all_intip = set(range(startip, endip + 1))
                subnets = _get_subnets(cfg)
                for subnet in subnets['subnets']:
                    try:
                        network = _ipaddress.ip_address(unicode(subnet['network']))
                        netmask = _ipaddress.ip_address(unicode(subnet['netmask']))
                        if _is_in_ipnetwork(startip, network, netmask) and \
                                _is_in_ipnetwork(endip, network, netmask):
                            for dynrange in subnet['dynamicRanges']:
                                low = _ipaddress.ip_address(unicode(dynrange['low']))
                                high = _ipaddress.ip_address(unicode(dynrange['high']))
                                all_intip -= set(range(low, high + 1))
                    except ValueError:
                        pass
                all_intip = list(all_intip)
                all_intip.sort()

            for host in reserved:
                # validation of the inner structure
                if not isinstance(host, dict):
                    messages.append({'level': 'error',
                                     'msg': 'reserved must contain a dict with '
                                            'host informations, not {}'.format(str(host)),
                                     'rel': host})
                    continue
                keys = set(host.keys())
                if 'macaddress' not in keys:
                    messages.append({'level': 'error',
                                     'msg': "L'adresse MAC est obligatoire pour une réservation",
                                     'rel': host})
                    continue
                keys -= set(['macaddress', 'id'])
                if 'rangename' in host:
                    host['dyn'] = True
                elif 'ip' in host:
                    host['dyn'] = False
                elif hosts_range is not None:
                    if 'rangename' in hosts_range:
                        host['dyn'] = True
                    elif 'low' in hosts_range:
                        host['dyn'] = False
                    else:
                        raise ValueError('unexpected hosts_range parameter')
                else:
                    messages.append({'level': 'error',
                                     'msg': "Pour créer l'hôte {} il faut une IP ou un nom de plage".format(host['macaddress']),
                                     'rel': host})
                    continue

                if host['dyn'] == True:
                    keys -= set(['rangename'])
                else:
                    keys -= set(['ip'])
                if not 'hostname' in keys and prefix is None:
                    messages.append({'level': 'error',
                                     'msg': "Le nom d'hôte est obligatoire pour la réservation de {}".format(host['macaddress']),
                                     'rel': host})
                    continue
                keys -= set(['hostname'])
                if keys != set([]):
                    messages.append({'level': 'error', 'msg': 'host must not contain parameters {}'.format(', '.join(keys)),
                                     'rel': host})
                    continue


                validate = False
                id_dhcp = cfg.dhcp.dhcp.id_dhcp.getattr('id_dhcp', validate=validate)

                c_id = host.get('id')
                if host['dyn'] is False and 'ip' not in host:
                    try:
                        host['ip'] = _make_ip(hosts_range, cfg.dhcp.dhcp.id_dhcp.getattr('ip', validate=validate), host['macaddress'])
                    except ValueError, err:
                        messages.append({'level': 'error', 'msg': str(err),
                                         'rel': host})
                        continue
                # if no ID, assume that it's a new host
                if c_id is None:
                    try:
                        c_id = _add_new_host(host)
                    except ValueError, err:
                        messages.append({'level': 'error', 'msg': str(err),
                                         'rel': host})
                        continue
                    new = True
                else:
                    new = False
                hostname = cfg.dhcp.dhcp.id_dhcp.getattr('hostname', validate=validate)
                macaddress = cfg.dhcp.dhcp.id_dhcp.getattr('macaddress', validate=validate)
                dyn = cfg.dhcp.dhcp.id_dhcp.getattr('dyn', validate=validate)
                ip = cfg.dhcp.dhcp.id_dhcp.getattr('ip', validate=validate)
                name = cfg.dhcp.dhcp.id_dhcp.getattr('name', validate=validate)
                if 'hostname' not in host:
                    host['hostname'] = _make_hostname(prefix, hostname)
                try:
                    idx = id_dhcp.index(c_id)
                except ValueError:
                    messages.append({'level': 'error', 'msg': 'unknown ID for entry {}'.format(str(host)),
                                     'rel': host})
                    continue
                if not new:
                    macaddresses = list(macaddress)
                    macaddresses.pop(idx)
                    dyns = list(dyn)
                    dyns.pop(idx)
                    names = list(name)
                    names.pop(idx)
                    ips = list(ip)
                    ips.pop(idx)
                    hostnames = list(hostname)
                    hostnames.pop(idx)
                    try:
                        if host['dyn'] is False:
                            _validate_ip(ips, host)
                        _validate_macaddress(macaddresses,
                                             dyns,
                                             names,
                                             ips,
                                             hostnames,
                                             host, cfg)
                        _validate_hostname(host)
                        _validate_range_name(host)
                    except ValueError, err:
                        messages.append({'level': 'error', 'msg': str(err),
                                         'rel': host})
                        continue

                try:
                    hostname[idx] = host['hostname']
                    macaddress[idx] = host['macaddress']
                    dyn[idx] = {True: "oui", False: "non"}.get(host['dyn'])
                    if host['dyn']:
                        name[idx] = unicode(host['rangename'])
                    else:
                        ip[idx] = host['ip']
                except ValueError as err:
                    messages.append({'level': 'error', 'msg': str(err),
                                     'rel': host})
                    continue

            # applying the new DHCP configuration
            # do not save on errors, unless save_when_error is True
            if len(messages) == 0 or save_when_error:
                _save_values(cfg, messages)
    ret = {}
    if messages != []:
        ret['messages'] = messages
    return ret


def _del_reserved(cfg, messages, reserved, allreservations):
    if allreservations:
        del cfg.dhcp.dhcp.id_dhcp.id_dhcp
    else:
        if isinstance(reserved, list):
            for res in reserved:
                orig_reserved = _get_reserved(cfg, res['id'], res)['reserved'][0]
                if res != orig_reserved:
                    messages.append({'level': 'error',
                                     'msg': 'host with id {} has not '
                                            'same informations'.format(res['id']),
                                     'rel': res})
                else:
                    idx = cfg.dhcp.dhcp.id_dhcp.id_dhcp.index(res['id'])
                    cfg.dhcp.dhcp.id_dhcp.id_dhcp.pop(idx)
        else:
            messages.append({'level': 'error', 'msg': 'entries must be a list of reserved',
                             'rel': reserved})


def _save_values(cfg, messages=None):
    _config_save_values(cfg, _MODNAME, reload_config=_reload_config)
    if '__salt__' in globals():
        results = __salt__['state.apply'](_MODNAME, saltenv=_EWTAPP)
    else:
        results = None
    #COPIED FROM ewt-actions/ewt/generic_form.py
    msg = []
    if isinstance(results, list):
        msg = results
    elif isinstance(results, dict):
        for result in results.values():
            if not result['result']:
                # Action failed, try to extract error message
                err_object = result['name']
                if result.get('changes', {}):
                    # return cmd.run error ouput, or standard output if error is empty
                    err_output = result['changes'].get('stderr', '') or \
                                 result['changes'].get('stdout', '')
                else:
                    err_output = result['comment']
                    if not err_output.endswith('.'):
                        err_output += "."
                # if output is a python Traceback, only return last line
                if err_output.startswith('Traceback'):
                    err_output = err_output.split('\n')[-1].lstrip('Exception:')
                comment = "{0} ({1})".format(err_output, err_object)
                msg.append(comment)
    if messages is None:
        messages = []
    for error in msg:
        messages.append({'level': 'error', 'msg': str(error)})
    return messages


@_check_activation
def del_reserved(reserved=None, allreservations=False, *args, **kwargs):
    """
    Delete an existant DHCP route
    return True or False with error message
    """
    messages = []
    if reserved is None and allreservations is False:
        messages.append({'level': 'error', 'msg': 'del_reserved need reserved or allreservations',
                         'rel': {}})
    else:
        if kwargs["cfg"] is None:
            cfg = _get_loader()
        else:
            cfg = kwargs["cfg"]
        _del_reserved(cfg, messages, reserved, allreservations)
        if messages == []:
            messages = _save_values(cfg)
    ret = {}
    if messages != []:
        ret['messages'] = messages
    return ret
