# -*- coding: utf-8 -*-
from os.path import isfile as _isfile, dirname as _dirname, abspath as _abspath, basename as _basename
from copy import copy as _copy
import re as _re

import ipaddress as _ipaddress
from datetime import datetime as _datetime

from creole.loader import creole_loader as _creole_loader, config_save_values as _config_save_values
from tiramisu.option import DomainnameOption as _DomainnameOption

_CONFIG_FILE = '/var/lib/eole/config/eoledhcp.cfg'

_HERE = _dirname(_abspath(__file__))
_MODNAME = _basename(_HERE)
if _MODNAME == 'salt':
    #Only for test
    _HERE = _dirname(_HERE)
    _MODNAME = _basename(_HERE)
_EWTAPP = _basename(_dirname(_HERE))

# __________________________________________________________

_force_dirs = None
_force_eoleextradico = None
_force_configeol= None
_force_eoleextraconfig = None
_reload_config = True
_MAC_RE = _re.compile(r'^([0-9a-fA-F]{2})(:[0-9a-fA-F]{2}){5}$')
unicode = str

def _check_activation(func):
    def wrapper(*args, **kwargs):
        cfg = _get_loader()
        if cfg.dhcpactivation.dhcp_activation.dhcp_activation_ead3 == "non":
            return { 'messages': [{'level': 'error', 'msg': "Veuillez d'abord activer l'action DHCP de l'EAD3."}] }
        else:
            kwargs["cfg"] = cfg
            return func(*args, **kwargs)
    return wrapper


def _get_loader(mandatory_permissive=True):
    return _creole_loader(load_extra=True, rw=True, owner=_MODNAME,
                         mandatory_permissive=mandatory_permissive, force_dirs=_force_dirs,
                         force_eoleextradico=_force_eoleextradico, force_configeol=_force_configeol,
                         force_eoleextraconfig=_force_eoleextraconfig, reload_config=_reload_config)


def _next_int():
    i = 0
    while True:
        yield i
        i += 1


def _extract_gen_config_data(cfg):
    """Return conveniently structured data from server configuration

    :param cfg: Configuration loader
    :returns: tuple containing subnets, ranges and ips in declared subnets data

    """
    range_specifications = list(zip([_ipaddress.ip_network(u'{}/{}'.format(ip, netmask))
                                for ip, netmask in zip(list(cfg.creole.dhcp.adresse_network_dhcp.adresse_network_dhcp),
                                    list(cfg.creole.dhcp.adresse_network_dhcp.adresse_netmask_dhcp))
                               ],
                               list(cfg.creole.dhcp.adresse_network_dhcp.nom_plage_dhcp),
                               [_ipaddress.ip_address(unicode(ip))
                                for ip in list(cfg.creole.dhcp.adresse_network_dhcp.ip_basse_dhcp)
                               ],
                               [_ipaddress.ip_address(unicode(ip))
                                for ip in list(cfg.creole.dhcp.adresse_network_dhcp.ip_haute_dhcp)
                               ],
                               list(cfg.creole.dhcp.adresse_network_dhcp.interdire_hotes_inconnus),
                               [True if s == 'oui' else False
                                for s in list(cfg.creole.dhcp.adresse_network_dhcp.adressage_statique)]
                               ))

    subnets = {}
    for range_spec in range_specifications:
        subnets.setdefault(range_spec[0], []).append(range_spec[1])

    ranges = {r[1]: {'range': set([_ipaddress.ip_address(ip) for ip in range(int(r[2]), int(r[3]) + 1)]),
                     'restricted': r[4],
                     'static': r[5],
                     'subnet': r[0]}
              for r in range_specifications}

    addressable_ips = set()
    for subnet in subnets:
        addressable_ips |= set(subnet.hosts())

    for ip_range in ranges:
        if ranges[ip_range]['static'] is False:
            addressable_ips -= ranges[ip_range]['range']
    return subnets, ranges, addressable_ips


def _subnet_host_is_in(host, subnets, ranges):
    """Return subnet and range the host is in.
    :param host: host description
    :type host: dict('ip'=str, 'rangename'=str)
    """
    in_subnet = None
    rangename = host.get('rangename')
    ip = host.get('ip')
    if rangename in ranges:
        in_subnet = ranges[rangename]['subnet']
    elif ip:
        targeted_range = [r for r in ranges if _ipaddress.ip_address(unicode(ip)) in ranges[r]['range']]
        if targeted_range:
            rangename = targeted_range[0]
            in_subnet = ranges[rangename]['subnet']
        else:
            rangename = None
            for subnet in subnets:
                if  _ipaddress.ip_address(unicode(ip)) in subnet:
                    in_subnet = subnet
                    break
    return in_subnet, rangename


def _is_in_ipnetwork(ipaddr, subnet):
    if ipaddr == subnet.network_address:
        raise ValueError("L’adresse IP {} est l'adresse du réseau".format(ipaddr))
    if ipaddr == subnet.broadcast_address:
        raise ValueError("L’adresse IP {} est l'adresse du broadcast".format(ipaddr))
    return ipaddr in subnet


def _validate_ip_range(ip_addr, range_low_ip, range_high_ip):
    """Raise if IP is in range defined between low and high ip.
    :param ip_addr: IP address to validate
    :type ip_addr: ipaddress.IPv4Address
    :param range_low_ip: Lower end of IP range
    :type range_low_ip: ipaddress.IPv4Address
    :param range_high_ip: Higher end of IP range
    :type range_high_ip: ipaddress.IPv4Address
    """
    ip_in_range = any((ipaddr == _ipaddress.IPv4Address(intip)
                       for intip in xrange(range_low_ip, range_high_ip + 1)))
    if ip_in_range:
        raise ValueError("L’adresse IP {} ne doit pas être entre {} et {}"
                             "".format(str(ipaddr), str(startip), str(endip)))


def _validate_rangename(host_rangename, ranges):
    if range_name in ranges:
        if ranges[range_name]['restricted'] is False:
            raise ValueError('La plage "{}" ne peut pas contenir d’IP réservée'.format(range_name))
    else:
        raise ValueError("Le nom {} ne correspond à aucune plage".format(host_rangename))


# __________________________________________________________

def _get_config(name):
    if not _isfile(_CONFIG_FILE):
        raise Exception("Configuration file not present")
    from configparser import ConfigParser
    cfg = ConfigParser(allow_no_value=True)
    cfg.read(_CONFIG_FILE)
    if name == "SUBNETS":
        # c'est un tuple (variable multiple)
        return eval(cfg.get('eole', 'subnets'))
        #return eval(cfg.get('eole', 'eole.dhcp.adresse_network_dhcp'))
    elif name == "LEASES_FILE":
        return cfg.get('eole', 'dhcp_path')


@_check_activation
def get_leases(*args, **kwargs):
    """
    Send list of existing leases
    leases : machines connected to the network that have received a dynamic ip

    :return: dict of leases, for example::
                 {
                     "leases": [
                         {
                         "mac": "02:00:c0:a8:00:66",
                         "rangename": "pcxubuntu",
                         "address": "192.168.0.200"
                         },
                         {
                         "mac": "02:00:c0:a8:00:67",
                         "rangename": "pcxubuntu",
                         "address": "192.168.0.201"
                         },
                         {
                         "mac": "02:00:c0:a8:00:68",
                         "rangename": "pcxubuntu",
                         "address": "192.168.0.202"
                         }
                     ]
                 }

    """
    leases = {}
    dates = {}
    blocs = open(_get_config("LEASES_FILE")).read().split('}')
    now = _datetime.utcnow()
    for bloc in blocs:
        if 'lease ' not in bloc:
            continue
        lines = bloc.split('lease ')[-1].splitlines()
        address = lines[0].split()[0]
        if '  client-hostname ' in lines[-1]:
            name = lines[-1].split('"')[1]
            try:
                _DomainnameOption('test', '', name, type_='domainname', allow_ip=False, allow_without_dot=True, allow_uppercase=True)
            except ValueError:
                name = None
        else:
            name = None
        invalid_date = False
        date = None
        for line in lines:
            line = line.strip()
            if line.startswith('ends '):
                curdate = line.split(' ', 2)[-1].replace(';', '')
                date = _datetime.strptime(curdate, "%Y/%m/%d %H:%M:%S")
                if date < now:
                    invalid_date = True
                break
        if invalid_date:
            continue
        for line in lines:
            line = line.strip()
            if line.startswith('hardware ethernet '):
                mac = line.split('hardware ethernet ')[1][0:-1]
                if mac in leases:
                    if date and date > dates[mac]:
                        del leases[mac]
                    else:
                        break
                dates[mac] = date
                leases[mac] = dict(name=name, address=address, mac=mac, expiration=str(date))
                break
    return dict(leases=list(leases.values()))
    #dhcp-lease-list --lease /var/lib/dhcp/dhcpd.leases


def _get_subnets(cfg):
    subnetlist = []
    adresse_network_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.adresse_network_dhcp)
    adresse_netmask_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.adresse_netmask_dhcp)
    nom_plage_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.nom_plage_dhcp)
    ip_haute_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.ip_haute_dhcp)
    ip_basse_dhcp = list(cfg.creole.dhcp.adresse_network_dhcp.ip_basse_dhcp)
    interdire_hotes_inconnus = list(cfg.creole.dhcp.adresse_network_dhcp.interdire_hotes_inconnus)
    range_type = list(cfg.creole.dhcp.adresse_network_dhcp.adressage_statique)
    for idx, network in enumerate(adresse_network_dhcp):
        netmask = adresse_netmask_dhcp[idx]
        dynrange = dict(name=nom_plage_dhcp[idx],
                        high=ip_haute_dhcp[idx],
                        low=ip_basse_dhcp[idx],
                        restricted=interdire_hotes_inconnus[idx] == 'oui',
                        static=range_type[idx] == 'oui')
        for subnet in subnetlist:
            if network == subnet['network'] and netmask == subnet['netmask']:
                subnet['ranges'].append(dynrange)
                break
        else:
            subnetlist.append(
                dict(network=network,
                     netmask=adresse_netmask_dhcp[idx],
                     ranges=[dynrange]
                    )
            )
    return subnetlist


@_check_activation
def get_subnets(*args, **kwargs):
    """
    Send list of declared subnets
    exemple:
    # salt-call ead.dhcp_get_subnets
    :returns: a list network design like this:
              {"subnets": [{
                  "dynamicRanges": [
                      {
                          "rangename": "test1",
                          "high": "192.168.0.120",
                          "reservable": false,
                          "low": "192.168.0.110"
                      },
                      {
                          "rangename": "test2",
                          "high": "192.168.0.130",
                          "reservable": true,
                          "low": "192.168.0.125"
                      },
                      {
                          "rangename": "test3",
                          "high": "192.168.0.140",
                          "reservable": false,
                          "low": "192.168.0.135"
                      }
                  ],
                  "netmask": "255.255.255.0",
                  "network": "192.168.0.0"
              }]}
    """
    if kwargs["cfg"] is None:
        cfg = _get_loader()
    else:
        cfg = kwargs["cfg"]
    return {"subnets": _get_subnets(cfg)}


#-------------------------------------
# reservation


def _get_reserved(cfg, reserved_id, rel):
    def _get_host_by_id(id_dh, idx):
        dico = dict(id=id_dh, hostname=hostname[idx], macaddress=macaddress[idx])
        if dyn[idx] == 'oui':
            dico['rangename'] = name[idx]
        else:
            if name[idx]:
                dico['rangename'] = name[idx]
            dico['ip'] = ip[idx]
        return dico
    id_dhcp = list(cfg.dhcp.dhcp.id_dhcp.id_dhcp)
    hostname = list(cfg.dhcp.dhcp.id_dhcp.hostname)
    macaddress = list(cfg.dhcp.dhcp.id_dhcp.macaddress)
    dyn = list(cfg.dhcp.dhcp.id_dhcp.dyn)
    ip = list(cfg.dhcp.dhcp.id_dhcp.ip)
    name = list(cfg.dhcp.dhcp.id_dhcp.name)
    reserved = []
    messages = []
    if reserved_id is not None:
        try:
            idx = id_dhcp.index(reserved_id)
        except ValueError:
            messages.append({'level': 'error', 'msg': "L’ID de réservation {} n'existe pas".format(reserved_id),
                             'rel': rel})
        else:
            reserved.append(_get_host_by_id(reserved_id, idx))
    else:
        for idx, id_dh in enumerate(id_dhcp):
            reserved.append(_get_host_by_id(id_dh, idx))
    ret = {'reserved': reserved}
    if messages != []:
        ret['messages'] = messages
    return ret


@_check_activation
def get_reserved(reserved_id=None, *args, **kwargs):
    """
    Send list of reserved IP
    return list of tuple (id, machine name, IP, MAC Adress)
    """
    if kwargs["cfg"] is None:
        cfg = _get_loader()
    else:
        cfg = kwargs["cfg"]
    return _get_reserved(cfg, reserved_id, reserved_id)

@_check_activation
def upsert_reserved(reserved, hosts_range=None, prefix=None, save_when_error=False, del_reservations=False, *args, **kwargs):
    """
    Creates or modifies an existing DHCP bookings
    :param dict kwargs:  - with the 'id' key -> update
                         - without the 'id' key -> insert
                         {'id': 1, 'hostname': '..', 'ip': '..', 'macaddress': '..', 'name': '..'}
    :return: True or False (FIXME : an error message has to be returned)
    """
    # La liste des validations nécessaires est :
    # - est-ce que l’IP est une IPv4 valide
    # - est-ce que l’IP est disponible à la réservation
    # - est-ce que l’adresse MAC n’est pas déjà associée à une IP dans le même sous-réseau
    # - est-ce que le nom d’hôte suit la syntaxe d’un nom de domaine
    # - est-ce que le nom de plage existe

    def _assign_ip(rangename, candidates_ips, ranges, available_ips):
        """Return first available IP in range associated or raise if no IP available.
        """
        if rangename is not None:
            dispendable_ips = ranges[rangename]['range'].intersection(available_ips)
            low = min(ranges[rangename]['range'])
            high = max(ranges[rangename]['range'])
        else:
            dispendable_ips = candidates_ips.intersection(available_ips)
            low = min(candidates_ips)
            high = max(candidates_ips)
        if not dispendable_ips:
            raise ValueError('Pas d’adresse IP disponible dans la plage "{}-{}" pour {}'.format(low, high, mac))
        else:
            ip = sorted(list(dispendable_ips)).pop(0)
            return ip

    def _make_name(hosts_range):
        if set(['rangename']) != set(hosts_range.keys()):
            raise ValueError("Le dictionnaire du nom de la plage est invalide, il faut la clef 'rangename'")
        return hosts_range['rangename']

    def _make_hostname(prefix, hostnames, reuse=True):
        auto_hostname_re = _re.compile(r'{}(?P<suffixe>[0-9]+)'.format(prefix))
        hostnames_with_prefix = [pref for pref in hostnames if pref is not None and pref.startswith(prefix)]
        suffixes = []
        for hostname_with_prefix in hostnames_with_prefix:
            re_match = auto_hostname_re.match(hostname_with_prefix)
            if re_match:
                suffixes.append(int(re_match.group('suffixe')))
        if reuse is False:
            if not suffixes:
                suffixe = 0
            else:
                suffixe = max(suffixes) + 1
        else:
            n = _next_int()
            suffixe = next(n)
            while suffixe in suffixes:
                suffixe = next(n)

        new_name = '{}{}'.format(prefix, suffixe)
        return new_name

    # retrieving the existing booking list in the cfg.dhcp.dhcp config variable
    messages = []
    try:
        _DomainnameOption('test', '', prefix, type_='hostname', allow_ip=False, allow_without_dot=True, allow_uppercase=True)
    except ValueError as err:
        messages.append({'level': 'error',
                         'msg': "Le préfixe n'est pas valide : \"{}\"".format(err),
                         'rel': {'source': prefix}})
    else:
        if kwargs["cfg"] is None:
            cfg = _get_loader()
        else:
            cfg = kwargs["cfg"]

        if del_reservations:
            _del_reserved(cfg, messages, None, True)
        if messages != []:
            pass

        subnets, ranges, addressable_ips = _extract_gen_config_data(cfg)

        # récupération des informations des réservations existantes
        validate = False

        available_ips = set()
        available_ips |= addressable_ips

        if not isinstance(reserved, list):
            messages.append({'level': 'error', 'msg': 'reserved parameter must be a list of dict',
                             'rel': reserved})
        else:
            for host in reserved:
                # validation of the inner structure
                if not isinstance(host, dict):
                    messages.append({'level': 'error',
                                     'msg': 'reserved must contain a dict with '
                                            'host informations, not {}'.format(str(host)),
                                     'rel': host})
                    continue

                # Fetch known reservations
                known_id_dhcp = cfg.dhcp.dhcp.id_dhcp.getattr('id_dhcp', validate=validate)
                known_hostnames = cfg.dhcp.dhcp.id_dhcp.getattr('hostname', validate=validate)
                known_macaddresses = cfg.dhcp.dhcp.id_dhcp.getattr('macaddress', validate=validate)
                known_dyns = cfg.dhcp.dhcp.id_dhcp.getattr('dyn', validate=validate)
                known_ips = cfg.dhcp.dhcp.id_dhcp.getattr('ip', validate=validate)
                known_rangenames = cfg.dhcp.dhcp.id_dhcp.getattr('name', validate=validate)

                # Find if this is new or update case
                c_id = host.get('id')
                if c_id is None:
                    c_id = cfg.dhcp.dhcp.last_id + 1
                    new = True
                else:
                    new = False
                    try:
                        idx = known_id_dhcp.index(c_id)
                    except ValueError:
                        messages.append({'level': 'error', 'msg': 'unknown ID for entry {}'.format(str(host)),
                                         'rel': host})
                        continue

                reserved_ips = set([_ipaddress.ip_address(unicode(reservation)) if not isinstance(reservation, Exception) else None
                                    for reservation in known_ips
                                    if reservation is not None])
                available_ips -= reserved_ips

                # MAC address
                macaddress = host.get('macaddress')
                if macaddress:
                    if not _MAC_RE.match(macaddress):
                        messages.append({'level': 'error',
                                        'msg': "L’adresse MAC {} n’est pas valide".format(macaddress),
                                        'rel': host})
                        continue
                else:
                    messages.append({'level': 'error',
                                     'msg': "L’adresse MAC est obligatoire pour une réservation",
                                     'rel': host})
                    continue

                hostname = host.get('hostname').lower()
                try:
                    _DomainnameOption('test', '', hostname, type_='domainname', allow_ip=False, allow_without_dot=True, allow_uppercase=True)
                except ValueError:
                    messages.append({'level': 'error',
                                     'msg': "Le nom d’hôte {} n’est pas valide".format(hostname),
                                     'rel': host})
                    continue
                if not hostname and prefix is None:
                    messages.append({'level': 'error',
                                     'msg': "Le nom d’hôte est obligatoire pour la réservation de {}".format(host['macaddress']),
                                     'rel': host})
                    continue
                elif hostname and (new or hostname != known_hostnames[idx]) :
                    conflicting_macaddress = None
                    for known_hostname, known_macaddress in zip(known_hostnames, known_macaddresses):
                        if hostname == known_hostname and macaddress != known_macaddress:
                            conflicting_macaddress = known_macaddress
                            break
                    if conflicting_macaddress:
                        messages.append({'level': 'error',
                                         'msg': "Le nom d’hôte {} est déjà attribué".format(hostname),
                                         'rel': host})
                        continue


                # Range name and subnet
                targeted_range = host.get('rangename')
                if targeted_range and targeted_range not in ranges:
                    messages.append({'level': 'error',
                                     'msg': "Le nom {} ne correspond à aucune plage".format(targeted_range),
                                     'rel': host})
                    continue
                if targeted_range:
                    targeted_subnet = ranges[targeted_range].get('subnet')

                # vérifier que la plage passée en paramètre pour l’affectation automatique des IP est valide.
                # la plage est valide si elle est incluse dans l’ensemble des IP attribuables
                auto_range = set()
                if hosts_range is not None:
                    if hosts_range.get('rangename', None) is not None:
                        range_desc = ranges[hosts_range['rangename']]
                        if range_desc['static'] is False and range_desc['restricted'] is False:
                            messages.append({'level': 'error',
                                            'msg': "La plage n’accepte pas de réservation",
                                            'rel': host})
                            continue
                        host['rangename'] = hosts_range['rangename']
                        targeted_range = hosts_range['rangename']
                        targeted_subnet = ranges[targeted_range].get('subnet')
                    elif hosts_range.get('low', None) is not None and hosts_range.get('high', None) is not None:
                        try:
                            range_floor = _ipaddress.ip_address(unicode(hosts_range['low']))
                            range_ceil = _ipaddress.ip_address(unicode(hosts_range['high']))
                        except:
                            messages.append({'level': 'error',
                                            'msg': "Les limites de la plage ne sont pas des IP valides",
                                            'rel': host})
                            continue
                        auto_range |= set([_ipaddress.ip_address(ip) for ip in range(int(range_floor), int(range_ceil) + 1)])
                        targeted_subnet, _ = _subnet_host_is_in({'ip': hosts_range['low']}, subnets, ranges)
                        if not auto_range.intersection(addressable_ips): # intersection instead of subset ?
                            messages.append({'level': 'error',
                                            'msg': "La plage n’est pas constituée d’adresses utilisables",
                                            'rel': host})
                            continue
                        elif auto_range.isdisjoint(available_ips):
                            messages.append({'level': 'error',
                                             'msg': 'Pas d’adresse IP disponible dans la plage "{}-{}" pour {}'.format(hosts_range['low'], hosts_range['high'], macaddress),
                                            'rel': host})
                            continue

                # Contrôle sur l’IP
                ip = host.get('ip')
                if ip:
                    # vérifier que l’IP est valide et disponible dans la plage si déclarée, ou dans un sous-réseau
                    try:
                        ip = _ipaddress.ip_address(unicode(host['ip']))
                    except Exception as err:
                        messages.append({'level': 'error',
                                         'msg': "L’adresse IP déclarée n’est pas valide : {}".format(str(err)),
                                         'rel': host})
                        continue
                    else:
                        if not ip in available_ips:
                            if new or str(ip) != known_ips[idx]:
                                messages.append({'level': 'error',
                                                'msg': "L’adresse IP {} n’est pas attribuable".format(str(ip)),
                                                'rel': host})
                                continue
                        if targeted_range is not None:
                            if ranges[targeted_range]['static'] is False:
                                messages.append({'level': 'error',
                                                'msg': "La plage déclarée n’accepte pas les IP statiques",
                                                'rel': host})
                                continue
                            elif not ip in ranges[targeted_range]['range']:
                                messages.append({'level': 'error',
                                                'msg': "L’IP n’est pas cohérente avec la plage déclarée",
                                                'rel': host})
                                continue
                            targeted_subnet = ranges[targeted_range]['subnet']
                        else:
                            targeted_subnet, targeted_range = _subnet_host_is_in(host, subnets, ranges)
                elif not targeted_range and not auto_range:
                    messages.append({'level': 'error',
                                     'msg': 'Pour créer l’hôte {} il faut une IP ou un nom de plage'.format(macaddress),
                                     'rel': host})
                    continue

                else:
                    ip = None

                # Consolidation des données de l’hôte
                # une entrée est finalement valide si elle a :
                # - une MAC, un nom dans le cas d’une autorisation d’accéder à une plage dynamique (static == False et restricted == True)
                # - une MAC, un nom et une IP dans le cas d’une réservation dans une plage statique ou hors plage.
                # On complète donc si il manque quelque chose.
                # On associe aussi, finalement, l’attribut 'dyn' qui servira à renseigner le host de la bonne façon dans le template

                if targeted_range is None or ranges[targeted_range]['static'] is True:
                    if not ip:
                        host['ip'] = str(_assign_ip(targeted_range, auto_range, ranges, available_ips))
                    host['dyn'] = False
                else:
                    host['dyn'] = True

                if targeted_range is not None:
                    host['rangename'] = targeted_range

                hostnames = cfg.dhcp.dhcp.id_dhcp.getattr('hostname', validate=validate)
                if not host.get('hostname'):
                    host['hostname'] = _make_hostname(prefix, hostnames)

                # tests complémentaires sur les valeurs :
                if known_id_dhcp:
                    macaddress_reservations = ({'ip': kr[2], 'rangename': kr[3]}
                                               for kr in zip(known_macaddresses, known_id_dhcp, known_ips, known_rangenames)
                                               if macaddress == kr[0] and c_id != kr[1])

                    if macaddress_reservations:
                        for resa in macaddress_reservations:
                            conflicting_subnet, conflicting_range = _subnet_host_is_in(resa, subnets, ranges)
                            if conflicting_subnet == targeted_subnet:
                                if conflicting_range:
                                    if ranges[conflicting_range]['static']:
                                        conflict_desc = 'a déjà une adresse IP réservée dans la plage statique {}'.format(conflicting_range)
                                    else:
                                        conflict_desc = 'est déjà associé à la plage dynamique {}'.format(conflicting_range)
                                else:
                                    conflict_desc = 'a déjà une adresse IP réservée hors plage'
                                messages.append({'level': 'error',
                                                 'msg': "L’hôte {} ({}) {} dans le réseau {}".format(host['hostname'], host['macaddress'], conflict_desc, str(conflicting_subnet)),
                                                 'rel': host})
                                continue

                if not new:
                    macaddresses = list(known_macaddresses)
                    macaddresses.pop(idx)
                    dyns = list(known_dyns)
                    dyns.pop(idx)
                    names = list(known_rangenames)
                    names.pop(idx)
                    ips = list(known_ips)
                    ips.pop(idx)
                    hostnames = list(known_hostnames)
                    hostnames.pop(idx)
                else:
                    cfg.dhcp.dhcp.last_id = c_id
                    known_id_dhcp.append(c_id)

                try:
                    idx = known_id_dhcp.index(c_id)
                except ValueError:
                    messages.append({'level': 'error', 'msg': 'unknown ID for entry {}'.format(str(host)),
                                     'rel': host})
                    continue



                known_hostnames = cfg.dhcp.dhcp.id_dhcp.getattr('hostname', validate=validate)
                known_macaddresses = cfg.dhcp.dhcp.id_dhcp.getattr('macaddress', validate=validate)
                known_dyns = cfg.dhcp.dhcp.id_dhcp.getattr('dyn', validate=validate)
                known_ips = cfg.dhcp.dhcp.id_dhcp.getattr('ip', validate=validate)
                known_rangenames = cfg.dhcp.dhcp.id_dhcp.getattr('name', validate=validate)

                try:
                    known_hostnames[idx] = host['hostname'].lower()
                    known_macaddresses[idx] = host['macaddress']
                    known_dyns[idx] = {True: "oui", False: "non"}.get(host['dyn'])
                    # FIXME enregistrer le nom de la plage dès qu’il existe
                    if host['dyn']:
                        known_rangenames[idx] = unicode(host['rangename'])
                    else:
                        if 'rangename' in host:
                            known_rangenames[idx] = unicode(host['rangename'])
                        known_ips[idx] = host['ip']
                except ValueError as err:
                    messages.append({'level': 'error', 'msg': str(err),
                                     'rel': host})
                    continue

            # applying the new DHCP configuration
            # do not save on errors, unless save_when_error is True
            if len(messages) == 0 or save_when_error:
                _save_values(cfg, messages)
    ret = {}
    if messages != []:
        ret['messages'] = messages
    return ret


def _del_reserved(cfg, messages, reserved, allreservations):
    if allreservations:
        del cfg.dhcp.dhcp.id_dhcp.id_dhcp
    else:
        if isinstance(reserved, list):
            for res in reserved:
                orig_reserved = _get_reserved(cfg, res['id'], res)['reserved'][0]
                if res != orig_reserved:
                    messages.append({'level': 'error',
                                     'msg': 'host with id {} has not '
                                            'same informations'.format(res['id']),
                                     'rel': res})
                else:
                    idx = cfg.dhcp.dhcp.id_dhcp.id_dhcp.index(res['id'])
                    cfg.dhcp.dhcp.id_dhcp.id_dhcp.pop(idx)
        else:
            messages.append({'level': 'error', 'msg': 'entries must be a list of reserved',
                             'rel': reserved})


def _save_values(cfg, messages=None):
    _config_save_values(cfg, _MODNAME, reload_config=_reload_config)
    if '__salt__' in globals():
        results = __salt__['state.apply'](_MODNAME, saltenv=_EWTAPP)
    else:
        results = None
    #COPIED FROM ewt-actions/ewt/generic_form.py
    msg = []
    if isinstance(results, list):
        msg = results
    elif isinstance(results, dict):
        for result in results.values():
            if not result['result']:
                # Action failed, try to extract error message
                err_object = result['name']
                if result.get('changes', {}):
                    # return cmd.run error ouput, or standard output if error is empty
                    err_output = result['changes'].get('stderr', '') or \
                                 result['changes'].get('stdout', '')
                else:
                    err_output = result['comment']
                    if not err_output.endswith('.'):
                        err_output += "."
                # if output is a python Traceback, only return last line
                if err_output.startswith('Traceback'):
                    err_output = err_output.split('\n')[-1].lstrip('Exception:')
                comment = "{0} ({1})".format(err_output, err_object)
                msg.append(comment)
    if messages is None:
        messages = []
    for error in msg:
        messages.append({'level': 'error', 'msg': str(error)})
    return messages


@_check_activation
def del_reserved(reserved=None, allreservations=False, *args, **kwargs):
    """
    Delete an existant DHCP route
    return True or False with error message
    """
    messages = []
    if reserved is None and allreservations is False:
        messages.append({'level': 'error', 'msg': 'del_reserved need reserved or allreservations',
                         'rel': {}})
    else:
        if kwargs["cfg"] is None:
            cfg = _get_loader()
        else:
            cfg = kwargs["cfg"]
        _del_reserved(cfg, messages, reserved, allreservations)
        if messages == []:
            messages = _save_values(cfg)
    ret = {}
    if messages != []:
        ret['messages'] = messages
    return ret
