# -*- coding: UTF-8 -*-
"""Ce module contient la définition des acteurs (au sens 'Entité qui
effectue une action' d'un EAD
"""
__metaclass__ = type

from sets import Set

class Role(object): # Role(str) ? 
    """Définition de ce qu'est un rôle"""
    def __init__(self, name, description="", baseroles=()):
        """
        @param name: Le nom du rôle
        @param description: description du rôle
        @param baseroles: Un iterable qui la liste des rôles à partir desquels
                          est construit le rôle courant
        """
        self.name = name
        self.description = description
        self.baseroles = Set(baseroles)

    def __str__(self):
        return self.name

    def __repr__(self):
        return '<Role %s at %s>' % (self.name, hex(id(self)))
    
    def __eq__(self, other):
        if other is None:
            return False
        try:
            return other.name == self.name
        except AttributeError:
            # On permet la comparaison avec une chaine
            return other == self.name

    def __hash__(self):
        return hash(self.name)

    def has_baseroles(self):
        "renvoie True ou False suivant si <self> est basé sur d'autres rôles"
        return len(self.baseroles) > 0
        
    def based_on(self, role):
        """renvoie True ou False suivant si <self> est basé sur <role>"""
        return role in self.baseroles

    

class User:
    """Utilisateur

    Un utilisateur peut avoir un ou plusieurs rôles.
    Les autorisations peuvent être attribuées soit à un rôle,
    soit à utilisateur.

    Une autorisation donnée à un utilisateur a plus de priorité
    qu'une autorisation donnée à l'un de ses rôles
    """
    def __init__(self, name, description="", roles=None):
        self.name = name
        self.description = description
        self.roles = roles or Set()

    def add_role(self, role):
        """Donne un rôle supplémentaire à l'utilisateur"""
        self.roles.add(role)

    def remove_role(self, role):
        """Retire à un rôle à l'utilisateur"""
        try:
            self.roles.remove(role)
        except KeyError:
            print "Role '%s' pas attribué à %s" % (role, self)

    def __str__(self):
        return self.name
