# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action Bacula
#
# Programme des sauvegardes bacula
#
###########################################################################

""" Actions permettant de gérer les sauvegardes bacula """
from twisted.python import log

from pyeole.bacula import add_job, del_job, display_bacula_jobs, DAY_TO_STRING, \
        run_backup, bacula_active_dir
from pyeole.lock import is_locked

from ead2.backend.config.config import BACULA_DIR, SCRIPT_DIR
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.backend.actions.tool.bacula import (USED_TEMPLATES,
                            get_bacula_form, bacula_configured,
                            get_niveau)
from ead2.backend.actions.lib.widgets import main as M, ajax

# ajout du chemin pour bacula dans le pythonpath
import sys
import re
from datetime import datetime, timedelta
sys.path.append(BACULA_DIR)
#import baculaeole as bacula
SSTITRE_DICT = {'monthly':"Programmation de sauvegarde mensuelle",
                'weekly':"Programmation de sauvegarde hebdomadaire",
                'daily':'Programmation de sauvegarde quotidienne',
                'direct':"Programmation de sauvegarde immédiate",
                'defered':"Programmation d'une sauvegarde unique différée"}


def get_time(hour, minute):
    """
    Return complete date from hour.
    :param hour: scheduled hour
    :type hour: str
    """

    if re.match(r'[0-2][0-9]', hour) and re.match(r'[0-5][0-9]', minute):
        hour = int(hour)
        minute = int(minute)
        sched_time = datetime.now().replace(hour=hour, minute=minute, second=0)
        if datetime.now() > sched_time:
            sched_time = sched_time + timedelta(days=1)
        sched_time = sched_time.strftime('%Y-%m-%d %H:%M:%S')
    else:
        raise Exception('horaire mal formaté')
    return sched_time


class Bacula(Action):
    """
    interface de planification de sauvegarde
    quatre formulaires sont proposes pour les sauvegardes
    (hebdomadaire, quotidienne, immediate et differee)
    """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'bacula'
    libelle = 'Programmation'
    category = 'Sauvegardes'
    description = 'gestionnaire de sauvegarde bacula'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'job_suppr', 'page', 'aff_save'])
    form_result = Dict(default={},
                       doc="retour de formulaire en JSON",
                       keys=['sauvegarde', 'page'])
    template = 'bacula'

    def execute(self):
        """ renvoie les donnees pour la mise en forme de l'action
            et gere les retours
        """
        # on recup le numero du serveur en cours de traitement et la requete
        params, self.server_nb = tools.get_request(self.request)
        if not bacula_active_dir():# bacula est inactif
            datas = dict(message="Les sauvegardes Bacula sont désactivées \
dans le dictionnaire",
                         redirect=ajax.call(self.server_nb,
                                            'main_status',
                                             only=False))
            return self.send(datas, frag=True)

        elif not bacula_configured(): # bacula n'est pas configuré
            datas = dict(message= "Pensez à configurer Bacula avant \
toutes sauvegardes",
                         redirect=ajax.call(self.server_nb,
                                            'bacula_config',
                                            only=False))
            return self.send(datas, frag=True)

        elif is_locked(name='eolesauvegarde', level='system'):
            # sauvegarde en cours
            if self.form_result:
                result = dict(message="!!! Programmation impossible !!!:\\n \
Une sauvegarde est en cours.",
                              redirect=ajax.call(self.server_nb,
                                                self.name,
                                                only=False))
                self.form_result = {}
            else:
                result = {'titre':self.description}
                self.page = params.get('page', ['monthly'])[0]
                result['current_save'] = "<br/>Une sauvegarde est en cours..."
            return self.send(result)

        elif params.has_key('job_suppr'): # annulation d'une sauvegarde
            self.page = params.get('page', ['monthly'])[0]
            self.cancel_save(params['job_suppr'][0])
            datas = dict(jobs=self._get_actual_saves())
            return self.send(datas, frag=True)

        elif self.form_result:
            self.page = params.get('page', ['monthly'])[0]
            try:
                datas = self.save()
                href = ajax.call(self.server_nb,
                                              self.name,
                                              only=False,
                                              page=self.page)
                if self.page == 'direct':
                    # si sauvegarde immédiate,
                    # on rajoute un délai pour rappeler la page
                    # le temps que la sauvegarde se lance
                    href = href.replace("'", "\\'")
                    href = "setTimeout('%s', 8000)" % (href,)

                datas['redirect'] = href
            except Exception, err:
                datas = dict(message=err)
            self.form_result = {}
            return self.send(datas, frag=True)

        else:
            self.page = params.get('page', ['monthly'])[0]
            result = {'titre':self.description}
            result['toexec'] = ajax.call(self.server_nb,
                                         "rapport_sav",
                                         container="save_report")
            ## 2 -
            result['sstitre'] = SSTITRE_DICT[self.page]
            result['menus'] = self._get_menu()
            result['jobs'] = self._get_actual_saves()
            result.update(get_bacula_form(self.page))
            result['valid'] = self._get_valid_btn()
            return self.send(result)

    def send(self, datas, frag=False):
        """
            Renvoie les données
        """
        if frag:
            send_method = self.send_frag
        else:
            send_method = self.send_all
        return send_method(datas,
                    template=self.template,
                    templates=USED_TEMPLATES)


    def _get_menu(self):
        """ renvoit la description du menu """
        monthly = M.Bouton(href=ajax.call(self.server_nb, self.name,
                                         page='monthly'),
                          libelle='Mensuelle',
                          icone='/image/mail.gif',
                          title='Sauvegarde mensuelle')
        weekly = M.Bouton(href=ajax.call(self.server_nb, self.name,
                                         page='weekly'),
                          libelle='Hebdomadaire',
                          icone='/image/mail.gif',
                          title='Sauvegarde hebdomadaire')
        daily = M.Bouton(href=ajax.call(self.server_nb, self.name,
                                        page='daily'),
                         libelle='Quotidienne',
                         icone='/image/mail.gif',
                         title='Sauvegarde quotidienne')
        direct = M.Bouton(href=ajax.call(self.server_nb, self.name,
                                         page='direct'),
                          libelle='Immédiate',
                          icone='/image/mail.gif',
                          title='Sauvegarde immédiate')
        defered = M.Bouton(href=ajax.call(self.server_nb, self.name,
                                         page='defered'),
                          libelle='Différée',
                          icone='/image/mail.gif',
                          title='Sauvegarde unique différée')
        if self.page == 'monthly':
            monthly['href'] = ''
        elif self.page == 'weekly':
            weekly['href'] = ''
        elif self.page == 'daily':
            daily['href'] = ''
        elif self.page == 'direct':
            direct['href'] = ''
        else:
            defered['href'] = ''
        return monthly, weekly, daily, direct, defered

    def _get_valid_btn(self):
        """
            renvoie la description du bouton valider pour le formulaire
        """
        validate = ajax.valid(self.server_nb,
                              self.name,
                              ['sauvegarde', 'page'],
                              container='return_msg_container',
                              only=False,
                              page=self.page)
        if self.page == 'direct':
            message = "Etes-vous sûr de vouloir lancer cette sauvegarde ?"
        else:
            message = "Etes-vous sûr de vouloir programmer cette sauvegarde ?"
        href = ajax.confirm(message, validate)
        return M.Submit(href=href,
                        libelle="OK",
                        title="Programmer la sauvegarde"
                        )

    def _get_actual_saves(self, lock=False):
        """ renvoit la liste des sauvegardes planifiees """
        #saves = bacula.chargedico()

        loaded_jobs = display_bacula_jobs()
        if loaded_jobs == None:
            if lock:
                return []
            else:
                return [{'sauvegarde': "Aucune sauvegarde n'est programmée"}]

        jobs = []
        for jobnumber, sauvegarde in loaded_jobs:
            suppr = ajax.call(self.server_nb,
                             self.name,
                             job_suppr=jobnumber,
                             container="joblist",
                             page=self.page)
            message = "Etes-vous sûr de vouloir supprimer cette programmation ?"
            href = ajax.confirm(message, suppr)
            btn = M.Bouton(href=href,
                            title="Annulation de la sauvegarde",
                            libelle="Annuler",
                            icone='/image/supprimer.gif',
                            _class='')
            jobs.append({'btn':btn, 'sauvegarde': sauvegarde})
        return jobs

    def save(self):
        """ gère la distribution des sauvegardes """
        if not self.form_result.has_key('sauvegarde'):
            log.err("Il manque des données pour la programmation \
de sauvegarde.")
            raise Exception("Erreur: il manque des données pour \
la programmation de sauvegarde")
        if self.page == 'monthly':
            return self.sauvegarde_mensuelle()
        elif self.page == 'weekly':
            return self.sauvegarde_hebdo()
        elif self.page == 'daily':
            return self.sauvegarde_quotidienne()
        elif self.page == 'direct':
            return self.sauvegarde_immediate()
        elif self.page == 'defered':
            return self.sauvegarde_differee()
        else:
            return 1, ''

    def sauvegarde_mensuelle(self):
        """ programme une sauvegarde mensuelle """
        result = self.form_result['sauvegarde']
        options = tools.format_form_result(result)

        try:
            heure = options['heure']
            niveau = options['niveau']
            jourdebut = options['jourdebut']
            if '' in [niveau, jourdebut, heure]:
                raise Exception('error')
        except:
            error = "Il manque des données pour la programmation d'une sauvegarde mensuelle."
            log.err(error)
            raise Exception("Erreur: {0}".format(error))
        try:
            add_job(job='monthly', level=niveau, day=jourdebut, hour=heure)
        except Exception, e:
            error = "erreur lors de la programmation de la sauvegarde mensuelle : {0}.".format(e)
            log.err(error)
            raise Exception(error)
        return dict(message="Une sauvegarde de type %s a bien été \
programmée. \\n \\n La sauvegarde s'effectuera toutes les premières nuits du %s du mois à %s:00."%(
                                  get_niveau(niveau),
                                  DAY_TO_STRING[int(jourdebut)],
                                  heure)
                   )


    def sauvegarde_hebdo(self):
        """ programme une sauvegarde hebdomadaire """
        result = self.form_result['sauvegarde']
        options = tools.format_form_result(result)

        try:
            heure = options['heure']
            niveau = options['niveau']
            jourdebut = options['jourdebut']
            if '' in [niveau, jourdebut, heure]:
                raise Exception('error')
        except:
            error = "Il manque des données pour la programmation d'une sauvegarde hebdomadaire."
            log.err(error)
            raise Exception("Erreur: {0}".format(error))
        try:
            add_job(job='weekly', level=niveau, day=jourdebut, hour=heure)
        except Exception, e:
            error = "erreur lors de la programmation de la sauvegarde hebdomadaire : {0}.".format(e)
            log.err(error)
            raise Exception("Erreur : {0}".format(error))
        return dict(message="Une sauvegarde de type %s a bien été \
programmée. \\n \\n La sauvegarde s'effectuera toutes les nuits du %s à %s:00."%(
                                  get_niveau(niveau),
                                  DAY_TO_STRING[int(jourdebut)],
                                  heure)
                   )

    def sauvegarde_quotidienne(self):
        """ programme une sauvegarde quotidienne """
        result = self.form_result['sauvegarde']
        options = tools.format_form_result(result)
        try:
            heure = options['heure']
            niveau = options['niveau']
            jourdebut = options['jourdebut']
            jourfin = options['jourfin']
            if '' in [heure, niveau, jourdebut, jourfin]:
                raise Exception('error')
        except:
            error = "Il manque des données pour la programmation d'une sauvegarde quotidienne."
            log.err(error)
            raise Exception("Erreur : {0}".format(error))
        try:
            add_job(job='daily', level=niveau, day=jourdebut, end_day=jourfin, hour=heure)
        except Exception, e:
            error = "erreur lors de la programmation de la sauvegarde quotidienne : {0}.".format(e)
            log.err(error)
            raise Exception("Erreur : {0}".format(error))

        message = "Une sauvegarde quotidienne de type %s \
a bien été programmée.\\n\\n" % get_niveau(niveau)
        message += "La sauvegarde s'effectuera de la nuit du %s à la nuit du %s à %s:00." % (
                   DAY_TO_STRING[int(jourdebut)],
                   DAY_TO_STRING[int(jourfin)],
                   heure)
        return dict(message=message)

    def sauvegarde_immediate(self):
        """
            Lance une sauvegarde immédiate
        """
        options = self.form_result['sauvegarde']
        result = tools.format_form_result(options)
        try:
            if result.has_key('niveau'):
                run_backup(level=result['niveau'])
            else:
                run_backup()
        except Exception, e:
            error = "Erreur au lancement de la sauvegarde : {0}".format(e)
            log.err(error)
            raise Exception(error)
        return dict(message="Une sauvegarde est lancée.")


    def sauvegarde_differee(self):
        """
            Lance une sauvegarde immédiate
        """
        options = self.form_result['sauvegarde']
        result = tools.format_form_result(options)
        try:
            when = get_time(result['scheduled_hour'], result['scheduled_minute'])
            run_backup(level=result['niveau'], when=when)
        except Exception, e:
            error = "Erreur au lancement de la sauvegarde : {0}".format(e)
            log.err(error)
            raise Exception(error)
        return dict(message="Une sauvegarde unique est programmée à {0}:{1}.".format(result['scheduled_hour'], result['scheduled_minute']))


    def cancel_save(self, num):
        """ Annule une sauvegarde programmee """
        del_job(int(num) + 1)
        #cmd = '%sbaculaeole.py -s %s' % (BACULA_DIR, num)
        #code, retour  = tools.command_statusoutput(cmd)

