#-*-coding:utf-8-*-
"""
    Module d'édition des acls
"""
from os.path import isdir
from pyeole.process import system_out

try:
    from horus.backend import get_groups as list_groups
    from horus.backend import get_users
    module = 'horus'
except ImportError:
    from scribe.eolegroup import Group
    list_groups = Group().get_groups
    from scribe.enseignants import Enseignant
    from scribe.eleves import Eleve
    from scribe.administratifs import Administratif
    module = 'scribe'

from ead2.backend.actions.lib.widgets import main, form, ajax
from ead2.backend.actions.lib.acls.filesystem import deformat_path_from_html, \
format_path_for_html, js_unescape

# instruction javascript pour la suppression d'un élément
# dans la liste des acls à éditer
js_suppr = "javascript:setVoid('tr_%s_%s');"
js_search = "javascript:searchAndSelect('%s', '%s', '%s')"

# dictionnaire de translation du type à son suffixe html
keydict = {'r':'read', 'w':'write', 'x':'exec'}
inverted_keydict = dict([(value, key)for key, value in keydict.items()])

# active les prints
debug = False

def get_acl_editor(server_nb, name, path):
    """ renvoie la description du formulaire d'édition des acls pour le path
        @path: chemin à éditer (fichier ou répertoire) au format
        dir__subdir__subsubdir
    """
    path = deformat_path_from_html(path)
    users = get_user_form()
    groups = get_groups_form()
    user_search = form.Input(name='user_search_input',
            libelle='Rechercher un utilisateur',
            _id='user_search_input')
    href = js_search % ('user_search_input', 'datas', 'user')
    user_search_valid = main.Bouton(href=href, title='Chercher')

    group_search = form.Input(name='group_search_input',
                              libelle='Rechercher un groupe',
                              _id='group_search_input')
    href = js_search % ('group_search_input', 'datas', 'group')
    group_search_valid = main.Bouton(href=href, title='Chercher')

    acls = get_path_acls(server_nb, name, path)
    btn = get_transf_btn("datas", "acls_form")
    close_btn = get_close_btn('acls_workspace')
    valid_btn = get_valid_btn(server_nb, name, path, "acls_form",
                              "acls_workspace_msg")
    # construction du titre
    if isdir(path):
        acl_message = "Acls du répertoire"
    else:
        acl_message = "Acls du fichier"
    return dict(titre="Édition des acls de %s" % js_unescape(path.replace('\ ', ' ')),
                users=users,
                user_search=user_search,
                user_search_valid=user_search_valid,
                groups=groups,
                group_search=group_search,
                group_search_valid=group_search_valid,
                btn=btn,
                acl_message=acl_message,
                acls=acls,
                close_btn=close_btn,
                valid_btn=valid_btn
                )

def get_user_form():
    """
        renvoie les données pour la mise en forme de la partie du formulaire
        pour la gestion des acls par user
    """
    users_select = form.Select(name='users',
            libelle='Utilisateurs',
            _id='user',
            size='20',
            multi=True)
    if module == "horus":
        users = get_users()
        users.sort()
        for user in users:
            users_select.add_option(user)
    else:
        ens = Enseignant()
        ele = Eleve()
        adm = Administratif()
        ens.ldap_admin.connect()
        ele.ldap_admin = ens.ldap_admin
        adm.ldap_admin = ens.ldap_admin
        for user in ens._get_enseignants():
            users_select.add_option(user, group='Enseignant')
        for user in adm._get_administratifs():
            users_select.add_option(user, group='Administratif')
        for user in ele._get_eleves():
            users_select.add_option(user, group='Eleve')
        ens.ldap_admin.close()
    return users_select

def get_groups_form():
    """
        renvoie les données de mise en forme de la partie du formulaire
        pour la gestion des acls par groupe
    """
    group_select = form.Select(name='group',
                               libelle='Choix du groupe',
                               _id='group',
                               size='20',
                               multi=True)
    if module == "horus":
        group_names = _get_groups()
        group_names.sort()
        for grp in group_names:
            group_select.add_option(grp)
    else:
        gr_types = ['Groupe', 'Matiere', 'Equipe',
                    'Classe', 'Option', 'Service']
        for key in gr_types:
            group_names = _get_groups(key)
            group_names.sort()
            for grp in group_names:
                group_select.add_option(grp, group=key)
        # groupes spéciaux de Scribe
        group_select.add_option('eleves', group='Autres')
        group_select.add_option('professeurs', group='Autres')
        group_select.add_option('administratifs', group='Autres')

    return group_select

def get_transf_btn(orig, dest):
    """ renvoie les données pour le transfert de données
        du formulaire orig vers le formulaire dest
        @orig: id du formulaire d'origine
        @dest: id du formulaire
    """
    href = "javascript:AclsTransferElement('%s', '%s')" % (orig, dest)
    return main.Bouton(href=href, icone="/image/leftright.png",
                        title="Ajouter des Acls")

def get_close_btn(divname):
    """
        renvoie la description du bouton pour la fermeture de la balise
    """
    href = "javascript:$('%s').style.display='none';setVoid('%s');" % (divname,
                                                                       divname)
    return main.Bouton(href=href,
            icone="/image/supprimer.gif",
            libelle="Fermer",
            _class="close_btn")

def get_valid_btn(server_nb, name, path, formname, balise):
    """
        renvoie les données pour la mise en forme du bouton
        valider pour la modification des acls
    """
    href = ajax.valid(server_nb, name,
            [formname],
            container=balise,
            path=format_path_for_html(path))
    return main.Bouton(href=href,
            libelle="Valider",
            title="Modifier les acls")

def get_path_acls(server_nb, name, path):
    """
        renvoie la description du formulaire d'édition
        des acls du chemin path
        renvoie {group_chekboxes:[(groupname, checkboxes, bouton)),
                 user_chekboxes:[(username, checkboxes, bouton)),
                 other_chekboxes:[(name, checkboxes bouton)),
                }
    """
    acls = parse_acls(path)

    owner, rights = acls['owner']
    user_checkboxes = [ get_checkboxes_and_suppr_btn(server_nb, name,
                            owner+'*', rights, path, 'user') ]
    users = acls['users']
    for user, rights in users.items():
        user_checkboxes.append(get_checkboxes_and_suppr_btn(server_nb, name,
                                user, rights, path, 'user'))

    gowner, rights = acls['gowner']
    group_checkboxes = [ get_checkboxes_and_suppr_btn(server_nb,
                            name, gowner+'*', rights, path, 'group') ]
    groups = acls['groups']
    for group, rights in groups.items():
        group_checkboxes.append(get_checkboxes_and_suppr_btn(server_nb,
                                    name, group, rights, path, 'group'))

    other = acls['other']
    other_checkboxes = [get_checkboxes_and_suppr_btn(server_nb, name,
                           'other*', other, path, 'other')]

    return dict(group_checkboxes=group_checkboxes,
                user_checkboxes=user_checkboxes,
                other_checkboxes=other_checkboxes
                )

def get_checkboxes_and_suppr_btn(server_nb, name, item, rights, path, _type='group'):
    """
        renvoie les données pour la mise en forme de checkbox
        d'édition d'acls pour l'élément item
    """
    test = lambda key, rights: key in rights

    checkboxes = []
    for key in ('r', 'w', 'x'):
        suffix = keydict[key]
        check_name = "%s_%s_%s" % (item, _type, suffix)
        checkbox = form.Checkbox(name=check_name, inline=True)
        if test(key, rights):
            checkbox['checked'] = True
        checkboxes.append(checkbox)

    href = ajax.call(server_nb,
                     name,
                     container='acls_workspace_msg',
                     _type=_type,
                     todel=item,
                     path=format_path_for_html(path))
    suppr_btn = main.Bouton(href=href,
                            title="Supprimer les Acls pour %s" % item,
                            _class="close_btn",
                            icone="/image/supprimer.gif")
    return item, checkboxes, suppr_btn


def parse_acls(path):
    """
        parse le résultat de getfacl
    """
    status, acls_lines, err_msg = system_out(['getfacl', js_unescape(path)])
    rights = {'groups':{}, 'users':{}, 'owner':{}, 'gowner':{}}
    if debug:
        print "### Les acls de %s" % path

    for line in acls_lines.splitlines():
        line = line.strip()

        if line.startswith('# owner: '):
            owner = line[9:]
            if debug:
                print "+++ Le proprio est : ", owner

        elif line.startswith('# group: '):
            group_owner = line[9:]
            if debug:
                print "+++ Le groupe proprio : ", group_owner

        elif line.startswith('user::'):
            owner_rights = line[6:]
            if debug:
                print "+++ Les droits utilisateurs : ", owner_rights

        elif line.startswith('group::'):
            group_rights = line[7:]
            if debug:
                print "+++ Les droits du groupe : ", group_rights

        elif line.startswith('group:'):
            line = line[6:].split(':')
            if len(line) == 2:
                group = line[0]
                right = line[1].strip()
                rights['groups'][group] = right
                if debug:
                    print "++ groupe avec droit %s : %s" % (group, right)
            elif len(line) >2 : # cas où on a un #effective à la suite des droits du groupe (mask qui n'a rien à faire là)
                group = line[0]
                right = line[1].strip()
                if len(group)>0:
                    rights['groups'][group] = right
                    if debug:
                        print "++ groupe avec droit %s : %s" % (group, right)
            else:
                raise Exception, "Erreur au parsing d'Acls :récupération de nom de groupe"

        elif line.startswith('user:'):
            line = line[5:].split(':')
            if len(line) == 2:
                user = line[0]
                right = line[1]
                rights['users'][user] = right
                if debug:
                    print "++ utilisateurs avec droits %s : %s" % (user, right)
            elif len(line) >2 : # cas où on a un #effective à la suite des droits du groupe (mask qui n'a rien à faire là)
                user = line[0]
                right = line[1].strip()
                if len(user)>0:
                    rights['users'][user] = right
                    if debug:
                        print "++ utilisateurs avec droits %s : %s" % (user, right)
            else:
                raise Exception, "Erreur au parsing d'Acls : récupération de nom de user"

        elif line.startswith('other::'):
            right = line[7:]
            rights['other'] = right
            if debug:
                print "++ droits autres : %s" % right

    # gestion des droits Unix à part
    rights['owner'] = (owner, owner_rights)
    rights['gowner'] = (group_owner, group_rights)

    return rights

### VALIDATION DES MODIFICATIONS D'ACLS
def valid_acl_mod(result, path):
    """
        valide la modification d'acls pour le path : path
    """
    def make_acl_dict(result):
        """
            renvoie un dico {'u':{'username':'rwx'},
                             'g':{'groupname':'rwx'}}
        """
        type_keys = {'group':'g', 'user':'u', 'other':'o'}
        get_right_type = lambda key: key.split('_')[-1]
        get_elem_type = lambda key:key.split('_')[-2]
        get_name = lambda key:'_'.join(key.split('_')[0:-2])
        translate_right = lambda key:inverted_keydict[key]
        acl_dict = {'u':{},
                    'g':{},
                    'o':{}}
        for key, value in result.items():
            _type = get_elem_type(key)
            right = get_right_type(key)
            name = get_name(key)
            _type = type_keys[_type]
            acl_dict[_type].setdefault(name, {}).setdefault(
                                translate_right(right),value)
        return acl_dict

    def valid_right(right, value):
        """
            renvoie les droits
        """
        if value:
            return right
        else:
            return '-'

    # on remplace les __ par des slashs
    path = deformat_path_from_html(path)
    acls_template = "%(_type)s:%(name)s:%(r)s%(w)s%(x)s"
    acls_lines = []
    for _type, elements in make_acl_dict(result).items():
        for name, rights in elements.items():
            # cas des propriétaires
            if name.endswith('*'):
                rname = ''
            else:
                rname = name
            acls_lines.append(acls_template % dict(_type=_type,
                                               name=rname,
                                               r=valid_right('r', rights['r']),
                                               w=valid_right('w', rights['w']),
                                               x=valid_right('x', rights['x'])))
    for cmd in acls_lines:
        setacl(cmd, path)
    return dict()

def valid_del_acl(item, _type, path):
    """
        supprime les acls pour l'item de type _type pour le path path
    """
    path = deformat_path_from_html(path)
    type_keys = {'group':'g', 'user':'u'}

    if item.endswith('*'):
        item = item[:-1]
    cmd = ['setfacl', '-RPx',
           "%s:%s" % (type_keys[_type], item),
           js_unescape(path)]
    status, msg, err_msg = system_out(cmd)
    if status:
        raise Exception, "Erreur à l'exécution de %s : %s " % (cmd, err_msg)
    if isdir(path):
        set_default_acl(path)
    return dict()


def setacl(rights, path):
    """
        exécute la commande setacl
        les droits sont prêts à l'avance
    """
    cmd = ['setfacl', '-RPm', rights, js_unescape(path)]
    status, msg, err = system_out(cmd)
    if status:
        exc = "Erreur à la mise en place des acls\\n"
        exc += "Commande : %s\\nMessage : %s"
        raise Exception, exc % (cmd, err)
    # pas d'acl par défaut sur un fichier ;)
    if isdir(path):
        set_default_acl(path)
    return True


def set_default_acl(path):
    """
        met à jour les acls par défaut pour l'héritage
        pour les repertoires mais ne gueule pas pour les fichiers
    """
    # suppression des acl par défaut
    system_out(['setfacl', '-PRk', js_unescape(path)])
    # récupération des acl actuelles
    cmd = ['getfacl', '--access', '--absolute-names', js_unescape(path)]
    ret, acl, err = system_out(cmd)
    if ret:
        exc = "Erreur lors de la récupération des acls\\n"
        exc += "Commande : %s\\nMessage : %s"
        raise Exception, exc % (str(cmd), err)
    # mise à jour des acl par défaut
    cmd = ['setfacl', '-RPd', '-M-', path]
    ret, acl, err = system_out(cmd, stdin=acl)
    if ret:
        exc = "Erreur lors de la mise en place des acls\\n"
        exc += "Commande : %s\\nEntrée : %s\\nMessage : %s"
        raise Exception, exc % (cmd, acl, err)
    return True

def _get_groups(cle=''):
    """
        fonction de mapping pour gérer le bi-module (horus, scribe)
        renvoie la liste des groupes pour la clé cle
    """
    if cle: # cas scribe
        return list_groups(cle)
    else: # cas horus
        return list_groups()

