# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action printers
# outil de gestion des imprimantes (gestion simplifiée et lien vers cups)
#
###########################################################################
from twisted.python import log
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.lib.error import MissingKey, MissingValue
from ead2.backend.actions.lib import printers_tools
from ead2.config.config import ip_locale, dico
from ead2.backend.actions.lib.widgets import ajax

class Printers(Action):
    """ outils de gestion d'imprimante
        trois types de page:
            la gestion simplifiée des imprimantes
            la liste des travaux en cours
            l'interface cups sauf pour les profs admin
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'printers'
    libelle = "Imprimantes"
    category = "Imprimantes"
    description = "Gestion des imprimantes"
    request = Dict(default={},
              doc="arguments de la requete en cours cote frontend",
              keys=['server', 'action', 'page', 'printer',
                    'stop', 'start', 'job_suppr'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json", keys=[])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie les données:
                a -récupère le nom de la page demandée
                   (statut des imprimantes par defaut)
                b - renvoie le menu et la page
            2 - renvoie les travaux en cours
            3 - action sur les imprimantes:
                a - start
                b - stop
                c - annuler une impression
        """
        params, self.server_nb = tools.get_request(self.request)

        if dico['activer_cups'] == 'non':
            datas = dict(message="La gestion des imprimantes (CUPS) \
est désactivée dans le dictionnaire",
                         redirect=ajax.call(self.server_nb,
                               'main_status',
                               only=False))
            return self.send_all(datas, template='printers')

        ## 1 - a
        self.page = 'all'
        if params.has_key('page'):
            self.page = params['page'][0]
        ## 3 - c
        if params.has_key('job_suppr'):
            try:
                datas = dict(message=self.cancel_job(params['job_suppr'][0]))
                if params.has_key('printer'):
                    name = params['printer'][0]
                    datas['toexec'] = tools.make_js_link(self.server_nb,
                                                         self.name,
                                                         balise='div_%s' % name,
                                                         confirm=True,
                                                         printer=name,
                                                         page=self.page)
                else:
                    datas['toexec'] = tools.make_js_link(self.server_nb,
                                                         self.name,
                                                         confirm=True,
                                                         page=self.page)
            except Exception, mess:
                datas = dict(message=mess)
            return self.frag(datas)
        ## 3 - b
        if params.has_key('stop'):
            try:
                datas  = self.stop_printer(params['stop'][0])
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name)
            except (MissingKey, MissingValue, Exception), mess:
                log.err("Erreur dans printers.py : %s" % mess)
                datas = {'message':str(mess)}
            params.pop('stop')
            return self.frag(datas)
        ## 3 - a
        if params.has_key('start'):
            try:
                datas  = self.start_printer(params['start'][0])
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name)
            except (MissingKey, MissingValue, Exception), mess:
                log.err("Erreur dans printers.py : %s" % mess)
                datas = {'message':str(mess)}
            params.pop('start')
            return self.frag(datas)
        ## 2 -
        if params.has_key('printer'):
            try:
                datas  = self._get_printer_jobs(params['printer'][0])
            except (MissingKey, MissingValue, Exception), mess:
                log.err("Erreur dans printers.py : %s" % mess)
                datas = {'message':str(mess)}
            params.pop('printer')
            return self.frag(datas)
        ## 1 - b
        result = {'titre':self.description, 'soustitre':''}
        result.update(self._get_datas())
        result.update(self._get_menu())
        return self.send_all(result, template='printers')

    def frag(self, datas):
        """ retourne un fragment """
        return self.send_frag(datas, template='printers')

    def _get_datas(self):
        """ renvoie les datas pour la page d'imprimante """
        if self.page == 'all':
            return self._get_printer_form()
        if self.page == 'jobs':
            return self._get_printer_jobs()

    def _get_printer_form(self):
        """ renvoie la description pour la page d'action sur les imprimantes
            :stopped: illustration du statut de l'imprimante
            :name_link: lien appelant les travaux en attente
                        de l'imprimante concernée
            :stop_link: lien appelant l'arrêt de l'imprimante
        """
        try:
            printers = printers_tools._get_printers()
        except:
            printers = []
        for printer in printers:
            if printer['stopped'] == 'ok':
                printer['stopped'] = '/image/upload/printer-stopped.gif'
                printer['stop_link'] = {'href':tools.make_js_link(self.server_nb,
                                                    self.name,
                                                    start=printer['name'],
                                                    balise='div_message_container'),
                                        'title':'Redémarrer cette imprimante',
                                        'libelle':'',
                                        'icone':'/image/redemarrer_serv.gif',
                                        '_class':'btn'}
            elif printer['stopped'] == 'idle':
                printer['stopped'] = '/image/upload/printer-idle.gif'
                printer['stop_link'] = {'href':tools.make_js_link(self.server_nb, self.name,
                                                    stop=printer['name'],
                                                    balise='div_message_container'),
                                        'title':'Arrêter cette imprimante',
                                        "libelle":'',
                                        'icone':'/image/arreter_serv.gif',
                                        '_class':'btn'}
            else:
                printer['stopped'] = '/image/upload/printer-processing.gif'
                printer['stop_link'] = {'href':tools.make_js_link(self.server_nb, self.name,
                                                    stop=printer['name'], balise='div_message_container'),
                                                    'title':'Arrêter cette imprimante', "libelle":"",
                                                    "icone":"/image/arreter_serv.gif", '_class':'btn'}
            printer['name_link'] = {'href':tools.make_js_link(self.server_nb, self.name,
                                                    printer=printer['name'], balise="div_%s" % printer['name']),
                                                    'title':'Voir cette imprimante',
                                                    "libelle":"Imprimante %s"%printer['name'], 'icone':"",
                                                    '_class':'simple_link'}
        return dict(printers=printers)

    def _get_printer_jobs(self, printer_name=None):
        """ renvoie les travaux en cours pour l'imprimante printer_name
            Toutes les imprimantes par défaut
            :close_btn: bouton pour vider le container recevant les travaux d'un imprimante
        """
        try:
            completed_jobs = printers_tools._get_printers_job(printer_name,
                                                              completed=True)
            for job in completed_jobs:
                texte = "L'impression %s lancée par '%s' le %s à %s est terminée." % (job['printid'],
                            job['user'], job['date'], job['heure'])
                job['texte'] = texte
                job.pop('user')
                job.pop('date')
                job.pop('heure')
        except:
            completed_jobs = []
        try:
            uncompleted_jobs = printers_tools._get_printers_job(printer_name)
            for job in uncompleted_jobs:
                texte = "L'utilisateur '%s' a lancé l'impression %s le %s à %s." % (job['user'],
                            job['printid'], job['date'], job['heure'])
                job['btn'] = self._get_job_cancel_btn(job['printid'],
                                                      printer_name)
                job['texte'] = texte
                job.pop('user')
                job.pop('date')
                job.pop('heure')
        except:
            uncompleted_jobs = []
        if completed_jobs == []:
            completed_jobs = [{'texte':"Aucune impression"}]
        else:
            # les 25 dernières impressions
            completed_jobs = completed_jobs[-25:]
            completed_jobs.reverse()
        if uncompleted_jobs == []:
            uncompleted_jobs = [{'texte':"Aucune impression en cours"}]
        close_btn = {'href':"javascript:setVoid('div_%s')" % printer_name,
                     'title':"Fermer",
                     'libelle':'Fermer',
                     '_class':'btn',
                     'icone':'/image/suppr.gif'}

        if printer_name != None:
            return dict(jobs=[{'entete':"<b>Travaux actifs</b>",
                               'jobs':uncompleted_jobs},
                              {'entete':"<b>Travaux terminés</b>",
                               'jobs':completed_jobs}],
                        close_btn=close_btn)
        else:
            return dict(jobs=[{'entete':"<b>Travaux actifs</b>",
                               'jobs':uncompleted_jobs},
                              {'entete':"<b>Travaux terminés</b>",
                               'jobs':completed_jobs}],
                        job_page='true')

    def _get_job_cancel_btn(self, printid, printer_name=None):
        if printer_name != None:
            # annulation depuis la page imprimantes
            href = tools.make_js_link(self.server_nb, self.name,
                                      balise="div_message_container",
                                      job_suppr=printid,
                                      printer=printer_name)
        else:
            # annulation depuis la page travaux
            href = tools.make_js_link(self.server_nb, self.name,
                                      balise="div_message_container",
                                      job_suppr=printid,
                                      page='jobs')
        title = "Annuler l'impression %s" % printid
        libelle = "Annuler"
        icone = ""
        return dict(href=href, title=title, libelle=libelle,
                    icone=icone, _class='btn')

    def stop_printer(self, printer):
        """ stoppe une imprimante """
        message = printers_tools.stop_printer(printer)
        return dict(message=message)

    def start_printer(self, printer):
        """ démarre une imprimante """
        message = printers_tools.start_printer(printer)
        return dict(message=message)

    def cancel_job(self, printid):
        """ annule l'impression printid """
        return printers_tools.cancel_print(printid)

    def _get_menu(self):
        """ renvoie le menu pour les imprimantes (toutes les imprimantes,
                               travaux d'impression en cours, cups)
        """
        all = {'title':'', 'icone':'/image/printer_create.png',
               'href':tools.make_js_link(self.server_nb, "printers",
                                         page='all'),
               'libelle':'Imprimantes'}
        jobs = {'title':"Travaux d'impression en cours",
                'icone':'/image/quota.gif',
                'href':tools.make_js_link(self.server_nb, "printers",
                                          page='jobs'),
                'libelle':"Travaux"}
        # prise en compte du mode conteneur
        if dico['mode_conteneur_actif'] == 'oui':
            adresse = dico['adresse_ip_fichier_link']
        else:
            adresse = ip_locale
        cups_url = "https://%s:631/" % adresse
        cups = {'title':"Interface d'administration CUPS",
                'icone':'/image/machines.gif',
                'href':"javascript:OuvreFenetre('%s', 'cups')" % (cups_url),
                'libelle':"Cups"}
        if self.page == 'all':
            all['href'] = ''
        elif self.page == 'jobs':
            jobs['href'] = ''
        if self.user_description['role'] == 'prof' or \
           self.user_description['role'] == 'prof_admin':
            return dict(menus=(all, jobs))
        return dict(menus=(all, jobs, cups))
