# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action scribe_devoir
# outil de gestion des devoirs
#
###########################################################################
from urllib import unquote
import tempfile, os
from os import listdir
from os.path import isdir, join, isfile
from shutil import rmtree
from string import ascii_letters, digits

from twisted.python import log
from scribe.ldapconf import SUPPORT_ETAB
from scribe import eolegroup
if SUPPORT_ETAB:
    from scribe.eoleuser import User

from ead2.backend.lib.action import Action, Dict, Text
from ead2.backend.actions import tools
from ead2.lib.error import MissingKey, MissingValue
from ead2.backend.config.config import TEMP_DIR
from ead2.backend.actions.lib.scribe_devoir import gest_devoir, pbgest
from ead2.backend.actions.lib.widgets import main as M, form as F

C = pbgest.CliPB()
getrootobject, disconnect = C.getrootobject, C.disconnect

used_templates = ['main', 'formselect', 'form']
RESERVED_NAMES = ['.distribues', '.ftp', 'Maildir', 'corrections', 'devoirs',
                  'devoir', 'donnees', 'groupes', 'perso', 'ramasses']

def comes_from_windows(filename):
    """ renvoie true si le filename est de type windows (avec \ au lieu de /)
    """
    ## FIXME : cela s'améliore peut être
    for char in ascii_letters:
        if filename.startswith(char+":\\"):
            return True
    return False

def validate_filename(dev_name):
    """
    Vérifie si le nom de devoir est valide
    """
    msg = "Nom de devoir invalide : le caractère '%s' n'est pas autorisé"
    valid_chars = "-_.%s%s" % (ascii_letters, digits)
    for car in dev_name:
        if car not in valid_chars:
            raise MissingKey, msg % car.replace('"', '\\"')

def get_menu(server_nb, page=''):
    """ renvoie la description du menu
        :page: page en cours d'affichage
    """
    distribuer = M.Bouton(href=tools.make_js_link(server_nb,
                                                  "scribe_devoir_distribuer"),
                          icone='/image/scribe/distribuer_devoir.png',
                          libelle="Distribuer")
    ramasser = M.Bouton(href=tools.make_js_link(server_nb,
                                                "scribe_devoir_ramasser"),
                        libelle="Ramasser",
                        icone='/image/scribe/ramasser_devoir.png')
    rendre = M.Bouton(href=tools.make_js_link(server_nb, "scribe_devoir_rendre"),
                      icone='/image/scribe/rendre_devoir.png',
                      libelle="Rendre")
    supprimer = M.Bouton(href=tools.make_js_link(server_nb,
                                                 "scribe_devoir_supprimer"),
                         icone="/image/scribe/supprimer_devoir.png",
                         libelle='Supprimer')
    if page == 'distribuer':
        distribuer['href'] = ''
    elif page == 'ramasser':
        ramasser['href'] = ''
    elif page == 'rendre':
        rendre['href'] = ''
    elif page == 'supprimer':
        supprimer['href'] = ''
    return dict(menus=(distribuer, ramasser, rendre, supprimer))

class DevoirDistribuer(Action):
    """ outils de gestion de la distribution de devoir
        upload de fichier, distribution ou renvoie de formulaire
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_devoir_distribuer'
    libelle = "Distribuer des documents"
    category = "Documents"
    description = "Distribuer des documents"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'fichier', 'filename',
                         'fichier_datas', 'nomdefichier', 'delete_file',
                         'delete_data'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['group', 'distrib_option'])
    filename = Text(default='', doc="Nom du fichier devoir temporaire")
    datafile = Text(default='', doc="Nom du fichier de donnees")

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie le formulaire de distribution
            2 - upload un fichier:
                a - devoir
                b - données
            3 - supprime un fichier
                a - devoir
                b - données
            4 - valide l'action de distribution
        """
        self.params, self.server_nb = tools.get_request(self.request)
        self.user_login = self.user_description['name']
        if SUPPORT_ETAB:
            self.etab = User().get_etab(self.user_login)
        else:
            self.etab = None
        ## 2 -
        if self.params.has_key('fichier') or self.params.has_key('fichier_datas'):
            try:
                datas = self._write_files()
            except (MissingKey, MissingValue, Exception), mess:
                log.err("Erreur dans devoir.py 1: %s" % mess)
                log.err()
                datas = {'message':mess.message}
            return self.frag(datas)
        ## 3 -a
        if self.params.has_key('delete_file'):
            try:
                datas = self._delete_file(self.params['delete_file'][0])
            except (Exception, MissingKey, MissingValue), mess:
                log.err("Erreur dans devoir.py 2: %s"%mess)
                log.err()
                datas = {'message':mess.message}
            return self.frag(datas)
        ## 3 - b
        if self.params.has_key('delete_data'):
            try:
                datas = self._delete_file(self.params['delete_data'][0], 'data')
            except (Exception, MissingKey, MissingValue), mess:
                log.err("Erreur dans devoir.py 3: %s"%mess)
                log.err()
                datas = {'message':mess.message}
            return self.frag(datas)
        self._set_pbroot()
        d = self.call_gd_get_devs()
        if self.form_result != {}:
            ## 4 -
            d.addCallback(self._valid_form)
        else:
            ## 1 -
            d.addCallback(self.main)
        d.addErrback(self.main_erreur)
        return d

    def frag(self, datas):
        """ utilitaires pour le renvoi de données pour un fragment de page """
        return self.send_frag(datas, template='scribe_devoir',
                              templates=used_templates)

    def _set_pbroot(self):
        """ initialise du gestionnaire de devoir """
        log.msg("Initialisation du gestionnaire de devoir")
        self.gd = gest_devoir.GestDevoir( self.user_login)

    def call_gd_get_devs(self):
        b = self.gd.get_devs()
        return b

    def main(self, message=None):
        """ fonction de renvoi en cas d'appel de l'action sans commande
        (renvoie le formulaire) """
        self.form_result = {}
        result = {'titre':"Gestion des documents", 'sstitre':self.description}
        result.update(get_menu(self.server_nb, 'distribuer'))
        if type(message) == dict:
            result.update(message)
        result['toexec'] = self._get_jstoexec_for_filelists()
        result['explication'] = "<b>La distribution s'effectue en trois étapes:</b><br />  -téléchargement des documents<br />   -téléchargement des fichiers de données (facultatif)<br />  - choix du groupe et distribution des documents"
        result.update(self._get_form())
        result.update(self._get_file_download_link())
        result.update(self._get_data_file_download_link())
        result.update(self._get_distribute_btn())
        return self.send_all(result, template='scribe_devoir',
                             templates=used_templates)

    def main_erreur(self, data=None):
        """ fonction appelée en cas d'erreur """
        log.err("Main erreur distribution : %s"%data)
        if data.type in (MissingKey, MissingValue):
            message = data.getErrorMessage()
        else:
            message = "Erreur : une erreur inconnue s'est produite à la distribution de documents."
        disconnect()
#        self.purge_temp_files()
        self.form_result = {}
        return self.frag(dict(message=message, toexec=self._get_jstoexec_for_filelists()))

    def _write_files(self, toto=None):
        """ appel l'écriture d'éventuels fichiers pour les devoirs """
        log.msg("Ecriture de fichiers.")
        toexec = None
        filedir = join(TEMP_DIR, "tempfiledir", self.user_login)
        datadir = join(TEMP_DIR, "tempdatadir", self.user_login)
        result = {'message':'Le fichier a bien été chargé'}
        if self.params.has_key('fichier'):
            ## initialisation du rep de devoir
            tempfiledir = join(TEMP_DIR, "tempfiledir")
            if not isdir(tempfiledir):
                os.mkdir(tempfiledir)
            if not isdir(filedir):
                os.mkdir(filedir)
            #fichier du devoir
            file_datas = unquote(self.params['fichier'][0]).strip()
            if file_datas != '':
                if comes_from_windows(self.params['nomdefichier'][0]):
                    filename = self.params['nomdefichier'][0].split('\\')[-1]
                else:
                    filename = self.params['nomdefichier'][0].split('/')[-1]
                file_name = join(filedir, filename) #self.params['nomdefichier'][0])
                try:
                    self.filename = self.write_tempfile(file_datas, file_name)
                except Exception, mess:
                    log.err("Erreur dans devoir.py 4: %s"%mess)
                    log.err()
                    result['message'] = mess.message
            else:
                result['message'] = "Le fichier téléchargé est vide ou inexistant."
        if self.params.has_key('fichier_datas'):
            ## initialisation du répertoire de données pour les devoirs
            tempdatadir = join(TEMP_DIR, "tempdatadir")
            if not isdir(tempdatadir):
                os.mkdir(tempdatadir)
            if not isdir(datadir):
                os.mkdir(datadir)
            # fichiers de donnees pour les devoirs
            file_datas = unquote(self.params['fichier_datas'][0]).strip()
            if file_datas != '':
                if comes_from_windows(self.params['nomdefichier'][0]):
                    filename = self.params['nomdefichier'][0].split('\\')[-1]
                else:
                    filename = self.params['nomdefichier'][0].split('/')[-1]
                file_name = join(datadir, filename) #self.params['nomdefichier'][0])
                try:
                    self.datafile = self.write_tempfile(file_datas, file_name)
                except Exception, mess:
                    log.err("Erreur dans devoir.py 5: %s"%mess)
                    log.err()
                    result['message'] = mess.message
            else:
                result['message'] = "Le fichier téléchargé est vide ou inexistant."
        toexec = self._get_jstoexec_for_filelists()
        if toexec != None:
            result['toexec'] = toexec
        return result

    def _get_form(self):
        """
        renvoie les données de mise en forme
        du formulaire de distribution de devoirs
        """
        group_select = F.Select(name='group',
                                libelle='Choix du groupe',
                                inline=True)
        gr_types = ['Classe', 'Option', 'Groupe',
                    'Matiere', 'Equipe']
        for key in gr_types:
            group_names = eolegroup.Group().get_groups(key, self.etab)
            group_names.sort()
            for gr in group_names:
                group_select.add_option(gr, group=key)
        devname = F.Input(name='devname',
                          libelle="Nom du document",
                          inline=True,
                          required=True)
        # seulement aux élèves ?
        eleves_only = F.Checkbox(name='eleves_only', inline=True,
                                 checked=True, value='eleve')
        eleve_libelle = "Uniquement aux élèves du groupe"
        # choix du repertoire
        option_entete = "ÉTAPE 4 : Choix du répertoire de destination"
        option1 = F.Radio(name='devoir_dir', value='perso',
                          inline=True, checked=True)
        option1['_libelle'] = "Dans le dossier 'perso\devoirs'"
        option2 =  F.Radio(name='devoir_dir', value='devoir', inline=True)
        option2['_libelle'] = "Dans le partage 'devoirs' (non accessible par défaut)"
        option_radios = (option1, option2)

        return dict(main_entete="ÉTAPE 1: choix du groupe et du nom du document",
                    group_select = group_select,
                    devname=devname,
                    eleves_only=eleves_only,
                    eleve_libelle=eleve_libelle,
                    option_entete=option_entete,
                    option_radios=option_radios
                )

    def _valid_form(self, toto=None):
        """ valide les retours d'action """
        if self.form_result.has_key('group') and self.form_result.has_key('distrib_option'):
            return self._distribute()
        raise Exception, "Erreur : problème de validation de formulaire, il manque des données."

    def _distribute(self):
        """ distribue des devoirs """
        result = tools.format_form_result(self.form_result['group'], 1)
        options = tools.format_form_result(self.form_result['distrib_option'])
        if not result.has_key('group'):
            raise MissingKey, "Erreur : il manque un groupe pour la distribution des documents."
        if not result.has_key('devname') or not result['devname']:
            raise MissingKey, "Erreur : il manque un nom pour le document à distribuer."
        if not result.has_key('eleves_only'):
            raise MissingKey, "Erreur : seulement aux élèves, ou pas ?"
        devoir = self._get_downloaded_filelist()
        datas = self._get_downloaded_datalist()
        group = result['group']
        devname = result['devname'].lower()
        if devname in self.gd.devs:
            raise MissingKey, "Erreur: Un document portant ce nom existe déjà"
        elif devname in RESERVED_NAMES:
            raise MissingKey , "Erreur: nom de document invalide : '%s'" % devname
        # vérification des caractères interdits (#3022)
        validate_filename(devname)
        eleves_only = result['eleves_only']
        if options.get('devoir_dir', 'perso') == 'perso':
            in_perso = True
        else:
            in_perso = False
        self.form_result = {}
        if eolegroup.Group().get_members(group) == []:
            raise MissingKey, "Erreur : ce groupe est vide."
        log.msg("distribution de %s avec %s pour le groupe %s, option in_perso:%s , eleves_only:%s" % \
                 (devoir,datas,group, in_perso, eleves_only))
        a = self.gd.distribute(devoir, datas, group, eleves_only=eleves_only,
                               in_perso=in_perso, dev_name=devname)
        a.addCallback(self.reussite_distrib)
        a.addErrback(self.erreur_distrib)
        return a

    def reussite_distrib(self, compte_rendu):
        """ fonction appelé en cas de réussite de la distribution de devoir """
        disconnect()
        files = self._get_downloaded_filelist(True)
        datas = self._get_downloaded_datalist(True)
        self.purge_temp_files()
        result = {}
        toexec = self._get_jstoexec_for_filelists()
        if toexec != None:
            result['toexec'] = toexec
#        name = self.gd.get_dist_devoirs()[-1]
        name = compte_rendu[2]
        num_devoir = compte_rendu[0]
        message = "La distribution du document %s s'est bien passée.\\n    - %s documents ont été distribués." % (name, num_devoir)
        message += "\\n    - Les documents : \\n"
        for fic in files:
            message += "          * %s \\n" % fic
        if datas != []:
            message += "\\n    - Les données jointes :\\n"
            for dat in datas:
                message += "          * %s" % dat
        result['message'] = message
        return self.frag(result)

    def erreur_distrib(self, mess):
        """ fonction d'erreur de la distribution """
        log.err("Erreur de distribution.")
        log.err()
        disconnect()
        self.purge_temp_files()
        self.form_result = {}
        return self.frag(dict(message=mess.getErrorMessage(),
                              toexec=self._get_jstoexec_for_filelists()))

## outil de gestion des fichiers téléchargés
    def write_tempfile(self, datas, tempfilename=''):
        """ utilitaire d'écriture de données"""
        if not tempfilename:
            while True:
                tempfilename = tempfile.mktemp()

                if not os.path.isfile(tempfilename) and not os.path.isdir(tempfilename):
                    break
        if datas.strip() != '':
            if isfile(tempfilename):
                raise Exception, "Erreur : ce fichier a déjà été chargé"
            fil = file(tempfilename, 'w')
            fil.write(datas)
            fil.close()
            return tempfilename
        return ""

    def purge_temp_files(self, toto=None):
        """ supprime les fichiers temporaires """
        log.msg('On purge les fichiers téléchargés.')
        filedir = join(TEMP_DIR, "tempfiledir", self.user_login)
        datadir = join(TEMP_DIR, "tempdatadir", self.user_login)
        try:
            if isdir(filedir):
                rmtree(filedir)
            if isdir(datadir):
                rmtree(datadir)
        except:
            return False
            #raise Exception, "Erreur : erreur de suppression des fichiers temporaires."
        return True

    def _delete_file(self, name, _type='file'):
        """ supprime un fichier
            :_type: 'file': fichier devoir, 'autre valeur': données pour les devoirs
        """
        if _type == 'file':
            filedir = join(TEMP_DIR, "tempfiledir", self.user_login)
        else:
            filedir = join(TEMP_DIR, "tempdatadir", self.user_login)
        filename = join(filedir, name)
        try:
            os.remove(filename)
        except Exception ,mess:
            log.err("Erreur dans devoir.py 6: %s"%mess)
            log.err()
            raise Exception, "Erreur : erreur à la suppression du fichier %s." % name
        return dict(message="Le fichier %s a été supprimé." % name,
                    toexec=self._get_jstoexec_for_filelists())

    def _get_downloaded_filelist(self, nameonly=False):
        filedir = join(TEMP_DIR, "tempfiledir", self.user_login)
        if isdir(filedir):
            files = listdir(filedir)
            if files != []:
                if nameonly :
                    return files
                return [join(filedir, fil) for fil in files]
        if nameonly:
            return []
        raise MissingKey, "Erreur: aucun document n'a été téléchargé."

    def _get_downloaded_datalist(self, nameonly=False):
        datadir = join(TEMP_DIR, "tempdatadir", self.user_login)
        if isdir(datadir):
            files = listdir(datadir)
            if nameonly : return files
            return [join(datadir, fil) for fil in files]
        return []

    def _get_jstoexec_for_filelists(self):
        # on crée les fonctions javascript à exécuter pour rafraichir la liste des fichiers
        files = self._get_downloaded_filelist(True)
        if files == []:
            files_texte = "Aucun document n'a été téléchargé<br />"
        else:
            files_texte = '<b>Les documents téléchargés :</b> <br />'
        for f in files:
            link = tools.make_js_link(self.server_nb, self.name,
                                      balise='div_message_container',
                                      delete_file=f)
            link.replace("'", "\\'")
            link.replace('"', '\\"')
            files_texte += "<a href=\\\"%s\\\"><img src='/image/supprimer.gif'/></a>  %s<br />" % (link, f)
        #toexec = "window.parent.document.getElementById('downloaded_file').innerHTML = \"%s\";"%files_texte
        toexec = "$('downloaded_file').innerHTML = \"%s\";" % files_texte
        # liste des données
        datas = self._get_downloaded_datalist(True)
        if datas == []:
            datas_texte = "Aucune donnée n'a été téléchargée<br />"
        else:
            datas_texte = '<b>Les données téléchargées : </b><br />'
        for f in datas:
            link = tools.make_js_link(self.server_nb, self.name, balise='div_message_container', delete_data=f)
            link.replace("'", "\\'")
            link.replace('"', '\\"')
            datas_texte += "<a href=\\\"%s\\\"><img src='/image/supprimer.gif'/></a>  %s<br />" % (link, f)
        #toexec += "window.parent.document.getElementById('downloaded_data').innerHTML = \"%s\";"%datas_texte
        toexec += "$('downloaded_data').innerHTML = \"%s\";" % datas_texte
        return toexec

    def _get_file_download_link(self):
        """ renvoie le lien vers le formulaire de téléchargement de fichier """
        return dict(devoir_file="./file_download/?server=%s&filename=fichier&action=%s" % (self.server_nb, self.name),
                    devoir_file_entete="ÉTAPE 2: télécharger les documents à distribuer")

    def _get_data_file_download_link(self):
        """ renvoie le lien vers le formulaire de téléchargement de fichier de données du devoir"""
        return dict(devoir_data="./file_download/?server=%s&filename=fichier_datas&action=%s"%(self.server_nb, self.name),
                    devoir_data_entete="ÉTAPE 3: télécharger les données à joindre au document")

    def _get_distribute_btn(self):
        """ renvoie les données pour la mise en forme du bouton distribuer """
        href = tools.make_form_link(self.server_nb, self.name, True, ['group', 'distrib_option'], "div_message_container")
        return dict(distribute_btn=M.Bouton(href=href,
                                            libelle="Distribuer"))

class DevoirRamasser(Action):
    """ outils de gestion du ramassage de devoir
        envoie de formulaire, ramassage de devoir
    """
    user_description = Dict(default={}, doc="description de l'exécutant", keys=['ip', 'name', 'role'])
    name = 'scribe_devoir_ramasser'
    libelle = "Ramasser des documents"
    category = "Documents"
    description = "Ramasser un document"
    request = Dict(default={}, doc="arguments de la requete en cours cote frontend",
                                keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json", keys=['collect'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie le formulaire de ramassage
            2 - valide l'action de ramassage
        """
        self.params, self.server_nb = tools.get_request(self.request)
        self.user_login = self.user_description['name']
        self._set_pbroot()
        d = self.call_gd_get_devs()
        if self.form_result != {}:
            ## 2 -
            d.addCallback(self._valid_form)
        else:
            ## 1 -
            d.addCallback(self.main)
        d.addErrback(self.main_erreur)
        return d

    def frag(self, datas):
        """ utilitaires pour le renvoi de données pour un fragment de page """
        return self.send_frag(datas, template='scribe_devoir', templates=used_templates)

    def _set_pbroot(self):
        """ initialise du gestionnaire de devoir """
        log.msg("Initialisation du gestionnaire de devoir")
        self.gd = gest_devoir.GestDevoir( self.user_login)

    def call_gd_get_devs(self):
        b = self.gd.get_devs()
        return b

    def main(self, message=None):
        """ fonction de renvoi en cas d'appel de l'action sans commande (renvoie le formulaire) """
        self.form_result = {}
        result = {'titre':"Gestion des documents", 'sstitre':self.description}
        result.update(get_menu(self.server_nb, 'ramasser'))
        result.update(self._get_form())
        result.update(self._get_collect_btn())
        return self.send_all(result, template='scribe_devoir',
                             templates=used_templates)

    def main_erreur(self, data=None):
        """ fonction appelée en cas d'erreur """
        log.err("Main erreur ramassage : %s"%data)
        disconnect()
        self.form_result = {}
        return self.frag({'message':"Erreur : une erreur inconnue s'est produite lors du ramassage de document."})

    def _get_form(self):
        """ renvoie les données de mise en forme du formulaire de rammassage de devoir """
        devoir = self.gd.get_dist_devoirs()
        log.msg("Documents distribués: %s" % devoir)
        if devoir == []:
            return dict(collect_msg="Il n'y a aucun document à ramasser.")
        else:
            collect = F.Select(name='devoir')
            for nom in devoir:
                collect.add_option(str(nom), group='Documents en attente de ramassage')
            return dict(collect=collect, collect_entete="Document à ramasser")

    def _valid_form(self, toto=None):
        """ valide les retours d'action """
        if self.form_result.has_key('collect'):
            return self._ramasse()
        raise Exception, "Erreur : problème de validation de formulaire."

    def _ramasse(self):
        """ ramasse un devoir """
        result = tools.format_form_result(self.form_result['collect'])
        if not result.has_key('devoir'):
            raise Exception, "Erreur : impossible de ramasser un document des données manquent."
        self.devoir = result['devoir'].strip()
        self.form_result = {}
        if self.devoir not in self.gd.get_dist_devoirs():
            raise Exception, "Erreur : le document sélectionné n'existe pas."
        a = self.gd.ramasse(self.devoir)
        a.addCallback(self.reussite_ramasse)
        a.addErrback(self.erreur_ramasse)
        return a

    def reussite_ramasse(self, toto=None):
        """ renvoie les données de mise en forme lors de la réussite du ramassage """
        disconnect()
        result = {'titre':"Gestion des documents", 'sstitre':self.description}
        result.update(get_menu(self.server_nb, 'ramasser'))
        result.update(self._get_form())
        result.update(self._get_collect_btn())
        message = "Le ramassage du document %s s'est bien passé." % self.devoir
        self.devoir = ''
        result['message'] = message
        return self.frag(result)

    def erreur_ramasse(self, mess):
        """ renvoie les données de mise en forme lors de l'échec du ramassage """
        log.err("Erreur de ramassage.")
        disconnect()
        self.form_result = {}
        return self.frag(dict(message=mess.getErrorMessage()))


    def _get_collect_btn(self):
        """ renvoie les données pour la mise en forme du bouton de rammassage """
        href = tools.make_form_link(self.server_nb, self.name, True, ['collect'])
        return dict(collect_btn=M.Bouton(href=href,
                                         libelle="Ramasser"))

class DevoirRendre(Action):
    """ outils de gestion du rendu de devoir
        envoie de formulaire, rendu de devoir
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_devoir_rendre'
    libelle = "Rendre des documents"
    category = "Documents"
    description = "Rendre un document (corrigé)"
    request = Dict(default={}, doc="arguments de la requete en cours cote frontend",
                                keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json", keys=['rendre'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie le formulaire de rendu
            2 - valide l'action de rendu
        """
        self.params, self.server_nb = tools.get_request(self.request)
        self.user_login = self.user_description['name']
        self._set_pbroot()
        d = self.call_gd_get_devs()
        if self.form_result != {}:
            ## 2 -
            d.addCallback(self._valid_form)
        else:
            ## 1 -
            d.addCallback(self.main)
        d.addErrback(self.main_erreur)
        return d

    def frag(self, datas):
        """ utilitaires pour le renvoi de données pour un fragment de page """
        return self.send_frag(datas, template='scribe_devoir',
                              templates=used_templates)

    def _set_pbroot(self):
        """ initialise du gestionnaire de devoir """
        log.msg("Initialisation du gestionnaire de devoir")
        self.gd = gest_devoir.GestDevoir(self.user_login)

    def call_gd_get_devs(self):
        b = self.gd.get_devs()
        return b

    def main(self, message=None):
        """ fonction de renvoi en cas d'appel de l'action sans commande (renvoie le formulaire) """
        self.form_result = {}
        result = {'titre':"Gestion des documents", 'sstitre':self.description}
        result.update(get_menu(self.server_nb, 'rendre'))
        result.update(self._get_form())
        result.update(self._get_rendre_btn())
        return self.send_all(result, template='scribe_devoir',
                             templates=used_templates)

    def main_erreur(self, data=None):
        """ fonction appelée en cas d'erreur """
        log.err("Main erreur rendu : %s"%data)
        disconnect()
        self.form_result = {}
        return self.frag({'message':"Erreur : une erreur inconnue s'est produite lors du rendu de document."})

    def _get_form(self):
        """ renvoie les données de mise en forme du formulaire de rendu de devoir """
        devoir = self.gd.get_ram_devoirs()
        log.msg("Docuementss déjà ramassés : %s" % devoir)
        if devoir == []:
            return dict(rendre_msg="Il n'y a aucun document à rendre.")
        else:
            rendre = F.Select(name='devoir')
            for nom in devoir:
                rendre.add_option(str(nom), group='Documents en attente d\'être rendus')
            return dict(rendre=rendre, rendre_entete="Document à rendre")

    def _valid_form(self, toto=None):
        """ valide les retours d'action """
        if self.form_result.has_key('rendre'):
            return self._rendre()
        raise Exception, "Erreur : problème de validation de formulaire."

    def _rendre(self):
        """ rend un devoir """
        result = tools.format_form_result(self.form_result['rendre'])
        if not result.has_key('devoir'):
            raise Exception, "Erreur : impossible de rendre un document, des données manquent."
        self.devoir = result['devoir'].strip()
        self.form_result = {}
        if self.devoir not in self.gd.get_ram_devoirs():
            raise Exception, "Erreur : le document sélectionné n'existe pas."
        a = self.gd.rendre(self.devoir)
        a.addCallback(self.reussite_rendre)
        a.addErrback(self.erreur_rendre)
        return a

    def reussite_rendre(self, toto=None):
        """ renvoie les données de mise en forme lors de la réussite du rendage """
        disconnect()
        result = {'titre':"Gestion des documents", 'sstitre':self.description}
        result.update(get_menu(self.server_nb, 'rendre'))
        result.update(self._get_form())
        result.update(self._get_rendre_btn())
        message = "Le rendu du document %s s'est bien passé." % self.devoir
        self.devoir = ''
        result['message'] = message
        return self.frag(result)

    def erreur_rendre(self, mess):
        """ renvoie les données de mise en forme lors de l'échec du rendage """
        log.err("Erreur de rendage.")
        disconnect()
        self.form_result = {}
        return self.frag(dict(message=mess.getErrorMessage()))

    def _get_rendre_btn(self):
        """ renvoie les données pour la mise en forme du bouton de rendu """
        href = tools.make_form_link(self.server_nb, self.name, True, ['rendre'])
        libelle = "Rendre"
        return dict(rendre_btn=M.Bouton(href=href,
                                        libelle="Rendre"))

class DevoirSupprimer(Action):
    """ outils de gestion de suppression des données de devoir
        envoie de formulaire, suppression de devoir
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_devoir_supprimer'
    libelle = "Supprimer les données"
    category = "Documents"
    description = "Supprimer les données relatives à un document rendu"
    request = Dict(default={}, doc="arguments de la requete en cours cote frontend",
                                keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['supprimer'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie le formulaire de suppression
            2 - valide l'action de suppression
        """
        self.params, self.server_nb = tools.get_request(self.request)
        self.user_login = self.user_description['name']
        self._set_pbroot()
        d = self.call_gd_get_devs()
        if self.form_result != {}:
            ## 2 -
            d.addCallback(self._valid_form)
        else:
            ## 1 -
            d.addCallback(self.main)
        d.addErrback(self.main_erreur)
        return d

    def frag(self, datas):
        """ utilitaires pour le renvoi de données pour un fragment de page """
        return self.send_frag(datas, template='scribe_devoir',
                              templates=used_templates)

    def _set_pbroot(self):
        """ initialise du gestionnaire de devoir """
        log.msg("Initialisation du gestionnaire de documents")
        self.gd = gest_devoir.GestDevoir( self.user_login)

    def call_gd_get_devs(self):
        b = self.gd.get_devs()
        return b

    def main(self, message=None):
        """ fonction de renvoi en cas d'appel de l'action sans commande (renvoie le formulaire) """
        self.form_result = {}
        result = {'titre':"Gestion des documents", 'sstitre':self.description}
        result.update(get_menu(self.server_nb, 'supprimer'))
        result.update(self._get_form())
        result.update(self._get_supprimer_btn())
        return self.send_all(result, template='scribe_devoir',
                             templates=used_templates)

    def main_erreur(self, data=None):
        """ fonction appelée en cas d'erreur """
        log.err("Main erreur suppression : %s"%data)
        disconnect()
        self.form_result = {}
        return self.frag({'message':"Erreur : une erreur inconnue s'est produite lors de la suppression des données d'un document."})

    def _get_form(self):
        """ renvoie les données de mise en forme du formulaire de suppression de devoir """
        devoir = self.gd.get_dev_todel()
        log.msg("Documents dont il reste des données à supprimer : %s" % devoir)
        if devoir == []:
            return dict(supprimer_msg="Il n'y aucune donnée à supprimer.")
        else:
            supprimer = F.Select(name='devoir')
            for nom in devoir:
                supprimer.add_option(str(nom), group='Documents en attente de suppression')
            return dict(supprimer=supprimer, supprimer_entete="Document rendu dont les données sont à supprimer")

    def _valid_form(self, toto=None):
        """ valide les retours d'action """
        if self.form_result.has_key('supprimer'):
            return self._supprimer()
        raise Exception, "Erreur : problème de validation de formulaire."

    def _supprimer(self):
        """ supprime les données d'un devoir """
        result = tools.format_form_result(self.form_result['supprimer'])
        if not result.has_key('devoir'):
            raise Exception, "Erreur : impossible de supprimer les données d'un document, des données manquent."
        self.devoir = result['devoir'].strip()
        self.form_result = {}
        if self.devoir not in self.gd.get_dev_todel():
            raise Exception, "Erreur : le document sélectionné n'existe pas."
        a = self.gd.suppr(self.devoir)
        a.addCallback(self.reussite_supprimer)
        a.addErrback(self.erreur_supprimer)
        return a

    def reussite_supprimer(self, toto=None):
        """ renvoie les données de mise en forme lors de la réussite de la suppression """
        disconnect()
        result = {'titre':"Gestion des documents", 'sstitre':self.description}
        result.update(get_menu(self.server_nb, 'supprimer'))
        result.update(self._get_form())
        result.update(self._get_supprimer_btn())
        message = "La suppression des données du document %s s'est bien passée." % self.devoir
        self.devoir = ''
        result['message'] = message
        return self.frag(result)

    def erreur_supprimer(self, mess):
        """ renvoie les données de mise en forme lors de l'échec de la suppression """
        log.err("Erreur de suppression.")
        disconnect()
        self.form_result = {}
        return self.frag(dict(message=mess.getErrorMessage()))

    def _get_supprimer_btn(self):
        """ renvoie les données pour la mise en forme du bouton de suppression """
        href = tools.make_form_link(self.server_nb, self.name, True, ['supprimer'])
        title = 'Supprimer les données relatives au document'
        return dict(supprimer_btn=M.Bouton(href=href,
                                           libelle="Supprimer",
                                           title=title))
