# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action scribe_connexion
# Utilisateur connectés
# Machine connectée
# Gestion des quotas utilisateurs
# Gestion des acls des partages
# Historique des connexions
#
###########################################################################
"""Actions permettant de gérer les connexions"""
from twisted.python import log
from re import escape
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.backend.actions.lib.widgets import form as F, main as M
from ead2.lib.error import MissingKey, MissingValue
from ead2.backend.config.filenames import get_client_scribe_script
from ead2.backend.actions.scribe.tool import scribe_tools
from pyeole.process import system_out

# liste des templates complémentaires utilisés
used_templates = ['main', 'form', 'formselect', 'accordion']

def get_menu(server_nb, page=''):
    """ renvoie la description du menu
        :page: page en cours d'affichage
    """
    index = M.Bouton(icone='/image/refresh.gif',
                     href=tools.make_js_link(server_nb,
                                             "scribe_connexion_index"),
                     libelle="Connectés")
    history = M.Bouton(icone='/image/hour.png',
                       href=tools.make_js_link(server_nb,
                                               "scribe_connexion_history"),
                       libelle="Historique")
    if page == 'index':
        index['href'] = ''
    elif page == 'history':
        history['href'] = ''
    return dict(menus=(history, index))

class Connected(Action):
    """
    gestion des utilisateurs connectés
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_connexion_index'
    libelle = "Connectés"
    category = "Outils/Connexion/"
    description = 'Afficher les utilisateurs connectés'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'user_file',
                         'sendmsg', 'sendmsgs'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['sendmsg', 'user_list'])

    def execute(self):
        """ renvoit les données pour l'affichage
             1 - renvoie la liste des utilisateurs connectés
             2 - renvoie les fichiers utilisés par un user
             3 - envoie un message
                a - à un utilisateur
                b - à plusieurs utilisateurs

        """
        tools.record_action(self.description, self.user_description)
        params, self.server_nb = tools.get_request(self.request)
        ## 2 -
        if params.has_key('user_file'):
            user_machine = params['user_file'][0]
            datas = self._get_user_files(user_machine)
            return self.frag(datas)
        ## 3 - a
        if params.has_key('sendmsg'):
            user_machine = params['sendmsg'][0]
            datas = self._get_sendmsg_form(user_machine)
            return self.frag(datas)
        ## 3 - b
        if params.has_key('sendmsgs'):
            datas = self._get_sendmsg_form()
            return self.frag(datas)
        if self.form_result != {}:
            if self.form_result.has_key('user_list'):
                try:
                    datas = self._valid_sendmsgs()
                except (MissingKey, MissingValue, Exception), mess:
                    log.err()
                    datas = {'message':mess.message}
                self.form_result = {}
                return self.frag(datas)
            if self.form_result.has_key('sendmsg'):
                try:
                    datas = self._valid_sendmsg()
                except (MissingKey, MissingValue, Exception), mess:
                    log.err()
                    datas = {'message':mess.message}
                self.form_result = {}
                return self.frag(datas)
            self.form_result = {}
        ## 1 -
        result = {'titre':"Gestion des connexions",
                  'sstitre':"Gestion des utilisateurs connectés"}
        result.update(get_menu(self.server_nb, page='index'))
        result.update(self._get_users())
        result.update(self._get_messages_btn())
        return self.send_all(result, template='scribe_connexion_index',
                             templates=used_templates)

    def frag(self, datas):
        return self.send_frag(datas, template='scribe_connexion_index',
                              templates=used_templates)

    def _get_users(self):
        """
        renvoie la liste des utilisateurs avec leur fichier
        en cours de consultation, nom de machine, ip
        """
        #a = tools.command_output("/usr/bin/smbstatus")
        code, a, err = system_out(['/usr/bin/smbstatus'], container='fichier')
        users = tools.parse_smbstatus(a)
        for user in users:
            user['btn'] = self._get_message_btn(user['nom'], user['machine'])
            user['checkbox'] = {'name':user['machine'], 'libelle':'',
                                'value':user['machine']}
            user['show'] = self._get_show_btn(user['nom'], user['machine'])
            user['hide'] = self._get_hide_btn(user['nom'], user['machine'])
        if len(users) >0 :
            if len(users) == 1:
                title = "Il y a %d utilisateur connecté" % len(users)
            else:
                title = "Il y a %d utilisateurs connectés" % len(users)
            return dict(users=users, user_title=title)
        return dict(users=users)

    def _get_sendmsg_form(self, user_machine=''):
        """ renvoie la description du formulaire d'envoi de message """
        close_btn = M.Bouton(href="javascript:setVoid('div_message_container')",
                             libelle="Fermer",
                             _class='btn',
                             icone="/image/suppr.gif")
        if user_machine:
            sendmsg = {'name':'message',
                       'libelle':"Message à envoyer à %s" % user_machine,
                       'rows':8, 'cols':15}
            user = F.Input(name='user_machine', default=user_machine)
            validate = self._get_valid_sendmsg_btn()
            return dict(sendmsg=sendmsg, close_btn=close_btn, user=user,
                        validate=validate)
        else:
            sendmsg = {'name':'message',
                       'libelle':"Message à envoyer aux utilisateurs sélectionnés",
                       'rows':8, 'cols':15}
            validate = self._get_valid_sendmsg_btn(True)
            return dict(sendmsg=sendmsg, close_btn=close_btn, validate=validate)

    def _valid_sendmsg(self):
        """ envoi un message à un utilisateur concerné """
        result = tools.format_form_result(self.form_result['sendmsg'])
        if not result.has_key('user_machine') or not result.has_key('message'):
            raise MissingKey, "Erreur : il manque des données pour l'envoi de message"
        machine = result['user_machine'].strip()
        message = result['message'].strip()
        return_msg = self.send_msg([machine], message)
        return dict(message=return_msg)

    def _valid_sendmsgs(self):
        """ envoi un message à un groupe d'utilisateur"""
        if not self.form_result.has_key('sendmsg'):
            raise MissingKey, "Erreur : Manque de données pour l'envoi groupé de message"
        result = tools.format_form_result(self.form_result['sendmsg'])
        user_list = tools.format_form_result(self.form_result['user_list'])
        if not result.has_key('message'):
            raise MissingKey, "Erreur : il manque des données pour l'envoi de message"
        machines = [machine.strip() for machine in user_list.keys()]
        message = result['message'].strip()
        return_msg = self.send_msg(machines, message)
        return dict(message=return_msg)

    def send_msg(self, machines, message):
        """envoi les messages"""
        if machines == []:
            raise MissingValue, "Erreur : aucune machine n'a été sélectionnée pour l'envoi de message."
        # on recoit parfois le message en unicode utf-8
        try:
            message = message.encode('utf-8')
        except:
            pass
        for machine in machines:
            #code, retour = tools.command_statusoutput("echo %s" % escape(message),
            #                                          "smbclient -NM %s" % escape(machine))
            code, retour, err = system_out(['smbclient', '-NM', escape(machine)], stdin=escape(message), container='fichier')
            if code:
                raise MissingKey, "Erreur : une erreur s'est produite lors de l'envoi d'un message à %s : \\n %s" % \
                                  (escape(machine), str(retour))
        return "L'envoi du message aux utilisateurs s'est bien passé."

    def _get_valid_sendmsg_btn(self, multi=False):
        """ renvoie la description du bouton valider pour l'envoi de message
        :multi: si True, on valide la liste des utilisateurs sélectionnés
        """
        if multi:
            href = tools.make_form_link(self.server_nb, self.name, True,
                    ['sendmsg', 'user_list'], "div_message_container")
        else:
            href = tools.make_form_link(self.server_nb, self.name, True,
                    ['sendmsg'], "div_message_container")
        return M.Bouton(href=href, libelle='Valider', _class='btn')

    def _get_message_btn(self, user, machine):
        """ renvoie la description du bouton d'envoi de message à l'utilisateur user """
        href = tools.make_js_link(self.server_nb, self.name,
                                  balise='div_message_container',
                                  sendmsg=machine)
        return M.Bouton(href=href,
                        title="Envoi de message à %s"%user,
                        icone="/image/mail_mini.png",
                        _class='btn')

    def _get_messages_btn(self):
        """ renvoie la description du bouton d'envoi groupé de message """
        href = tools.make_js_link(self.server_nb, self.name,
                                  balise='div_message_container',
                                  sendmsgs="ok")
        return dict(group_sending=M.Bouton(href=href,
                                           libelle="Envoi groupé",
                                           icone="/image/mail.gif",
                                           _class='btn'))

    def _get_show_btn(self, user, machine):
        """
        renvoie la description du bouton
        d'affichage des fichiers en cours d'utilisation
        """
        href = tools.make_js_link(self.server_nb, self.name,
                                  balise='div_file_%s_container' % machine,
                                  user_file=machine)
        return M.Bouton(href=href,
                        title="Afficher les fichiers de à %s" % user,
                        libelle="Afficher",
                        icone="",
                        _class='btn')

    def _get_hide_btn(self, user, machine):
        """
        renvoie la description du bouton de masquage
        des fichiers en cours d'utilisation
        """
        href = "javascript:setVoid('div_file_%s_container');" % machine
        return M.Bouton(href=href,
                        title="Masquer les fichiers de à %s" % user,
                        libelle="Masquer",
                        icone="",
                        _class='btn')

    def _get_user_files(self, usermachine):
        users = self._get_users()['users']
        for user in users:
            if user['machine'] == usermachine:
                if user['files'] is not None:
                    return dict(user_files=[fichier['nom'] for fichier in user['files']if fichier['nom'].strip() != '.'])
        return dict(user_files=[])

class ConnectedMachine(Action):
    """
    affichage des stations du réseau
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_connexion_machine'
    libelle = "Machines"
    category = "Outils/Stations/"
    description = 'Stations connectées au réseau'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'machine_type' ])
    form_result = Dict(default={},
                       doc="Retour de formulaire en Json",
                       keys=['user_list', 'actions'])

    def execute(self):
        """ renvoit les données pour l'affichage
            1 - récupère le type de station à afficher
            2 - renvoie la liste des staions du type choisie
                a - ainsi que le menu
                b - dans un fragment
            3 - exécute une action sur un client du domaine
        """
        params, self.server_nb = tools.get_request(self.request)
        ## 1 -
        self.machine_type = 'client'
        if params.has_key('machine_type'):
            ## 2 - b
            self.machine_type = params['machine_type'][0]
            try:
                datas = self._get_machines(self.machine_type)
                datas.update(self._get_controleur_btn())
                datas.update(self._get_maitre_btn())
                datas.update(self._get_all_btn())
                datas.update(self._get_client_btn())
            except (MissingKey, MissingValue, Exception), mess:
                log.err()
                datas = {'message':mess.message}
            return self.frag(datas)
        ## 3 -
        if self.form_result != {}:
            if self.form_result.has_key('actions'):
                try:
                    datas = self._valid_action()
                    datas['toexec'] = tools.make_js_link(self.server_nb,
                                                  self.name, confirm=True)
                except (MissingKey, MissingValue, Exception), mess:
                    log.err()
                    datas = {'message':mess.message}
                self.form_result = {}
                return self.frag(datas)
        ## 2 - a
        result = {'titre':"Gestion des connexions",
                  'sstitre':self.description}
        #result.update(get_menu(self.server_nb, page='machine'))
        try:
            result.update(self._get_machines('client'))
        except:
            result.update({'message':"Erreur inconnue au listing des machines."})
        result.update(self._get_controleur_btn())
        result.update(self._get_maitre_btn())
        result.update(self._get_all_btn())
        result.update(self._get_client_btn())
        return self.send_all(result, template='scribe_connexion_index',
                             templates=used_templates)

    def frag(self, datas):
        """ renvoie les données pour un fragment de page """
        return self.send_frag(datas, template='scribe_connexion_index',
                              templates=used_templates)

    def _get_machines(self, _type='client'):
        """ renvoie la liste des machines de type _type du réseau """
        if _type == 'client':
            return self._get_client_scribe()
        elif _type in [None, 'maitre', 'controleur', '']:
            return dict(machines=tools.list_valid_workstations(_type))
        else:
            raise MissingKey, "Erreur : Problème de validation."

    def _get_client_scribe(self):
        """ renvoie la liste des clients scribe connectés au domaine """
        #a, b = tools.command_statusoutput("%s -l -v" % get_client_scribe_script)
        a, b, err = system_out([get_client_scribe_script, '-l', '-v'], container='fichier')
        if a :
            raise MissingKey, "Erreur : impossible de récupérer les clients du domaine."
        elif b == '':
            # un strip sur une chaine vide ne renvoie pas une liste vide
            return dict(clients=[])
        else:
            clients = [cl.split(',') for cl in b.split(';')]
            client_list = [{'netbiosname':a[1].strip(),
                            'nom':a[2].strip(),
                            'ip':a[0].strip(),
                            'checkbox':{'name':a[0].strip(),
                                        'libelle':'',
                                        'value':a[0].strip()}}for a in clients]
            return dict(clients=client_list,
                        actions_radio=self._get_action_form(),
                        actions_validate=self._get_valid_actions_btn()
                       )

    def _get_action_form(self):
        """
        renvoie la description des radios
        pour les actions sur les clients connectés
        """
        stop = F.Radio(name='action', value='stop', libelle="Eteindre",
                       inline=True, checked=True)
        restart = F.Radio(name='action', value='restart',
                          libelle="Redémarrer", inline=True)
        session_close = F.Radio(name='action', value='close_session',
                                libelle="Fermer la session", inline=True)
        return [stop, restart, session_close]

    def _valid_action(self):
        """ valide l'exécution d'action sur les clients scribe """
        if not self.form_result.has_key('user_list'):
            raise MissingKey, "Erreur : il manque des utilisateurs pour l'exécution d'action."
        result = tools.format_form_result(self.form_result['actions'])
        if not result.has_key('action'):
            raise MissingKey, "Erreur : il manque des valeurs pour l'exécution d'action."
        user_list = tools.format_form_result(self.form_result['user_list'])
        action = result['action']
        users = [user for user in user_list.keys()]
        if action == 'stop':
            return self._stop_client(users)
        elif action == 'restart':
            return self._restart_client(users)
        elif action == 'close_session':
            return self._close_session(users)
        return dict(message="action inconne : %s" % action)

    def _stop_client(self, users):
        """ stop un client """
        if users == []:
            raise MissingValue, "Erreur : aucun utilisateurs n'a été sélectionné."
        if len(users) == 1:
            message = "La station : \\n "
        else:
            message = "Les stations : \\n "
        for user in users:
            #a, b = tools.command_statusoutput("%s -s %s -t 0" % (get_client_scribe_script, user))
            a, b, err = system_out([get_client_scribe_script, '-s', user, '-t', '0'], container='fichier')
            if a:
                message += "- %s (une erreur s'est produite)\\n" % user
            else:
                message += "  - %s\\n" % user
        if len(users) == 1:
            message += "a bien été arrêtée."
        else:
            message += "ont bien été arrêtées."
        return dict(message = message)

    def _restart_client(self, users):
        """ redémarrage d'un client """
        if users == []:
            raise MissingValue, "Erreur : aucun utilisateurs n'a été sélectionné."
        if len(users) == 1:
            message = "La station : \\n "
        else:
            message = "Les stations : \\n "
        for user in users:
            #a, b = tools.command_statusoutput("%s -s %s -t 1" % (get_client_scribe_script, user))
            a, b, err = system_out([get_client_scribe_script, '-s', user, '-t', '1'], container='fichier')
            if a:
                message += "- %s (une erreur s'est produite)\\n" % user
            else:
                message += "  - %s\\n" % user
        if len(users) == 1:
            message += "a bien été redémarrée."
        else:
            message += "ont bien été redémarrées."
        return dict(message = message)

    def _close_session(self, users):
        """ fermeture de session du client """
        if users == []:
            raise MissingValue, "Erreur : aucun utilisateurs n'a été sélectionné."
        if len(users) == 1:
            message = "La session sur la station : \\n "
        else:
            message = "Les sessions sur les stations : \\n "
        for user in users:
            #a, b = tools.command_statusoutput("%s -s %s -t 2" % \
            #             (get_client_scribe_script, user))
            a, b, err = system_out([get_client_scribe_script, '-s', user, '-t', '2'], container='fichier')
            if a:
                message += "- %s (une erreur s'est produite)\\n" % user
            else:
                message += "  - %s\\n" % user
        if len(users) == 1:
            message += "a bien été fermée."
        else:
            message += "ont bien été fermées."
        return dict(message = message)

    def _get_valid_actions_btn(self):
        """
        renvoie la description du bouton de validation
        de l'execution d'action sur les postes clients
        """
        href = tools.make_form_link(self.server_nb, self.name, True,
                                    ['actions', 'user_list'], 'div_message')
        return M.Bouton(href=href,
                    title="Exécuter une action sur le client",
                    libelle="Exécuter")

    def _get_controleur_btn(self):
        """
        renvoie la description des boutons d'affichage
        des controleurs de domaine
        """
        href = tools.make_long_js_link(self.server_nb, self.name,
                                       balise='div_station_container',
                                       machine_type="controleur")
        title = "Afficher les contrôleurs de domaine du réseau"
        libelle = "Contrôleurs de domaine"
        if self.machine_type == 'controleur':
            return dict(controleur_btn=M.Bouton(href=href,
                                                title=title,
                                                libelle=libelle,
                                                icone="",
                                                _class='scribe_machine_active'))
        else:
            return dict(controleur_btn=M.Bouton(href=href,
                                                title=title,
                                                libelle=libelle,
                                                icone="",
                                                _class='scribe_machine'))

    def _get_maitre_btn(self):
        """
        renvoie la description des boutons d'affichage des maitres explorateurs
        """
        href = tools.make_long_js_link(self.server_nb, self.name,
                balise='div_station_container', machine_type="maitre")
        title = "Afficher les maîtres explorateurs du réseau"
        libelle = "Maîtres explorateurs"
        if self.machine_type == 'maitre':
            return dict(maitre_btn=M.Bouton(href=href,
                                            title=title,
                                            libelle=libelle,
                                            icone="",
                                            _class='scribe_machine_active'))
        else:
            return dict(maitre_btn=M.Bouton(href=href,
                                            title=title,
                                            libelle=libelle,
                                            icone="",
                                            _class='scribe_machine'))

    def _get_all_btn(self):
        """
        renvoie la description des boutons d'affichage
        de toutes les stations du réseau
        """
        href = tools.make_long_js_link(self.server_nb, self.name,
                balise='div_station_container', machine_type="")
        return dict(all_btn=M.Bouton(href=href,
                                     title="Afficher toutes les stations du réseau",
                                     libelle="Toutes les stations du réseau",
                                     icone="",
                                     _class='scribe_machine'))

    def _get_client_btn(self):
        """
        renvoie la description des boutons
        d'affichage des clients du domaine
        """
        href = tools.make_long_js_link(self.server_nb, self.name,
                balise='div_station_container', machine_type='client')
        title = "Afficher les clients Scribe connectés au domaine"
        libelle = "Clients Scribe"
        if self.machine_type == 'client':
            return dict(client_btn=M.Bouton(href=href,
                                            title=title,
                                            libelle=libelle,
                                            icone="",
                                            _class='scribe_machine_active'))
        else:
            return dict(client_btn=M.Bouton(href=href,
                                            title=title,
                                            libelle=libelle,
                                            icone="",
                                            _class='scribe_machine'))

class ConnectedQuotas(Action):
    """
    affichage des quotas des utilisateurs du réseau
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_connexion_quota'
    libelle = "Quotas disque"
    category = "Outils/"
    description = 'Affichage des quotas utilisateurs'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'quota'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json", keys=[])

    def execute(self):
        """ renvoit les données pour l'affichage
            1 - renvoie le formulaire de choix des quotas à afficher (proche, dépassé ou tous)
            2 - valide le retour de formulaire et renvoie la liste des utilisateurs concernés
        """
        params, self.server_nb = tools.get_request(self.request)
        result = {'titre':"Gestion des connexions", 'sstitre':self.description}
        ## 2 -
        if params.has_key('quota') and params['quota'][0] != 'aucun':
            try:
                datas = self._valid_quota(params)
            except (MissingValue, MissingKey, Exception), mess:
                log.err()
                datas = {'message':mess.message}
            self.form_result = {}
            return self.send_frag(datas, template='scribe_connexion_quota',
                                  templates=used_templates)
        ## 1 -
        result.update(self._get_form())
        #result.update(get_menu(self.server_nb, page='quota'))
        return self.send_all(result, template='scribe_connexion_quota',
                             templates=used_templates)

    def _get_form(self):
        """ renvoie les données pour la mise en page du formulaire """
        # Appel javascript de l'action avec pour option quota=l'option sélectionnée
        onchange = tools.make_js_link(self.server_nb, self.name,
                                      code=True,
                                      balise="quota_table",
                                      quota='this.options[this.selectedIndex].value')
        quota_select = F.Select(name='quota',
                                libelle='Afficher les quotas selon le filtre: ',
                                inline=True,
                                onchange=onchange)
        quota_select.add_option(value='depasse', libelle="dépassement de quota")
        quota_select.add_option(value='proche', libelle='quotas à surveiller')
        quota_select.add_option(value='tous', libelle='tous les quotas')
        quota_select.add_option(value='aucun', libelle=' ', default=True)
        return dict(quota_select=quota_select)

    def _valid_quota(self, params):
        """ valide le formulaire pour l'affichage des quotas """
        quota_type = params['quota'][0]
        quotas = scribe_tools._get_quotas(quota_type)
        return dict(quotas = quotas,
                    quota_entete1="Utilisateur",
                    quota_entete2="Espace utilisé",
                    quota_entete3="Délai éventuel")

class ConnectedHistory(Action):
    """
    affichage de l'historique de connexion
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_connexion_history'
    libelle = "Historique"
    category = "Outils/Connexion/"
    description = 'Historique des connexions'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'quota'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json", keys=[])

    def execute(self):
        """ renvoit les données pour l'affichage """
        params, self.server_nb = tools.get_request(self.request)
        result = {'titre':"Gestion des connexions", 'sstitre':self.description}
        result.update(self._get_history())
        result.update(get_menu(self.server_nb, page='history'))
        return self.send_all(result, template='scribe_connexion_quota',
                             templates=used_templates)

    def _get_history(self):
        """ renvoie les données pour la mise en page du formulaire """
        try:
            return dict(history=scribe_tools._get_history(),
                        hist_entete1="Date",
                        hist_entete2="Utilisateurs",
                        hist_entete3="Ordinateurs",
                        hist_entete4="OS")
        except:
            return dict(history=[{'default':"Aucune entrée n'a été enregistrée"}],
                        hist_entete1="Date",
                        hist_entete2="Utilisateurs",
                        hist_entete3="Ordinateurs",
                        hist_entete4="OS")

