# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action gestion_user*
#
# Cree Modifie et Supprime des utilisateurs
#
###########################################################################

"""Actions permettant de gérer les utilisateurs"""
from twisted.python import log
from fichier.quota import set_quota
from horus import backend as horus
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.backend.actions.lib.widgets import main as M, form as F, ajax
from ead2.backend.actions.horus import gestion_tools as G
from ead2.lib.error import MissingKey, MissingValue, \
BadPassword, BadLogin, BadDriveLetter

titre = 'Gestion des utilisateurs'
used_templates = ['form', 'main', 'formselect', 'listlist',
                  'accordion', 'checklist']

def _get_groups():
    """ renvoie la liste des groupes """
    try:
        grs = horus.get_groups()
    except:
        grs = []
    return grs

def _get_users():
    """ renvoie la liste des utilisateurs """
    try:
        usrs = horus.get_users()
    except:
        usrs = []
    return usrs


def get_role_association_btn(server_nb, username):
    """ renvoie la descritpion du bouton d'appel pour l'édition de rôle """
    return M.Bouton(href=ajax.call(server_nb, 'role_manager',
                               'div_role_container',
                               form='usertoassociate',
                               user=username),
                    libelle="Associer un rôle à cet utilisateur",
                    icone="/image/action_role.png",
                    _class='scribe_action_btn'
                    )

class GestionUserCreate(Action):
    """ création d'utilisateur
    """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'gestion_user_create'
    libelle = "Création"
    category = "Gestion/Utilisateurs"
    description = 'Créer un utilisateur'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', '_type'])
    form_result = Dict(default={},
                       doc="Retour de formulaire en Json",
                       keys=['user'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - Renvoie le formulaire de création d'utilisateur ainsi que le menu
            2 - Renvoie seulement le formulaire (si la création a réussi)
            3 - Traite le retour formulaire et valide la création utilisateur
        """
        params, self.server_nb = tools.get_request(self.request)
        ## 2 -
        if params.has_key('_type'):
            datas = self._get_form()
            datas.update(self._get_valid_btn())
            return self.send_frag(datas,
                                  template='horus_gestion_user',
                                  templates=used_templates)
        ## 3 -
        if self.form_result != {}:
            try:
                datas = self.create_user(self.form_result)
                datas['toexec'] = ajax.call(self.server_nb, self.name,
                                            'user_form_div_container',
                                            only=False,
                                            _type='form')
            except (Exception, MissingKey, MissingValue, BadLogin, BadPassword), mess:
                log.err()
                datas = dict(message=mess.message)
            self.form_result = {}
            return self.send_frag(datas,
                                  template='horus_gestion_user',
                                  templates=used_templates)
        ## 1 -
        menu = G.get_user_menu(self.server_nb, self.name)
        sstitre = self.description
        result = {'sstitre':sstitre, 'titre':titre, 'menus':menu}
        result.update(self._get_form())
        result.update(self._get_valid_btn())
        result['users'] = ['<b>Utilisateurs existants:</b>']
        result['users'].extend(_get_users())
        return self.send_all(result,
                             template='horus_gestion_user',
                             templates=used_templates)

    def _get_form(self):
        """ renvoie les datas pour la creation du formulaire """
        user_name = F.Input('user_name', "Nom de l'utilisateur",
                            required=True, inline=True)
        password = F.Input('password', "Mot de passe",
                           required=True, inline=True)
        force_password = F.Checkbox('force_password',
                                    'Forcer la modification du mot de passe à la 1ère connexion',
                                    inline=True)
        quota = F.Input('quota', "Quota utilisateur",
                        default='0', inline=True)

        grs = _get_groups()
        if "DomainAdmins" in grs:
            grs.remove("DomainAdmins")
        if 'DomainUsers' in grs:
            grs.remove('DomainUsers')
        group_select = F.Select(name='group',
                                libelle='Groupe principal',
                                inline=True)
        group_select.add_option('DomainUsers', default=True)
        for grp in grs:
            group_select.add_option(grp)

        profil = F.Select('profil', 'Profil utilisateur', inline=True)
        for pro in ('itinerant', 'local', 'obligatoire'):
            profil.add_option(pro, group='Profils disponibles')
        profil.default('local', group='Profils disponibles')

        # lettre de lecteur pour repertoire perso
        disque = F.Select(name='disque',
                    libelle="Lettre de lecteur ('<b>U:</b>' conseillé)",
                    inline='ok')
        for dd in G._get_dds():
            if dd == "U:":
                disque.add_option(dd, default=True)
            else:
                disque.add_option(dd)

        enable = F.Checkbox('enable', "Activer l'utilisateur", value='on',
                            checked=True, inline=True)
        group_copy = F.Select('copy',
                              "Copier les groupes d'un autre utilisateur",
                              inline=True)
        group_copy.add_option('', default=True)
        for user in _get_users():
            group_copy.add_option(user)
        admin = F.Checkbox('admin',
                           "Membre du groupe DomainAdmins",
                           inline=True)
        shell = F.Checkbox('shell',
                           "Activation du shell (gestion de clients Linux)",
                           inline=True)
        return dict(user_name=user_name,
                    password=password,
                    force_password=force_password,
                    quota=quota,
                    group_select=group_select,
                    profil=profil,
                    disque=disque,
                    enable=enable,
                    group_copy=group_copy,
                    admin=admin,
                    shell=shell,
                    focus='user_name')

    def _get_valid_btn(self):
        """ renvoie la description du bouton valider """
        href = ajax.valid(self.server_nb, self.name, ['user'],
                          'user_msg_div_container')
        validate = M.Submit(href, title="Créer ce nouvel utilisateur")
        validation = tools.make_key_form_link(self.server_nb,
                                              self.name, False,
                                              ['user'],
                                              'user_msg_div_container')
        return dict(validate=validate, validation=validation)

    def create_user(self, dico):
        """ crée un utilisateur avec un dico récupéré depuis l'interface
            :user_name: nom utilisateur
            :group: groupe principal
            :disque: lettre du lecteur associe
            :enable: actif ou pas
            :profil: 'local' ou 'itinerant'
            :copy: utilisateur modele pour la copie des groupes d'appartenance
            :shell: shell actif ou non
        """
        if not dico.has_key('user'):
            raise MissingKey, "Erreur : il manque des données pour créer l'utilisateur."

        result = tools.format_form_result(dico['user'], True)
        for key in ['user_name', 'enable', 'disque', 'quota', 'password',
                    'copy', 'profil', 'admin', 'group', 'shell']:
            if key not in result.keys():
                raise Exception, "Erreur : il manque des données \
pour la création d'utilisateur."

        name = result['user_name']
        if not tools.test_login(name):
            raise BadLogin

        if result['profil'] == 'itinerant':
            profil = True
        elif result['profil'] == 'obligatoire':
            profil = 'obligatoire'
        else:
            profil = False

        dd = result['disque']

        copy = result['copy']
        if copy != '':
            try:
                grs = horus.get_user_groups(copy)
            except:
                raise Exception, "Erreur : erreur à la récupération \
des groupes de %s" % copy
        else:
            grs = []
        gr = result['group'].strip()

        if result['admin']:
            grs.append(gr)

        if result['quota'] != '':
            quota = result['quota']
        else:
            quota = '0'
        if not tools.test_number(quota):
            raise Exception, "Erreur : le quota doit être un nombre."

        if result['password'] != '':
            pwd = result['password']
        if not tools.test_pwd(pwd):
            raise BadPassword

        for i in [name, pwd, gr]:
            if i == '' or not tools.test_ldap_attribute(i):
                raise Exception, "Erreur : une variable est mal \
formée ou non renseignée."

        activity = result['enable']
        shell = result['shell']

        if not horus.is_user(name):
            add = horus.add_user(name, gr, grs, dd, profil, shell)
            if add:
                set_quota(name, quota)
                horus.mod_password(name, pwd)
                horus.set_active_users(name, activity)
                if result.has_key('force_password'):
                    if result['force_password']:
                        horus.password_must_change(name)
                return dict(message="L'utilisateur %s a bien été créé." % name)
            else:
                raise Exception, "Erreur : erreur à la \
création de l'utilisateur."
        else:
            raise Exception, "Erreur : cet utiilisateur existe déjà"

class GestionUserModify(Action):
    """ modification d'utilisateur """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'gestion_user_modify'
    libelle = "Modification"
    category = "Gestion/Utilisateurs"
    description = 'modifier un utilisateur'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'current_user'])
    form_result = Dict(default={},
                       doc="retour de formulaire en Json",
                       keys=['user', 'active', 'unactive'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - Tente de récupérer le nom de l'utilisateur
                en cours de traitement
            2 - Renvoie le formulaire de modification utilisateur
            3 - Traite le retour formulaire et valide la
                modification d'utilisateur
        """
        params, self.server_nb = tools.get_request(self.request)
        self.current_user = ''
        ## 3 -
        if self.form_result != {}:
            try:
                datas = self.modify_user(self.form_result)
                datas['toexec'] = ajax.call(self.server_nb, self.name,
                                            current_user=self.current_user)
            except (Exception, MissingKey, MissingValue, BadPassword), mess:
                log.err()
                datas = dict(message=mess.message)
            self.form_result = {}
            return self.send_frag(datas,
                                  template='horus_gestion_user',
                                  templates=used_templates)
        ## 1 -
        if 'current_user' in params.keys():
            self.current_user = params['current_user'][0]
        ## 2 -
        menu = G.get_user_menu(self.server_nb, self.name)
        sstitre = self.description
        result = {'sstitre':sstitre,
                  'titre':titre,
                  'menus':menu}
        result.update(self._get_form())
        if self.current_user:
            result['usertoassociate'] = {'name':'user',
                                         'default_value':self.current_user}
            result['role'] = get_role_association_btn(self.server_nb,
                                                      self.current_user)
        return self.send_all(result,
                             template='horus_gestion_user',
                             templates=used_templates)

    def _get_user_listlist(self):
        """ renvoie les datas pour les listes de choix des groupes """
        listlist = F.ListList("Associer des groupes à l'utilisateur",
                              'user_group_list')
        listlist.set_title('left', "Groupes disponibles")
        listlist.set_title('right', "Groupes de l'utilisateur")
        listlist.set_formname('left', "unactive")
        listlist.set_formname('right', "active")
        for gr in self._get_unactive_groups(self.current_user):
            listlist.add_item('left', {'name':gr})
        for gr in self._get_active_groups(self.current_user):
            listlist.add_item('right', {'name':gr})
        listlist.add_btns()
        return listlist

    def _get_form(self):
        """ renvoie les données pour le formulaire de modification utilisateur """
        usrs = _get_users()
        # choix du nom de user
        onchangelink = 'this.options[this.selectedIndex].value'
        user_name = F.Select(name='user_name',
                             libelle="Nom de l'utilisateur",
                             onchange=ajax.call(self.server_nb,
                                                self.name,
                                                js_args=True,
                                                current_user=onchangelink))

        if not self.current_user: # on a pas encore choisi de user
            default = ''
            user_name.add_option(default, default=True)
            for usr in usrs:
                user_name.add_option(usr)
            return dict(user_select=user_name)
        else:
            usrs.remove(self.current_user)
            default = self.current_user
        user_name.add_option(default, default=True)
        for usr in usrs:
            user_name.add_option(usr)

        datas = self._get_user_data()

        password = F.Input('password', "Mot de passe",
                          inline=True)
        force_password = F.Checkbox('force_password',
          'Forcer la modification du mot de passe à la prochaine connexion',
                                    inline=True)
        # quota
        current_quota = datas.get('current_quota', '')
        quota = F.Input('quota', "Quota utilisateur",
                        default=current_quota, inline=True)
        #construction de la liste des groupes
        current_gr = datas.get('current_gr', '')
        group_select = F.Select('group', 'Groupe principal',
                                                 inline=True)
        for gr in _get_groups():
            if current_gr == gr:
                group_select.add_option(gr, default=True)
            else:
                group_select.add_option(gr)

        # construction de la liste des profils
        current_profil = datas.get('current_profil', 'local')
        profil = F.Select('profil', 'Profil utilisateur',
                           inline=True)
        for pr in ('itinerant', 'local', 'obligatoire'):
            if current_profil == pr:
                profil.add_option(pr, default=True)
            else:
                profil.add_option(pr)

        #construction de la liste des lettres de lecteur
        current_dd = datas.get('current_dd', '')
        disque = F.Select('disque',
                          "Lettre de lecteur ('<b>U:</b>' conseillé)",
                          inline=True)
        for dd in G._get_dds():
            if current_dd == dd:
                disque.add_option(dd, default=True)
            else:
                disque.add_option(dd)

        #statut d'activite d'un utilisateur
        enable = F.Checkbox(name='enable',
                            value='on',
                            libelle="Activer l'utilisateur",
                            inline=True,
                            checked=datas.get('enable', True))
        # activation du shell
        shell = F.Checkbox(name='shell',
                           libelle="Activation du shell (gestion de clients Linux)",
                           inline='ok',
                           checked=datas.get('shell', False))
        # copie de groupes
        group_copy = F.Select(name='copy',
                              libelle="Copier les groupes d'un autre utilisateur",
                              inline=True)
        group_copy.add_option('', default=True)
        for usr in usrs:
            if usr != self.current_user:
                group_copy.add_option(usr)

        user_group_list = self._get_user_listlist()
        return dict(user_select=user_name,
                    password=password,
                    force_password=force_password,
                    quota=quota,
                    group_select=group_select,
                    profil=profil,
                    mod_disque=disque,
                    enable=enable,
                    group_copy=group_copy,
                    validate=self._get_valid_btn(),
                    user_group_list=user_group_list,
                    shell=shell)

    def modify_user(self, dico):
        """ modifie les attributs d'un utilisateur depuis un dico de type: {'user':dico de modif, 'unactive':, 'active':}
            :user:
                :user_name: nom de l'utilisateur
                :password: mot de passe
                :profil:
                :quota:
                :disque:
                :enable: active l'utilisateur ?
                :group: groupe principal
                :copy: nom d'un utilisateur dont on doit copier les groupes
            :active:
                liste des groupes de l'utilisateur
            :unactive:
                liste des groupes auxquels l'utilisateur n'appartient pas
        """
        # recuperation du formulaire
        if not dico.has_key('user'):
            raise MissingKey, "Erreur : il manque des données pour la validation de formulaire."
        recup = tools.format_form_result(dico['user'], True)
        # recuperation du nom d'utilisateur
        self.current_user = recup['user_name']
        if not self.current_user:
            raise Exception, "Erreur : il manque le nom de l'utilisateur à modifier."
        message="L'utilisateur %s a bien été modifié" % self.current_user
        # recuperation des infos utilisateurs
        user_datas = self._get_user_data()

        # modification du quota de l'utilisateur
        quota = recup.get('quota', None)
        if quota != None and quota != user_datas.get('current_quota', ''):
            if not tools.test_number(quota):
                raise Exception, "Erreur : Le quota doit être un nombre."
            elif set_quota(self.current_user, quota) != 0:
                raise Exception, "Erreur : erreur d'attribution de quota."
        # modification du mot de passe
        pwd = recup.get('password', '')
        if pwd:
            if not tools.test_pwd(pwd):
                raise BadPassword
            else:
                horus.mod_password(self.current_user, pwd)

        if recup.get('force_password', False):
            horus.password_must_change(self.current_user)

        # Activer le compte utilisateur
        enable = recup.get('enable', False)
        if enable == user_datas['enable']:
            enable = None

        # profil
        profil = recup['profil']
        if profil == user_datas['current_profil']:
            profil = None
        elif profil == 'itinerant':
            profil = True
        elif profil == 'local':
            profil = False
        elif profil:
            profil = 'obligatoire'

        # disque dur
        dd = recup.get('disque', '')
        dd = dd.upper()
        if ':' not in dd:
            dd += ':'
        if not tools.test_drive_letter(dd):
            raise BadDriveLetter
        elif dd == user_datas['current_dd']:
            dd = None

        # shell
        shell = recup.get('shell', False)
        if shell == user_datas['shell']:
            shell = None

        # groupe principal
        gr = recup.get('group', '')
        if gr == user_datas.get('current_gr', ''):
            gr = None
        res = horus.mod_user(self.current_user, homedrive=dd, enable=enable,
                             profile=profil, gid=gr, shell=shell)
        if not res:
            raise Exception, "Echec lors de la modification de l'utilisateur %s" % self.current_user

        # copier les groupes d'un autres user et groupe de l'utilisateur
        if recup.has_key('copy'):
            try:
                groups_to_copy = horus.get_user_groups(recup['copy'])
            except:
                groups_to_copy = []
        actual_groups = self._get_active_groups(self.current_user)
        actual_no_groups = self._get_unactive_groups(self.current_user)
        for gr in dico.get('active', []):
            if gr['name'] not in actual_groups:
                horus.inscription(self.current_user, gr['name'])
        for ugr in dico.get('unactive', []):
            if ugr['name'] not in groups_to_copy:
                if ugr['name'] not in actual_no_groups:
                    if ugr['name'] == recup.get('group', ''):
                        message = "Désinscription du groupe %s impossible :\\n"
                        message+= "C'est le groupe principal de l'utilisateur %s."
                        message = message % (ugr['name'], self.current_user)
                    else:
                        horus.desinscription(self.current_user, ugr['name'])
            else:
                horus.inscription(self.current_user, ugr['name'])

        return dict(message=message)

    def _get_user_data(self):
        """ renvoie les attributs de user """
        user = self.current_user
        if not user:
            return {}
        if not horus.is_user(user):
            return {}
        try:
            datas = horus.get_user_data(user)
        except:
            return {}
        if 'sambaProfilePath' in datas.keys():
            if datas['sambaProfilePath'][0].endswith("\\netlogon\\profil"):
                profil = 'obligatoire'
            else:
                profil = 'itinerant'
        else:
            profil = 'local'
        try:
            actives = horus.get_active_users()
            if user in actives:
                enable = True
            else:
                enable = False
        except:
            enable = True
        dd = datas['sambaHomeDrive'][0]
        gr = horus.get_gid(user)
        if datas.has_key('loginShell') and datas['loginShell'][0] == '/bin/bash':
            shell = True
        else:
            shell = False
        quota = horus.get_quota(user)
        return {'current_gr':gr, 'current_profil': profil, 'current_dd': dd,
                'current_quota':str(quota), 'enable':enable, 'shell':shell}

    def _get_active_groups(self, user):
        """ renvoie les groupes actifs pour user"""
        if user is None:
            return []
        try:
            usr_grs = horus.get_user_groups(user)
        except:
            usr_grs = []
        return usr_grs

    def _get_unactive_groups(self, user):
        """ renvoie les groupes inactifs pour user"""
        if user is None:
            return []
        try:
            usr_grs = horus.get_user_no_groups(user)
        except:
            usr_grs = []
        return usr_grs

    def _get_valid_btn(self):
        """ renvoie la description du bouton valider """
        href = ajax.valid(self.server_nb, self.name,
                        ['user', 'active', 'unactive'],
                        'user_msg_div_container')
        return M.Bouton(href=href,
                        libelle="Valider",
                        title="Modifier l'utilisateur")

class GestionUserDelete(Action):
    """ Suppression d'utilisateur """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'gestion_user_suppr'
    libelle = "Suppression"
    category = "Gestion/Utilisateurs"
    description = 'Supprimer un utilisateur'
    request = Dict(default={},
            doc="arguments de la requete en cours cote frontend",
            keys=['server', 'action'])
    form_result = Dict(default={},
            doc='retour formulaire pour la suppression',
            keys=['user'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie le formulaire de suppression d'utilisateur
            2 - Traite le retour formulaire et supprime un utilisateur
        """
        params, self.server_nb = tools.get_request(self.request)
        menu = G.get_user_menu(self.server_nb, self.name)
        ## 2 -
        if self.form_result != {}:
            try:
                datas = self.del_user(self.form_result)
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name,
                                                     confirm=True)
            except Exception, mess:
                datas = dict(message=mess.message)
                log.err()
            self.form_result = {}
            return self.send_frag(datas, template='horus_gestion_user',
                                  templates=used_templates)
        sstitre = self.description
        result = {'sstitre':sstitre, 'titre':titre, 'menus':menu}
        ## 1 -
        result.update(self._get_form())
        result['validate'] = self._get_valid_btn()
        return self.send_all(result, template='horus_gestion_user',
                             templates=used_templates)

    def _get_form(self):
        """ renvoie la description du formulaire de suppression utilisateur """
        try:
            usrs = horus.get_users()
        except:
            usrs = []

        user_select = F.Select(name='user_name', libelle="Nom d'utilisateur")
        for usr in usrs:
            user_select.add_option(usr)
        del_rep = F.Checkbox(name='remove_data', checked=True,
                             libelle="Supprimer son répertoire")
        return dict(user_select=user_select, enable=del_rep)

    def del_user(self, dico):
        """ supprimer un utilisateur """
        if not dico.has_key('user'):
            raise Exception, "Erreur : il manque des données pour la suppression utilisateur."
        result = tools.format_form_result(dico['user'], True)
        name = result['user_name']
        removing = result['remove_data']
        if not horus.del_user(name, removing):
            raise Exception, "Erreur : une erreur est survenue lors de la suppression de %s." % name
        message = "L'utilisateur %s a bien été supprimé.\\n" % name
        if removing:
            message += "Ainsi que son répertoire."
        return dict(message=message)

    def _get_valid_btn(self):
        """ renvoie la description du bouton valider """
        href = tools.make_form_link(self.server_nb, self.name, False, ['user'],
                                    balise='user_msg_div_container')
        href = tools.make_confirm_link("Etes-vous sûr de vouloir supprimer l'utilisateur", href)
        return M.Bouton(href=href, libelle="Valider",
                        title="Supprimer cet utilisateur")
