# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG - 2010
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
###########################################################################
"""
Librairie pour la gestion des services
et des groupes de services
"""
from twisted.python import log
from ConfigParser import ConfigParser
from pyeole.service import manage_services, ServiceError
from creole.config import VIRTMASTER
from creole.client import CreoleClient
from ead2.backend.config.config import EAD_SERVICES, NON_STOP_SERVICES, \
EOLE_MODULE as module
from ead2.backend.config.filenames import simple_services as services_list
from ead2.backend.actions import tools

client = CreoleClient()

METHOD_FILTER = [u'service', u'upstart', u'network']
SERVICES = []
for service in client.get_services():
    if service.get('activate', False) == True \
            and service['method'] in METHOD_FILTER:
        SERVICES.append((str(service['name']), str(service['container_group'])))
SERVICES = set(SERVICES)



def service_action(service, container, action):
    """
    effectue l'action (stop/restart) pour un service
    """
    try:
        manage_services(action, service, container=container)
    except ServiceError:
        return 1
    return 0

def stop(service, container):
    """
    Arrêt d'un service
    """
    log.msg('Arrêt de %s' % service)
    if (service, container) not in SERVICES:
        return "Le service %s est inconnu" % service
    if service in EAD_SERVICES or service in NON_STOP_SERVICES:
        return "Arrêt interdit pour le service %s" % service
    if service_action(service, container, 'stop') > 0:
        return "Erreur lors de l'arrêt du service %s" % service
    return "Le service %s a été arrêté." % service

def restart(service, container):
    """
    Redémarrage d'un ou plusieurs services
    """
    log.msg('Redémarrage de %s' % service)
    if (service, container) not in SERVICES:
        return "Le service %s est inconnu" % service
    if service in EAD_SERVICES:
        # utilisation de at pour les services Ead2 ...
        tools.command_statusoutput('echo "/etc/init.d/%s restart"' % service, 'at now')
        return "Le service %s va être redémarré." % service
    if service_action(service, container, 'restart') > 0:
        return "Erreur lors du redémarrage du service : %s" % service
    return "Le service %s a été redémarré." % service

def restart_all():
    """
    Redémarrage de tous les services
    """
    log.msg("Redémarrage de tous les services")
    manage_services('restart', exclude=(('root', 'network'),
                                        ('root', 'ead-server'),
                                        ('root', 'ead-web')))
    return "Tous les services ont été redémarrés"

class ServicesParser(ConfigParser):
    """ parser pour le fichier de configuration des groupes de services """
    def parse_file(self, filename):
        """ parse un fichier """
        self.read(filename)

    def get_services(self):
        """ renvoie les groupes de services """
        if module in self.sections():
            for libelle, services in self.items(module):
                yield({'libelle':libelle, 'services':services.split(',')})

    def get_libelles(self):
        """ renvoie la liste des groupes de services """
        if module in self.sections():
            for libelle, value in self.items(module):
                yield libelle

def get_group_services(group_name):
    """ renvoie la liste des services associés à un groupe """
    list_services = set()
    sparser = ServicesParser()
    sparser.parse_file(services_list)
    for i in sparser.get_services():
        if i['libelle'] == group_name:
            for srv in i['services']:
                if '#' in srv:
                    res = srv.split('#')
                else:
                    res = (srv, VIRTMASTER)
                list_services.add((res[0], res[1]))
            break
    return list_services

def get_groups():
    """ renvoie la liste des groupes du module """
    sparser = ServicesParser()
    sparser.parse_file(services_list)
    return list(sparser.get_libelles())

def get_group_available_services(group_name):
    """ renvoie la différence entre la liste des services
        du groupe et des services disponibles
    """
    list_services = SERVICES.difference(get_group_services(group_name))
    return list_services

## action sur le fichier de config
def create_service_group(group_name, services):
    """ cree un groupe de service """
    if group_name in get_groups():
        raise Exception, "Ce groupe de service existe déjà sur %s." % module
    if not services :
        raise Exception, "Veuiller associer au moins un service à la création de ce groupe."
    sparser = ServicesParser()
    sparser.parse_file(services_list)
    if not sparser.has_section(module):
        sparser.add_section(module)
    sparser.set(module, group_name, ','.join(services))
    fic = file(services_list, 'w')
    sparser.write(fic)
    fic.close()

def mod_service_group(group_name, services):
    """ modifie un groupe de services """
    if type(services) != list:
        services = [services]
    if list(set(get_group_services(group_name)).difference(set(services))) == [] and list(set(services).difference(set(get_group_services(group_name)))) == []:
        # pas de modification
        return True
    sparser = ServicesParser()
    sparser.parse_file(services_list)
    if not sparser.has_section(module):
        sparser.add_section(module)
    sparser.remove_option(module, group_name)
    if services:
        sparser.set(module, group_name, ','.join(services))
    fic = file(services_list, 'w')
    sparser.write(fic)
    fic.close()

def del_service_group(group_name):
    """ supprime un groupe de services """
    if group_name not in get_groups():
        raise Exception, "Erreur : Ce groupe n'existe pas, impossible de le supprimer."
    sparser = ServicesParser()
    sparser.parse_file(services_list)
    sparser.remove_option(module, group_name)
    fic = file(services_list, 'w')
    sparser.write(fic)
    fic.close()

if __name__ == '__main__':
    groupe = 'xxx_test_xxx'
    try:
        del_service_group(groupe)
    except:
        pass
    print "Création du groupe de services %s" % groupe
    create_service_group(groupe, ['bareos-fd#root', "test#machin", "ead-server"])
    print "résultat :", get_group_services(groupe)
    print "modification :"
    mod_service_group(groupe, ['bareos-fd#root', "truc#bidule", "ead-web"])
    print "résultat :", get_group_services(groupe)


