# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
    Classe de base pour le téléchargement de fichier
"""
from tempfile import mktemp
from urllib2 import unquote

from twisted.python import log
from twisted.internet import defer
from twisted.python.failure import DefaultException, Failure

from pyeole.process import system_out

from ead2.backend.lib.action import Dict
from ead2.backend.actions.lib.widgets import main as M, ajax, form as F

from ead2.backend.actions.lib.error import KnownError, _clean_errors
from ead2.backend.actions.lib.importation.config import EXTRACTION_TEMP_DIR
from ead2.backend.actions.lib.importation.config import (PARSE_LOG_FILE,
                                     IMPORT_LOG_FILE, IMPORTATION_SCRIPT,
                                     REFRESH_FREQUENCY)
from ead2.backend.actions.lib.importation.base import BaseImportation

from ead2.backend.actions.lib.importation.forms import get_category_libelle
from ead2.backend.actions.lib.importation.tools import (write_file,
                                        save_register, get_register)
from ead2.backend.config.config import debug

PARSE_CMD = "%s --parse=%%s %%s" % IMPORTATION_SCRIPT
IMPORTATION_CMD = "%s --import --type=%%s %%s" % IMPORTATION_SCRIPT

def _thread_message((message, txtmessage), return_dict):
    """
        Ajoute une popup et du texte dans le dictionnaire
    """
    if message:
        return_dict['message'] = message
    return_dict['txtmessage'] = txtmessage
    return return_dict

def reinit_file(filename):
    """
        Ecrase un fichier
    """
    if debug:
        log.msg("écrasement de %s" % filename)
    file(filename, 'w').close()
    return True

class BaseImportationFichier(BaseImportation):
    """
        Classe de base pour le téléchargement de fichier
    """
    data_type = 'base'
    widget_templates = ['main', 'form', 'formselect', 'filedownload']
    template = "scribe_extraction_download"
    all_file_descriptions = {'eleve':tuple(),
                             'enseignant':tuple()}
    request_keys = ['server',
                    'action',
                    'category_type',
                    'data_type',
                    'nomdefichier',
                    'filename',
                    'finished',
                    'parse',
                    'refresh_parse',
                    'importation',
                    'refresh_importation',
                    ]
    filevarnames = []
    origfilevarnames = []
    request = Dict(default={}, keys=request_keys)
    file_download_entete = "Importation de %s au format : BASE"

    register_file = mktemp('.register', 'ead-importation-',
                                        EXTRACTION_TEMP_DIR)

    def execute(self):
        """
            Renvoie le dictionnaire de mise en forme au serveur xmlrpc
        """
        self.parse_request()
        if debug:
            log.msg("# Importation %s #" % (self.data_type))
        if self.filevarnames:
            if debug:
                log.msg(self.filevarnames)
        return_dict = dict(title=self.description)
        downloaded_file = self._get_downloaded_file()

        if self.params.has_key('category_type'):
            if debug:
                log.msg("Téléchargement de fichier : étape 1")
            # c'est le premier passage dans l'action,
            # on initialise le 'pool' de nom de fichier dispo
            category_type = self.params['category_type'][0]
            import_type = self.params['import_type'][0]
            if debug:
                log.msg("%s %s" % (category_type, import_type))
            del self.filevarnames[:]
            self.clean_memory()
            if debug:
                log.msg(self.request_keys)
            for name in self.origfilevarnames[category_type]:
                self.filevarnames.append(name)
            self.remember('import_type', import_type)
            self.remember('category_type', category_type)

            # initialisation du fichier registre
            reinit_file(self.register_file)
            defered = defer.Deferred()
            defered.callback(return_dict)
            defered.addCallback(self._download_form)

        elif downloaded_file:
            if debug:
                log.msg("Téléchargement de fichier : étape 2 un " + \
                    "fichier a été téléchargé")
            defered = self._download(downloaded_file)

        elif self.params.has_key('finished'):
            if debug:
                log.msg("Téléchargement de fichier : étape 3" + \
                    " tous les fichiers ont été téléchargés")
            defered = defer.Deferred()
            defered.callback(return_dict)
            defered.addCallback(self._bouton_parse)

        elif set(self.params.keys()).intersection(self.origfilevarnames):
            msg = "Ce fichier a déjà été téléchargé"
            if debug:
                log.msg("Téléchargement de fichier : erreur " + \
                    msg)
            defered = defer.Deferred()
            defered.errback(DefaultException(msg))

        elif self.params.has_key('parse'):
            if debug:
                log.msg("Parsing de fichier : étape 1" + \
                    " lancement du parsing")
            defered = self._parse(return_dict)

        elif self.params.has_key('refresh_parse'):
            if debug:
                log.msg("Parsing de fichier : étape 2 " + \
                    "rafraichissement")
            defered = defer.Deferred()
            defered.callback(
                        dict(title="Lecture des fichiers d'importation"))
            defered.addCallback(self._refresh_parse)
            defered.addErrback(_clean_errors,
                "Erreur de récupération des journaux de lecture")

        elif self.params.has_key('importation'):
            if debug:
                log.msg("Importation des comptes : étape 1")
            defered = self._importation()
            defered.addCallback(self._test_lock)
            defered.addErrback(self._err_lock)

        elif self.params.has_key('refresh_importation'):
            if debug:
                log.msg("Importation des comptes : étape 2 " + \
                    "rafraichissement")
            defered = defer.Deferred()
            defered.callback(
                    dict(title="Importation des comptes"))
            defered.addCallback(self._refresh_importation)
            defered.addErrback(_clean_errors,
                "Erreur de récupération des journaux d'importation")

        else:
            if debug:
                log.msg("Une erreur inconnue s'est produite")
            defered = defer.Deferred()
            defered.errback(DefaultException("Erreur inconnue"))

        defered.addCallback(self.main)
        defered.addErrback(self.send_error)
        return defered

# FORMULAIRE DE TÉLÉCHARGEMENT
    def _download_tag(self, file_descriptions):
        """
            Renvoie les objets Download pour le téléchargement
        """
        for filetype, libelle in file_descriptions:
            yield F.Download(self.server_nb,
                             self.name,
                             filetype,
                             libelle=libelle)

    def _download_form(self, return_dict):
        """
            formulaire de téléchargement de fichier
        """
        category_type = self.params['category_type'][0]
        libelle = get_category_libelle(self.data_type, category_type)
        file_descriptions = self.all_file_descriptions[category_type]
        import_file_tags = tuple(self._download_tag(file_descriptions))
        return_dict['import_fichiers'] = import_file_tags
        return_dict['import_fichier_entete'] = libelle
        return return_dict

# Bouton de lancement de parsing
    def _bouton_parse(self, return_dict):
        """
            Renvoie un bouton pour lancer le parsing des fichiers
        """
        href = ajax.call(self.server_nb,
                         self.name,
                         parse=True
                         )
        bouton = M.Lien(href=href,
                libelle="<h1 style='text-decoration:underline'>\
Lancer la lecture des fichiers</h1>")
        return_dict['bouton_parse'] = bouton
        return return_dict

# TÉLÉCHARGEMENT DE FICHIER
    def _get_downloaded_file(self):
        """
            Renvoie le nom du fichier télécharger (si il y en a un)
        """
        intersection = set(self.params.keys()).intersection(
                                       set(self.filevarnames))
        if intersection:
            return list(intersection)[0]
        else:
            return None

    def _download(self, filevarname):
        """
            Lance l'enregistrement du fichier téléchargé
            Récupération des infos
            Écriture du fichier
            Enregistrement de informations d'écriture
            Gestion de l'avancement du téléchargement
        """
        defered = defer.Deferred()
        defered.callback(filevarname)
        defered.addCallback(self._file_infos)
        defered.addErrback(_clean_errors,
        "Erreur à la récupération des informations du fichier" + \
        "téléchargé")
        defered.addCallback(write_file)
        defered.addErrback(_clean_errors,
        "Erreur à l'écriture du fichier téléchargé")
        defered.addCallback(self._register_file_infos, filevarname)
        defered.addErrback(_clean_errors,
        "Erreur : une opération a échoué (register). \\n " + \
        "Veuillez re-télécharger le fichier" )
        defered.addCallback(self._download_status)
        defered.addErrback(_clean_errors,
        "Erreur : erreur à la gestion de l'avancement du téléchargement")
        return defered

    def _file_infos(self, filevarname):
        """
            Renvoie les informations pour l'écriture de fichier
        """
        if debug:
            log.msg("Renvoie des informations de fichiers")
        filename = self.params.get('nomdefichier', ['default'])[0]
        # fix bug avec Opera #2011
        if filename.startswith('C:\\fakepath\\'):
            filename = filename.replace('C:\\fakepath\\', '')
        filecontent = unquote(self.params.get(filevarname)[0])
        if debug:
            log.msg('Téléchargement du fichier %s' % filename)
        return (filename, filecontent)

    def _register_file_infos(self, (filepath, return_dict), filevarname):
        """
            Enregistre les infos sur les fichiers télécharger
        """
        if debug:
            log.msg("Enregistrement du fichier de registre dans %s" % (
                                                self.register_file,))
        register_dict = get_register(self.register_file)
        register_dict[filevarname] = filepath
        save_register(self.register_file, register_dict)
        return return_dict, filevarname

    def _download_status(self, (return_dict, filevarname)):
        """
            Renvoie le bouton de lancement de lecture
            si le tous les fichiers ont été téléchargés
        """
        self.filevarnames.remove(filevarname)
        if not self.filevarnames:
            # On passe à l'étape suivante
            return_dict = dict(toexec=ajax.call(self.server_nb,
                                                self.name,
                                                only=False,
                                                finished=True
                                                ))
        else:
            # On vire le formulaire de téléchargement
            # ex : form_eleve pour la var eleve
            js = "$('form_%s').parentNode.removeChild($('form_%s'));" % (
                                                filevarname, filevarname)
            return_dict['toexec'] = js
        return return_dict

# PARSING DES DONNÉES
    def _parse(self, return_dict):
        """
            renvoie le deferred de parsing
        """
        reinit_file(PARSE_LOG_FILE)
        defered = defer.Deferred()
        defered.callback(self.register_file)
        defered.addCallback(self._do_parse)
        defered.addErrback(_clean_errors,
            "Erreur au lancement de la lecture de fichier")
        defered.addCallback(_thread_message, return_dict)
        defered.addCallback(self._refresh_parse_call)
        return defered

    def _do_parse(self, filevarname):
        """
            Lance la commande de parsing dans un 'at'
        """
        cmd = PARSE_CMD % (self.data_type, filevarname)
        ret_code, value, ret_value = system_out(['at', 'now'], stdin=cmd)
        if ret_code:
            log.err(ret_value)
            Failure(KnownError("Erreur au lancement du " + \
                               "parsing")).raiseException()
        else:
            return ("Lecture des fichiers en cours, " + \
                    "veuillez conserver cette page.",
                    "Lancement de la lecture ...")

#Log du parsing
    def _refresh_parse(self, return_dict):
        """
            Renvoie la chaine d'éxécution pour
            le rafraichissement des logs du parsing
        """
        logfile = file(PARSE_LOG_FILE, 'r')
        logcontent = logfile.read()
        logfile.close()
        return_dict['txtmessage'] = logcontent
        if 'FIN' not in logcontent:
            return_dict = self._refresh_parse_call(return_dict)
        else:
            if debug:
                log.msg("Fin du parsing des fichiers %s" % self.data_type)
            if "ERREUR" in logcontent: # On a une erreur de parsing
                self.params['category_type'] = [self.memory['category_type']]
                self.params['import_type'] = [self.memory['import_type']]
                return_dict.update(self._get_bouton_retour(step=4))
                msg = "<br /><b>Une erreur est survenue à la lecture " + \
                      "des fichiers, veuillez les re-télécharger.</b>"
                return_dict['txtmessage'] += msg
            else:
                return_dict = self._bouton_importation(return_dict)
        return return_dict

    def _refresh_parse_call(self, return_dict):
        """
            Renvoie le script d'appel de rafraichissement de page
        """
        ajax_call = ajax.call(self.server_nb,
                              self.name,
                              only=True,
                              refresh_parse=True)
        return_dict['toexec'] = "setTimeout(\"%s\", %s);" % (ajax_call,
                                                     REFRESH_FREQUENCY)
        return return_dict

# Bouton pour lancer l'importation
    def _bouton_importation(self, return_dict):
        """
            Renvoie le bouton de lancement d'importation
        """
        ajax_call = ajax.call(self.server_nb,
                              self.name,
                              importation=True)
        bouton = M.Lien(href=ajax_call,
                        libelle="<h1 style='text-decoration:underline'>\
Lancer l'importation</h1>",
                        title="Lancer l'importation %s " % self.data_type + \
                              "depuis les fichiers téléchargés")
        return_dict['bouton_importation'] = bouton
        return return_dict

# IMPORTATION
    def _importation(self):
        """
            Lance l'importation
        """
        category_type = self.memory.get('category_type', None)
        if debug:
            log.msg(self.memory)
        if not category_type:
            Failure(KnownError("Pas de category_type à l'import")
                    ).raiseException()
        import_type = self.memory.get('import_type', None)
        if not import_type:
            Failure(KnownError("Pas de import_type à l'import")
                    ).raiseException()
        return_dict = dict()
        reinit_file(IMPORT_LOG_FILE)
        defered = defer.Deferred()
        defered.callback(category_type)
        defered.addCallback(self._do_importation, import_type=import_type)
        defered.addErrback(_clean_errors,
        "Erreur au lancement de l'importation")
        defered.addCallback(_thread_message, return_dict)
        defered.addCallback(self._refresh_importation_call)
        return defered

    def _do_importation(self, category_type, import_type):
        """
            Lance l'importation des comptes
        """
        cmd = IMPORTATION_CMD % (import_type, category_type)
        ret_code, value, ret_value = system_out(['at', 'now'], stdin=cmd)
        if ret_code:
            log.err(ret_value)
            Failure(KnownError("Erreur au lancement de l'importation")
                    ).raiseException()
        else:
            return ("Importation des comptes en cours, veuillez conserver" + \
                    " cette page. Cette opération peut être longue.",
                    "Lancement de l'importation ...")

# logs d'avancement de l'importation
    def _refresh_importation(self, return_dict):
        """
            Renvoie les logs de l'éxécution pour l'importation
        """
        logfile = file(IMPORT_LOG_FILE, 'r')
        logcontent = logfile.read()
        logfile.close()
        return_dict['txtmessage'] = logcontent
        if 'FIN' not in logcontent:
            return_dict = self._refresh_importation_call(return_dict)
        else:
            if 'ERREUR' in logcontent:
                self.params['category_type'] = [self.memory['category_type']]
                self.params['import_type'] = [self.memory['import_type']]
                return_dict.update(self._get_bouton_retour(step=4))
                msg = "<br /><b>Une erreur est survenue à l'importation " + \
                       "des comptes, veuillez recommencer la procédure.</b>"
                return_dict['txtmessage'] += msg
            else:
                log.msg("Fin de l'importation des comptes")
            self.clean_memory()
        return return_dict

    def _refresh_importation_call(self, return_dict):
        """
            rafraichit l'importation
        """
        ajax_call = ajax.call(self.server_nb,
                              self.name,
                              only=True,
                              refresh_importation=True)
        return_dict['toexec'] = "setTimeout(\"%s\", %s);" % (ajax_call,
                                                     REFRESH_FREQUENCY)
        return return_dict
