# -*- coding: UTF-8 -*-
from twisted.python import log
from ConfigParser import ConfigParser, NoSectionError
from sets import Set
from actor import Role
from os.path import isfile
from ead2.backend.lib import error

class PermissionParser(ConfigParser):

    def parse_file(self, filename):
        """charge les permissions depuis un fichier"""
        self.read(filename)

    def parse_string(self, permissions):
        """charge les permissions depuis une chaine"""
        from cStringIO import StringIO
        self.readfp(StringIO(permissions))

    def parse_remote(self, url, method="get_table", *args, **kwargs):
        """charge les permissions depuis une chaine"""
        import xmlrpclib
        proxy = xmlrpclib.ServerProxy(url)
        meth = getattr(proxy, method)
        return self.parse_string(meth(*args, **kwargs))

    def get_trusted_clients(self):
        """renvoie la liste des machines pouvant se connecter"""
        if not hasattr(self, '_trusted'):
            self._trusted = {}
            for application, addresses in self.items('trusted'):
                self._trusted[application] = addresses.split(',')
        return self._trusted

    def get_roles(self):
        """renvoie la liste des roles"""
        # Shortcut: remove me if possible
        if not hasattr(self, '_roles'):
            try:
                self._roles = dict([(name, Role(name, descr)) for name, descr
                                in self.items('roles')])
            except NoSectionError, message:
                error.get_out("Il manque une section obligatoire (%s)" % message)
        return self._roles

    def get_permissions(self):
        """renvoie la liste des permissions définies dans le .ini"""
        if not hasattr(self, '_permissions'):
            self.get_roles()
            self._permissions = {}
            for action_name, rolenames in self.items('permissions'):
                for rolename in rolenames.split(','):
                    try:
                        role = self._roles[rolename.strip()]
                        self._permissions.setdefault(action_name, []).append(role)
                    except KeyError:
                        log.err("!!! Le rôle \"%s\" est inconnu !!!" % (
                                                              rolename,))
        return self._permissions

class RoleParser(ConfigParser):
    """Utilisé pour parser un fichier role.ini
    (fichiers qui définissent à quels rôles correspondent
     des infos susceptibles de se trouver dans un annuaire
     LDAP)
    """
    def parse_file(self, filename):
        """charge les permissions depuis un fichier"""
        self.read(filename)

    def parse_string(self, permissions):
        """charge les permissions depuis une chaine"""
        from cStringIO import StringIO
        self.readfp(StringIO(permissions))

    def _extend_role(self, ldapkey, ldapvalue, role):
        act_value = self.items(ldapkey)[1]
        new_value = "%s,%s" % ( act_value, role)
        self.set(ldapkey, ldapvalue, new_value)

class RoleManager(dict):
    """ manager de rôle """

    def _add_ldapkey(self, ldapkey):
        if not self.has_key(ldapkey):
            self[ldapkey] = dict()

    def add_roles(self, ldapkey, ldapvalue, roles):
        """ ajoute une définition de role dans le manager
            ldapkey: nom de la clé à associer
            ldapvalue: nom de la valeur de la clé
            role: rôle associer au user ayant les attributs ldapkey=ldapvalue
        """
        self._add_ldapkey(ldapkey)
        self._associate_value_and_role(ldapkey, ldapvalue, roles)

    def _associate_value_and_role(self, ldapkey, ldapvalue, roles):
        """ renvoie le dictionnaire de description de rôle """
        if ldapvalue in self[ldapkey].keys():
            act_roles = self[ldapkey][ldapvalue]
            for role in roles.split(','):
                if role not in act_roles.split(','):
                    act_roles = "%s,%s" % (act_roles, role)
            self[ldapkey][ldapvalue] = act_roles
        else:
            self[ldapkey][ldapvalue] = roles

    def get_roles(self, infos):
        """ calcule les roles en fonction des infos utilisateur """
        roles = Set()
        for ldapkey in self.keys():
            if ldapkey in infos:
                for ldapvalue, role in self[ldapkey].items():
                    inf = infos[ldapkey]
                    if not hasattr(inf, '__iter__'):
                        inf = [inf]
                    if ldapvalue in inf:
                        roles.union_update(role.split(','))
        return roles

    def get_users(self):
        """ retourne la liste des utilisateurs système """
        users = []
        for username in self.get('pam', {}):
            users.append(username)
        return users

def parse_and_update(perm_manager, filename):
    """Parse le fichier <filename> et met à jour <perm_manager>"""
    parser = PermissionParser()
    parser.parse_file(filename)
    for role in parser.get_roles().values():
        perm_manager.role_created(role)
    for action_id, roles in parser.get_permissions().items():
        for role in roles:
            perm_manager.allow_named_action(action_id, role)

def parse_and_update_roles(role_manager, filename):
    """ Parse filename et update le role_manager """
    parser = RoleParser()
    parser.parse_file(filename)
    for ldapkey in parser.sections():
        for ldapvalue, roles in parser.items(ldapkey):
            role_manager.add_roles(ldapkey, ldapvalue, roles)

def get_unavailable_actions(filename):
    """ renvoie la liste des actions qui n'existent plus
        (pour les permissions locales déjà définis)
    """
    if isfile(filename):
        f_lines = file(filename).read().splitlines()
        return [action.strip() for action in f_lines if action.strip() != '']
    return []

def parse_and_update_local(perm_manager, filename, filter_filename=''):
    """ Parse le fichier <filename et met à jour <perm_manager>
        et filtre les actions qui n'existe plus
    """
    if filter_filename:
        unavailable_actions = get_unavailable_actions(filter_filename)
    else:
        unavailable_actions = []
    parser = PermissionParser()
    parser.parse_file(filename)
    for role in parser.get_roles().values():
        perm_manager.role_created(role)
    for action_id, roles in parser.get_permissions().items():
        if action_id not in unavailable_actions:
            for role in roles:
                perm_manager.allow_named_action(action_id, role)
