# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
Libraire de fonctions
pour l'EAD de Scribe
"""
from twisted.python import log
from os.path import isfile, join
from scribe import ldapconf
from scribe import eoleldap
from scribe import eoleuser
from scribe import eolegroup
from ead2.backend.actions import tools
from ead2.backend.actions.scribe.tool import validate
from ead2.config.config import CONTAINER_PATH_FICHIER, NFS_ACTIVATE, CLIENT_LTSP_INSTALLED

## outil de listing utilisateur

def get_quota_classe(ldapuser, username):
    """
    calcul de la chaine à afficher pour qualifier l'utilisateur
    """
    try:
        user_type = ldapuser._get_type(username)
    except:
        # utilisateur système par exemple
        user_type = 'autre'
    if user_type == 'eleve':
        return "élève de %s" % ldapuser._get_attr(username, 'Divcod')[0]
    elif user_type == 'enseignant':
        return "enseignant"
    elif user_type == 'administratif':
        return 'personnel administratif'
    else:
        return 'autre'

def filter_by_alpha(users, alpha):
    """ filtre une liste d'utilisateur sur la première lettre """
    result = []
    for user in users:
        if user.lower().startswith(alpha.lower()):
            result.append(user)
    return result

def user_filter(_type='', domain='', _class='', namepart='', alpha_key='', group='', sort=''):
    """ filtre les utilisateurs selon des parametres
        on cree le filtre ldap et on lance une recherche
        :alpha_key: première lettre du nom de l'utilisateur
        :_type: type d'utilisateur (pupil/teacher/administratif/responsable)
        :domain: domaine mail
        :group: groupe des utilisateurs
        :class: classe de l'utilisateur (pour les élèves à priori,
                marche aussi pour les profs admin de classe)
        :size: nombre d'utilisateurs par page
    """
    filtre = "(&"
    if _type:
        if _type == 'pupil':
            filtre += ldapconf.ELEVE_FILTER
        elif _type == 'teacher':
            filtre += ldapconf.PROF_FILTER
        elif _type == 'administratif':
            filtre += ldapconf.ADMINISTRATIF_FILTER
        elif _type == 'responsable':
            filtre += ldapconf.RESPONSABLE_FILTER
        elif _type == 'other':
            filtre += ldapconf.AUTRE_FILTER
        else:
            log.err("type d'utilisateur inconnu : %s" % _type)
            filtre += ldapconf.USER_FILTER
    else:
        filtre += ldapconf.USER_FILTER
    if domain == 'empty':
        filtre += '(!(mail=*@*))'
    elif domain == 'externe':
        filtre += '(mail=*@*)'
        for dom in ldapconf.MAIL_DOMAIN.values():
            filtre += '(!(mail=*@%s))' % dom
    elif domain:
        filtre += "(mail=*@%s)" % domain
    if _class:
        filtre += "(Divcod=%s)" % _class
        # c'est forcément un élève également
        filtre += ldapconf.ELEVE_FILTER
    if namepart:
        filtre += "(sn=*%s*)" % namepart
    filtre += ")"
    ldapentry = eoleldap.LdapEntry()
    ldapentry.ldap_admin.connect()
    usrs = ldapentry.ldap_admin._search(filtre, ['uid', 'sn'])
    if sort == 'nom':
        # /!\ outil de tri tout pourri /!\
        dic = {}
        for user in usrs:
            dic[(user[1]['sn'][0].lower(), user[1]['uid'][0])] = user[1]['uid'][0]
        sns = dic.keys()
        sns.sort()
        users = [dic[sn] for sn in sns]
    else:
        users = [user[1]['uid'][0]for user in usrs]
        users.sort()
    if group:
        gr_users = ldapentry._get_members(group)
        users = [user for user in users if user in gr_users]
    if alpha_key:
        users = filter_by_alpha(users, alpha_key)
    ldapentry.ldap_admin.close()
    return users

def _get_alpha_links():
    """ renvoit la description des liens alphabetique pour un objet select"""
    alphas = [_get_select_default_value()]
    alphas.extend([{'name':chr(i).upper(), 'libelle':chr(i).upper()}for i in range(97, 123)])
    return alphas

def _get_select_default_value():
    """ renvoie l'option vide pour un objet select """
    return {'name':'', 'libelle':'', 'default':'ok'}

def sort_users(users):
    """ trie les nom des utilisateurs par ordre alphabétique
        prenom.nom par nom
    """
    _list = []
    for user in users:
        login = user.split('.')
        if len(login)>1:
            _list.append("%s.%s"%(login[1], login[0]))
        else:
            _list.append("%s"%login[0])
    _list.sort()
    users = []
    for user in _list:
        login = user.split('.')
        if len(login)>1:
            users.append("%s.%s"%(login[1], login[0]))
        else:
            users.append("%s"%login[0])
    return users

def sort_list(myset):
    """
    transforme un set en liste triée
    """
    res = list(myset)
    res.sort()
    return res

## quotas

def _get_quotas(_type='depasse'):
    """ renvoie les quotas des utilisateurs du domaine
        dans le cas _type: qui prend les valeurs "proche", "depasse" ou "tous"
    """
    def _get_name_in_login(login):
        """ récupère le nom dans un login de type p.nom """
        login = login.split('.')
        if len(login)> 1 :
            return login[1]
        else:
            return login[0]

    def cmp_username(a, b):
        """ sort function to compare by username
        """
        ua = _get_name_in_login(a.get('username', '0.0'))
        ub = _get_name_in_login(b.get('username', '0.0'))
        if ua > ub:
            return 1
        elif ua == ub:
            return 0
        else:
            return -1

    cmd = "/usr/sbin/repquota -a"
    return_code, quota_lines = tools.command_statusoutput(cmd)
    if return_code:
        raise Exception, "Erreur : un problème s'est produit à la récupération des quotas"
    #FIXME: gestion de plusieurs partitions avec quota
    quota_lines = quota_lines.splitlines()[5:]
    result = []
    ldapuser = eoleuser.User()
    ldapuser.ldap_admin.connect()
    for line in quota_lines:
        if line.startswith('***'):
            break
        user = line.split()
        if len(user) < 5:
            continue
        username = user[0]
        if username.startswith('#'):
            continue
        fullquota = float(user[3])
        quota = int(fullquota) / 1024
        if quota == 0:
            continue
        fullused = float(user[2])
        used = int(fullused) / 1024
        delai = user[5]
        if fullused > fullquota:
            classe = get_quota_classe(ldapuser, username)
            result.append({'username':username, 'used':used, 'quota':quota,
                           'delai':delai, 'color':'#fd8758', 'classe':classe})
        elif fullused+3072 > fullquota and _type in ['proche', 'tous']:
            classe = get_quota_classe(ldapuser, username)
            result.append({'username':username, 'used':used, 'quota':quota,
                           'color':'#fddc58', 'classe':classe})
        elif _type == 'tous':
            result.append({'username':username, 'used':used, 'quota':quota})
    result.sort(cmp_username)
    return result


## historique de connexion

def _get_history():
    """ renvoie l'historique des connexions """
    filename = join('/', CONTAINER_PATH_FICHIER, 'var/log/samba/connexions.log')
    if isfile(filename):
        cmd = '/usr/bin/uniq {0}'.format(filename)
    return_code, result = tools.command_statusoutput(cmd)
    if return_code:
        raise Exception, "Erreur à la récupération de l'historique des connexions."
    res_lines = result.splitlines()
    res_lines.reverse()
    lines = []
    for line in res_lines:
        tab_line = line.split()
        lines.append({'date': " ".join(tab_line[1:6]),
                  'user': tab_line[6],
                  'poste': "%s (%s)"%(tab_line[8], tab_line[10]),
                  'system': tab_line[9]})
    if lines == []:
        return [{'default':"Aucune entrée n'a été enregistrée."}]
    return lines


## Gestion des utilisateurs

## outils pour la gestion des roles (prof_admin et prof)
def get_prof_class(login):
    """ renvoie les classes du prof login """
    classes = eolegroup.Group().get_classes()
    return filter_class_by_prof(login, classes)

def get_prof_class_list(login):
    """ renvoie la liste des classes d'un prof """
    cls = []
    for classe in get_prof_class(login).values():
        cls.extend(classe)
    return cls

def filter_class_by_prof(prof, classes):
    """ renvoie les classes du prof prof """
    groups = eoleuser.User().get_attr(prof, 'Divcod')
    for niveau, classe_list in classes.items():
        nivo = []
        for classe in classe_list:
            if classe in groups:
                nivo.append(classe)
        if nivo != []:
            classes[niveau] = nivo
        else:
            classes.pop(niveau)
    return classes

def is_eleve_from_prof(eleve, prof):
    """ l'eleve est il membre d'une classe administrée par prof ?"""
    ldapuser = eoleuser.User()
    ldapuser.ldap_admin.connect()
    if ldapuser._get_attr(eleve, 'typeadmin'):
        #c'est un prof
        ldapuser.ldap_admin.close()
        return False
    cls = get_prof_class_list(prof)
    for classe in cls:
        if eleve in ldapuser._get_members(classe):
            ldapuser.ldap_admin.close()
            return True
    ldapuser.ldap_admin.close()
    return False

def _valid_mail(mailtype, mailperso):
    """
    fabrication de l'adresse mail
    en fonction des infos du formulaire
    """
    if mailtype == 'perso':
        return validate.test_mailperso(mailperso)
    else:
        return mailtype

# gestion des rôles sur scribe
def is_admin(roles):
    """
        Renvoie True si l'utilisateur est admin
    """
    return ('admin_pedago' in roles) or ('admin' in roles)

def is_prof_admin(roles):
    """
        L'utilisateur est prof_admin ?
    """
    return (not is_admin(roles)) and ('prof_admin' in roles)

def is_prof(roles):
    """
        L'utilisateur est prof ?
    """
    return (not is_prof_admin(roles)) and ('prof' in roles) \
            and (not is_admin(roles))

def is_administratif(roles):
    """
        L'utilisateur est administratif_scribe ?
    """
    return (not is_admin(roles)) and ('administratif_scribe' in roles)

def get_role(roles):
    """
        Renvoie un des trois rôles prof/admin/prof_admin
    """
    if is_admin(roles):
        return 'admin'
    elif is_prof_admin(roles):
        return 'prof_admin'
    else:
        return 'prof'

def use_eclair():
    """
    détermine si Eclair est utilisé
    """
    if NFS_ACTIVATE:
        # Scribe ou Horus + eole-nfs
        return True
    if CLIENT_LTSP_INSTALLED:
        # AmonEcole+
        return True
    return False
