# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
librairie contenant les définitions des champs de formulaires
"""
from scribe import ldapconf
from scribe.ldapconf import MAIL_DOMAIN, EOLE_AD
from scribe import eoleuser
from scribe.eoletools import formate_date
from ead2.backend.actions import tools
from ead2.backend.actions.lib.widgets import main as M, form as F
from ead2.backend.actions.scribe.tool.user import ADRESSE

def make_logininput(user):
    """
    definition du formulaire pour l'affichage du login
    :param user: login de l'utilisateur
    """
    return F.Input(name='login',
            libelle="Login",
            disabled=True,
            default=user,
            inline=True)

def make_fornameinput(datas=None):
    """
    definition du formulaire de gestion du prénom
    :param datas: données originales
    """
    name = F.Input(name='forname',
                   libelle="Prénom",
                   inline=True,
                   required=True)
    if datas:
        name.default(datas['givenName'][0])
    return name

def make_nameinput(datas=None):
    """
    definition du formulaire de gestion du nom de famille
    :param datas: données originales
    """
    name = F.Input(name='name',
                   libelle="Nom",
                   inline=True,
                   required=True)
    if datas:
        name.default(datas['sn'][0])
    return name

def make_identityinput(datas):
    """
    definition du formulaire de gestion de gestion de l'identité
    :param datas: données originales
    """
    name = datas.get('displayName', [''])[0]
    return F.Input(name='displayName',
                   libelle="Identité",
                   default=name,
                   inline=True)

def make_mailinput(user, datas):
    """
    definition du formulaire de gestion de l'adresse e-mail
    :param user: login de l'utilisateur
    :param datas: données originales
    """
    # pas de modif pour admin (#1304)
    if user == 'admin':
        disabled = 'disabled'
    else:
        disabled = ''
    maildir = datas.get('mailDir', None)
    act_mail = datas.get('mail', [None])[0]
    mail_perso = datas.get('mailPerso', datas.get('FederationKey', ['']))[0]
    radioperso = F.Radio(name='mailtype',
                        value='perso',
                        disabled=disabled,
                        inline=True)
    radiointernet = F.Radio(name='mailtype',
                        value='internet',
                        disabled=disabled,
                        inline=True)
    radiorestreint = F.Radio(name='mailtype',
                        value='restreint',
                        disabled=disabled,
                        inline=True)
    radioaucun = F.Radio(name='mailtype',
                        value='aucun',
                        disabled=disabled,
                        inline=True)

    # adresse fournie par l'importation
    mailinput = F.Input(name="mailperso", default=mail_perso)
    if maildir: # mail local
        if act_mail.endswith(MAIL_DOMAIN['restreint']):
            radiorestreint.check()
        else:
            radiointernet.check()
    elif act_mail: # compte mail perso
        radioperso.check()
        mailinput.default(act_mail)
    else:
        act_mail = "mail non communiqué"
        radioaucun.check()
    libelle = "Adresse e-mail (personalisée <b>ou</b> locale) :<br/>"
    if user != '':
        libelle += "<br/><center><b>%s</b></center>" % act_mail
    else:
        # hack pour formulaire de création
        user = 'login'
    return dict( libelle=libelle,
                 radios=[radioperso, radiointernet, radiorestreint, radioaucun],
                 radios_libelle=['Adresse personalisée :',
                                 'Adresse locale Internet :<br/><i>%s@%s</i>' % (user, MAIL_DOMAIN['internet']),
                                 'Adresse locale restreinte :<br/><i> %s@%s</i>' % (user, MAIL_DOMAIN['restreint']),
                                 'Aucune adresse',
                                ],
                 input=mailinput)

def make_profilinput(user=None):
    """
    definition du formulaire de gestion des profils
    user : login de l'utilisateur
    """
    profile = F.Select(name='_sambaProfilePath',
             libelle="Profil Windows",
             inline=True)
    for indice, libelle in ldapconf.PROFILS:
        profile.add_option(indice, libelle=libelle)
    if user:
        profile.default(str(eoleuser.User().get_profil(user)))
    else:
        # profil local par défaut
        profile.default('1')
    return profile

def make_civiliteinput(datas, name='codecivilite'):
    """
    definition du formulaire de gestion de la civilité
    datas : données originales
    """
    # FIXME : changer l'arité à l'occasion ;)
    civilite = F.Select(name=name,
                        libelle="Civilité",
                        inline=True)
    for civ in ldapconf.CIVILITES:
        civilite.add_option(civ['code'], civ['title'])
    if datas:
        civilite.default(str(datas.get('codecivilite', ['0'])[0]))
    return civilite

def make_shellinput(datas={}):
    """
    definition du formulaire de gestion du shell
    datas : données originales
    """
    shell = F.Checkbox(name='shell',
             libelle="Activation du shell (gestion de clients Linux)",
             inline=True)
    if datas.get('loginShell', [None])[0] == '/bin/bash':
        shell.check()
    return shell

def make_attrprofinput(jeSuisUnProf):
    """
    definition du formulaire d'ajout d'attributs prof aux administrateurs (#9353)
    jeSuisUnProf : True si on a déjà les attributs prof
    """
    attrprof = F.Checkbox(name='attrprof',
             libelle="Conférer les droits du profil professeur à cet utilisateur (action irréversible)",
             inline=True)
    if jeSuisUnProf:
        attrprof.check()
        attrprof['disabled'] = 'ok'
    return attrprof

def make_quotainput(user):
    """
    definition du formulaire de gestion du quota
    user : login de l'utilisateur
    """
    _quota = eoleuser.get_quota(user)
    quota = F.Input(name='quota',
                     libelle="Quotas disque en Mo (0 pour inactif)",
                     default=_quota,
                     inline=True)
    return quota

def make_dateinput(datas):
    """
    definition du formulaire de gestion de la date de naissance
    datas : données originales
    """
    _date = datas.get('dateNaissance', [''])[0]
    _date = formate_date(_date)
    date = F.Input(name='dateNaissance',
                   libelle="Date de naissance (jj/mm/aaaa)",
                   inline=True,
                   default=_date)
    return date

def make_elenoetinput(datas={}):
    """
    definition du formulaire de gestion du numéro interne
    datas : données originales
    """
    user_number = F.Input(name='employeeNumber',
                          libelle="Numéro interne de l'élève (ELENOET)",
                          inline=True,
                          required=True)
    num = datas.get('employeeNumber', [''])[0]
    user_number.default(num)
    return user_number

def make_ineinput(datas={}):
    """
    definition du formulaire de gestion du numéro national
    datas : données originales
    """
    ine = F.Input(name='Ine',
                  libelle="Numéro national de l'élève (INE)",
                  inline=True)
    num = datas.get('Ine', [''])[0]
    ine.default(num)
    return ine

def make_force_passwd_checkbox(disabled=False, checked=True):
    """
    définition de la checkbox pour "force_password"
    """
    if EOLE_AD:
        # Hack pour le mode AD #27595
        disabled = True
    if disabled:
        checked = False
    return F.Checkbox(name='force_password',
                      libelle='Forcer la modification du mot de passe à la 1ère connexion',
                      inline=True,
                      checked=checked,
                      disabled=disabled)

def get_role_association_btn(server_nb, username):
    """ renvoie la descritpion du bouton d'appel pour l'édition de rôle """
    return M.Bouton(href=tools.make_js_link(server_nb, 'role_manager',
                                            balise='div_role_container',
                                            form='usertoassociate',
                                            user=username),
                     libelle="Associer un rôle à cet utilisateur",
                     icone="/image/action_role.png",
                     _class='scribe_action_btn')

def make_adresseinput(datas={}):
    """
    definition du formulaire de gestion des adresses
    datas : données originales
    """
    form = []
    for attr, label, _ in ADRESSE:
        form.append(F.Input(name=attr, libelle=label, inline=True,
                            default=datas.get(attr, [''])[0]))
    return form

def get_transf_btns(racine='admin_class'):
    """ renvoit les differents btns de transfert de donnees d'une select à une liste
        :racine:racine des noms html des balises concernées
    """
    href1 = "javascript:formRemoveFromList('%s_form', '%s_select');" % (racine, racine)
    href2 = "javascript:formAddNameToList3('%s_select','%s_table');" % (racine, racine)
    icone1 = "/image/scribe/rightleft_red.png"
    icone2 = "/image/scribe/leftright.png"
    libelle1 = "Retirer"
    libelle2 = "Ajouter"
    btns = [M.Bouton(href=href1, libelle=libelle1, icone=icone1),
            M.Bouton(href=href2, libelle=libelle2, icone=icone2)]
    return btns

def get_close_btn():
    """
        renvoie la description du bouton de fermeture de la section modify
    """
    html_tag = 'mod_user_div_container'
    _class = "scribe_close_btn"
    return dict(close_btn=M.CloseBtn(_class=_class, balise=html_tag))

def make_etabinput(user):
    """
    definition du formulaire pour l'affichage du login
    :param user: login de l'utilisateur
    """
    etab = str(eoleuser.User().get_etab(user))
    return F.Input(name='etab',
            libelle="Etablissement",
            disabled=True,
            default=etab,
            inline=True)
