# -*- coding: UTF-8 -*-
###########################################################################
# EOLE - 2011
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
# Gestion du DHCP Statique
#
###########################################################################
import re
from os.path import join

from twisted.python import log

from pyeole.process import system_out

from ead2.config.config import DHCP_PATH, DHCP_ACTIVATION_EAD3
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.backend.actions.seshat import seshat_tools as t
from ead2.backend.config.config import EOLE_CONFIG_DIR
from ead2.backend.actions.lib.widgets import form
from ead2.backend.config.config import debug
from ead2.backend.actions.lib.widgets import ajax

EAD_DHCP = join(EOLE_CONFIG_DIR, 'dhcp.conf')
LEASES_FILE = join('/', DHCP_PATH, 'var/lib/dhcp/dhcpd.leases')
used_templates = ['main', 'form', 'formselect', 'accordion']
VALID_DOMAIN = re.compile('^[a-zA-Z\d\-]*$')

def gen_dhcp():
    """
    Appel au script de génération des configurations DHCP
    """
    res = system_out(['/usr/share/eole/sbin/gen_dhcp'])
    messages = res[1]+res[2]
    if messages != '':
        if debug:
            log.msg(messages.rstrip())
    return messages

def get_leases():
    """
    Lecture des baux DHCP
    """
    leases = []
    blocs = file(LEASES_FILE).read().split('}')
    for bloc in blocs:
        if '  client-hostname ' not in bloc:
            # bail expiré
            continue
        lines = bloc.split('lease ')[-1].splitlines()
        adresse = lines[0].split()[0]
        name = lines[-1].split('"')[1]
        for line in lines:
            if line.startswith('  hardware ethernet'):
                mac = line.split('  hardware ethernet ')[1][0:-1]
                leases.append((name, adresse, mac))
                break
    return leases

class Dhcp(Action):
    """ Page """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = "dhcp" #celui de perm_nomdumodule.ini
    libelle = "DHCP statique"
    category = "Outils"
    description = "Gestion des adresses DHCP"

    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'delroute'])#server et action = base

    form_result = Dict(default={},
                       doc='retour de formulaire',
                       keys=['addroute'])

    def execute(self):
        """
        submit
        """
        messages = ''
        params, self.server_nb = tools.get_request(self.request)
        # Dans le cas d'un formulaire:
        if DHCP_ACTIVATION_EAD3 == "oui":
            datas = dict(message="La gestion du DHCP dans l'EAD2 a été désactivée par l'EAD3."
                                 "Pour gérer le DHCP, connectez vous à l'EAD3 (https://<machine>/ead)",
                         redirect=ajax.call(self.server_nb,
                                            'main_status',
                                             only=False))
            return self.send_all(datas,
                    template='dhcp',
                    templates=used_templates)
        else:
            if self.form_result.has_key('addroute'):
                res = tools.format_form_result(self.form_result['addroute'])
                # datas: dico utilisé pour la templatisation
                machine, adresse, mac = res['machine'], res['adresse'], res['macadresse']
                valid = True
                if len(mac) != 17 or mac.count(':') != 5:
                    messages = "Adresse MAC invalide"
                    valid = False
                elif adresse.count('.') != 3:
                    messages = "Adresse IP invalide"
                    valid = False
                if valid:
                    try:
                        sadresse = adresse.split('.')
                        for adr in sadresse:
                            int(adr)
                    except:
                        messages = "Adresse IP invalide"
                        valid = False
                if valid and not VALID_DOMAIN.search(res['machine']):
                    messages = "Nom de machine invalide"
                    valid = False
                if valid:
                    if debug:
                        log.msg("+ ajout de machine %s %s %s" % (machine, adresse, mac))
                    res = t.add_mac(EAD_DHCP, machine, adresse, mac)
                    messages = gen_dhcp()
                # nettoyage du formulaire
                self.form_result = {}
            elif params.has_key('delroute'):
                machine, adresse, mac = params['delroute'][0].split('#')
                if debug:
                    log.msg("+ suppression de machine %s %s %s" % (machine, adresse, mac))
                res = t.del_mac(EAD_DHCP, machine, adresse, mac)
                messages = gen_dhcp()
        result = {} # variable renvoyée utilisée dans le template associé
        if messages:
            result['message'] = "ATTENTION :\\n%s" % messages.replace('\n', '\\n')
        result.update(self._get_list(t.get_routes(EAD_DHCP)))
        result.update(self._get_form())
        result.update(self._get_leases(get_leases()))
        return self.send_all(result, template='dhcp',
                             templates=used_templates)

    def _get_list(self, routes):
        """
        mise en forme des adresses déjà réservées
        """
        routes_list = []
        for machine, add, mac in routes:
            libelle = "%s (%s) -> %s" % (machine, add, mac)
            suppr_btn = dict(href=tools.make_js_link(self.server_nb, self.name,
                                                     delroute='%s#%s#%s' % (machine, add, mac)),
                             libelle="", title="", icone="/image/supprimer.gif", _class='btn')
            routes_list.append((libelle, suppr_btn))
        return dict(routes=routes_list)

    def _get_leases(self, leases):
        """
        mise en forme des baux dhcp actifs
        """
        onchange = 'javascript:fill_leases(this)'
        select = form.Select(name='leases',
                             libelle='Baux en cours',
                             inline=True,
                             onchange=onchange,
                            )
        # tri des triples sur le nom de machine
        leases = sorted(leases, key=lambda lease: lease[0])
        select.add_option(value='', libelle='')
        for machine, add, mac in leases:
            value = '{0}#{1}#{2}'.format(machine, add, mac)
            libelle = '{0} ({1})'.format(machine, add)
            select.add_option(value=value, libelle=libelle)
        return dict(leases=select)

    def _get_form(self):
        """
        Eléments du formulaire
        """
        machine = {'libelle': 'Nom de la machine',
                   'name':'machine', 'inline':'ok',
                   'onblur':"testVoid('machine', 'error_machine');"}
        adresse = {'libelle':'Adresse IP',
                   'name':'adresse', 'inline':'ok',
                   'onblur':"testVoid('adresse', 'error_adresse');"}
        macadresse = {'libelle':'Adresse MAC',
                    'name':'macadresse', 'inline':'ok',
                    'onblur':"testVoid('macadresse', 'error_macadresse');"}
        valid_btn = {'libelle':'Valider', 'title':'', 'icone':'/image/ok.gif',
                     'href':tools.make_form_link(self.server_nb, self.name,
                                                 True, ['addroute'])}
        return dict(machine=machine, adresse=adresse, macadresse=macadresse,
                    valid_btn=valid_btn)

