# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action Maj
#
# Programme des mises à jour du serveur de commande depuis l'ead
#
###########################################################################
""" Actions de programmation de mise à jour du serveur """
import urllib
from os.path import isfile
from twisted.python import log

from pyeole.process import system_out
from creole import maj

from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.lib.libead import uni
from ead2.config.config import SCHEDULE_CUSTOMIZE_TASKS

TEXT_AUTO = "La mise à jour s'effectuera tous les %(weekday)s"
TEXT_AUTO += " à partir de %(hour)s:%(minute)s"

def get_maj_mode():
    return "La mise à jour est configurée."

class Maj(Action):
    """ Page de programmation de mise à jour"""
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'maj'
    libelle = "Mise à jour"
    category = "Système"
    description = "Mise à jour du système"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'active', 'cancel_maj', 'paquet'])
    form_result = Dict(default={}, doc='retour de formulaire', keys=['maj'])

    def execute(self):
        """ gestionnaire de l'application de mise à jour
            1 - Renvoie la description du formulaire de programmation de mise à jour
            2 - Programme des mises à jour
            3 - Annule des mises à jour
            4 - Liste les paquets à mettre à jour
            5 - Active la mise à jour auto
        """
        params, self.server_nb = tools.get_request(self.request)
        result = {'titre':'Gestionnaire de mise à jour', "soustitre":"Mise à jour programmée"}
        result['mode_maj'] = get_maj_mode()
        retour_msg, retour_code = [None, None]
        ## 2 -
        if self.form_result.has_key('maj'):
            retour_msg = self._set_maj()
            self.form_result = {}
        ## 5 -
        if params.has_key('active'):
            retour_msg = self._set_maj(True, params['active'][0])
        ## 3 -
        if params.has_key('cancel_maj'):
            retour_msg = self._unset_maj()
        ## 4 -
        if params.has_key('paquet'):
            result['titre'] = "Liste des paquets disponibles"
            result['datas'] = self._get_pkgs()
            result['retour'] = self._get_retour_btn()
            dic = {'template':'maj_paquet', 'data':{'content':result}}
            return 0, uni(urllib.quote(str(dic)))
        if retour_msg is not None:
            result['retour_msg'] = retour_msg
        else:
            result['retour_msg'] = ''
        ## 1 -
        result['paquets'] = self._get_paquet_btn()
        result['formulaires'] = [{'name':'maj', 'formulaire':self._get_form(),
                                  'titre':'Programmez une mise à jour',
                                 'valide':self._get_valide_btn('maj')}]
        result.update(self._get_maj(True))
        result.update(self._get_maj())
        return self.send_all(result, template='maj', templates=['main'])

# methode d'envoi des donnees de mise en forme

    def _get_form(self):
        """ renvoit le formulaire """
        tag = {'_type':'choice', 'name':'majheure',
#                'default_value': {'name':'0'},
                'values':self._get_hours(),
                'libelle':'Mettre à jour '}
        return [tag]

    def _get_pkgs(self):
        """ renvoie la liste des paquets """
        res = system_out(['/usr/bin/Query-Auto', '-W'])
        if res[0] != 0:
            log.err("Erreur lors du lancement de '/usr/bin/Query-Auto -W' :")
            log.err(res[2])
        else:
            return '<pre>{0}</pre>'.format(res[1])

    def _get_retour_btn(self):
        """
        renvoie le bouton de retour pour la page de listing des paquets
        """
        return_btn = {}
        return_btn['href'] = tools.make_js_link(self.server_nb, self.name)
        return_btn['icone'] = "/image/back.png"
        return_btn['libelle'] = "Retour"
        return_btn['title'] = "Retour à la page de mise à jour"
        return return_btn

    def _get_valide_btn(self, formname):
        """ renvoit la description du bouton valider """
        validate = {}
        message = "Etes vous sûr de vouloir programmer cette mise à jour ?"
        link = tools.make_form_link(self.server_nb, self.name,
                                   False, [formname])
        validate['href'] = tools.make_confirm_link(message, link)
        validate['icone'] = "/image/ok.gif"
        validate['libelle'] = "OK"
        validate['title'] = "Programmer la mise à jour"
        return validate

    def _get_active_btn(self, active=False):
        """ renvoie la description du bouton activer/désactiver la mise à jour auto """
        btn = {}
        if active:
            btn['icone'] = "/image/ok.gif"
            btn['href'] = tools.make_js_link(self.server_nb, self.name,
                                             active='True')
            btn['libelle'] = "Activer"
            btn['title'] = "Activer la mise à jour auto"
        else:
            btn['icone'] = "/image/suppr.gif"
            btn['href'] = tools.make_js_link(self.server_nb, self.name,
                                             active='False')
            btn['libelle'] = "Désactiver"
            btn['title'] = "Désactiver la mise à jour auto"
        return btn

    def _get_paquet_btn(self):
        """ description du bouton pour l'affichage des paquets dispo """
        btn  = {}
        btn['href'] = tools.make_js_link(self.server_nb, self.name, paquet='liste')
        btn['libelle'] = "Voir la liste des paquets à mettre à jour (cette action peut prendre du temps)"
        btn['title'] = "Liste les paquets disponibles pour la mise à jour"
        return btn

    def _get_hours(self):
        """ renvoie les heures (depuis 1->23) pour les formulaires """
        hours = [
                {'name':'0', 'libelle':'maintenant'},
                {'name':'once', 'libelle':'cette nuit'},
                {'name':'1', 'libelle':'dans 1 heure'},
                ]
        for hrs in range(24)[2:]:
            hour = {'name':str(hrs), 'libelle':'dans %d heures' % hrs}
            hours.append(hour)
        return hours

    def _get_maj(self, auto=False):
        """ renvoie la liste des mises à jour programmées ou hebdo"""
        if auto:
            # maj hebomadaire
            if SCHEDULE_CUSTOMIZE_TASKS == 'oui':
                text = "Les préférences de mise à jour sont définies dans l'interface de configuration du module"
                return dict(active={'texte':text,
                                    'msg':"Mise à jour automatique"})
            if maj.maj_enabled():
                text = TEXT_AUTO % maj.get_maj_day()
                btn = self._get_active_btn(False)
            else:
                btn = self._get_active_btn(True)
                text = "La mise à jour hebdomadaire est désactivée"
            return dict(active={'texte':text,
                                'btn':btn,
                                'msg':"Mise à jour automatique"})
        else:
            # maj programmée
            majdiff_filename = maj.DIFF_FILENAME
            if isfile(majdiff_filename):
                fichier = file(majdiff_filename, 'r')
                lines = fichier.read()
                fichier.close()
                if lines:
                    return dict(maj={'texte':lines,
                                "btn":self._get_cancel_maj_btn()})
            return dict(maj={'texte':"Aucune mise à jour n'est programmée"})

    def _get_cancel_maj_btn(self):
        """ description du bouton d'annulation de la mise a jour programmee """
        cancel = {}
        cancel['href'] = tools.make_js_link(self.server_nb, self.name,
                                           cancel_maj='true')
        cancel['icone'] = "/image/suppr.gif"
        cancel['libelle'] = "Annuler"
        cancel['title'] = "Annuler la mise à jour"
        return cancel

# methode d'action sur les programmations de mise a jour
    def _unset_maj(self):
        """ annule les mise a jour programmee """
        retour = maj.cancel_maj_differee()
        if retour:
            return "La mise à jour a été annulée."
        else:
            log.err("Erreur lors du lancement de creole.maj.cancel_maj_differee()")
            return "Echec lors de l'annulation de la mise à jour."

    def _set_maj(self, auto=False, *kw):
        """ programme la mise à jour """
        if not auto:
            result = tools.format_form_result(self.form_result['maj'])
            heure = result['majheure']
            return self._set_maj_differe(heure)
        else:
            ## activite vient du frontend (str et pas booleen)
            activite = kw[0]
            if activite == 'True':
                return self._set_maj_auto()
            elif activite == 'False':
                return self._set_maj_auto(False)
            else:
                return "Erreur lors de l'activation/désactivation de la mise à jour automatique."

    def _set_maj_differe(self, nb_heure):
        """ programme une mise à jour différée dans nb_heure"""
        retour = maj.prog_maj_differee(nb_heure)
        if retour:
            if nb_heure == '0':
                return "La mise à jour va être effectuée."
            elif nb_heure == 'once':
                return "La mise à jour s'effectuera cette nuit"
            elif nb_heure == '1':
                return "La mise à jour s'effectuera dans %s heure." % nb_heure
            else:
                return "La mise à jour s'effectuera dans %s heures." % nb_heure
        else:
            log.err("Erreur lors de la programmation de mise à jour.")
            return "Erreur lors de la programmation de la mise à jour."

    def _set_maj_auto(self, activation=True):
        """ active la mise à jour automatique
            :activation: True/False, active/desactive
        """
        if activation:
            maj.enable_maj_auto()
            return "La mise à jour hebdomadaire a été activée"
        else:
            maj.disable_maj_auto()
            return "La mise à jour hebdomadaire a été désactivée"

