# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Actions isis, isis_disconnect, isis_stop, isis_login, isis_quota,
# isis_machine, isis_sendmsg
#
# Gere les connexions au domaine
#
###########################################################################
"""
Actions permettant de gérer les connexions sur un horus (application Isis)
"""
from twisted.python import log
import commands
from re import escape

from pyeole.process import system_code
from horus.backend import get_active_users, set_active_users
from ead2.backend.lib.action import Action, Dict, Boolean, Text
from ead2.lib.libead import uni
from ead2.backend.actions import tools
# import des fonctions specifiques a isis
from ead2.backend.actions.horus.connexion_tools import get_menu, \
list_valid_workstations, get_back_to_main, parse_smbstatus

used_templates = ['main', 'listlist', 'form', 'alphas',
                  'checklist', 'formselect']

class Isis(Action):
    """ entrée de l'interface isis, presente dans le menu
        :stopping: booleen indiquant qu'on attend une deconnexion
                   de tout le monde
    """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'isis'
    libelle = 'Connexion'
    category = 'Outils'
    description = 'gestion des connexions'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'stopping'])
    # variable specifique a cette action
    stopping = Boolean(default=False, doc="stop le gestionnaire")

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - récupère les connexions en cour
            2 - renvoie le menu adapté
                (message et déconnexion si un user est connecté)
            3 - déconnecte tout le monde
        """
       # on recup le numero du serveur en cours de traitement et la requete
        params, server_nb = tools.get_request(self.request)
        ## 3 -
        if 'stopping' in params.keys():
            self.stop()
        ## 1 -
        contenu = self.get_content()
        if not contenu:
            contenu = {'message':"Aucun utilisateur n'est connecté"}
            menu = get_menu(server_nb, page='index')
        else:
            menu = get_menu(server_nb, page='index', extended=True)
        ## 2 -
        result = {'menus':menu, 'users':contenu,
                  'sstitre':'index', 'titre':self.name}
        return self.send_all(result, template='horus_isis_index',
                             templates=used_templates)

    def get_content(self):
        """
        renvoie la liste des utilisateurs connectés et de leur activité
        """
        code, result = commands.getstatusoutput('/usr/bin/smbstatus')
        if code == 0:
            try:
                users = parse_smbstatus(result)
            except:
                log.err()
                return None
            return users
        return None

    def stop(self):
        """ coupe les connexions et deconnecte les utilisateurs """
        users = get_active_users()
        if len(users)>0:
            for user in get_active_users():
                set_active_users(user, False)
        code, result = commands.getstatusoutput('/usr/bin/smbstatus')
        if code == 0:
            try:
                users = parse_smbstatus(result)
            except:
                users = []
            if len(users)>0:
                for user in users:
                    commands.getstatusoutput('kill -9 %s' % user['pid'])
        return "La déconnexion s'est bien passée"


class IsisStop(Action):
    """ section d'arret du gestionnaire de connexion
    """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'isis_stop'
    libelle = 'Arret des Connexions'
    category = None
    description = 'Arret des connexions'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])

    def execute(self):
        """ renvoie les données pour l'affichage """
        params, self.server_nb = tools.get_request(self.request)
        result = {}
        result['menu'] = get_menu(self.server_nb, page='stop')
        result['valid'] = self.get_valid_btn()
        result['msg1'] = "Êtes-vous sûr de vouloir :"
        result['msg2'] = "Interdire tous les utilisateurs de pouvoir se connecter ?"
        result['msg3'] = "Et de déconnecter tous les utilisateurs actuellement connectés ?"
        result['retour'] = get_back_to_main(self.server_nb)
        return self.send_all(result, template='horus_isis_stop')

    def get_valid_btn(self):
        """ renvoit le bouton de validation de l'arret des connexions """
        href = tools.make_js_link(self.server_nb, 'isis', stopping="true")
        return {'href':href, 'icone':'/image/ok.gif'}

class IsisLogin(Action):
    """ Gestion des logins et des droits de connexion
    """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'isis_login'
    libelle = 'Gestion des droits de connexion'
    category = None
    description = 'Gestionnaire de droits de connexion'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'user_alpha', 'frag'])
    form_result = Dict(default={}, doc="arguments Json de la requete",
                       keys=['active', 'unactive'])

    def execute(self):
        """
            renvoie les données pour l'affichage
            1 - renvoie le formulaire (liste actif, inactif) pour gérer les autorisations
            2 - renvoie une partie des utilisateurs (filtré par première lettre du login)
            3 - traite le retour formulaire en autorisant ou interdisant des users
        """
        params, self.server_nb = tools.get_request(self.request)
        result = {}
        result['menus'] = get_menu(self.server_nb, page='login')
        result['retour'] = get_back_to_main(self.server_nb)
        result['titre'] = 'Connexion'
        result['sstitre'] = self.libelle
        auth = None
        self.user_alpha = ''
        self.frag = ''
        ## 2 -
        if params.has_key('user_alpha'):
            self.user_alpha = params['user_alpha'][0]
            self.frag = params['frag'][0]
            return self.user_frag()
        ## 3 -
        if self.form_result != {}:
            try:
                auth = self.autorization(self.form_result)
            except:
                result['messages'] = " Une erreur s'est produite lors de la gestion des autorisations"
                self.form_result = {}
                return self.send_frag(result, template='error')
            self.form_result = {}
        if auth is not None:
            result['messages'] = auth
            return self.send_frag(result, template='error')
        ## 1 -
        result.update(self._get_form())
        return self.send_all(result, template='horus_isis_login')

    def user_frag(self):
        """ fragment utilisateur
            renvoie les donnés pour une liste d'utilisateur actif ou inactif
            dont le login commence par user_alpha
        """
        result = {}
        if self.frag == 'active':
            result['datas'] = self._get_active_users()
        else:
            result['datas'] = self._get_active_users(False)
        d = {'template':'checklist', 'data':{'content':result}}
        return 0, uni(str(d))

    def _get_form(self):
        """
        renvoie la description du formulaire
        d'interdiction-autorisation des utilisateurs
        """
        right = {'title':"Utilisateurs autorisés", 'form':"active",
                 'datas':self._get_active_users(True)}
        left = {'title':'Utilisateurs interdits', 'form':"unactive",
                'datas':self._get_active_users(False)}
        title = ''
        name = 'horus_autorized_users'
        unactive_user_alphas = self._get_alpha_links()
        active_user_alphas = self._get_alpha_links(False)
        btns = self._get_table_links()
        btns.append(self._get_valid_btn())
        user_table = {'name':name, 'btns':btns, 'left':left,
                      'right':right, 'title':title}
        return dict(user_table=user_table,
                    active_user_alphas=active_user_alphas,
                    unactive_user_alphas=unactive_user_alphas)

    def _get_active_users(self, bool=True):
        """ renvoie une liste d'utilisateurs:
            bool : True pour utilisateur actif
                    False pour utilisateur inactif
        """
        try:
            usrs = get_active_users(bool)
        except:
            usrs = []
        result = []
        if self.user_alpha:
            alpha = self.user_alpha.lower()
        else:
            alpha = 'tous'
        if alpha == 'tous':
            return [{'name':usr}for usr in usrs]
        if alpha:
            for usr in usrs:
                if '.' in usr:
                    usrname = usr.split('.')[1].lower()
                    if usrname[0] == alpha:
                        result.append({'name':usr})
                else:
                    if usr[0] == alpha:
                        result.append({'name':usr})
        else:
            result = [{'name':usr}for usr in usrs]
        return result

    def _get_alpha_links(self, active=True):
        """ renvoit la description des liens alphabetique
            active: True si c'est pour les users actifs
        """
        links = []
        alpha = self.user_alpha
        for i in range(97, 123):
            if chr(i) != alpha:
                if active:
                    href = tools.make_js_link(self.server_nb, self.name,
                            user_alpha=chr(i), frag='active',
                            balise='active_table')
                else:
                    href = tools.make_js_link(self.server_nb, self.name,
                            user_alpha=chr(i), frag='unactive',
                            balise='unactive_table')
            else:
                href = ''
            libelle = chr(i).upper()
            links.append({'href':href, 'libelle':libelle})
        if 'tous' != alpha:
            if active:
                href = tools.make_js_link(self.server_nb, self.name,
                        user_alpha='tous', frag='active',
                        balise='active_table')
            else:
                href = tools.make_js_link(self.server_nb, self.name,
                        user_alpha='tous', frag='unactive',
                        balise='unactive_table')
            links.append({'href':href, 'libelle':'Tous'})
        else:
            links.append({'href':'', 'libelle':'Tous'})
        return links

    def _get_table_links(self):
        """ renvoit les boutons de transfert d'élément de liste à liste
        """
        href1 = "javascript:formTransferListElement('active','unactive_table');"
        href2 = "javascript:formTransferListElement('unactive','active_table');"
        icone1 = "/image/rightleft_red.png"
        icone2 = "/image/leftright.png"
        msg1 = "Interdire"
        msg2 = "Autoriser"
        btns = [{'href':href1, 'libelle':msg1, 'icone':icone1, 'title':msg1},
                {'href':href2, 'libelle':msg2, 'icone':icone2, 'title':msg2}]
        return btns

    def _get_valid_btn(self):
        """ renvoit la description du bouton valider """
        msg = 'Valider vos changements'
        href = tools.make_form_link(self.server_nb, self.name, True,
                                    ['active', 'unactive'])
        description = 'valider les changements effectués'
        icone = '/image/ok.gif'
        return {'libelle':msg, 'icone':icone, 'href':href, 'title':description}

    def autorization(self, dico):
        """ autorise (active) et interdit (desactive) les utilisateurs
        """
        _list_active = []
        if dico.has_key('active'):
            if dico['active'] != []:
                for item in dico['active']:
                    nom = item['name']
                    value = item['value']
                    _list_active.append(nom)
        else:
            return "Impossible de valider les changements."
        _list_unactive = []
        if dico.has_key('unactive'):
            if dico['unactive'] != []:
                for item in dico['unactive']:
                    nom = item['name']
                    value = item['value']
                    _list_unactive.append(nom)
        else:
            return "Impossible de valider les changements."
        a = set_active_users(_list_active, True)
        b = set_active_users(_list_unactive, False)
        if not a or not b:
            return "Erreur dans la gestion des autorisations"
        return None

class IsisQuota(Action):
    """
        renvoie le statut des quotas des utilisateurs
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'isis_quota'
    libelle = "Quotas disque"
    category = "Outils"
    description = "Affichage de l'utilisation de l'espace disque"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])

    def execute(self):
        """ renvoit les données pour l'affichage des quotas
        """
        params, self.server_nb = tools.get_request(self.request)
        result = {}
        result['refresh'] = self._get_refresh_btn()
        try:
            result['quotas'] = self._get_quotas()
        except:
            result['quotas'] = []
#        result['menus'] = get_menu(self.server_nb, page='quota')
        result['titre'] = 'Isis'
        result['sstitre'] = self.libelle
        return self.send_all(result, template='horus_isis_quota',
                             templates=used_templates)

    def _get_refresh_btn(self):
        """ renvoit la description du bouton refresh """
        href = tools.make_js_link(self.server_nb, self.name)
        return {'href':href, 'icone':"/image/refresh.gif",
                'title':"Rafraichir l'affichage", 'libelle':'Rafraichir'}

    def _get_quotas(self):
        """ script de récupération de la sortie de repquota -a
        """
        # nous lançons findsmb et récupérons la sortie
        commande = "/usr/sbin/repquota -a"
        code, sortie = commands.getstatusoutput(commande)
        users = get_active_users(True)
        users.extend(get_active_users(False))
        if code == 0:
            # nous mettons chaque connexion dans une liste
            lines = sortie.splitlines()
            # parsing des lignes:
            for index, line in enumerate(lines) :
                if '-------' in line:
                    spaces = lines[(index+1):]
                    break
            result = []
            # parsing des espaces disques
            for space in spaces:
                b = {}
                a = space.split()
                if len(a)>0:
                    if a[0] in users:
                        b['user'] = a[0]
                        try:
                            taille = int(a[2])
                            dispo  = int(a[3])
                            if dispo == 0:
                                b['usedspace'] = str(taille/1024)+' (Mo)'
                            else:
                                b['usedspace'] = "%d/%d (Mo)" % (taille/1024, dispo/1024)
                            if a[1] == '+-':
                                b['delay'] = a[5].replace('days',' jours').replace(':',' heures ')
                            else:
                                b['delay'] = ''
                        except:
                            b['usedspace'] = 'undefined'
                            b['delay'] = ''
                        result.append(b)
            return result

class IsisMachines(Action):
    """ Recherche des stations connectées au réseau
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'isis_machine'
    libelle = 'Machines du réseau'#'Machines connectées au réseau'
    category = "Outils"
    description = 'Liste les machines'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', '_type'])
    # variable spécifique
    _type = Text(default='tout', doc="type de machine à afficher")

    def execute(self):
        """ renvoit les données pour l'affichage
            1 - récupération du type àa afficher
             (controleur de domaine, maitre explorateur, toutes les stations)
            2 - renvoie la liste des stations, ainsi que le menu
        """
        params, self.server_nb = tools.get_request(self.request)
        result = {}
        ## 1 -
        if '_type' in params.keys():
            self._type = str(params['_type'][0])
        ## 2 -
#        result['menus'] = get_menu(self.server_nb, page='machine')
        result['titre'] = 'Isis'
        result['sstitre'] = "Les stations connectés au réseau"
        result['refresh'] = self.get_refresh_btn()
        result['links'] = self.get_type_btns()
        result['msg'] = "Type de station"
        result['msg_gen'] = " au domaine"
        try:
            result['workstations'] = list_valid_workstations(self._type)
        except:
            result['workstations'] = []
        msg_dict = {'tout': 'Toutes les stations connectées',
                    'maitre': 'Les maîtres explorateurs connectés',
                    'controleur':'Les controleurs connectés' }
        result['machinetype'] = msg_dict[self._type]
        return self.send_all(result, template='horus_isis_machine',
                             templates=used_templates)

    def get_type_btns(self):
        """
        renvoie la description des boutons de choix
        du type de station à afficher
        """
        href1 = tools.make_long_js_link(self.server_nb,
                    self.name, _type="tout")
        href2 = tools.make_long_js_link(self.server_nb,
                    self.name, _type="maitre")
        href3 = tools.make_long_js_link(self.server_nb,
                    self.name, _type="controleur")
        msg1 = "Toutes les stations"
        msg2 = "Maîtres explorateurs"
        msg3 = "Controleurs de domaine"
        return {'href':href1, 'libelle':msg1, 'title':msg1, 'icone':''}, \
               {'href':href2, 'libelle':msg2, 'title':msg2, 'icone':''}, \
               {'href':href3, 'libelle':msg3, 'title':msg3, 'icone':''}

    def get_refresh_btn(self):
        """ renvoit le bouton de rarfaichissement """
        href = tools.make_js_link(self.server_nb, self.name,
                                 _type=self._type)
        icone = "/image/refresh.gif"
        return {'href':href, 'icone':icone, 'title':'Rafraichir',
                'libelle':'Rafraichir'}

class IsisDisconnect(Action):
    """ section de deconnexion des utilisateurs
    """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'isis_disconnect'
    libelle = 'Déconnexion des utilisateurs'
    category = None
    description = 'Déconnecte et désactive des utilisateurs'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={},
                       doc="retour de formulaire en syntaxe Json",
                       keys=['isis_index', 'disconnect'])

    def execute(self):
        """
            renvoie les données pour l'affichage
            1 - récupère la liste des utilisateurs à déconnecter
                (on vient de la page d'accueil)
            2 - renvoie le formulaire de confirmation
            3 - valide la déconnexion d'utilisateur
        """
        params, self.server_nb = tools.get_request(self.request)
        users = []
        result = {}
        result['titre'] = self.libelle
        result['retour'] = get_back_to_main(self.server_nb)
        result['valid'] = self.get_valid_btn()
        if self.form_result != {}:
            try:
                if self.form_result.has_key('isis_index'):
                    ## 1 -
                    users = self.get_users(self.form_result['isis_index'])
                    self.form_result = {}
                elif self.form_result.has_key('disconnect'):
                    ## 3 -
                    return_msg = self.disconnect(self.form_result['disconnect'])
                    result['messages'] = return_msg
                    result['retour'] = get_back_to_main(self.server_nb)
                    self.form_result = {}
                    return self.send_frag(result,
                                          template='horus_isis_action_return')
            except:
                result['messages'] = " Une erreur s'est produite lors de la déconnexion "
                self.form_result = {}
                return self.send_frag(result, template='error')
        result['msg'] = "Déconnecter  et désactiver les utilisateurs:"
        result['users'] = [{'name':user, 'default_value':user}for user in users]
        if users == [] or users == None:
            result['message'] = "Aucun utilisateur n'a été sélectionné."
            result['toexec'] = tools.make_js_link(self.server_nb, "isis")
        result['formid'] = 'disconnect' # nom du formulaire
        return self.send_all(result, template='horus_isis_disconnect')

    def get_users(self, users_list):
        """ renvoit la liste des utilisateurs que l'on va déconnecter
        """
        users = tools.format_form_result(users_list, False).keys()
        return users

    def disconnect(self, form_result):
        """ deconnecte les utilisateurs """
        #FIXME : on ne peut plus forcer la déconnexion
        # (voit pas en quoi juste killer le pid bloque l'utilisateur)
        users = self.get_users(form_result)
        return_msg = ["Compte rendu de déconnexion :\n"]
        code, result = commands.getstatusoutput('/usr/bin/smbstatus')
        if code == 0:
            try:
                act_users = parse_smbstatus(result)
            except:
                act_users = []
            for user in act_users:
                if user['nom'] in users:
                    code, result = commands.getstatusoutput('kill -9 %s ' % user['pid']) # on deconnecte
                    if code:
                        return_msg.append("Echec de deconnexion (1) pour  %s.\n" % user['nom'])
                    else:
                        return_msg.append("Déconnexion réussie pour %s.\n" % user['nom'])
        if users != []:
            for user in users:
                act = set_active_users(user, False)
                if not act:
                    return_msg = "Echec de déconnexion (2) pour %s.\\n " % user
        return return_msg

    def get_valid_btn(self):
        """ renvoie le bouton de validation de l'arret des connexions """
        ref = tools.make_form_link(self.server_nb, self.name,
                                   False, ['disconnect'])
        href = tools.make_confirm_link("Êtes-vous sûr de vouloir couper les connexions ?", ref)
        return {'href':href, 'icone':'/image/ok.gif',
                'title':'Envoyer le message', 'libelle':'Envoyer'}

class IsisSendMsg(Action):
    """ section d'envoi de message du gestionnaire de connexion
    """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'isis_sendmsg'
    libelle = 'Envoi de message'
    category = None
    description = 'Envoi des messages aux utilisateurs'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', '_type'])
    form_result = Dict(default={},
                       doc="retour de formulaire en syntaxe Json",
                       keys=['isis_index', 'sendmsg'])

    def execute(self):
        """ renvoie les donnees pour le mise en forme
            1 - récupère la liste des utilisateurs
                (lorsqu'on vient de la page d'accueil)
            2 - renvoie le formulaire d'envoi de message
            3 - traite le retour formulaire, envoie le message et renvoie le retour
        """
        params, self.server_nb = tools.get_request(self.request)
        users = []
        return_msg = None
        result = {}
        result['titre'] = self.libelle
        result['retour'] = get_back_to_main(self.server_nb)
        result.update(self.get_valid_btn())
        if self.form_result != {}:
            if self.form_result.has_key('isis_index'):
                ## 1 -
                users = self.get_users(tools.format_form_result(self.form_result['isis_index'], False))
                self.form_result = {}
            elif self.form_result.has_key('sendmsg'):
                ## 3 -
                return_msg = self.send_msg(self.form_result['sendmsg'])
                self.form_result = {}
                result['messages'] = return_msg
                result['retour'] = get_back_to_main(self.server_nb)
                return self.send_frag(result,
                                      template='horus_isis_action_return',
                                      templates=['main', 'form', 'formselect'])
        ## 2 -
        result['msg'] = "Envoyer un message aux utilisateurs:"
        result['users'] = [{'name':user, 'default_value':user}for user in users]
        if users == [] or users == None:
            result['message'] = "Aucun utilisateur n'a été sélectionné"
            result['toexec'] = tools.make_js_link(self.server_nb, "isis")
        result.update(self._get_form())
        return self.send_all(result, template='horus_isis_sendmsg',
                                     templates=['main', 'form', 'formselect'])

    def _get_form(self):
        """
        renvoie la description du formulaire d'envoi de message
        """
        msg_input = {'name':'message', 'id':'message',
                     'libelle':'Message à envoyer',
                     'cols':'25', 'rows':'8'}
        return dict(msg_input=msg_input, focus='message')

    def get_users(self, users_list):
        """
        renvoie la liste des utilisateurs auxquels on va envoyer un message
        """
        users = users_list.keys()
        return users

    def get_valid_btn(self):
        """ renvoie le bouton de validation de l'arret des connexions """
        href = tools.make_form_link(self.server_nb, self.name,
                                    True, ['sendmsg'])
        valid = {'href':href, 'icone':'/image/ok.gif',
                'libelle':'OK', 'title':"Envoyer le message"}
        validation = tools.make_key_form_link(self.server_nb, self.name,
                                              False, ['sendmsg'])
        return dict(valid=valid, validation=validation)

    def send_msg(self, form_return):
        """ envoie le message aux utilisateurs """
        message = None
        result = tools.format_form_result(form_return)
        message = result.pop('message').strip()
        users = self.get_users(result)
        return_msg = ["Compte rendu de l'envoi du message :\n"]
        try:
            message = message.encode('utf-8')
        except:
            pass
        if users != []:
            for user in users:
                #code, retour = tools.command_statusoutput("echo %s" % escape(message),
                #                                      "smbclient -NM %s" % escape(user))
                code = system_code(['smbclient', '-NM', escape(user)], stdin=escape(message))
                if code:
                    return_msg.append( "Envoi à %s: échec\n" % user)
                else:
                    return_msg.append("Envoi à %s: réussi \n" % user)
        return return_msg

