# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action securite
#
# Edite les regles de pare-feu, interdiction de poste, destinations,
# peer to peer
#
###########################################################################

""" Actions de gestion des paramètres de sécurité """
from os.path import isfile
from twisted.python import log
from pyeole.service import service_out
from pyeole.process import system_out
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.backend.config.filenames import securite
from ead2.backend.actions.amon import menus
from ead2.backend.actions.amon.amon_tools import get_lines, write_file

# templates additionnels
used_templates = ['main', 'form', 'formselect']

#id html des formulaires à valider
form_name = {'regles':['mod_regles'],
             'interdits':['list_add', 'current_page'],
             'horaire':['list_add', 'current_page']}

# nom des fichiers a editer
p2p_filename = securite['p2p']
horaire_filename = securite['horaire']

def get_default_horaires():
    """ renvoie les horaires par defaut """
    return ["1 - -", "2 - -", "3 - -", "4 - -",
            "5 - -", "6 - -", "0 - -"]

def get_horaires():
    """ renvoie les horaires du pare-feu
        sous la forme ["numjour heuredeb heurefin", ...]
    """
    if isfile(horaire_filename):
        horaires = open(horaire_filename, 'r').read().splitlines()
        if len(horaires) <7:
            horaires = get_default_horaires()
    else:
        horaires = get_default_horaires()
    return parse_horaires(horaires)

def parse_horaires(horaires):
    """ parse les horaires du pare-feu """
    result = []
    for i in horaires:
        horaire = i.split()
        if len(horaire) == 3:
            result.append((horaire[0], horaire[1], horaire[2],))
        else:
            # si il y a une erreur dans le fichier
            return parse_horaires(get_default_horaires())
    return result

def write_horaire(jours):
    """ ecrit les horaires du pare-feu """
    for jour in jours:
        try:
            deb = int(jour[1])
            end = int(jour[2])
        except:
            deb = 0
            end = 1
        if deb > end:
            raise Exception("Erreur: l'heure d'ouverture du firewall" + \
                             "doit précéder celle de fermeture")
    fic = open(horaire_filename, 'w')
    for jour in jours:
        fic.write('%s %s %s \n' % (jour[0], jour[1], jour[2]))
    fic.close()
    return True

def get_hours(act_hour='-'):
    """ recupere les heures """
    hours = [{'libelle':"--------", "name":"-"},
             {'libelle':"0h 01", "name":"0"}]
    for i in range(1, 24):
        libelle = '%dh 00' % i
        value = '%d' % i
        hours.append({'libelle':libelle, "name":value})
    hours.append({'libelle':'23h 59', 'name':'24'})
    for i in hours:
        if i['name'] == act_hour:
            i['default'] = 'ok'
    return hours

class P2P(Action):
    """ interface de gestion du peer to peer """
    user_description = Dict(default={},
            doc="description de l'exécutant",
            keys=['ip', 'name', 'role'])
    name = "peertopeer"
    libelle = "P2P et horaires du pare-feu"
    description = "Activation du peer to peer et horaires"
    category = "Configuration générale"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'p2p'])

    def execute(self):
        """ renvoie les données pour la mise en forme
        """
        params, self.server_nb = tools.get_request(self.request)
        self.page = 'p2p'
        if 'p2p' in params:
            p2p_status = params['p2p'][0]
            self._valid_p2p(p2p_status)

        result =  {'titre': self.libelle}
        result['sstitre'] = ''
        result['menus'] = menus._get_menu(self.server_nb, self.page)
        result.update(self._get_p2p())
        return self.send_all(result, template='amon_p2p',
                             templates=used_templates)

    def _get_p2p(self):
        """ renvoie la description du formulaire de gestion p2p """
        msg = ["Les échanges de fichiers (peer to peer) sont "]
        lines = get_lines(p2p_filename)
        if lines and lines[0] == "KILLP2P=on":
            msg.append("interdits")
            color = "#b22"
            validate = {"href":tools.make_js_link(self.server_nb, self.name,
                                                    p2p="autoriser"),
                       'libelle':"Autoriser",
                       'icone':"/image/ok.gif",
                       'title':"Autoriser le peer to peer"}
        else:
            msg.append("autorisés")
            color = "#2b2"
            validate = {"href":tools.make_js_link(self.server_nb, self.name,
                                                    p2p="interdire"),
                        'libelle':"Interdire",
                        'icone':"/image/amon/refused.png",
                        'title':"Interdire le peer to peer"}
        entete = "Echanges de fichiers (peer to peer)"
        return dict(ptpmsg=msg,
                    forbid_p2p=validate,
                    entete=entete,
                    color=color)

    def _valid_p2p(self, p2p_status):
        """
        valide l'activation/désactivation du peer to peer
        {'p2p':autoriser/interdire}
        """
        if p2p_status == 'autoriser':
            write_file(p2p_filename, ["KILLP2P=off"])
            service_out('killp2p', 'stop')
        else:
            write_file(p2p_filename, ["KILLP2P=on"])
            service_out('killp2p', 'start')
        return {}

class Horaire(Action):
    """ interface de gestion des horaires de pare-feu """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = "horaire"
    libelle = 'Horaires du pare-feu'
    description = "horaires d'activation du pare-feu"
    category = None
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={},
                  doc="retour formulaire de la requete en cours coté frontend",
                  keys = ['list_add', 'list_suppr'])

    def execute(self):
        """ renvoie les données pour la mise en forme
        """
        params, self.server_nb = tools.get_request(self.request)
        self.page = 'horaire'
        if self.form_result != {}:
            try:
                datas = self._valid_horaire()
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name)
            except Exception as mess:
                log.err()
                log.err("Erreur: securite.py, %s" % mess)
                datas = dict(message=str(mess))
            self.form_result = {}
            return self.send_frag(datas, template='amon_horaire',
                                  templates=used_templates)
        result =  {'titre': self.libelle}
        result['sstitre'] = ''
        result['menus'] = menus._get_menu(self.server_nb, self.page)
        result.update(self._get_horaire())
        if self.page in form_name:
            result['validate'] = {'href':tools.make_long_form_link(self.server_nb,
                                                                   self.name,
                                                                   True,
                                                                   form_name[self.page],
                                                                   balise='return_msg_container'),
                                  'icone':'/image/amon/ok.gif',
                                  'libelle':'Valider',
                                  'title':'Valider les changements'}
            result['validation'] = tools.make_key_form_link(self.server_nb,
                                                            self.name,
                                                            False,
                                                            form_name[self.page],
                                                            balise='return_msg_container')
        return self.send_all(result, template='amon_horaire',
                             templates=used_templates)

    def _get_horaire(self):
        """
        renvoie la description du formulaire
        de gestion des horaires de pare-feu
        """
        semaine = ['dimanche', 'lundi', 'mardi',
                'mercredi', 'jeudi', 'vendredi', 'samedi']
        horaires = get_horaires()
        days = []
        for horaire in horaires:
            num_jour, houvert, hferme = horaire
            indice = int(num_jour)
            ouvert_select = {'name':'ouvert[%d]' % (indice),
                    'options':[{'name':'', 'options':get_hours(houvert)}]}
            ferme_select = {'name':'ferme[%d]' % (indice),
                    'options':[{'name':'', 'options':get_hours(hferme)}]}
            days.append({'libelle':semaine[indice],
                         'ouvert':ouvert_select,
                         'ferme':ferme_select})
        return dict(horaire=days)

    def _valid_horaire(self):
        """ valide la configuration des horaires du pare-feu """
        if 'list_add' not in self.form_result:
            raise Exception("Erreur: il manque des données " + \
                             "pour la gestion des horaires du pare-feu.")
        config = tools.format_form_result(self.form_result['list_add'])
        jours = []
        for i in range(7)[1:]:
            keyopen = 'ouvert[%d]' % i
            keyclosed = 'ferme[%d]' % i
            indice = i
            jours.append(('%d'%indice, config[keyopen], config[keyclosed]))
        keyopen, keyclosed = 'ouvert[0]', 'ferme[0]'
        jours.append(('0', config[keyopen], config[keyclosed]))
        write_horaire(jours)
        res = system_out('bastion', 'restart')
        if res[0] == 0:
            return dict(message="Le pare-feu a bien été redémarré, " + \
                                "les nouveaux horaires sont pris en compte.")
        else:
            log.err("Erreur: securite.py, une erreur s'est produite au lancement de 'bastion restart'(3).")
            log.err(res[1])
            log.err(res[2])
            raise Exception("Erreur: une erreur est survenue au redémarrage du pare-feu.")
