# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
librairie contenant les définitions des champs de formulaires
"""
from scribe import eolegroup
from scribe.ldapconf import MAIL_DOMAIN, SUPPORT_ETAB, HAS_SYMPA
from ead2.backend.actions.lib.widgets import form as F
from ead2.backend.actions.scribe.tool import scribe_tools
from ead2.backend.actions.scribe.tool import formencode
from ead2.backend.actions.scribe.tool.group import LISTE_TYPES
if SUPPORT_ETAB:
    from ead2.backend.actions.scribe.tool.etablissements import get_etabs

SHARE_TYPES = (("rw", 'lecture/écriture (membres)'),
               ("ro", 'lecture seule (élèves)'),
               ('dt', 'modèle données/travail'))

def _get_common_create_form():
    """
        Renvoie la description de formulaire commune
        à la création d'un prof ou d'un élève
    """
    user = F.Input(name='user', libelle='Login (prenom.nom conseillé)',
                   inline=True, required=True)

    name = formencode.make_nameinput()
    forname = formencode.make_fornameinput()

    date = F.Input(name='date', libelle="Date de naissance (format jj/mm/aaaa)",
                   inline=True, required=True)
    password = F.Input(name='password', libelle="Mot de passe",
                       inline=True, required=True)
    if scribe_tools.use_eclair():
        shell = formencode.make_shellinput({'loginShell':['/bin/bash']})
        force_password = formencode.make_force_passwd_checkbox(checked=False)
    else:
        shell = formencode.make_shellinput()
        force_password = formencode.make_force_passwd_checkbox()

    # civilite
    civilite = formencode.make_civiliteinput(None, name='civilite')

    # les quotas
    quota = F.Input(name='user_quota',
                    libelle="Quota disque (0 pour inactif)",
                    inline=True,
                    _type='number',
                    default='0')
    # les profils
    profile = formencode.make_profilinput()

    return dict(user=user,
                name=name,
                forname=forname,
                date=date,
                password=password,
                shell=shell,
                force_password=force_password,
                civilite=civilite,
                quota=quota,
                profile=profile
                )

def _get_mail_form():
    """
    gestion spécifique des adresses mail (hors élèves)
    choix adresse personnalisée ou boite locale
    """
    return formencode.make_mailinput('', {})

def _get_responsable_form():
    """
    fabrication du formulaire pour les responsables
    """
    form = _get_common_create_form()
    form['adresse'] = formencode.make_adresseinput()
    form['mail'] = _get_mail_form()
    form['eleve'] = F.Input(name= 'eleve0',
                            libelle="Login élève(s)",
                            inline=True)
    del(form['profile'])
    del(form['quota'])
    del(form['force_password'])
    del(form['shell'])
    del(form['date']['onblur'])
    return form

def _get_administratif_form():
    """
    fabrication du formulaire pour les personnels administratifs
    """
    form = _get_common_create_form()
    # mail
    form['mail'] = _get_mail_form()

    form['admin'] = F.Checkbox(name='admin',
                       libelle="Membre du groupe DomainAdmins",
                       inline=True)
    if SUPPORT_ETAB:
        form['etab'] = F.Select(name='etab',
                                   libelle='Etablissement associé',
                                   inline=True)
        etabs = get_etabs()
        #etablissement de référence
        form['etab'].add_option(etabs[0])
        for etab in etabs[1]:
            form['etab'].add_option(etab)
    return form

def _get_other_form():
    """
    fabrication du formulaire pour les invités
    """
    form = _get_common_create_form()
    form['mail'] = _get_mail_form()
    del(form['profile'])
    del(form['quota'])
    del(form['force_password'])
    del(form['shell'])
    del(form['date']['onblur'])
    return form

def _get_teacher_form():
    """ renvoie la description du formulaire de création de professeur
    :user, name, forname, password, date de naissance: \
        inputs de login nom prenom, mot de passe, date de naissance
    :civilite: select pour Mr/Mme/Mlle
    :date:
    :mail:radios et input pour la configuration du mail
    :quota: input pour quotas disque attribué
    :profile: select pour profil utilisateur local/profil1/profil2
    :principal: select pour prof principal?
    :admin: checkbox pour administrateur?
    :shell: activer le shell (/bin/bash)
    """
    form = _get_common_create_form()
    # mail
    form['mail'] = _get_mail_form()
    # prof principal
    if SUPPORT_ETAB:
        form['etab'] = F.Select(name='etab',
                                   libelle='Etablissement associé',
                                   inline=True)
        etabs = get_etabs()
        #etablissement de référence
        form['etab'].add_option(etabs[0])
        for etab in etabs[1]:
            form['etab'].add_option(etab)
    principal = F.Select(name='principal',
                         libelle='Professeur principal',
                         inline=True)
    principal.add_option('0', libelle='non', default=True)
    for niveau, classes in list(eolegroup.Group().get_classes().items()):
        for _cl in classes:
            principal.add_option(_cl, libelle="de la classe %s" % \
                                      _cl, group=niveau)
    form['principal'] = principal

    form['admin'] = F.Checkbox(name='admin',
                       libelle="Membre du groupe DomainAdmins",
                       inline=True)
    return form

def _get_pupil_form(role='admin', login=''):
    """ renvoie la description du formulaire de creation d'eleve
    :user, name, forname, password, date: \
        inputs de login, nom, prenom, mot de passe, date de naissance
    :civilite: select pour Mr/Mme/Mlle
    :user_number:input pour le numéro elenoet de l'élève
    :ine:input pour le numéro national de l'élève
    :quota: input pour quotas disque attribué
    :profile: select pour profil utilisateur local/profil1/profil2
    :divcod: select pour le choix de la classe
    @role : role associé à l'utilisateur appelant
            (permet de restreindre la création aux classes d'un prof_admin)
    @login : login du prof_admin
    :shell: activer le shell (/bin/bash)
    """
    form = _get_common_create_form()
    # domaine mail
    domains = F.Select(name='domains', libelle='@', inline=True)
    for cle, value in list(MAIL_DOMAIN.items()):
        domains.add_option(cle, libelle=value)
    domains.default('restreint')
    form['domains'] = domains

    # numéro ELENOET
    form['user_number'] = formencode.make_elenoetinput()

    form['ine'] = formencode.make_ineinput()

    # classe de l'élève
    divcod = F.Select(name='divcod', libelle="Classe", inline=True)
    _niveaux_dict = eolegroup.Group().get_classes()
    if role == 'prof_admin':
        _niveaux_dict = scribe_tools.filter_class_by_prof(login, _niveaux_dict)
        if _niveaux_dict == {}:
            return dict(message="Erreur : vous n'êtes administrateur d'aucune classe")
    flag = False
    _niveaux = list(_niveaux_dict.keys())
    _niveaux.sort()
    for _niveau in _niveaux:
        _classes = _niveaux_dict[_niveau]
        _classes.sort()
        for _cl in _classes:
            flag = True
            divcod.add_option(_cl, group=_niveau)
    if not flag:
        return dict(message="Il faut au moins une classe pour créer un élève")
    form['divcod'] = divcod
    return form

def _get_user_listing_form(prof_admin=None):
    """ renvoie la description du formulaire de recherche utilisateur
        :prof_admin: nom du prof dans le cas d'un prof_admin
                     (permet de restreindre les critères de recherche)
    """
    alpha_keys = F.Select(name='alpha_key',
                 libelle="Première lettre du login")
    alpha_keys.add_option("", default=True)
    for abc in range(97, 123):
        alpha_keys.add_option(chr(abc).upper())

    if not prof_admin:
        user_type = F.Select(name='user_type', libelle="Type de l'utilisateur")
        user_type.add_option("", default=True)
        user_type.add_option("pupil", "élève")
        user_type.add_option("teacher", "professeur")
        user_type.add_option("administratif", "administratif")
        user_type.add_option("responsable", "responsable")
        user_type.add_option("other", "invité")
    else: # prof_admin ne peut modifier que des élèves
        user_type = F.Input(name='user_type', default='pupil')
    # classe
    classe_select = F.Select(name='user_class', libelle='Membre de la classe')
    _niveaux_dict = eolegroup.Group().get_classes()
    if not prof_admin:
        classe_select.add_option('', default=True)
    else:
        _niveaux_dict = scribe_tools.filter_class_by_prof(prof_admin,
                                                          _niveaux_dict)
    _niveaux = list(_niveaux_dict.keys())
    _niveaux.sort()
    _niveaux.reverse()
    for _niveau in _niveaux:
        _classes = _niveaux_dict[_niveau]
        _classes.sort()
        for _cl in _classes:
            classe_select.add_option(_cl, group=_niveau)
    # groupes
    if not prof_admin:
        group = F.Select(name='user_group', libelle='Membre du groupe')
        group.add_option('', default=True)
        _gr_types = ['Groupe', 'Matiere', 'Service' ,
                     'Equipe', 'Administrateurs']
        for _gr_type in _gr_types:
            _groups = eolegroup.Group().get_groups(_gr_type)
            _groups.sort()
            for _gr in _groups:
                group.add_option(_gr, group=_gr_type)
    else:
        group = F.Input(name='user_group')
        group.default('')
    #domaine
    domain = F.Select(name='domains',
                      libelle="Type d'adresse mail")
    domain.add_option('', default=True)
    for cle, value in list(MAIL_DOMAIN.items()):
        domain.add_option(value, libelle="domaine %s" % cle)
    domain.add_option('externe', libelle="externe à l'établissement")
    domain.add_option('empty', libelle="pas d'adresse")
    #nom
    namepart = F.Input(name='namepart', libelle="Partie du nom de famille")
    #nombre d'utilisateur par page
    size = F.Select(name='size', libelle='Nombre de résultats par page')
    for _num in range(20, 101, 20):
        size.add_option(str(_num))
    size.default('40')
    sort_method = F.Select(name='sort_method',
                           libelle='Tri par')
    sort_method.add_option('login')
    sort_method.add_option('nom')
    return dict(alpha_keys=alpha_keys,
                user_type=user_type,
                classe=classe_select,
                domain=domain,
                namepart=namepart,
                size=size, group=group,
                sort_method= sort_method)

def groupe_form(_type, msg):
    """
    Renvoie la description du formulaire de création de groupe
    :_type : type du groupe (en minuscule)
    :msg: message associé au type de groupe à créer
    @group_name : nom du groupe (input)
    @group_desc : description du groupe (input)
    @share_check : avec partage ? (checkbox)
    @share_status : statut du partage
    @liste_check : avec liste de diffusion ? (checkbox)
    @liste_status : statut de la liste de diffusion (internet ou restreint)
    @level : niveau associé (si classe)
    @etab : etablissement associé si activé
    """
    fields = {}

    ####################
    ## champs communs ##
    ####################
    # nom du groupe
    fields['group_name'] = F.Input(name='group_name',
                                   libelle='Nom %s' % msg,
                                   inline=True,
                                   required=True)
    # description du groupe
    fields['group_desc'] = F.Input(name='group_desc',
                                   libelle='Description %s' % msg,
                                   inline=True)
    # domaine pour les listes de diffuion
    if HAS_SYMPA:
        fields['liste_status'] = F.Select(name='liste_status',
                                          libelle="Liste de diffusion",
                                          inline=True)
        for code, libelle in LISTE_TYPES:
            fields['liste_status'].add_option(code, libelle=libelle)

    if SUPPORT_ETAB and _type not in ['classe', 'etablissement']:
        fields['etab'] = F.Select(name='etab',
                                   libelle='Etablissement associé',
                                   inline=True)
        etabs = get_etabs()
        #etablissement de référence
        fields['etab'].add_option(etabs[0])
        for etab in etabs[1]:
            fields['etab'].add_option(etab)

    ####################
    # champs variables #
    ####################
    if _type == 'classe':
        # classe : gestion des niveaux
        niveaux = eolegroup.Group().get_groups('Niveau')
        if niveaux == []:
            return dict(message="Erreur : il faut au moins un niveau pour créer une classe")
        fields['level'] = F.Select(name='level',
                                   libelle='Niveau associé',
                                   inline=True)
        for nivo in niveaux:
            fields['level'].add_option(nivo)
    elif _type in ['option', 'niveau']:
        # option : liste faculattive
        if HAS_SYMPA:
            fields['liste_check'] = F.Checkbox(name='liste_check',
                                               libelle="Avec liste de diffusion",
                                               inline=True,
                                               checked=True)
    elif _type == 'groupe':
        # groupe : tous les choix possibles
        if HAS_SYMPA:
            fields['liste_check'] = F.Checkbox(name='liste_check',
                                     libelle="Avec liste de diffusion",
                                     inline=True)
        fields['share_check'] = F.Checkbox(name='share_check',
                                           libelle="Avec Partage",
                                           inline=True,
                                           checked=True)
        fields['share_status'] = F.Select(name="share_status",
                                          libelle="Partage",
                                          inline=True)
        for code, libelle in SHARE_TYPES:
            fields['share_status'].add_option(code, libelle=libelle)
    elif _type in ['matiere', 'service', 'etablissement']:
        # choix du type de partage simplifié
        fields['share_check'] = F.Checkbox(name='share_check',
                                           libelle="Avec Partage",
                                           inline=True,
                                           checked=True)
        if HAS_SYMPA:
            fields['liste_check'] = F.Checkbox(name='liste_check',
                                     libelle="Avec liste de diffusion",
                                     inline=True)
    else:
        return dict(message="Erreur : type inconnu : %s" % _type)
    return fields
