# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action scribe_user*
# Creation d'utilisateur
# Recherche puis modification d'utilisateur
# modfication de mot de passe
# Supression d'utilisateur
#
###########################################################################
"""Actions permettant de modifier un utilisateur"""

from twisted.python import log
from scribe import eoleuser
from scribe.eleves import Eleve
from scribe.enseignants import Enseignant
from scribe.responsables import Responsable
from scribe.autres import Autre
from scribe.eolegroup import Group
from scribe.ldapconf import MAIL_DOMAIN, SUPPORT_ETAB
from scribe.eoletools import replace_cars
from ead2.backend.config.config import debug
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions.lib.widgets import main as M, form as F
from ead2.backend.actions import tools
from ead2.lib.error import MissingKey, MissingValue
from ead2.backend.actions.scribe.tool import formencode
from ead2.backend.actions.scribe.tool import scribe_tools, filterdata
from ead2.backend.actions.scribe.tool.user import used_templates, \
manage_groups, set_teacher_administration, set_managed_group, ADRESSE

# formulaires gérés par l'action
FORMULAIRES = ['mod_user', 'current_type', "equipe_form", 'admin_class_form',
               'matiere_form', 'option_form', 'groupe_form', 'service_form',
               'spec_group_form', 'managed_group_form']

class UserModify(Action):
    """ Modifie les utilisateurs
        accessible par prof_admin avec un controle
        pour les modifs et les renvoi de formulaire.
    """
    user_description = Dict(default={},
                       doc="description de l'exécutant",
                       keys=['ip', 'name', 'role'])
    name = 'scribe_user_modify'
    libelle = "Modification"
    category = None
    description = 'Modifier un utilisateur'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'user'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=FORMULAIRES)

    def execute(self):
        """ renvoit les données pour l'affichage
            1 - récupère le nom et le type d'utilisateur à modifier
            2 - renvoie le formulaire approprié
            3 - tente la modification et si la modification est réussie
                on rafraichit le formulaire
                a - récupère les données du formulaire
                b - filtre pour voir les valeurs modifiées
                    et vérifie leur intégrité
                c - applique les modifications
        """
        params, self.server_nb = tools.get_request(self.request)
        result = {} #'titre':"Gestion des utilisateurs"}
        ## 1-
        self.user = ''
        self._type = ''
        datas = {}
        if 'user' in params:
            self.user = params['user'][0]
        if self.form_result != {}: ## on modifie l'utilisateur
            if 'current_type' in self.form_result:
                self._type = tools.format_form_result(self.form_result['current_type'])['_type']
                self.form_result.pop('current_type')
            ## 3-
            try:
                datas = self._valid_form()
                datas['toexec'] = tools.make_js_link(self.server_nb,
                                  self.name, confirm=True,
                                  balise='mod_user_div_container',
                                  user=self.user)
            except (MissingKey, MissingValue, Exception) as mess:
                log.err("Erreur dans user.py 2: %s" % mess)
                log.err()
                datas = dict(message=str(mess))
            self.form_result = {}
            return self.send_frag(datas, template='scribe_users',
                                  templates=used_templates)
        ## 2-
        datas = self._get_user()
        result['_type'] = {'name':'_type', 'default_value':self._type }
        result.update(self._get_form(datas))
        result.update(self._get_valid_btn())
        result.update(formencode.get_close_btn())
        if scribe_tools.is_admin(self.user_description['role']):
            result['role'] = formencode.get_role_association_btn(self.server_nb,
                                                                 self.user)
        return self.send_all(result,
                            template='scribe_users',
                            templates=used_templates)

    def _get_user(self):
        """
        recupere tous les attributs ldap de l'utilisateur courant
        recupere egalement le type d'utilisateur
        """
        if self.user:
            user = eoleuser.User()
            user.ldap_admin.connect()
            datas = user._get_attrs(self.user, [])
            self._type = user._get_ead_type(self.user)
            user.ldap_admin.close()
            return datas
        else:
            return {}

### fonctions de description de formulaire
    def _get_form(self, datas):
        """
        renvoie la description du formulaire de modification d'utilisateur
        """
        if datas == {}:
            return {}
        if self._type == 'teacher':
            return self._get_teacher_form(datas)
        elif self._type == 'pupil':
            return self._get_pupil_form(datas)
        elif self._type == 'administratif':
            return self._get_administratif_form(datas)
        elif self._type == 'responsable':
            return self._get_responsable_form(datas)
        elif self._type == 'autre':
            return self._get_other_form(datas)
        else:
            raise Exception("type d'utilisateur inconnu : %s" % str(self._type))

    def _get_teacher_form(self, datas):
        """ renvoie la description du formulaire de modification de prof """
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # seul un admin peut modifier des profs
            return {'message':'Action non autorisée'}

        # définition des champs du formulaire
        user_login = formencode.make_logininput(self.user)
        forname = formencode.make_fornameinput(datas)
        name = formencode.make_nameinput(datas)
        user_name = formencode.make_identityinput(datas)
        civilite = formencode.make_civiliteinput(datas)
        mail = formencode.make_mailinput(self.user, datas)
        date = formencode.make_dateinput(datas)
        quota = formencode.make_quotainput(self.user)
        profile = formencode.make_profilinput(self.user)
        shell = formencode.make_shellinput(datas)
        if SUPPORT_ETAB:
            etab = formencode.make_etabinput(self.user)

        # administration de classe
        if datas['typeadmin'][0] == '2':
            prof_classes = Enseignant().get_attr(self.user, 'Divcod')
        else:
            prof_classes = []
        if SUPPORT_ETAB:
            etab_user = eoleuser.User().get_etab(self.user)
        else:
            etab_user = None
        ldapgroup = Group()
        ldapgroup.ldap_admin.connect()
        all_classes = ldapgroup._get_classes(etab=etab_user)

        # classes affectées
        _liste = {'title':'Classes à administrer', 'formid':'admin_class_form',
                  'id':'admin_class_table', 'datas':prof_classes}


        _select = F.Select(name='admin_class_select',
                           libelle="Classes disponibles",
                           _id='admin_class_select',
                           multi=True, size=10)

        # remplissage des classes disponibles
        keys = list(all_classes.keys())
        keys.sort()
        for _level in keys:
            for _cl in all_classes[_level]:
                if _cl not in prof_classes:
                    _select.add_option(_cl, group=_level)
        classadmin = {'name':'classadmin',
                      'btns':formencode.get_transf_btns('admin_class'),
                      'select':_select,
                      'liste':_liste,
                      'title':"Administration de classes"}

        user_all_groups = set(ldapgroup._get_user_groups(self.user))

        # administration de groupes
        if 'ManagedGroup' in datas:
            _groups = datas['ManagedGroup']
        else:
            _groups = []
        all_groups = ldapgroup._get_groups('Groupe', etab=etab_user)
        all_groups.sort()
        _liste = {'title':"Groupes à administrer", 'formid':'managed_group_form',
                  'id':'managed_group_table', 'datas':_groups}
        _select = F.Select(name='managed_group_select',
                           libelle="Groupes disponibles",
                           _id='managed_group_select',
                           multi=True, size=10)
        for _gr in all_groups:
            if _gr not in _groups:
                _select.add_option(_gr)
        managed_group = {'name':"managed_group",
                'btns':formencode.get_transf_btns('managed_group'),
                'select':_select, 'liste':_liste,
                'title':"Groupes à administrer"}


        #------------------------ gestion des groupes du prof ----------------#

        ## attachement à un groupe
        all_groups = set(all_groups)
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        _liste = {'title':"Groupes de l'enseignant", 'formid':'groupe_form',
                  'id':'groupe_table', 'datas':user_groups}
        _select = F.Select(name='groupe_select', libelle="Groupes disponibles",
                           _id='groupe_select',
                           multi=True, size=10)
        for _gr in available_groups:
            _select.add_option(_gr)
        groupe = {'name':"groupe", 'btns':formencode.get_transf_btns('groupe'),
                'select':_select, 'liste':_liste, 'title':"Groupes"}

        ## attachement à une matière
        all_groups = set(ldapgroup._get_groups('Matiere', etab=etab_user))
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        _liste = {'title':"Matières de l'enseignant", 'formid':'matiere_form',
                'id':'matiere_table', 'datas':user_groups}
        _select = F.Select(name='matiere_select',
                libelle="Matières disponibles",
                _id='matiere_select',
                multi=True, size=10)
        for _mat in available_groups:
            _select.add_option(_mat)
        matiere = {'name':"matiere",
                'btns':formencode.get_transf_btns('matiere'),
                'select':_select, 'liste':_liste,
                'title':"Matières dispensées"}

        ## attachement à une équipe pédagogique
        all_groups = set(ldapgroup._get_groups('Equipe', etab=etab_user))
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        _liste = {'title':"Equipes pédagogiques de l'enseignant",
                  'formid':'equipe_form',
                  'id':'equipe_table',
                  'datas':user_groups}
        _select = F.Select(name='equipe_select', libelle="Equipes disponibles",
                _id='equipe_select',
                multi=True, size=10)
        for _gr in available_groups:
            _select.add_option(_gr)
        equipe = {'name':"equipe", 'btns':formencode.get_transf_btns('equipe'),
                  'select':_select, 'liste':_liste,
                  'title':"Equipes pédagogiques"}

        ## groupes speciaux
        all_groups = set(ldapgroup._get_groups('Administrateurs'))
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        _liste = {'title':"Groupes spéciaux de l'enseignant",
                  'formid':'spec_group_form', 'id':'spec_group_table',
                  'datas':user_groups}
        _select = F.Select(name='spec_group_select',
                libelle="Groupes disponibles", _id='spec_group_select',
                multi=True, size=10)
        for _gr in available_groups:
            _select.add_option(_gr)
        spec_group = {'name':"spec_group",
                    'btns':formencode.get_transf_btns('spec_group'),
                    'select':_select,
                    'liste':_liste, 'title':"Groupes spéciaux"}

        ldapgroup.ldap_admin.close()

        dico = dict(mod_user_login=user_login, name=user_name,
                    civilite=civilite, mail=mail, profile=profile,
                    date=date, quota=quota, classadmin=classadmin,
                    matiere=matiere, equipe=equipe, groupe=groupe,
                    spec_group=spec_group, shell=shell,
                    managed_group=managed_group, sn=name, forname=forname)
        if SUPPORT_ETAB:
            dico['etab'] = etab
        return dico

    def _get_administratif_form(self, datas):
        """
        renvoie la description du formulaire
        de modification des administratifs
        """
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # seul un admin peut modifier les personnels
            return {'message':'Action non autorisée'}

        # définition des champs du formulaire
        user_login = formencode.make_logininput(self.user)
        forname = formencode.make_fornameinput(datas)
        name = formencode.make_nameinput(datas)
        user_name = formencode.make_identityinput(datas)
        civilite = formencode.make_civiliteinput(datas)
        mail = formencode.make_mailinput(self.user, datas)
        date = formencode.make_dateinput(datas)
        quota = formencode.make_quotainput(self.user)
        profile = formencode.make_profilinput(self.user)
        shell = formencode.make_shellinput(datas)

        ldapgroup = Group()
        ldapgroup.ldap_admin.connect()
        user_all_groups = set(ldapgroup._get_user_groups(self.user))

        jeSuisUnProf = 'professeurs' in user_all_groups
        attrprof = formencode.make_attrprofinput(jeSuisUnProf)

        if jeSuisUnProf:
                # administration de classe
                if datas['typeadmin'][0] == '2':
                    prof_classes = Enseignant().get_attr(self.user, 'Divcod')
                else:
                    prof_classes = []
                if SUPPORT_ETAB:
                    etab_user = eoleuser.User().get_etab(self.user)
                else:
                    etab_user = None

                all_classes = ldapgroup._get_classes(etab=etab_user)

                # classes affectées
                _liste = {'title':'Classes à administrer', 'formid':'admin_class_form',
                          'id':'admin_class_table', 'datas':prof_classes}


                _select = F.Select(name='admin_class_select',
                                   libelle="Classes disponibles",
                                   _id='admin_class_select',
                                   multi=True, size=10)

                # remplissage des classes disponibles
                keys = list(all_classes.keys())
                keys.sort()
                for _level in keys:
                    for _cl in all_classes[_level]:
                        if _cl not in prof_classes:
                            _select.add_option(_cl, group=_level)
                classadmin = {'name':'classadmin',
                              'btns':formencode.get_transf_btns('admin_class'),
                              'select':_select,
                              'liste':_liste,
                              'title':"Administration de classes"}

                ## attachement à une équipe pédagogique
                if SUPPORT_ETAB:
                    etab_user = eoleuser.User().get_etab(self.user)
                else:
                    etab_user = None

                all_groups = set(ldapgroup._get_groups('Equipe', etab=etab_user))
                user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
                available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
                _liste = {'title':"Equipes pédagogiques de l'enseignant",
                          'formid':'equipe_form',
                          'id':'equipe_table',
                          'datas':user_groups}
                _select = F.Select(name='equipe_select', libelle="Equipes disponibles",
                        _id='equipe_select',
                        multi=True, size=10)
                for _gr in available_groups:
                    _select.add_option(_gr)
                equipe = {'name':"equipe", 'btns':formencode.get_transf_btns('equipe'),
                          'select':_select, 'liste':_liste,
                          'title':"Equipes pédagogiques"}
        else:
                equipe = None
                classadmin = None

        # administration de groupes
        if 'ManagedGroup' in datas:
            _groups = datas['ManagedGroup']
        else:
            _groups = []
        # FIXME : à voir avec Gaston ;)
        available_groups = ldapgroup._get_groups('Groupe')
        available_groups.sort()
        _liste = {'title':"Groupes à administrer",
                  'formid':'managed_group_form',
                  'id':'managed_group_table', 'datas':_groups}
        _select = F.Select(name='managed_group_select',
                           libelle="Groupes disponibles",
                           _id='managed_group_select',
                           multi=True, size=10)
        for _gr in available_groups:
            _select.add_option(_gr)
        managed_group = {'name':"managed_group",
                'btns':formencode.get_transf_btns('managed_group'),
                'select':_select, 'liste':_liste,
                'title':"Groupes à administrer"}


        #------- gestion des groupes du personnel administratif  ------#

        ## attachement à un groupe
        all_groups = set(ldapgroup._get_groups('Groupe'))
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        _liste = {'title':"Groupes",
                  'formid':'groupe_form',
                  'id':'groupe_table', 'datas':user_groups}
        _select = F.Select(name='groupe_select', libelle="Groupes disponibles",
                           _id='groupe_select',
                           multi=True, size=10)
        for _gr in available_groups:
            _select.add_option(_gr)
        groupe = {'name':"groupe", 'btns':formencode.get_transf_btns('groupe'),
                'select':_select, 'liste':_liste, 'title':"Groupes"}

        ## attachement à un service
        all_groups = set(ldapgroup._get_groups('Service'))
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        _liste = {'title':"Services administratifs", 'formid':'service_form',
                'id':'service_table', 'datas':user_groups}
        _select = F.Select(name='service_select',
                libelle="Services disponibles",
                _id='service_select',
                multi=True, size=10)
        for _grp in available_groups:
            _select.add_option(_grp)
        service = {'name':"service",
                'btns':formencode.get_transf_btns('service'),
                'select':_select, 'liste':_liste,
                'title':"Services administratifs"}

        ## groupes speciaux
        all_groups = set(ldapgroup._get_groups('Administrateurs'))
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        _liste = {'title':"Groupes spéciaux",
                  'formid':'spec_group_form', 'id':'spec_group_table',
                  'datas':user_groups}
        _select = F.Select(name='spec_group_select',
                libelle="Groupes disponibles", _id='spec_group_select',
                multi=True, size=10)
        for _gr in available_groups:
            _select.add_option(_gr)
        spec_group = {'name':"spec_group",
                    'btns':formencode.get_transf_btns('spec_group'),
                    'select':_select,
                    'liste':_liste, 'title':"Groupes spéciaux"}

        ldapgroup.ldap_admin.close()

        retour = dict(mod_user_login=user_login, name=user_name,
                    civilite=civilite, mail=mail, profile=profile,
                    date=date, quota=quota, shell=shell, groupe=groupe,
                    service=service, spec_group=spec_group, attrprof=attrprof,
                    managed_group=managed_group, sn=name, forname=forname)
        if jeSuisUnProf:
                retour.update(dict(classadmin=classadmin, equipe=equipe))

        return retour

    def _get_responsable_form(self, datas):
        """
        renvoie la description du formulaire
        de modification des responsables
        """
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # seul un admin peut modifier un responsable
            return {'message':'Action non autorisée'}

        # définition des champs du formulaire
        user_login = formencode.make_logininput(self.user)
        forname = formencode.make_fornameinput(datas)
        name = formencode.make_nameinput(datas)
        user_name = formencode.make_identityinput(datas)
        civilite = formencode.make_civiliteinput(datas)
        date = formencode.make_dateinput(datas)
        mail = formencode.make_mailinput(self.user, datas)
        adresse = formencode.make_adresseinput(datas)

        entry = datas.get('eleve', '')
        if not entry:
            eleves = "<tr><td><i>Aucun Eleve</i></td></tr>"
        else:
            eleves = ''
            ele_tmpl = """<tr><td>Eleve %d : %s
                          </td><td>supprimer l'association
                          <input type="checkbox" name="delassoc%d" value="%s"/>
                          </td></tr>"""
            for ele in entry:
                eleves += ele_tmpl % (entry.index(ele)+1, ele,
                                      entry.index(ele)+1, ele)
        eleves += """<tr><td>Ajouter un élève
                    </td><td><input type="text" name="addassoc"/>
                    </td></tr> <br/>"""
        return dict(mod_user_login=user_login, name=user_name,
                    civilite=civilite, mail=mail, date=date,
                    sn=name, forname=forname, adresse=adresse,
                    eleves=eleves)

    def _get_other_form(self, datas):
        """
        renvoie la description du formulaire
        de modification des responsables
        """
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # seul un admin peut modifier un invité
            return {'message':'Action non autorisée'}

        # définition des champs du formulaire
        user_login = formencode.make_logininput(self.user)
        forname = formencode.make_fornameinput(datas)
        name = formencode.make_nameinput(datas)
        user_name = formencode.make_identityinput(datas)
        civilite = formencode.make_civiliteinput(datas)
        date = formencode.make_dateinput(datas)
        mail = formencode.make_mailinput(self.user, datas)

        return dict(mod_user_login=user_login, name=user_name,
                    civilite=civilite, mail=mail, date=date,
                    sn=name, forname=forname)

    def _get_pupil_form(self, datas):
        """ renvoie la description du formulaire de modification d'élève """
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # seul l'admin peut modifier tous les eleves
            prof = self.user_description['name']
            if not scribe_tools.is_eleve_from_prof(self.user, prof):
                return {'message':'Action non autorisée'}

        # définition des champs du formulaire
        user_login = formencode.make_logininput(self.user)
        forname = formencode.make_fornameinput(datas)
        name = formencode.make_nameinput(datas)
        user_name = formencode.make_identityinput(datas)
        civilite = formencode.make_civiliteinput(datas)

        ## mail et domaine ##
        # les élèves ont forcément une adresse locale
        # seul le domaine est modifiable
        _mailadress, _domain = datas.get('mail', ['__none__@none'])[0].split('@')
        mail = F.Input(name="email", libelle="Adresse e-mail", disabled=True,
                       inline=True)
        if _mailadress != "__none__":
            mail.default(_mailadress)
        maildomain = F.Select(name="maildomain")
        for dom in list(MAIL_DOMAIN.values()):
            maildomain.add_option(dom)
        if _domain != 'none':
            maildomain.default(_domain)

        user_number = formencode.make_elenoetinput(datas)
        ine = formencode.make_ineinput(datas)
        date = formencode.make_dateinput(datas)
        quota = formencode.make_quotainput(self.user)
        profile = formencode.make_profilinput(self.user)
        shell = formencode.make_shellinput(datas)

        #nivo
        niveau = F.Input(name='Meflcf',
                         libelle="Niveau",
                         inline=True,
                         disabled=True)
        nivo = datas['Meflcf'][0]
        niveau.default(nivo)

        #classe
        classe = F.Select(name='_Divcod',
                          libelle="Classe",
                          inline=True)
        for nivo, classes in list(Group().get_classes().items()):
            for cls in classes:
                if cls == datas['Divcod'][0]:
                    classe.add_option(cls, default=True, group=nivo)
                else:
                    classe.add_option(cls, group=nivo)

        # Responsables (FIXME : à améliorer)
        ldapresp = Responsable()
        ldapresp.ldap_admin.connect()
        resps = ldapresp._get_responsables(attrs=['displayName', 'uid'],
                                          eleve=self.user)
        if not resps:
            responsables = "<tr><td><i>Aucun responsable</i></td></tr>"
        else:
            responsables = ''
            resp_tmpl = """<tr><td colspan="2">Responsable %d : %s (%s)</td></tr>"""
            for resp in resps:
                responsables += resp_tmpl % (resps.index(resp)+1,
                                             resp['displayName'][0],
                                             resp['uid'][0])
        ldapresp.ldap_admin.close()

        #---------- gestion des groupes de l'élève ---------------#

        ## connexion ldap
        ldapgroup = Group()
        ldapgroup.ldap_admin.connect()
        user_all_groups = set(ldapgroup._get_user_groups(self.user))

        ## attachement à un groupe
        all_groups = set(ldapgroup._get_groups('Groupe'))
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        _liste = {'title':"Groupes de l'élève", 'formid':'groupe_form',
                  'id':'groupe_table', 'datas':user_groups}
        _select = F.Select(name='groupe_select', libelle="Groupes disponibles",
                           _id='groupe_select',
                           multi=True, size=10)
        for _gr in available_groups:
            _select.add_option(_gr)
        groupe = {'name':"groupe", 'btns':formencode.get_transf_btns('groupe'),
                'select':_select, 'liste':_liste, 'title':"Groupes"}

        ## attachement à une option
        all_groups = set(ldapgroup._get_groups('Option'))
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        _liste = {'title':"Options de l'élève", 'formid':'option_form',
                  'id':'option_table', 'datas':user_groups}
        _select = F.Select(name='option_select', libelle="Options disponibles",
                           _id='option_select',
                           multi=True, size=10)
        for _gr in available_groups:
            _select.add_option(_gr)
        option = {'name':"option", 'btns':formencode.get_transf_btns('option'),
                  'select':_select, 'liste':_liste,
                  'title':"Options"}

        ## fin de la connexion
        ldapgroup.ldap_admin.close()

        return dict(mod_user_login=user_login,
                    name=user_name,
                    civilite=civilite,
                    maildomain=maildomain,
                    maileleve=mail,
                    profile=profile,
                    quota=quota,
                    user_number=user_number,
                    ine=ine,
                    divcod=classe,
                    niveau=niveau,
                    date=date,
                    option=option,
                    groupe=groupe,
                    shell=shell,
                    sn=name, forname=forname,
                    responsables=responsables)

### fonction de gestion du retour de formulaire de modification

    def _valid_form(self):
        """
        choix de la méthode à utiliser pour
        modifier un utilisateur selon son type
        """
        if self._type == 'pupil':
            return self._modify_pupil()
        elif self._type == 'teacher':
            return self._modify_teacher()
        elif self._type == 'responsable':
            return self._modify_responsable()
        elif self._type == 'administratif':
            return self._modify_administratif()
        elif self._type == 'autre':
            return self._modify_other()
        else:
            raise KeyError("Type d'utilisateur inconnu : %s" % self._type)


    def _modify_teacher(self):
        """ modifie un prof """
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # un prof_admin ne peut modifier que ses eleves
            return dict(message="Action interdite")

        ## recupération des résultats de formulaire et formatage
        modification = tools.format_form_result(self.form_result['mod_user'])
        self.user = modification['login']
        administration = tools.format_form_result(self.form_result['admin_class_form'],'2' )
        managed_group = tools.format_form_result(self.form_result['managed_group_form'],'2' )
        groupe = tools.format_form_result(self.form_result['groupe_form'], '2')
        matiere = tools.format_form_result(self.form_result['matiere_form'], '2')
        equipe = tools.format_form_result(self.form_result['equipe_form'], '2')
        spec_groupes = tools.format_form_result(self.form_result['spec_group_form'], '2')
        ## filtrage pour n'avoir que les données modifiées
        datas = self._get_user()
        modification = filterdata._filter_teacher_modif(self.user, datas,
                                                        modification)
        ## application des modifications
        if modification:
            self._set_teacher_modification(modification)

        # administration des classes
        set_teacher_administration(self.user, datas, list(administration.keys()))

        # administration des groupes
        set_managed_group(self.user, datas, list(managed_group.keys()))

        # -- gestion des inscriptions du professeur -- #
        ldapgroup = Group()
        ldapgroup.ldap_admin.connect()

        manage_groups(self.user, 'Groupe', ldapgroup, list(groupe.keys()))

        manage_groups(self.user, 'Matiere', ldapgroup, list(matiere.keys()))

        manage_groups(self.user, 'Equipe', ldapgroup, list(equipe.keys()))

        manage_groups(self.user, 'Administrateurs', ldapgroup,
                      list(spec_groupes.keys()))

        ldapgroup.ldap_admin.close()

        return dict(message="Les modifications ont bien été attribuées à %s" % self.user)

    def _modify_administratif(self):
        """ modifie un administratif """
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # un prof_admin ne peut modifier que ses eleves
            return dict(message="Action interdite")
        ## recupération des résultats de formulaire et formatage
        modification = tools.format_form_result(self.form_result['mod_user'])
        self.user = modification['login']
        groupe = tools.format_form_result(self.form_result['groupe_form'], '2')
        if 'equipe_form' in self.form_result:
                equipe = tools.format_form_result(self.form_result['equipe_form'], '2')
        else:
                equipe = None
        if 'admin_class_form' in self.form_result:
                administration = tools.format_form_result(self.form_result['admin_class_form'],'2' )
        else:
                administration = None
        service = tools.format_form_result(self.form_result['service_form'], '2')
        spec_groupes = tools.format_form_result(self.form_result['spec_group_form'], '2')
        managed_group = tools.format_form_result(self.form_result['managed_group_form'], '2')
        ## filtrage pour n'avoir que les données modifiées
        datas = self._get_user()
        modification = filterdata._filter_teacher_modif(self.user, datas, modification)

        ## application des modifications
        # FIXME: même code pour profs & administratifs ?
        if modification:
            self._set_teacher_modification(modification)

        # administration des classes
        if administration is not None:
                set_teacher_administration(self.user, datas, list(administration.keys()))

        set_managed_group(self.user, datas, list(managed_group.keys()))

        # -- gestion des inscriptions du personnel administratif -- #
        ldapgroup = Group()
        ldapgroup.ldap_admin.connect()

        manage_groups(self.user, 'Groupe', ldapgroup, list(groupe.keys()))

        if equipe is not None:
                manage_groups(self.user, 'Equipe', ldapgroup, list(equipe.keys()))

        manage_groups(self.user, 'Service', ldapgroup, list(service.keys()))

        manage_groups(self.user, 'Administrateurs', ldapgroup, list(spec_groupes.keys()))

        ldapgroup.ldap_admin.close()

        return dict(message="Les modifications ont bien été attribuées à %s" % self.user)

    def _modify_responsable(self):
        """ modifie un responsable """
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # un prof_admin ne peut modifier que ses eleves
            return dict(message="Action interdite")
        ## recupération des résultats de formulaire et formatage
        modification = tools.format_form_result(self.form_result['mod_user'])
        self.user = modification['login']
        datas = self._get_user()
        modification = filterdata._filter_responsable_modif(datas, modification)
        if modification:
            self._set_responsable_modification(modification)
        return dict(message="Les modifications ont bien été attribuées à %s" % self.user)

    def _modify_other(self):
        """ modifie un compte invité """
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # un prof_admin ne peut modifier que ses eleves
            return dict(message="Action interdite")
        ## recupération des résultats de formulaire et formatage
        modification = tools.format_form_result(self.form_result['mod_user'])
        self.user = modification['login']
        datas = self._get_user()
        modification = filterdata._filter_other_modif(datas, modification)
        if modification:
            self._set_other_modification(modification)
        return dict(message="Les modifications ont bien été attribuées à %s" % self.user)

    def _modify_pupil(self):
        """ modifie un élève """
        ## recuperation des valeurs
        modification = tools.format_form_result(self.form_result['mod_user'])
        self.user = modification['login']
        if scribe_tools.is_prof_admin(self.user_description['role']):
            # un prof_admin ne peut modifier que ses eleves
            prof = self.user_description['name']
            if not scribe_tools.is_eleve_from_prof(self.user, prof):
                return dict(message='Action non autorisée.')
        groupe = tools.format_form_result(self.form_result['groupe_form'], '2')
        option = tools.format_form_result(self.form_result['option_form'], '2')
        ## filtrage des donnees
        datas = self._get_user()
        modification = filterdata._filter_pupil_modif(self.user, datas, modification)
        ## application des changements
        if modification:
            self._set_pupil_modification(modification)

        # -- gestion des inscriptions de l'élève -- #
        ldapgroup = Group()
        ldapgroup.ldap_admin.connect()

        manage_groups(self.user, 'Groupe', ldapgroup, list(groupe.keys()))

        manage_groups(self.user, 'Option', ldapgroup, list(option.keys()))

        ldapgroup.ldap_admin.close()

        return dict(message="Les modifications ont bien été attribuées à %s" % self.user)


    def set_displayName(self, ldapuser, value):
        """
        Applique une modification sur le nom complet
        d'un utilisateur
        """
        ldapuser._set_attr(self.user, 'displayName', value)
        ldapuser._set_attr(self.user, 'cn', value)
        ldapuser._set_attr(self.user, 'gecos', replace_cars(value))


### fonctions d'application des modifications

    def _set_pupil_modification(self, modif):
        """ modifie les attributs d'un élève"""
        if debug:
            log.msg("set_pupil_modification for %s" % self.user)
            log.msg(modif)

        ldapuser = Eleve()
        ldapuser.ldap_admin.connect()
        datas = ldapuser._get_attrs(self.user, [])
        # FIXME : attributs facultatifs
        datas['Ine'] = ''

        # modification des attributs simples
        for key, value in list(modif.items()):
            if key == 'codecivilite':
                ldapuser._mod_civilite(self.user, value)
            elif key == 'dateNaissance':
                ldapuser._mod_date(self.user, value)
            elif key in ['displayName']:
                self.set_displayName(ldapuser, value)
            elif key in list(datas.keys()):
                ldapuser._set_attr(self.user, key, value)
            elif key == 'shell':
                ldapuser._update_shell(self.user, value)
            elif key == 'name':
                ldapuser._set_attr(self.user, 'sn', value)
            elif key == 'forname':
                ldapuser._set_attr(self.user, 'givenName', value)
        #gestion des quotas:
        if '_quota' in modif:
            eoleuser.set_quota(self.user, modif['_quota'])
        # gestion du mail: FIXME : code à améliorer ?
        if 'maildomain' in modif:
            mail = "%s@%s" % (self.user, modif['maildomain'])
            ldapuser._set_attr(self.user, 'mail', mail)
        ## changement de classe:
        if '_Divcod' in modif:
            ldapuser._change_classe(self.user, modif['_Divcod'])
        ## profile
        if '_sambaProfilePath' in modif:
            ldapuser._set_profil(self.user, modif['_sambaProfilePath'])

        ldapuser._touch(self.user)
        ldapuser.ldap_admin.close()


    def _set_teacher_modification(self, modif):
        """ modifie les attributs d'un prof
            modif: dictionnaire {'clé':valeur} revenant du formulaire puis filtré
        """
        if debug:
            log.msg("set_teacher_modification for %s" % self.user)
            log.msg(modif)

        ldapuser = Enseignant()
        ldapuser.ldap_admin.connect()
        datas = ldapuser._get_attrs(self.user, [])

        for key, value in list(modif.items()):
            if key == 'codecivilite':
                ldapuser._mod_civilite(self.user, value)
            elif key == 'dateNaissance':
                ldapuser._mod_date(self.user, value)
            elif key in ['displayName']:
                self.set_displayName(ldapuser, value)
            ## on ne modifie que les termes déjà existant
            elif key in list(datas.keys()):
                ldapuser._set_attr(self.user, key, value)
            elif key == 'shell':
                ldapuser._update_shell(self.user, value)
            elif key == 'attrprof':
                # Pour le cas des administratifs ayant les droits prof
                # Ce champ n'est jamais filtré puisqu'il n'est présent que dans
                # le formulaire de modification d'un personnel administratif
                ldapgroup = Group()
                ldapgroup.ldap_admin.connect()
                user_all_groups = set(ldapgroup._get_user_groups(self.user))
                if 'professeurs' not in user_all_groups:
                    # Ajout des attributs prof uniquement la 1ère fois
                    # lors du cochage de "Conférer les droits du profil professeur à cet utilisateur (action irréversible)"
                    ldapuser._inscription(self.user, 'professeurs')
                    ldapuser._set_attr(self.user, 'typeadmin', '0')
            elif key == 'name':
                ldapuser._set_attr(self.user, 'sn', value)
            elif key == 'forname':
                ldapuser._set_attr(self.user, 'givenName', value)

        ## gestion des quotas:
        if '_quota' in modif:
            eoleuser.set_quota(self.user, modif['_quota'])

        ## modification du mail
        if 'mailtype' in modif:
            ldapuser._mod_mail(self.user, modif['mailtype'], modif['mailperso'])

        ## profile
        if '_sambaProfilePath' in modif:
            ldapuser._set_profil(self.user, modif['_sambaProfilePath'])
        ## quota
        if 'quota' in modif:
            eoleuser.set_quota(self.user, int(modif['quota']))

        ldapuser._touch(self.user)
        ldapuser.ldap_admin.connect()


    def _set_responsable_modification(self, modif):
        """ modifie les attributs d'un responsable
            modif: dictionnaire {'clé':valeur} revenant du formulaire
                   puis filtré
        """
        if debug:
            log.msg("set_responsable_modification for %s" % self.user)
            log.msg(modif)

        ldapuser = Responsable()
        ldapuser.ldap_admin.connect()

        adresse = [add[0] for add in ADRESSE]

        for key, value in list(modif.items()):
            if key == 'codecivilite':
                ldapuser._mod_civilite(self.user, value)
            elif key == 'dateNaissance':
                ldapuser._mod_date(self.user, value)
            elif key in ['displayName']:
                self.set_displayName(ldapuser, value)
            elif key == 'name':
                ldapuser._set_attr(self.user, 'sn', value)
            elif key == 'forname':
                ldapuser._set_attr(self.user, 'givenName', value)
            elif key in adresse:
                ldapuser._set_attr(self.user, key, value)
            elif key == 'addassoc':
                ldapuser.c_add_association(self.user, value)
            elif key.startswith('delassoc'):
                ldapuser.c_del_association(self.user, value)

        ## modification du mail
        if 'mailtype' in modif:
            ldapuser._mod_mail(self.user, modif['mailtype'], modif['mailperso'])

        ldapuser._touch(self.user)
        ldapuser.ldap_admin.close()


    def _set_other_modification(self, modif):
        """ modifie les attributs d'un compte invité
            modif: dictionnaire {'clé':valeur} revenant du formulaire
                   puis filtré
        """
        if debug:
            log.msg("set_other_modification for %s" % self.user)
            log.msg(modif)

        ldapuser = Autre()
        ldapuser.ldap_admin.connect()

        for key, value in list(modif.items()):
            if key == 'codecivilite':
                ldapuser._mod_civilite(self.user, value)
            elif key == 'dateNaissance':
                ldapuser._mod_date(self.user, value)
            elif key in ['displayName']:
                self.set_displayName(ldapuser, value)
            elif key == 'name':
                ldapuser._set_attr(self.user, 'sn', value)
            elif key == 'forname':
                ldapuser._set_attr(self.user, 'givenName', value)

        ## modification du mail
        if 'mailtype' in modif:
            ldapuser._mod_mail(self.user, modif['mailtype'], modif['mailperso'])

        ldapuser._touch(self.user)
        ldapuser.ldap_admin.close()

### description des boutons pour la modification des utilisateurs
    def _get_valid_btn(self):
        """ renvoie la description du bouton valider """
        html_container = 'return_msg_container'
        href = tools.make_form_link(self.server_nb, self.name, True,
                                    FORMULAIRES, html_container)
        return dict(validate=M.Submit(href=href))
