# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action scribe_grouped_edition
# Edition d'utilisateur par groupe
#
###########################################################################
from twisted.python import log
from os.path import isfile, join
from os import unlink
from scribe import linker
from scribe import eoleuser
from scribe.eoleldap import Ldap
from scribe.eoletools import nscd_start, nscd_stop
from scribe.ldapconf import EOLE_AD, MIN_PASSWORD_CLASS
from ead2.lib.error import MissingKey, MissingValue
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.backend.actions.lib.widgets import form as F, main as M
from ead2.backend.actions.scribe.tool import scribe_tools
from ead2.backend.actions.scribe.tool import getform, grpedit
from ead2.backend.config.filenames import pwd_dir

rapport_dir = pwd_dir['rapport']

used_templates = ['main', 'form', 'formselect', 'accordion',
                  'listlist', 'checklist']

class GroupedEdition(Action):
    """ permet l'édition groupée d'utilisateur """
    user_description = Dict(default={}, doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_grouped_edition'
    libelle = "Edition groupée"
    category = "Gestion/"
    description = "édition groupée d'utilisateurs"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'btns', 'form', 'btn', 'purge'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['search', 'userlist',
                             'inscription_edit', 'quota_edit',
                             'domain_edit', 'profil_edit',
                             'password_edit', 'shell_edit'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - récupère le role de l'utilisateur et son login pour le cas d'un prof_admin
            2 - renvoie la description du formulaire de listing d'utilisateur
            3 - renvoie la liste des utilisateurs
            4 - renvoie les boutons d'action
            5 - renvoie les formulaires d'action
            6 - valide les actions effectuées
            7 - purge les fichiers de mot de passe
        """

        params, self.server_nb = tools.get_request(self.request)
        ## 1 -
        self.user = self.user_description['name']
        self.roles = self.user_description['role']

        result = {'titre':self.description}
        ## 5 -
        if 'form' in params:
            datas = self._get_action_form(params['form'][0])
            if params['form'][0] == 'password' and EOLE_AD and MIN_PASSWORD_CLASS > 1:
                # delete weak password choices
                del(datas['radio_date'])
                del(datas['radio_name'])
            return self.frag(datas)
        ## 4 -
        if 'btn' in params:
            return self.frag(self._get_action_btns(int(params['btn'][0])))
        if 'btns' in params:
            return self.frag(self._get_action_btns())
        if 'purge' in params:
            try:
                datas = self.purge_pwd_files()
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name,
                                            balise='div_password_container',
                                            confirm=True, btn='4')
            except Exception as mess:
                log.err("Erreur dans grouped_edition.py: %s" % mess)
                datas = dict(message=str(mess))
            return self.frag(datas)
        ## 6 - 3 -
        if self.form_result != {}:# validation de formulaire
            try:
                filtered_datas = self._valid_form()
            except (MissingKey, MissingValue, Exception) as mess:
                log.err("Erreur dans grouped_edition.py: %s"%mess)
                log.err()
                filtered_datas = {'message':str(mess)}
            filtered_datas.update({'toexec':tools.make_js_link(self.server_nb,
                                                    self.name,
                                                    balise='action_div_container',
                                                    confirm=True,
                                                    btns='True')})
            self.form_result = {}
            return self.frag(filtered_datas)
        ## 2 -
        if scribe_tools.is_prof_admin(self.roles):
            result.update(getform._get_user_listing_form(prof_admin=self.user))
        else:
            result.update(getform._get_user_listing_form())
        result.update(dict(validate=self._get_valid_btn()))
        return self.send_all(result, template='scribe_edition',
                             templates=used_templates)

    def frag(self, filtered_datas):
        """ renvoie les datas pour l'affichage
        de la liste des utilisateurs correpondant à la recherche"""
        return self.send_frag(filtered_datas, template='scribe_edition',
                              templates=used_templates)

    def _valid_form(self):
        """ valide les retours de formulaire """
        if 'search' in self.form_result:
            return self._valid_user_listing()
        elif 'inscription_edit' in self.form_result:
            return self._valid_inscription()
        elif 'quota_edit' in self.form_result:
            return self._valid_quota()
        elif 'domain_edit' in self.form_result:
            return self._valid_domain()
        elif 'profil_edit' in self.form_result:
            return self._valid_profil()
        elif 'password_edit' in self.form_result:
            return self._valid_password()
        elif 'shell_edit' in self.form_result:
            return self._valid_shell()

    def _valid_inscription(self):
        """ valide les inscriptions à des groupes """
        if 'userlist' not in self.form_result:
            raise MissingKey("Erreur : il manque des données pour l'inscription.")
        userlist = self._get_user_list()
        resultat = tools.format_form_result(self.form_result['inscription_edit'])
        if 'group_name' not in resultat:
            raise MissingKey("Erreur : il manque des données pour l'inscription.")
        group = resultat['group_name']
        conn = Ldap()
        conn.connect()
        nscd_stop()
        for user in userlist:
            ldapuser = linker._user_factory(user, conn.connexion)
            ldapuser._inscription(user, group, sync=True)
        nscd_start()
        conn.close()
        return dict(message="Les utilisateurs ont bien été inscrits au groupe")

    def _valid_quota(self):
        """ valide l'attribution de quota à une liste d'utilisateur """
        if 'userlist' not in self.form_result:
            raise MissingKey("Erreur : il manque des données pour l'édition de quota.")
        userlist = self._get_user_list()
        resultat = tools.format_form_result(self.form_result['quota_edit'])
        if 'quota' not in resultat:
            raise MissingKey("Erreur : il manque des données pour l'édition de quota.")
        quota = resultat['quota']
        conn = Ldap()
        conn.connect()
        for user in userlist:
            ldapuser = linker._user_factory(user, conn.connexion)
            if ldapuser.has_samba:
                eoleuser.set_quota(user, quota)
        conn.close()
        return dict(message="%s Mo de quota disque ont été attribué aux utilisateurs sélectionnés" % quota)

    def _valid_domain(self):
        """ valide le changement de domaine mail d'une liste d'utilisateur """
        if 'userlist' not in self.form_result:
            raise MissingKey("Erreur : il manque des données pour le changement de domaine mail.")
        userlist = self._get_user_list()
        resultat = tools.format_form_result(self.form_result['domain_edit'])
        if 'domain' not in resultat:
            raise MissingKey("Erreur : il manque des données pour le changement de domaine mail.")
        domain = resultat['domain']
        conn = Ldap()
        conn.connect()
        for user in userlist:
            ldapuser = linker._user_factory(user, conn.connexion)
            ldapuser._set_localmail(user, domaine=domain)
        conn.close()
        return dict(message="Les utilisateurs sélectionnés ont désormais un mail de type %s." % domain)

    def _valid_profil(self):
        """ valide le changement de profil d'une liste d'utilisateurs """
        if 'userlist' not in self.form_result:
            raise MissingKey("Erreur : il manque des données pour le changement de profil.")
        userlist = self._get_user_list()
        resultat = tools.format_form_result(self.form_result['profil_edit'])
        if '_sambaProfilePath' not in resultat:
            raise MissingKey("Erreur : il manque des données pour le changement de profil.")
        profil = int(resultat['_sambaProfilePath'])
        conn = Ldap()
        conn.connect()
        for user in userlist:
            ldapuser = linker._user_factory(user, conn.connexion)
            ldapuser._set_profil(user, profil)
        conn.close()
        return {'message':"Le profil des utilisateurs sélectionnés a bien été modifié."}

    def _valid_shell(self):
        """ valide le changement de shell d'une liste d'utilisateurs """
        if 'userlist' not in self.form_result:
            raise MissingKey("Erreur : il manque des données pour le changement de profil.")
        conn = Ldap()
        conn.connect()
        userlist = self._get_user_list()
        resultat = tools.format_form_result(self.form_result['shell_edit'])
        for user in userlist:
            ldapuser = linker._user_factory(user, conn.connexion)
            ldapuser._update_shell(user, 'shell' in resultat)
        conn.close()
        return {'message':"Le shell des utilisateurs sélectionnés a bien été modifié."}

    def _valid_password(self):
        """
        valide le changement de mot de passe pour une liste d'utilisateur
        """
        if 'userlist' not in self.form_result:
            raise MissingKey("Erreur : il manque des données pour le changement de mot de passe.")
        userlist = self._get_user_list()
        resultat = tools.format_form_result(self.form_result['password_edit'],
                                            False)
        if 'password' not in resultat:
            raise Exception("Erreur: il manque des données pour le changement de mot de passe.")
        force = (resultat.get('force_password') == "force_password")
        # calcul du nom du fichier à écrire (fichier local)
        rootname = self.user_description['name']
        for increment in range(15):
            name = rootname + str(increment) + '.txt'
            filename = join(rapport_dir, name)
            if not isfile(filename):
                break
        if isfile(filename):
            raise Exception("Veuillez purger vos fichiers de mot de passe avant de continuer.")
        conn = Ldap()
        conn.connect()
        message = "Modification des mots de passe:\\n"
        openfile = open(filename, 'w')
        openfile.write("LOGIN;MOT DE PASSE;NOM;PRENOM;CLASSE;\r\n")
        for user in userlist:
            ret_code, ret_msg = grpedit.modify_password(user, openfile,
                                   resultat['password'],
                                   resultat.get('input_password', None),
                                   force,
                                   conn)
            if not ret_code:
                message += ret_msg
            elif len(userlist) < 15:
                ret_msg = ret_msg.replace('"', '\\"')
                ret_msg = ret_msg.replace("'", "\\'")
                message += ret_msg
        #openfile.write("\r\n\r\nPENSEZ À SUPPRIMER CE FICHIER APRÈS DIFFUSION DES MOTS DE PASSE AUX UTILISATEURS.")
        openfile.close()
        conn.close()
        ret_message = grpedit.copy_pwd_file(filename,
                                            self.user_description['name'])
        message += ret_message
        return dict(message = message)

    def purge_pwd_files(self):
        """ purge les fichiers de mot de passe """
        message = ""
        rootname = self.user_description['name']
        name = rootname
        for increment in range(15):
            name = rootname + str(increment) + '.txt'
            filename = join(rapport_dir, name)
            if isfile(filename):
                unlink(filename)
                message += "Suppression de %s\\n" % name
        return dict(message=message)

    def _get_user_list(self):
        """ renvoie la liste des utilisateurs sélectionnés (checkbox)
            pour les actions(teste les droits de l'appelant)
        """
        userlist = list(tools.format_form_result(self.form_result['userlist'], False).keys())
        if userlist == []:
            raise Exception("Aucun utilisateur n'a été sélectionné.")
        # contrôle des droits d'édition, si prof_admin on filtre sinon on est admin
        if scribe_tools.is_prof_admin(self.roles):
            for user in userlist:
                if not scribe_tools.is_eleve_from_prof(user, self.user) \
                and user != self.user:
                    log.err("Erreur : tentative abusive d'édition de %s par %s." % (user, self.user))
                    raise Exception("Erreur : action non autorisée.")
        return userlist

    def _valid_user_listing(self):
        """ renvoie la description de la liste des utilisateurs
            (libelle et checkbox)
            d'après les critéres de recherche
        """
        request = tools.format_form_result(self.form_result['search'])
        compels = ('user_type', 'domains', 'user_class', 'namepart',
                   'alpha_key', 'user_group')
        if set(request.keys()) & set(compels) == set():
            return ""
        args = [request[compel] for compel in compels]
        profadmin = scribe_tools.is_prof_admin(self.roles)
        ldapuser = eoleuser.User()
        ldapuser.ldap_admin.connect()
        userlist = []
        for user in scribe_tools.user_filter(*args):
            # si prof admin, on n'affiche que ses élèves
            if profadmin and not \
            scribe_tools.is_eleve_from_prof(user, self.user):
                continue
            _type = ldapuser._get_type(user)
            if user == 'admin':
                checked=False
            else:
                checked=True
            userlist.append(F.Checkbox(name=user,
                                       libelle="%s (%s)" % (user, _type),
                                       value=user,
                                       checked=checked,
                                       inline=True))
        ldapuser.ldap_admin.close()
        return dict(userlist=userlist,
                    nb_users="Nombre d'utilisateurs : %s" % len(userlist),
                    all_users_btn = grpedit.get_all_users_btn(),
                    none_users_btn = grpedit.get_none_users_btn())

    def _get_action_form(self, _type):
        """ renvoie la description du formulaire d'édition groupée """
        args = (self.server_nb, self.name, self.roles)
        attrs = ('inscription', 'quota', 'domain', 'profil',
                 'password', 'shell')
        if _type in attrs:
            return getattr(grpedit, 'get_%s_form' % _type)(*args)

    def _get_action_btns(self, num=''):
        """ renvoie la description des boutons pour les actions possibles
            :num: numéro du bouton demandé (on les renvoie pas tous à chaque fois)
        """
        # l'attribut name permet de créer la div où s'effectuera l'action
        btn1 = M.Bouton(href=tools.make_js_link(self.server_nb, self.name,
                                            balise='div_inscription_container',
                                            form='inscription'),
                        libelle="Inscrire ces utilisateurs à d'autres groupes",
                        icone="/image/scribe/action_groupe.png",
                        _class='scribe_action_btn')
        btn1['name'] = "div_inscription_container"

        btn2 = M.Bouton(href=tools.make_js_link(self.server_nb, self.name,
                                                balise='div_quota_container',
                                                form='quota'),
                        libelle="Définir des quotas disques pour ces utilisateurs",
                        icone="/image/scribe/action_quota.png",
                        _class='scribe_action_btn')
        btn2['name'] = "div_quota_container"

        btn3 = M.Bouton(href=tools.make_js_link(self.server_nb, self.name,
                                                balise='div_domain_container',
                                                form='domain'),
                        libelle="Changer le domaine mail pour ces utilisateurs",
                        icone="/image/scribe/action_liste.png",
                        _class='scribe_action_btn')
        btn3['name'] = "div_domain_container"

        btn4 = M.Bouton(href=tools.make_js_link(self.server_nb, self.name,
                                                balise='div_profil_container',
                                                form='profil'),
                        libelle="Changer le profil pour ces utilisateurs",
                        icone="/image/scribe/action_profil.png",
                        _class='scribe_action_btn')
        btn4['name'] = "div_profil_container"

        btn5 = M.Bouton(href=tools.make_js_link(self.server_nb, self.name,
                                                balise='div_password_container',
                                                form='password'),
                        libelle="Générer un nouveau mot de passe pour ces utilisateurs.",
                        icone="/image/scribe/action_password.png",
                        _class='scribe_action_btn')
        btn5['name'] = "div_password_container"

        btn6 = M.Bouton(href=tools.make_js_link(self.server_nb, self.name,
                                                balise='div_shell_container',
                                                form='shell'),
                        libelle="Modifier le shell associé à ces utilisateurs",
                        icone="/image/scribe/action_shell.png",
                        _class='scribe_action_btn')

        btn6['name'] = 'div_shell_container'

        btn7 = M.Bouton(href=tools.make_js_link(self.server_nb, 'role_manager',
                                                balise='div_role_container',
                                                form='userlist'),
                        libelle="Associer un rôle à ces utilisateurs",
                        icone="/image/action_role.png",
                        _class='scribe_action_btn')
        btn7['name'] = "div_role_container"

        if scribe_tools.is_admin(self.user_description['role']):
            action_btns = [btn1, btn2, btn3, btn4, btn5, btn6, btn7]
        else:
            action_btns = [btn1, btn2, btn3, btn4, btn5, btn6]
        if num != '' :
            return dict(action_btn=action_btns[num])
        return dict(action_btns=action_btns)

    def _get_valid_btn(self):
        """ renvoie la description du bouton de validation de la recherche
            la cible du retour est la balise : user_div_container
        """
        href = tools.make_form_link(self.server_nb, self.name, True,
                                    ['search'], 'user_td_container')
        title = 'Lister selon les filtres'
        return M.Submit(href=href, libelle='Lister', title=title)

