#!/usr/bin/env python
# -*- coding: UTF-8 -*-

import xmlrpc.client
import getpass
import sys
import socket

from ead2.lib.libfrontend import ServerParser
from ead2.lib.libead import EadKeyParser
from ead2.frontend.console.libconsole import *
from ead2.frontend.config.config import SERVERS, SERVERS_FILE, BACKEND_KEYS_FILE
from ead2.config.config import BACKEND_LISTEN_PORT
from eolesso import ticketcache
from ead2.lib.libsecure import TransportEole
import ead2.lib.i18n_frontend_console

class Ead:
    """fonctions ligne de commande pour l'Ead2
    """
    def __init__(self, auth_server_ip, session_id=''):
        self.language = os.environ['LANGUAGE']
        self.auth_server_address = auth_server_ip
        self.auth_server = xmlrpc.client.ServerProxy(auth_server_ip, transport=TransportEole())
        # chargement des clés de connexion aux backends enregistrés
        self.backend_key_dict = {}
        self.key_parser = EadKeyParser()
        self.key_parser.parse_file(BACKEND_KEYS_FILE)
        self.backend_key_dict = self.key_parser.get_key_dict()
        # id de la session en cours
        self.session_id = session_id
        # serveur actuellement sélectionné
        self.id_server = ''
        self.proxy = None
        self._action = None
        self.actions = []
        self.code = self.msg = ''
        self.APP_PATH = 'toto'
        #ticket d'application
        self.app_ticket = None
        #numero permettant de communiquer avec un serveur pour verifier notre identite
        self.magic_number = ''
        # authentification
        self.login = ''
        #authentification locale?
        self.local = False
    
    def _login(self, url_server):
        """On s'assure que l'utilisateur est authentifié"""
        first_time = True
        ok = False
        server_ip = url_server.split('//')[1]
        server_ip = server_ip.split(':')[0]
        server = xmlrpc.client.ServerProxy(url_server, transport=TransportEole())
        #self.session_id = ''
        while ok == False:
            if not first_time:
                print(_("Authentification error, please retry"))
            first_time = False
            login = input('login :')
            passwd = getpass.getpass('password :')
            self.login = login.strip()
            # authentification
            try:
                self.session_id = self.auth_server.authenticate(login.strip(), passwd.strip())
                self.local = False
                if self.session_id:
                    ok = True
                else:
                    print(_("remote authentification failed"))
                    print(_("trying local authentification"))
                    temp = server.local_authentification(self.backend_key_dict[server_ip], login.strip(), passwd.strip(), True)
                    if temp:
                        print(_("local authentification succeeded"))
                        self.magic_number = temp
                        self.local = True
                        ok = True
                        
            except:
                print(_("connexion error with authentification server"))
                print(_("trying local authentification"))
                self.magic_number = server.local_authentification(self.backend_key_dict[server_ip], login.strip(), passwd.strip(), True)
                if self.magic_number:
                    print(_("local authentification succeeded"))
                    self.session_id=''
                    self.local = True
                    ok = True
                else:
                    self.session_id = ''
                    self.app_ticket = None
                    ok = False

    def verif_parameter_val(self, funcdescr, args):
        """
        vérifie les paramètres déjà saisis
        """
        provided = len(args)-1
        cpt = 1
        for pname, pdescr, ptype, pvals in funcdescr[:provided]:
            if pvals[0] != "":
                while args[cpt] not in pvals[1]:
                    print(_("incorrect value for %s")%pname)
                    print(_("authorised values : %s")%",".join(pvals[1]))
                    args[cpt] = input(_("value")+' ? ')
            cpt += 1

    def ask_for_missing_params(self, funcdescr, args):
        """
        demande les parametres manquants si il y en a
        """
        provided = len(args)-1
        for pname, pdescr, ptype, pvals in funcdescr[provided:]:
            # on vérifie si on a une valeur par défaut
            # pvals : tuple (type, valeurs)
            # si type == "" : on peut avoir une valeur par défaut (valeurs[0])
            if pvals[0] == '' and pvals[1] != []:
                print(_("missing parameter (%s) [%s]") % (pdescr, pvals[1][0]))
                #print _("missing parameter (%s) [%s]" % (pdescr.encode('ISO-8859-1'), pvals[1][0]))
            else:
                print(_("missing parameter (%s)") % pdescr)
                #print _("missing parameter (%s)" % (pdescr.encode('ISO-8859-1')))
            # si type != "" : on récupère une liste de valeurs possibles
            if pvals[0] != '':
                print(_("authorised values : %s") % ",".join(pvals[1]))
            args.append(input("%s ? " % pname.encode('ISO-8859-1')))
            # on prend la valeur par défaut si rien n'est saisi
            if args[-1] == '' and pvals[0] == '' and pvals[1] != []:
               args[-1] = pvals[1][0] 
                
    
    def _execute(self, args):
        """On exécute 
        """
        if self.id_server == '':
            print(_("no active server"))
            return
        args = args.split()
        action_name = args[0]
        # on regarde si la fonction est disponible
        if not action_name in list(self.menu['actions'].keys()):
            print(_("action not available"))
            return
        else:
            act = self.menu['actions'][action_name]
        try:
            params = {}
            self.ask_for_missing_params(act[1], args)
            self.verif_parameter_val(act[1], args)
            for param_info, param_value  in zip(act[1], args[1:]):
                param_name = param_info[0]
                params[param_name] = param_value   
            try:
                self.code, self.msg = self.proxy.execute_action(action_name, params, self.magic_number, self.language)
            except  socket.error:
                print(_("Remote server doesn't respond, impossible to execute the function"))
                return
            self._action = action_name
            if self.code == 0:
                print(self.msg)
            else:
                print(_("Error : ")+self.msg)
                #code retour 3 ou 4 signifie session expiree ou absente, 
                #on enleve donc de l'espace de nommage les fonctions du serveur
                if self.code in [3,4]:
                    self.id_server = ''
                    self._action = None
                    self._delete_commands()
        except Exception as e:
            print(_("error when calling the function : %s")%e)
            self._action = None

    def do_get_status(self,args):
        """prints execution status of last action"""
        if self._action != None:
            if self.code == 0:
                print("OK")
            else:
                print("FAILED")

    def do_get_output(self,args):
        """prints stdout of last action"""
        if self._action != None:
            print(self.msg)

    def do_list_actions(self,args):
        """prints list of available actions"""
        if self.id_server == '':
            print(_("no active server"))
        else:
            print("\n"+_("Available actions on server %s:")%self.id_server+'\n')            
            menu = self.menu['menu']
            self._affiche(menu)
            print("")

    def get_magic_number_from_server(self, server, server_addr):
        """
        obtient un magic number d'un server
        on envoi aussi le session_id pour calculer les droits de l'utilisateur au niveau du backend
        @param server: proxy xmlrpc
        @param server_addr: adresse au format http(s)://xxx.xxx.xxx.xxx
        """
        ip_server = server_addr.split('//')[1]
        return server.get_magic_number(self.backend_key_dict[ip_server], self.app_ticket, self.login, self.session_id)

    def do_server(self,args=''):
        """choix du serveur actif"""
        if args == '':
            self.do_list_servers()
            return
        server = args.split()[0]
        if server in list(SERVERS.keys()):

            if self.session_id == '':
                #connexion au serveur d'authentification, ou authentification locale si besoin
                server_address = "%s:%s"%(SERVERS[server][0],SERVERS[server][1])
                self._login(server_address)
           
            #si on est pas en authentification locale
            if self.local == False:
                #on recupere un ticket d'application sur le serveur d'authentification
                #pour le moment, appurl recoit le login de l'utilisateur
                try:
                    self.app_ticket = self.auth_server.get_app_ticket(self.session_id, self.login)
                except:
                    print(_("the authentification server doesn't respond, please enter the local login"))
                    server_address = "%s:%s"%(SERVERS[server][0],SERVERS[server][1])
                    self._login(server_address)
                else:
                    if self.app_ticket == '':
                        print(_("Error, you are no longer registered on the authentification server"))
                        print(_("Please reconnect"))
                        server_address = "%s:%s"%(SERVERS[server][0],SERVERS[server][1])
                        self._login(server_address)
                        self.app_ticket = self.auth_server.get_app_ticket(self.session_id, self.login)

            proxy_sauv = self.proxy
            magic_sauv = self.magic_number
            self.proxy = xmlrpc.client.ServerProxy('%s:%s' % (SERVERS[server][0],SERVERS[server][1]),transport=TransportEole())
            #on envoie aussi le ticket d'application pour que le serveur verifie qu'on est bien authentifie
            #sur le serveur d'authentification
            try:
                #si on est pas en authentification locale
                if self.local == False:
                    #obtient le magic number du serveur
                    self.magic_number = self.get_magic_number_from_server(self.proxy, SERVERS[server][0])
                #obtient le menu associe au serveur selon les droits attribues a l'utilisateur 
                self.menu = self.proxy.get_menu(self.magic_number, self.language)
            except xmlrpc.client.Fault as fault:
                print(_("xmlrpclib error")+' : %s'%str(fault))
                self.menu = {'actions':{}, 'menu':{}}
                self.proxy = proxy_sauv
                self.magic_number = magic_sauv
            except socket.sslerror as e:
                print(_("ssl error")+' : %s' % str(e.args))
                self.proxy = proxy_sauv
                self.magic_number = magic_sauv
                return
            except Exception as e:
                print(_("Error when connecting to selected server")+" : %s" % e)
                self.proxy = proxy_sauv
                self.magic_number = magic_sauv
                return

            if self.menu in ['None', 'Expired']:
                if self.menu == 'None':
                    print(_("Error, no session registered on this server, please connect first"))
                else:
                    print(_("Error, the session has expired, please reconnect"))
                # supression des anciennes commandes
                self._delete_commands()
                return
 
            self.id_server = server
            
            # supression des anciennes commandes
            self._delete_commands()

            for cmd in list(self.menu['actions'].keys()):
                self.actions.append(cmd)
                # on ajoute la fonction dans l"espace de nommage
                self._add_command(cmd)
            self.do_list_actions("")
        else:
            print(_("unknown server"))
            
    def _add_command(self, commandname):
        """
        ajoute une commande a l'espace de nommage
        """
        def action(self, args):
            return self._execute('%s %s' % (commandname, args))
        setattr(self.__class__, 'do_%s' % commandname, action)

    def _delete_commands(self):
        """
        supprime les anciennes commandes de l'espace de nommage
        """
        # supression des anciennes commandes
        for cmd in self.actions:
            delattr(self.__class__, 'do_%s' % (cmd,))
        self.actions = []


    def do_list_servers(self, args=''):
        """définition des serveurs"""
        tmp = list(SERVERS.keys())
        tmp.sort()
        for key in tmp:
            value = SERVERS[key]
            print('%s : %s  (%s:%s)'%(key,value[2],value[0],value[1]))


    def do_change_password(self, num_server=''):
        """
        change le mot de passe local d'un serveur
        """
        ok = False
        #si on a fournit un numero de serveur et qu'il est bon
        if num_server and num_server in list(SERVERS.keys()):
            ok = True
        # numero pas bon ou pas de numero
        if ok == False:
            num_server = ''
            while num_server not in list(SERVERS.keys()):
                self.do_list_servers()
                num_server = input(_("Server number")+' : ')
                if num_server not in list(SERVERS.keys()):
                    print(_("unknown server"))
        #on a un bon numero de serveur dans num_server
        server = '%s:%s'%(SERVERS[num_server][0],SERVERS[num_server][1])
        print(_("Authentification on server %s (%s)")%(num_server, server))
        login = input('login : ')
        login = login.strip()
        old_passwd =  getpass.getpass(_("old password")+' :')
        old_passwd = old_passwd.strip()
        ok = False
        while ok == False:
            new_passwd = getpass.getpass(_("new password")+' :')
            new_passwd = new_passwd.strip()
            new_passwd_verif = getpass.getpass(_("please retype new password")+' :')
            new_passwd_verif = new_passwd_verif.strip()
            if new_passwd == new_passwd_verif:
                ok = True
            else:
                print(_("You didn't enter the same password twice, please retry"))

        proxy = xmlrpc.client.ServerProxy(server, transport=TransportEole()) 
        try:
            result, msg = proxy.change_local_passwd(login, old_passwd, new_passwd)
        except Exception as e:
            print(_("Unreachable server, giving up (%s)")%server)
            return
        #authentification reussie
        if result == 0:
            print(_("Password successfully changed"))
        #authentification ratée    
        else:
            print(msg)

        
    def do_add_server(self, args):
        """
        ajoute un serveur a la liste des serveurs
        """
        ip = input(_("Server IP (without https): "))
        if not ip.startswith('https'):
            url = 'https://'+ip
        for i in list(SERVERS.values()):
            if url == i[0]:
                print(_("There is already another server with the same IP address"))
                return
        port = input(_("Server port (leave it blank by default) [%s] : ")%BACKEND_LISTEN_PORT)
        if port == '':
            port = BACKEND_LISTEN_PORT
        comment = input(_("Comments about the server : "))
        if comment == '':
            comment = _("No informations")

        print(_("Asking for remote server agreement, please enter this server's local login/password"))
        login = input('login : ')
        login = login.strip()
        passwd =  getpass.getpass('password :')
        passwd = passwd.strip()
        
        server = '%s:%s'%(url,port)
        proxy = xmlrpc.client.ServerProxy(server, transport=TransportEole())
        # on procède à l'enregistrement sur le backend
        try:
            code, result = proxy.register_frontend(login, passwd, self.language)
            if code == 1:
                print(_("Error when registering : %s")%result)
                return
        except Exception as e:
            print(_("Unreachable server, giving up (%s)")%server)
            return
        # authentification reussie
        if result:
            # on sauvegarde la clé
            self.backend_key_dict[ip] = result
            # sauvegarde du fichier des clés
            if not self.key_parser.write_file(BACKEND_KEYS_FILE,self.backend_key_dict):
                print(_("error when saving the key"))
            else:
                # sauvegarde des informations sur le serveur
                cle = str(len(list(SERVERS.keys()))+1)
                SERVERS[cle] = (url, port, comment)
                parse = ServerParser(SERVERS_FILE)
                if parse.add_server(SERVERS) == True:
                    print(_("Server added successfully (%s)")%server)
                else:
                    print(_("Error when adding the server (%s)")%server)
        else:
            print(_("Authentification failed, server not added"))
        

    def _affiche(self, dico, level=0, padding='    '):
        """pour afficher le menu mis en forme
        """
        for category, (commands, subcategories) in list(dico.items()):
            print(color(padding*level + '-' + category + ' :', 'lred'))
            for funcname in commands:
                funcdoc = self.menu['actions'][funcname][0]
                print('%s%s : %s' % (padding*(level+1), color(funcname, 'lblue'), funcdoc))
            self._affiche(subcategories, level+1)
