# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2010
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action ServiceRestart et ServiceStop
#
# Redémarre et stoppe des services
#
###########################################################################
"""
Actions spécifiques au services qui tournent sur le serveur
"""
from ead2.backend.config.config import EAD_SERVICES, NON_STOP_SERVICES
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions.lib import services_tools as ss_tools
from ead2.backend.actions.tools import get_request, make_confirm_link, \
make_js_link


def get_retour_btn():
    """
    représentation de l'affichage du bouton "retour"
    """
    return {'href': "javascript:Refresh()",
            'libelle': "Revenir à la page d'accueil",
            'icone': "/image/back.png"}

class Daemon(Action):
    """
    Arrete ou redemarre un service
    """
    user_description = Dict(default={}, doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'daemon'
    libelle = "Services (mode expert)"
    description = "Stopper ou redémarrer un service"
    category = "Système"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'service_restart',
                         'service_stop', 'all_restart'])

    def execute(self):
        """
        exécution de l'action (arrêt ou redémarrage)
        """
        params, self.server_nb = get_request(self.request)
        message = ''
        result = {'titre':self.description}
        if 'service_restart' in params:
            srv = params['service_restart'][0].split('#')
            message = ss_tools.restart(*srv)
        elif 'service_stop' in params:
            srv = params['service_stop'][0].split('#')
            message = ss_tools.stop(*srv)
        elif 'all_restart' in params:
            message = ss_tools.restart_all()
        result['message'] = message
        result['all_restart'] = self._get_all_restart_btn()
        result['services'] = self._get_services()
        result['retour_btn']  = get_retour_btn()
        return self.send_all(result, template='services', templates=['main'])

    def _get_all_restart_btn(self):
        """
        renvoie la description du bouton de redémarrage de tous les services
        """
        jsl = make_js_link(self.server_nb, self.name, confirm=True,
                           all_restart='true')
        question = "Etes-vous sûr de vouloir redémarrer tous les services ?"
        rest_link = {'href':make_confirm_link(question, jsl),
                                    'libelle':'Redémarrer',
                                    'icone': '/image/redemarrer_serv.gif'}
        return {"restart":rest_link,
                'libelle':"Redémarrer tous les services (hors ead et sso)"}

    def _get_services(self):
        """
        renvoie la liste des services et la description des boutons pour les redémarrer
        """
        result = []
        for serv, container in ss_tools.SERVICES:
            # services critiques pour l'Ead2 ?
            libelle = "%s (%s)" % (serv, container)
            service = "%s#%s" % (serv, container)
            if serv in EAD_SERVICES:
                warn = "\\nVous risquez d'être déconnecté."
            else:
                warn = ""
            jsl = make_js_link(self.server_nb, self.name, confirm=True, service_restart=service)
            question = """Etes-vous sûr de vouloir redémarrer le service %s ?%s""" % (serv, warn)
            rest_link = {'href':make_confirm_link(question, jsl),
                    'libelle':'Redémarrer', 'icone': '/image/redemarrer_serv.gif'}
            jsl = make_js_link(self.server_nb, self.name, confirm=True, service_stop=service)
            if serv in EAD_SERVICES or serv in NON_STOP_SERVICES:
                # pas de clé 'stop'
                result.append({'restart':rest_link, 'libelle':libelle})
            else:
                question = """Etes-vous sûr de vouloir stopper le service %s ?%s""" % (serv, warn)
                stop_link = {'href':make_confirm_link(question, jsl),
                    'libelle':'Stopper', 'icone': '/image/arreter_serv.gif'}
                result.append({'restart':rest_link, 'stop':stop_link, 'libelle':libelle})
        return result

