# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action scribe_user*
# Creation d'utilisateur
# Recherche puis modification d'utilisateur
# modfication de mot de passe
# Supression d'utilisateur
#
###########################################################################
"""Actions permettant d'afficher les tableaux d'utilisateurs"""

from scribe.eoleuser import User
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions.lib.widgets import main as M, form as F, ajax
from ead2.backend.actions import tools
from ead2.backend.actions.scribe.tool import scribe_tools
from ead2.backend.actions.scribe.tool.user import used_templates

class UserTable(Action):
    """ action gérant les tableaux d'utilisateurs listés
    :alpha: alphabétique
    :domain: domaine mail
    :classe: classe de l'utilisateur
    :_type: eleve/prof
    :namepart: portion du nom des utilisateurs recherches
    :size: nombre d'utilisateur par page
    """
    user_description = Dict(default={}, doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_user_table'
    libelle = "Tableau d'utilisateur"
    category = None
    description = 'Tableau des utilisateurs'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'page', 'alpha_key', '_type',
                         'domain', '_class', 'size', 'namepart', 'user_group',
                         'sort_method'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['search'])

    def execute(self):
        """ renvoit les données pour l'affichage
            1 - renvoie la liste des utilisateurs selon les critères demandés
            2 - renvoie la page suivante si le listing est en cours
        """
        params, self.server_nb = tools.get_request(self.request)
        self.user_alpha = 'a'
        page = 1 # permet de savoir quelle page de listing
                 # afficher en cas de nombreux utilisateurs

        ## 2 -
        if 'page' in params: # dans le cas de page multiple
            page = int(params['page'][0])
            params.pop('page')
        if 'alpha_key' in params:
            alpha_key, _type, domain = params['alpha_key'][0], \
                                       params['_type'][0], params['domain'][0]
            _class, namepart = params['_class'][0], params['namepart'][0]
            if 'sort_method' in params:
                sort_method = params['sort_method'][0]
            else:
                sort_method = ''

            if 'user_group' in params:
                group = params['user_group'][0]
            else:
                group = ''

            size = int(params['size'][0]) #nombre d'item par page

            filtered_datas = self.filter_users(alpha_key, _type, domain,
                                _class, namepart, group, size, page, sort_method)

            return self.frag(filtered_datas)

        ## 1 -
        if self.form_result != {}:# validation de formulaire
            filtered_datas = self._valid_form()
            self.form_result = {}
            return self.frag(filtered_datas)
        return self.send_all(dict(),
                                  template='scribe_users',
                                  templates=used_templates)

    def frag(self, filtered_datas):
        """ renvoit les datas pour l'affichage de la liste
        des utilisateurs correpondant à la recherche"""
        return self.send_frag(filtered_datas,
                              template='scribe_users',
                              templates=used_templates)

    def _valid_form(self):
        """ valide les retours de formulaire """
        request = tools.format_form_result(self.form_result['search'])
        keys = list(request.keys())
        if 'alpha_key' not in keys or 'domains' not in keys or \
                'user_class' not in keys or 'user_type' not in keys or \
                'namepart' not in keys or 'size' not in keys:
            return ""
        alpha_key, _type, domain, _class, namepart = request['alpha_key'], \
                                  request['user_type'], request['domains'], \
                                  request['user_class'], request['namepart']

        if 'user_group' in request:
            group = request['user_group']
        else:
            group = ''
        size = int(request['size'])
        if 'sort_method' in request:
            sort_method = request['sort_method']
        else:
            sort_method = ''
        return self.filter_users(alpha_key, _type, domain, _class, namepart,
                                 group, size, sort_method=sort_method)

    def filter_users(self, alpha_key="", _type="", domain="", _class="",
                     namepart='', group='', size=40, page=1, sort_method=''):
        """ filtre les utilisateurs selon les parametres
            on cree le filtre ldap et on lance une recherche
            :alpha_key: première lettre du nom de l'utilisateur
            :_type: type d'utilisateur prof/eleve (pupil/teacher)
            :domain: domaine mail
            :class: classe de l'utilisateur (pour les élèves a priori,
                          marche aussi pour les profs admin de classe)
            :size: nombre d'utilisateur par page
        """
        namepart.strip()
        if scribe_tools.is_prof_admin(self.user_description['role']):
            if _class == "":
                return dict(message="Il manque des paramètres.")
            elif _class not in scribe_tools.get_prof_class_list(self.user_description['name']):
                return dict(message="Action non autorisée.")

        users = scribe_tools.user_filter(_type, domain, _class, namepart,
                                         alpha_key, group, sort_method)
        with open('/tmp/user_selection', 'w') as user_selection_fh:
            user_selection_fh.write('\n'.join(users))
        export_btn = self.get_export_btn()
        length = len(users)
        if length > size: # on affiche sur plusieurs pages
            max_index = size
            if page is not None:
                max_index = size * page
            min_index = max_index - size
            user_btns = self.get_btns(users, alpha_key, _type, domain, _class,
                                    namepart, group, size, page, sort_method)
            users = self._get_table_users(users[min_index:max_index])
            return dict(users=users, user_btns=user_btns, export_btn=export_btn,
                    nb_users="Nombre d'utilisateurs : %s" % length)
        users = self._get_table_users(users)
        return dict(users=users, export_btn=export_btn, nb_users="Nombre d'utilisateurs : %s"%length)

    def get_export_btn(self):
        return M.Bouton(href=tools.make_js_link(self.server_nb,
                                                'scribe_export_users_list',
                                                balise="mod_user_div_container",
                                                code=True,
                                                ),
                                                libelle="Exporter la liste des identifiants temporaires",
                                                _class="scribe_action_btn",
                                                icone='')
    def _get_table_users(self, users):
        """ renvoie la description du tableau de listing utilisateur
        """
        result = []
        ldapuser = User()
        ldapuser.ldap_admin.connect()
        for user in users:
            html_container = 'mod_user_div_container'
            edit_pwd = M.Bouton(href=tools.make_js_link(self.server_nb,
                                                        'scribe_user_modpassword',
                                                        balise=html_container,
                                                        user=user),
                                libelle="Changer le mot de passe",
                                _class='scribe_simple_link',
                                icone='')
            modify = M.Bouton(href=tools.make_js_link(self.server_nb,
                                                      'scribe_user_modify',
                                                      balise=html_container,
                                                      user=user),
                              libelle="Editer",
                              _class='scribe_simple_link',
                              icone="")
            href = ajax.call(self.server_nb,
                             'scribe_user_delete',
                             'mod_user_div_container',
                             user=user)
            delete = M.Bouton(href=href,
                              libelle="Supprimer",
                              icone="",
                              _class='scribe_simple_link')
            _type = ldapuser._get_type(user)
            longuser = "%s (%s)" % (user, _type)
            result.append({'name':longuser, 'pwd_link':edit_pwd,
                           'mod_link':modify, 'del_link':delete})
        ldapuser.ldap_admin.close()
        return result

    def get_btns(self, users, alpha_key, _type, domain, _class, namepart,
                 group, size, page, sort_method):
        """ renvoie la description des boutons de passage de page en page """
        html_container = "user_div_container"
        next = M.Bouton(href=tools.make_js_link(self.server_nb,
                                                self.name,
                                                balise=html_container,
                                                page=page+1,
                                                alpha_key=alpha_key,
                                                _type=_type,
                                                domain=domain,
                                                _class=_class,
                                                namepart=namepart,
                                                user_group=group,
                                                size=size,
                                                sort_method=sort_method),
                        libelle="suivant",
                        icone='')

        prec = M.Bouton(href=tools.make_js_link(self.server_nb, self.name,
                                                balise=html_container,
                                                page=page-1,
                                                alpha_key=alpha_key,
                                                _type=_type,
                                                domain=domain,
                                                _class=_class,
                                                namepart=namepart,
                                                user_group=group,
                                                size=size,
                                                sort_method=sort_method),
                        libelle="précédent",
                        icone='')
        if page == 1 :
            return [next]
        elif ((page) * size)>=len(users):
            return [prec]
        else:
            return [prec, next]
