#!/usr/bin/python3
#-*-coding:utf-8-*-
"""
    Script pour le lancement de l'importation en thread (par le biais de at)
    depuis l'Ead (les logs sont en HTML)
"""
DOC = """
Usage : sconet.py [OPTION] [ARGS]

-p --parse <data_type> : Parse des fichiers d'importation data_type=sconet/aaf/be1d/csv/csv2

-i --import              : Lance un import

-t, --type               : Type d'importation choisit ( 'maj' ou 'full' )
                           en complément de --import (Not implemented yet)

-u , --user              : utilisateur qui lance l'importation (default=admin)

Ex :
1) importation.py --parse sconet /tmp/extraction-00124-sconet.register
- Lie les fichiers d'importation sconet eleve depuis la conf dans /tmp/extraction-00124-sconet.register

2) importation.py --import -t maj enseignant
- Lance l'importation
"""
import getopt, sys
import json
import csv
from os.path import isfile

from scribe.eoleldap import Ldap
from scribe.ldapconf import SUPPORT_ETAB
from scribe.eoletools import nscd_start, nscd_stop
from scribe.storage import init_store
from scribe.parsing import sconet, aaf, be1d, scribecsv, scribecsv2
from scribe.importation import writer, config, log as importlog
from scribe.importation.config import CSV_PREFIX, WRITE_OPTS
if SUPPORT_ETAB:
    from scribe.importation import preferences

sys.path.append('/usr/share')
from ead2.backend.actions.lib.importation.config import (PARSE_LOG_FILE,
                                                        IMPORT_LOG_FILE)
from ead2.backend.actions.lib.thread.logger import Logger

def usage():
    """
        print la doc
    """
    print(DOC)

def main():
    """
        lance la commande
    """
    try:
        opts, args = getopt.getopt(sys.argv[1:],
                                   "t:p:i:u",
                                   ["type=","parse=","import","user="])
    except getopt.GetoptError as err:
        print(err)
        usage()
        sys.exit(2)
    action = None
    data_type = None
    import_type = None
    user = config.DEFAULT_USER
    for option, value in opts:
        if option in ('-h', '--help'):
            usage()
            sys.exit(0)
        elif option in ('-t', '--type'):
            import_type = value
        elif option in ('-p', '--parse'):
            action = 'parse'
            data_type = value
            break
        elif option in ('-i', '--import'):
            action = 'import'
        elif option in ('-u', '--user'):
            user = value
        else:
            usage()
            print("Option inconnue")
            sys.exit(2)

    if action not in ('parse', 'import'):
        print("Action inconnue")
        usage()
        sys.exit(2)

    if len(args) < 1:
        print("Il manque des arguments")
        usage()
        sys.exit(2)

    if action == 'parse':
        parser = Parser()
        conf_filename = args[0]
        if not isfile(conf_filename):
            msg = "Le chemin vers le fichier de configuration est inexistant"
            parser.log_err(msg)
        else:
            parser.conf_filename = conf_filename
        if not data_type:
            msg = "Il manque le type de données d'import sconet/aaf/be1d/csv/csv2"
            parser.log_err(msg)
        else:
            parser.data_type = data_type
        parser.do_parse()

    elif action == 'import':
        importer = Importer()
        importer.category_type = args[0]
        if not import_type:
            importer.log_err("Il manque le type d'import maj/annuel")
        else:
            importer.import_type = import_type
        importer.do_import(user=user)

class Parser(Logger):
    """
        Outil de parsing de fichier d'import
    """
    def __init__(self):
        super(Parser, self).__init__(PARSE_LOG_FILE, lockmanager=importlog,
                                                 root_logger=importlog.log)
        self.store = None
        self.data_type = None
        self.store = init_store(config.DB_DIR)
        self.end_message = "FIN DE LA LECTURE DE FICHIER"

    def do_parse(self):
        """
            Lance le parsing
            Écrit l'avancement dans un fichier temporaire
        """
        try:
            conf = json.loads(open(self.conf_filename).read())
        except:
            self.log_err("Le fichier de configuration n'est pas dans un format \
json valide")
        if not isinstance(conf, dict):
            self.log_err("Le fichier de configuration n'est pas dans un format \
valide ({'type':'chemin vers le fichier})")

        self.log("<h3># Lancement de la lecture des fichiers #</h3>")
        self.log("<hr>")

        if self.data_type == 'sconet':
            self.do_parse_sconet(conf)
        elif self.data_type == 'aaf':
            self.do_parse_aaf(conf)
        elif self.data_type == 'be1d':
            self.do_parse_be1d(conf)
        elif self.data_type == 'csv':
            self.do_parse_csv(conf)
        elif self.data_type == 'csv2':
            self.do_parse_csv2(conf)
        else:
            self.log_err("Format de fichier d'import inconnu")
        self.log_end(unlock=False)
        sys.exit(0)

    def do_parse_sconet(self, conf):
        """
            Parse des fichiers sconet
        """
        eleve = conf.get('eleve', '')
        enseignant = conf.get('enseignant', '')
        # Import de fichier élèves et responsables
        if eleve:
            responsable = conf.get('responsable', '')
            nomenclature = conf.get('nomenclature', '')
            structure = conf.get('structure', '')
            if not nomenclature or not structure:# or not responsable:
                self.log_err("Il manque des configurations pour les chemins \
vers fichiers d'import")
            self.log("Lecture des classes et des niveaux...<br />")
            try:
                sconet.parse_sco_divisions(self.store,
                                       structure,
                                       nomenclature)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
            self.log("Lecture des groupes (options)...<br />")
            try:
                sconet.parse_sco_groupes(self.store,
                                     structure)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
            self.log("Lecture des élèves...<br />")
            try:
                sconet.parse_sco_eleves(self.store,
                                    eleve)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
            if responsable:
                self.log("Lecture des responsables...<br />")
                try:
                    sconet.parse_sco_responsables(self.store,
                                            responsable)
                    self.log_ok()
                except Exception as error:
                    self.log_err(error=error)
        elif enseignant:
            try:
                self.log("Lecture des personnels...<br />")
                sconet.parse_sts_profs(self.store,
                                       enseignant)
            except Exception as error:
                self.log_err(error=error)

    def do_parse_aaf(self, conf):
        """
            Parsing aaf
        """
        eleve = conf.get('eleve', '')
        enseignant = conf.get('enseignant', '')
        if eleve:
            responsable = conf.get('responsable', '')
            if responsable:
                try:
                    self.log("Lecture des responsables...")
                    aaf.parse_aaf_responsables(self.store, responsable)
                    self.log_ok()
                except Exception as error:
                    self.log_err(error=error)

            try:
                self.log("Lecture des élèves...")
                aaf.parse_aaf_eleves(self.store, eleve)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        elif enseignant:
            try:
                self.log("Lecture des personnels...")
                aaf.parse_aaf_profs(self.store, enseignant)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        else:
            self.log_err("Erreur à la lecture de la configuration")

    def do_parse_be1d(self, conf):
        """
            Parsing be1d
        """
        eleve = conf.get('eleve', '')
        if eleve:
            try:
                self.log("Lecture des élèves...")
                be1d.parse_be1d_eleves(self.store, eleve)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
            responsable = conf.get('responsable', '')
            if responsable:
                try:
                    self.log("Lecture des responsables...")
                    be1d.parse_be1d_responsables(self.store, responsable)
                    self.log_ok()
                except Exception as error:
                    self.log_err(error=error)
        else:
            self.log_err("Erreur à la lecture de la configuration")

    def do_parse_csv(self, conf):
        """
            Parsing csv
        """
        eleve = conf.get('eleve', '')
        enseignant = conf.get('enseignant', '')
        if eleve:
            try:
                self.log("Lecture des élèves...")
                scribecsv.parse_csv_eleves(self.store, eleve)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        elif enseignant:
            try:
                self.log("Lecture des enseignants...")
                scribecsv.parse_csv_profs(self.store, enseignant)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        else:
            self.log_err("Erreur à la lecture de la configuration")

    def do_parse_csv2(self, conf):
        """
            Parsing csv version 2
        """
        eleve = conf.get('eleve', '')
        enseignant = conf.get('enseignant', '')
        administratif = conf.get('administratif', '')
        invite = conf.get('autre', '')
        if eleve:
            try:
                self.log("Lecture des élèves...")
                scribecsv2.parse_csv_eleves(self.store, eleve)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        elif enseignant:
            try:
                self.log("Lecture des enseignants...")
                scribecsv2.parse_csv_profs(self.store, enseignant)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        elif administratif:
            try:
                self.log("Lecture des personnels administratifs...")
                scribecsv2.parse_csv_administratifs(self.store, administratif)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        elif invite:
            try:
                self.log("Lecture des comptes invités...")
                scribecsv2.parse_csv_invites(self.store, invite)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        else:
            self.log_err("Erreur à la lecture de la configuration")

class Importer(Logger):
    """
        Outil d'importation de compte
    """
    def __init__(self):
        super(Importer, self).__init__(IMPORT_LOG_FILE, lockmanager=importlog,
                                                    root_logger=importlog.log)
        self.category_type = None
        self.import_type = None
        self.store = init_store(config.DB_DIR, delete=False)
        self.end_msg = "FIN DE L'IMPORTATION DE COMPTES"

    def do_import(self, user=config.DEFAULT_USER):
        """
            Importe des données dans le ldap du scribe
        """
        self.user = user
        try:
            importlog.add_lock()
            if self.category_type == 'eleve':
                self.do_import_eleve()
            elif self.category_type == 'eleve_ss_resp':
                self.do_import_eleve(resp=False)
            elif self.category_type == 'autre':
                self.do_import_invite()
            elif self.category_type == 'administratif':
                self.do_import_personnel(enseignants=False)
            else:
                self.do_import_personnel()
        except Exception as error:
            self.log_err(unlock=False, error=error)

    def do_import_eleve(self, resp=True):
        """
            écriture des élèves
        """
        self.log("Arrêt de LSC...<br />")
        importlog.debuglog("Arrêt de LSC...", title=True)
        nscd_stop()
        self.log_ok()
        connexion = Ldap()
        connexion.connect()
        if SUPPORT_ETAB:
            prefs = preferences.get_eleves_prefs()
            etab = prefs.get_default('etab')
            etab_prefix = prefs.get_default('etab_prefix')
            self.do_add_prefix_to_csv(etab, etab_prefix)
        else:
            etab = None
            etab_prefix = ''
        if self.import_type != 'maj' :
            self.log("Nettoyage des options")
            try:
                writer.purge_eleves_options(connexion=connexion, etab=etab)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        try:
            self.log("Intégration des niveaux...")
            if SUPPORT_ETAB:
                prefs = preferences.get_eleves_prefs()
                etab = prefs.get_default('etab')
                etab_prefix = prefs.get_default('etab_prefix')
                self.do_add_prefix_to_csv(etab,etab_prefix)
            else:
                etab = None
                etab_prefix = ''
            writer.write_niveau(store=self.store, connexion=connexion, etab=etab,
                    etab_prefix=etab_prefix)
            self.log_ok()
        except Exception as error:
            self.log_err(error=error)
        try:
            self.log("Intégration des classes...")
            writer.write_classe(store=self.store, connexion=connexion,
                    etab_prefix=etab_prefix)
            self.log_ok()
        except Exception as error:
            self.log_err(error=error)
        if WRITE_OPTS == 'oui':
            try:
                self.log("Intégration des options...")
                writer.write_option(store=self.store, connexion=connexion, etab=etab,
                        etab_prefix=etab_prefix)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        try:
            self.log("Intégration des élèves...")
            writer.write_eleve(store=self.store, connexion=connexion, etab=etab, current_ead_user=self.user)
            self.log_ok()
        except Exception as error:
            self.log_err(error=error)
        if resp:
            try:
                self.log("Intégration des responsables...")
                writer.write_responsable(store=self.store,
                                         connexion=connexion, current_ead_user=self.user)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        else:
            self.log("NB : Importation sans responsable<br>")
        self.log("Synchronisation des partages Samba...")
        writer.write_samba(connexion)
        self.log_ok()
        connexion.close()
        try:
            self.log("Démarrage de LSC...")
            importlog.debuglog("Démarrage de LSC...", title=True)
            nscd_start()
            self.log_ok()
        except Exception as error:
            self.log_err(error=error)
        self.log_end()
        importlog.end_importation('EAD', self.user)

    def do_import_personnel(self, enseignants=True):
        """
            écriture des enseignants et/ou personnels administratifs
        """
        self.log("Arrêt de LSC...<br />")
        importlog.debuglog("Arrêt de LSC...", title=True)
        nscd_stop()
        self.log_ok()
        connexion = Ldap()
        connexion.connect()
        if SUPPORT_ETAB:
            prefs = preferences.get_enseignants_prefs()
            etab = prefs.get_default('etab')
            etab_prefix = prefs.get_default('etab_prefix')
            self.do_add_prefix_to_csv(etab,etab_prefix)
        else:
            etab = None
            etab_prefix = ''
        if enseignants:
            if self.import_type != 'maj':
                self.log("Nettoyage des équipes pédagogiques<br />")
                try:
                    writer.purge_equipes(connexion=connexion, etab=etab)
                    self.log_ok()
                except Exception as error:
                    self.log_err(error=error)
            try:
                self.log("Vérification des classes...<br />")
                writer.verify_classe(store=self.store, connexion=connexion,
                                     etab_prefix=etab_prefix)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
            try:
                self.log("Intégration des matières...<br />")
                writer.write_matiere(store=self.store, connexion=connexion,
                                     etab=etab, etab_prefix=etab_prefix)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
            try:
                self.log("\nIntégration des options...<br />")
                writer.verify_option(store=self.store, connexion=connexion,
                                     etab_prefix=etab_prefix)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
            try:
                self.log("Intégration des enseignants...<br />")
                writer.write_enseignant(store=self.store, connexion=connexion,
                                        etab=etab, current_ead_user=self.user)
                self.log_ok()
            except Exception as error:
                self.log_err(error=error)
        try:
            self.log("Intégration des services administratifs...<br />")
            writer.write_service(store=self.store, connexion=connexion,
                                 etab=etab, etab_prefix=etab_prefix)
            self.log_ok()
        except Exception as error:
            self.log_err(error=error)
        try:
            self.log("Intégration des personnels administratifs...<br />")
            writer.write_administratif(store=self.store, connexion=connexion,
                                       etab=etab, current_ead_user=self.user)
            self.log_ok()
        except Exception as error:
            self.log_err(error=error)
        self.log("Synchronisation des partages Samba...")
        writer.write_samba(connexion)
        self.log_ok()
        # fin du mode connecté
        connexion.close()
        self.log("Démarrage de LSC...<br />")
        importlog.debuglog("Démarrage de LSC...", title=True)
        nscd_start()
        self.log_ok()
        self.log_end()
        importlog.end_importation('EAD', self.user)

    def do_import_invite(self):
        """
            écriture des comptes invités
        """
        # nb: arrêt de LSC inutile pour ces comptes
        connexion = Ldap()
        connexion.connect()
        try:
            self.log("Intégration des comptes invités...<br />")
            writer.write_invite(store=self.store, connexion=connexion, current_ead_user=self.user)
            self.log_ok()
        except Exception as error:
            self.log_err(error=error)
        # fin du mode connecté
        connexion.close()
        self.log_end()
        importlog.end_importation('EAD', self.user)

    def do_add_prefix_to_csv(self, etab, prefix):
        lines = []
        found = False
        if not isfile(CSV_PREFIX):
            open(CSV_PREFIX, 'w').close()
        with open(CSV_PREFIX, 'r') as prefixcsv:
            read_prefix = csv.reader(prefixcsv, delimiter=';')
            try:
                lines = list(read_prefix)
                for line in lines:
                    if etab == line[0]:
                        line[1] = prefix
                        found = True
                if not found:
                    lines.append([etab, prefix])
            except Exception as e:
                print(e)
        with open(CSV_PREFIX, 'w') as prefixcsv:
            write_prefix = csv.writer(prefixcsv, delimiter=';')
            for line in lines:
                write_prefix.writerow(line)


if __name__ == '__main__':
    main()
