#-*-coding:utf-8-*-
"""
Formeol: générateur de formulaire ClassicForm (Nevow standard), Form(Livepage)
"""
from twisted.python import log
from nevow import loaders, athena
from nevow.flat import flatten
from nevow import tags as T, rend
#from encoding import translate

class Form(athena.LiveFragment):
    """gestion de formulaire
       :requirement: type et contrôle de type associé
       :jsClass: module javascript associé au fragment
    """
    jsClass = "Formeol.Forms"
    docFactory = loaders.xmlstr('''\
            <div xmlns:nevow="http://nevow.com/ns/nevow/0.1"
              xmlns:athena="http://divmod.org/ns/athena/0.7"
             nevow:render="liveFragment">
             <div id="formeol_message"></div>
              <span nevow:render = 'form'/>
             </div>
             ''')

    # fonctions d'instantiation du formulaire
    def __init__(self, name, action="#", _class='formeol_form'):
        """initialisation
           :name: propriété name du formulaire
           :action: action exécuté à la validation du formulaire
           :_class: classe Css associée au formulaire
           :num_col: nombre de colonnes du formulaire
           """
        athena.LiveFragment.__init__(self)
        self.name = name
        self.form = T.form(name=name, _id=name, action=action, _class=_class)
        self.fields = []
        self.layout = TableLayout(2)
        self.field_layout = TableLayout(1)
        self.title = ''

    def add_title(self, title):
        """ ajoute un titre au formulaire """
        self.title = title

    def add_field(self, name, description, required=False, default=None,
                  js_validator=None):
        """ajoute un champ au formulaire
        :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        :required: champ requis?
        :js_validator: type de données (cf le dico de ref des types: self.requirement)
        """
        field = InputField(name, description, default, required, js_validator)
        field.layout = self.field_layout
        self.fields.append(field)

    def add_passwd_field(self, name, description, required=False, default=None,
                  js_validator=None):
        """ajoute un champ password au formulaire
        :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        :required: champ requis?
        :js_validator: type de données (cf le dico de ref des types: self.requirement)
        """
        field = InputPasswdField(name, description, default, required, js_validator)
        field.layout = self.field_layout
        self.fields.append(field)

    def add_hidden_field(self, name, description, default=None):
        """ajoute un champ invisible au formulaire
        :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        """
        field = HiddenInputField(name, description, default)
        self.fields.append(field)

    def add_select(self, name, description, required=False, default=None,  choices=[]):
        """ajoute un champ select au formulaire
       :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        :required: champ requis?
        :choices: liste des valeurs possibles ( couple (value, label))
        """
        field = SelectField(name, description,  default, choices, required)
        field.layout = self.field_layout
        self.fields.append(field)

    def add_hidden_select(self, name, description, required=False, default=None,  choices=[]):
        """ajoute un champ select au formulaire
       :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        :required: champ requis?
        :choices: liste des valeurs possibles ( couple (value, label))
        """
        field = HiddenSelectField(name, description, default, choices)
        self.fields.append(field)

    def add_button(self, libelle, action, description=''):
        """ ajoute un bouton à notre formulaire
        :name: nom du bouton
        :action: action javascript effectuée par le bouton
        """
        button = T.button(_type="button", onclick=action, title=description)[libelle]
        self.fields.append(button)

    def add_submit(self, libelle='Valider'):
        """ajoute un bouton submit """
        button = T.button(_type='submit')[libelle]
        self.fields.append(button)

    def set_field_layout(self, layout):
        """ définit le layout pour les champs """
        self.field_layout = layout

    def set_layout(self, layout):
        """ définit le layout pour le rendu """
        self.layout = layout

    # fonctions de rendu
    def render_form(self, ctx, data):
        """ rend le formulaire """
        if self.title != '':
            self.form.children.append(T.h2[self.title])
        self.form.children.append(self.layout.place(self.fields))
        return self.form

    def flat(self):
        return flatten(self.render())

    def __repr__(self):
        """ représentation de l'objet"""
        return "<%s %s>" % (self.__class__.__name__, self.name)

# fonctions de communication avec le client
    def send_error(self, message, variable):
        """renvoit un message d'erreur au client pour la variable 'variable' """
        tag_name = "error_%s"%variable
        tag_name = tag_name
        message = message
        if str(message) == "test_message":
            message = 'ce message est passe par le backend'
        self.callRemote("error", message, tag_name)
    athena.expose(send_error)

    def send_content(self, value, variable):
        """ change le contenu de variable """
        tag_name = 'input_%s'%variable
        tag_name = tag_name
        value = value
        self.callRemote('set_content', value, tag_name)

class ClassicForm(rend.Fragment):
    """gestion de formulaire
       :requirement: type et contrôle de type associé
    """
    docFactory = loaders.xmlstr('''\
            <div xmlns:nevow="http://nevow.com/ns/nevow/0.1">
             <div id="formeol_message"></div>
              <span nevow:render = 'form'/>
             </div>
             ''')

    # fonctions d'instantiation du formulaire
    def __init__(self, name, action="#", _class='formeol_form', method='POST'):
        """initialisation
           :name: propriété name du formulaire
           :action: action exécuté à la validation du formulaire
           :_class: classe Css associée au formulaire
           """
        self.name = name
        self.form = T.form(name=name, _id=name, action=action, _class=_class, method=method)
        self.fields = []
        self.layout = TableLayout(2)
        self.field_layout = TableLayout(1)
        self.title = ""

    def add_title(self, title):
        """ ajoute un titre au formulaire """
        self.title = title

    def add_field(self, name, description, required=False, default=None,
                  js_validator=None):
        """ajoute un champ au formulaire
        :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        :required: champ requis?
        :js_validator: type de données (cf le dico de ref des types: self.requirement)
        """
        field = InputField(name, description, default, required, js_validator)
        field.layout = self.field_layout
        self.fields.append(field)

    def add_checkbox(self, name, description, default=None, checked=None):
        """ajoute une checkbox  au formulaire
           :name: nom du champ (véhiculé à la validation)
           :description: libellé du champ
           :default: selectionné par défaut
        """
        field = CheckInputField(name, description, default=default)
        field.layout = self.field_layout
        self.fields.append(field)

    def add_passwd_field(self, name, description, required=False, default=None,
                  js_validator=None):
        """ajoute un champ password au formulaire
        :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        :required: champ requis?
        :js_validator: type de données (cf le dico de ref des types: self.requirement)
        """
        field = InputPasswdField(name, description, default, required, js_validator)
        field.layout = self.field_layout
        self.fields.append(field)

    def add_hidden_field(self, name, description, default=None):
        """ajoute un champ invisible au formulaire
        :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        """
        field = HiddenInputField(name, description, default)
        self.fields.append(field)

    def add_select(self, name, description, required=False, default=None,  choices=[]):
        """ajoute un champ select au formulaire
       :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        :required: champ requis?
        :choices: liste des valeurs possibles ( couple (value, label))
        """
        field = SelectField(name, description,  default, choices, required)
        field.layout = self.field_layout
        self.fields.append(field)

    def add_hidden_select(self, name, description, required=False, default=None,  choices=[]):
        """ajoute un champ select au formulaire
       :name: nom du champ (véhiculé à la validation)
        :description: libellé du champ
        :required: champ requis?
        :choices: liste des valeurs possibles ( couple (value, label))
        """
        field = HiddenSelectField(name, description, default, choices)
        self.fields.append(field)

    def add_button(self, libelle, action, description=''):
        """ ajoute un bouton à notre formulaire
        :name: nom du bouton
        :action: action javascript effectuée par le bouton
        """
        button = T.button(_type="button", onclick=action, title=description)[libelle]
        self.fields.append(button)


    def add_submit(self, libelle='Valider'):
        """ajoute un bouton submit """
        button = T.button(_type='submit')[libelle]
        self.fields.append(button)

    def set_field_layout(self, layout):
        """ définit le layout pour les champs """
        self.field_layout = layout

    def set_layout(self, layout):
        """ définit le layout pour le rendu """
        self.layout = layout

    # fonctions de rendu
    def render_form(self, ctx, data):
        """ rend le formulaire """
        if self.title != '':
            self.form.children.append(T.h2[self.title])
        self.form.children.append(self.layout.place(self.fields))
        return self.form

    def flat(self):
        return flatten(self.render())

    def __repr__(self):
        """ représentation de l'objet"""
        return "<%s %s>" % (self.__class__.__name__, self.name)

class Layout:
    """ objet de disposition """

    def place(self, objects):
        """
        :objects: liste d'objets à disposer
        """
        raise NotImplementedError('')

class TableLayout(Layout):
    """ disposition en tableau """

    def __init__(self, nb_col):
        self.nb_col = nb_col

    def place(self, objects):
        """ place et rend les objets
           :objects: liste de tags nevow
        """
        table = T.table()
        for i, obj in enumerate(objects):
            if i % self.nb_col == 0:
                row = T.tr()
                table.children.append(row)
            row.children.append(T.td[obj])
        if len(objects) % self.nb_col != 0:
            for i in range(len(objects) % self.nb_col, self.nb_col):
                row.children.append(T.td())
        return table

class LineLayout(Layout):
    """ disposition en ligne """

    def place(self, objects):
        """ place les objets en ligne """
        table = Table()
        row = T.tr()
        for obj in objects:
            row.children.append(obj)
        table.children.append(row)
        return table

class BaseField(rend.Fragment):
    """ classe de base pour les formulaires """

  ###  docFactory = loaders.xmlstr('''
  ###    <table class="" xmlns:nevow="http://nevow.com/ns/nevow/0.1">
  ###    <tr><td><nevow:attr name="class" nevow:render="required"></nevow:attr><span nevow:render="description"/></td>
  ###    <td class="formeol_field_description"><span nevow:render="field"/></td></tr>
  ###    <tr><td><span nevow:render="error_field"/></td></tr>
  ###    </table>
  ###    ''')
    docFactory = loaders.xmlstr('''
                    <div xmlns:nevow="http://nevow.com/ns/nevow/0.1">
                    <nevow:attr name="class" nevow:render="class"/>
                    <span nevow:render="basefield"/>
                    </div>
            ''')

    # dico recensant les types controlables en javascript
    requirement = {'IP':["testIp","Rentrez une IP xxx.xxx.xxx.xxx"],
                   'NUMBER':["testNumber", "Rentrez un Nombre"],
                   'NETMASK':["testNetMask", "Rentrez un NetMask xxx.xxx.xxx.xxx"]}

    def __init__(self, name, description, default=None, required=False,
                 js_validator=None):
        super(BaseField, self).__init__()
        self.name = name
        self.description = description
        self.required = required
        self.default = default
        self.js_validator = js_validator
        self.layout = TableLayout(1)

    def set_layout(self, layout):
        """ définit la trame pour le rendu
        :layout: objet layout"""
        self.layout = layout

    def render_basefield(self, ctx, data):
        """ rend l'ensemble du field """
        description = T.div( render=T.directive('description'))
        field = T.div(render=T.directive('field'))
        error = T.div(render=T.directive('error_field'))
        elements = [description, field, error]
        return self.layout.place(elements)

    def render_description(self, ctx, data):
        """ rend la description du champ """
        return T.div(_class="formeol_field_description")[self.description]

    def render_field(self, ctx, data):
        """ rend le champ de formulaire """
        raise NotImplementedError("concrete field class must implement the render_field method")

    def render_error_field(self, ctx, data):
        """ rend un champ d'erreur """
        return T.div(_id="error_%s"%self.name, _class="formeol_error")

    def render_class(self, ctx, data):
        """ rend l'attribut classe """
        if self.required:
            return "formeol_field_required"
        return "formeol_field"

class CheckInputField(BaseField):
    """input checkbox"""
    def render_field(self, ctx, data):
        field = T.input(_type='checkbox', name=self.name, _id="input_%s" % self.name)
        if self.default is not None:
            field.attributes['checked'] =  True
        return field

class HiddenInputField(BaseField):
    """ balise input caché"""
    def render_description(self, ctx, data):
        return ''
    def render_error_field(self, ctx, data):
        return ''
    def render_field(self, ctx, data):
        field = T.input(_type='hidden',name=self.name, _id="input_%s" % self.name)
        if self.default is not None:
            field.attributes['value'] = self.default
        return field
    def render_class(self, ctx, data):
        """ rend l'attribut classe """
        return "formeol_hidden_field"



class InputField(BaseField):
    """input classique"""

    def render_field(self, ctx, data):
        """ rend le champ et ajoute des appels javascript au changement de focus si besoin"""

        field= T.input(_type='text',name=self.name, _id="input_%s" % self.name,
                        _class='formeol_field_input')
        onBlur=''
        if self.js_validator is not None:
            try:
                onBlur = "%s('input_%s', 'error_%s');"%(self.requirement[self.js_validator][0], self.name, self.name)
                field.attributes['title'] = self.requirement[self.js_validator][1]
            except:
                field.attributes['title'] = "le parametre requis est un %s"%self.js_validator
                onBlur += "noError('error_%s');"%self.name
            field.attributes['onBlur'] = onBlur
        else:
            onBlur += "noError('error_%s');"%self.name
        if self.required:
            onBlur += "testVoid('input_%s', 'error_%s');"%(self.name, self.name)
            field.attributes['onBlur'] = onBlur
        if self.default is not None:
            field.attributes['value'] = self.default
        return field

class InputPasswdField(BaseField):
    """input password"""

    def render_field(self, ctx, data):
        """ rend le champ et ajoute des appels javascript au changement de focus si besoin"""

        field= T.input(_type='password',name=self.name, _id="input_%s" % self.name,
                        _class='formeol_field_input')
        onBlur=''
        if self.js_validator is not None:
            try:
                onBlur = "%s('input_%s', 'error_%s');"%(self.requirement[self.js_validator][0], self.name, self.name)
                field.attributes['title'] = self.requirement[self.js_validator][1]
            except:
                field.attributes['title'] = "le parametre requis est un %s"%self.js_validator
                onBlur += "noError('error_%s');"%self.name
            field.attributes['onBlur'] = onBlur
        else:
            onBlur += "noError('error_%s');"%self.name
        if self.required:
            onBlur += "testVoid('input_%s', 'error_%s');"%(self.name, self.name)
            field.attributes['onBlur'] = onBlur
        if self.default is not None:
            field.attributes['value'] = self.default
        return field

class HiddenSelectField(BaseField):
    """ balise select caché """

    def __init__(self, name, description, default=None, choices = (),
                 required=False):
        """
        :choices: liste de couples (valeur, label)
        """
        super(HiddenSelectField, self).__init__(name, description, default, required)

    def render_description(self, ctx, data):
        return ''
    def render_error_field(self, ctx, data):
        return ''

    def render_field(self, ctx, data):
        """ rend un select caché, en fait c'est un input
        mais vu qu'il est caché, ça change pas grand chose"""
        select = T.input(name=self.name, _id="select_%s" % self.name,
                           _type="hidden", value=self.default)
        return select

class SelectField(BaseField):
    """ select classique """

    def __init__(self, name, description, default=None, choices=(),
                 required=False):
        """
        :choices: liste de couples (valeur, label)
        """
        super(SelectField, self).__init__(name, description, default, required)
        self.choices = choices

    def render_field(self, ctx, data):
        select = T.select(name=self.name, _id="select_%s" % self.name,
                          _class='formeol_select_list')
        for value, label in self.choices:
            if value == self.default:
                opt = T.option(value=value, selected=True)[label]
            else:
                opt = T.option(value=value)[label]
            select.children.append(opt)
        return select
