#!/usr/bin/env python
# -*- coding: UTF-8 -*-
"""
    Classe du serveur xmlrpc du backend ead
"""
# imports python standard
import xmlrpc.client, socket, requests
from os import seteuid, setegid, getuid, getgid
from os.path import join, isfile
from ssl import SSLEOFError
from glob import glob

# imports twisted
from twisted.web import server
from twisted.internet import reactor, defer
from twisted.web.xmlrpc import XMLRPC
from twisted.python import log

# imports ead2
from ead2.lib.libbackend import PamAuth
from ead2.lib.libsecure import ServerContextFactory
from ead2.backend.lib import error
from ead2.backend.lib.action import get_action, get_action_classes
from ead2.backend.lib.actionlist import load_actions
from ead2.backend.lib.perm.parser import parse_and_update, \
        parse_and_update_local, parse_and_update_roles, RoleManager
from ead2.backend.lib.perm.registry import PermissionManager

#fichier de config specifique au backend
from ead2.backend.config.config import (NOBODY_UID, NOBODY_GID,
     FRONTEND_KEYS_FILE, CONFIG_DIR, TEMPLATE_DIR, EOLE_MODULE, debug)
#fichier de config de l'EAD, contenant les infos importantes
#sur les tickets de session du sso
from ead2.config.config import AUTH_SERVER, BACKEND_LISTEN_PORT, LEMON_SSO
# imports unicode ead
from ead2.lib.i18n_backend import _
from ead2.lib.libead import uni

# imports REFACTORING
from ead2.backend.lib.magicnumber import Magic
from ead2.backend.lib.frontendkeys import FrontendKeys
from ead2.backend.lib import message
from urllib.parse import quote
from lxml import etree

#try:
#    from creole.fonctionseole import get_module_name
#    module_name = get_module_name()
#except:
#    module_name = "unknown"

Fault = xmlrpc.client.Fault

def _get_template_from_disk(template_name):
    """
        :template_name: le nom (basename sans extenstion) du fichier
                                                        de template
        :return: string de template ou error si template non trouve
    """
    templatefile = join(TEMPLATE_DIR, template_name + '.tmpl')
    if isfile(templatefile):
        return open(templatefile, 'r').read()
    raise AssertionError

def _drop_privileges():
    """
        drop de privileges vers nobody/nobody passage en root
    """
    setegid(NOBODY_GID)
    seteuid(NOBODY_UID)

def _gain_privileges(uid, gid):
    """
        on donne les privileges uid, gid
    """
    seteuid(uid)
    setegid(gid)

def get_username(infos):
    """
        Renvoie le username depuis les informations utilisateurs
    """
    username = infos.get('uid', infos.get('pam'))
    if hasattr(username, '__iter__') \
        and not isinstance(username, str) \
        and not isinstance(username, str):
        username = username[0]
    if not username:
        username = "Inconnu"
    return username


class Backend(XMLRPC):
    """Serveur XMLRPC (backend Eole)
    """
    def __init__(self):
        #sauvegarde de notre UIG/GID courant
        self.uid = getuid()
        self.gid = getgid()
        # dictionnaire des clés de connexion autorisées
        # gestionnaire de magic_number
        self.magic_number = Magic()
        self.app_ticket = {}
        # gestionnaire de clé de frontend
        self.frontend = FrontendKeys(join(CONFIG_DIR, FRONTEND_KEYS_FILE))
        # chargement des actions
        load_actions()
        # initialisation du serveur d'authentification
        if AUTH_SERVER:
            self.auth_server = xmlrpc.client.ServerProxy(AUTH_SERVER)
        else:
            self.auth_server = None
        #initialisation des role et perm manager
        self._get_perms()
        self._get_roles()

        # utilisateurs systèmes
        self.allowed_users = self.role_manager.get_users()
        # drop de privileges vers nobody/nobody
        # on est plus root (FIXME : pb avec twisted 8)
        _drop_privileges()
        # initialisation de l'authentification locale (PAM)
        self.local_auth = PamAuth()
        XMLRPC.__init__(self)

    def _get_perms(self):
        """
            Initialise le gestionnaire de permissions
        """
        if debug:
            log.msg("*** Loading permissions ***")
        action_list = get_action_classes()
        #Initialisation du manager de rôle
        self.perm_manager = perm_manager = PermissionManager(action_list)

        # permissions communes
        perm_files = [join(CONFIG_DIR, 'perm.ini')]
        # permissions supplémentaires
        perm_files.extend(glob('%s/perms/perm_*.ini' % CONFIG_DIR))
        for perm_file in perm_files:
            if debug:
                log.msg("  + loading perms from %s" % perm_file)
            parse_and_update(perm_manager, perm_file)

        # permissions personnalisées
        # FIXME : encore utile sur 2.3 ?
        for niveau_gestion in ['acad', 'local']:
            perm_file = join(CONFIG_DIR, 'perm_%s.ini' % niveau_gestion)
            actions_perimes_file = join(CONFIG_DIR, 'unavailable_actions.txt')
            if isfile(perm_file):
                if debug:
                    log.msg("  + loading perms from %s" % perm_file)
                parse_and_update_local(perm_manager,
                                       perm_file,
                                       actions_perimes_file)

    def _get_roles(self):
        """
            Initialise le gestionnaire de rôles
        """
        if debug:
            log.msg("*** loading roles ***")
        #Initialisation du manager de rôle
        self.role_manager = role_manager = RoleManager()

        # permissions et rôles de base
        role_files = [join(CONFIG_DIR, 'roles.ini')]
        # permission et rôles supplementaires
        role_files.extend(glob('%s/roles/roles_*.ini' % CONFIG_DIR))
        # lecture et intégration des rôles
        for role_file in role_files:
            if debug:
                log.msg("  + loading roles from %s" % role_file)
            parse_and_update_roles(role_manager, role_file)

        # permissions et rôles personnalisés
        # FIXME : encore utile sur 2.3 ?
        for niveau_gestion in ['acad', 'local']:
            role_file = join(CONFIG_DIR, 'roles_%s.ini' % niveau_gestion)
            if isfile(role_file):
                if debug:
                    log.msg("  + loading roles from %s" % role_file)
                parse_and_update_roles(role_manager, role_file)

    def render(self, request):
        """
            rend les données
        """
        try:
            request.content.seek(0, 0)
            args, function_path = xmlrpc.client.loads(request.content.read())
            client_ip = request.getClientIP()
            try:
                # _getFunction a été remplacé par lookupProcedure (#2942)
                function = self.lookupProcedure(function_path)
            except Fault as f:
                self._cbRender(f, request)
            else:
                request.setHeader("content-type", "text/xml")
                defer.maybeDeferred(function, client_ip.encode(), *args).addErrback(
                    self._ebRender
                ).addCallback(
                    self._cbRender, request
                )
        except:
            log.err("----> Erreur dans la fonction render, la requête ne semble \
pas être de type xmlrpc")
            request.setHeader("content-length", str(len('Erreur')))
            request.write(b'Erreur')
            request.finish()
        return server.NOT_DONE_YET

    def _get_user_description(self, ip, avatar, roles):
        """ renvoie un dico de description de l'utilisateur
            renvoie le type de l'utilisateur, son nom, l'ip de son frontend

        """
        username = avatar.get_login()
        if hasattr(roles, '__iter__'):
            roles = list(roles)
        else:
            roles = ['eleve']
        return {'ip':ip, 'name':username, 'role':roles}

    def _validate_session(self, magic_number):
        """
            Vérifie la validité de la session
            par le biais de son magic number
        """
        #si magic_number non enregistre en local
        if self.magic_number.check(magic_number):
            return True, "OK"
        #si magic_number est une session expiree
        elif self.magic_number.check_expired(magic_number):
            language = self.magic_number.lang(magic_number)
            return False, uni(_(message.EXPIRED_SESSION, language))
        else:
            return False, uni(_(message.UNREGISTERED_SESSION))

    def _format_action(self, dico, language):
        """
        pour formater le menu des actions
        """
        result = {}
        for value in list(dico.values()):
            for script in value:
                #XXX gerer l'i18n ici (description et description de param)
                nom, description, param, libelle = script
                # on internationalise la description des parametres
                param_list = []
                #on internationalise la description de l'action
                desc = _(description, language)
                libe = _(libelle, language)
                if not isinstance(desc, str):
                    desc = uni(desc)
                if not isinstance(libe, str):
                    libe = uni(libe)
                for par in param:
                    nom_param, description_param, type_param , default_param, param_data = par
                    param_list.append((nom_param,
                                       uni(_(description_param, language)),
                                       type_param, default_param,
                                       param_data))
                result[str(nom)] = (desc, param_list, libe)
        return result

    def _recurs_menu(self, edict):
        """
        permet de construire la structure de donnes (range par catgories)
        @param edict: dictionnaire d'entree
                      (avec des clefs du types 'syst/general')
        @return: la structure de donne arborescence (cf doc/menu.html)
        """
        dico = {}
        tmp = {}
        for cle, value in list(edict.items()):
            # parse la clef (il peut s'agir d'un chemin)
            chemin = cle.split('/')

            if chemin[0] not in dico:
                # si la catgorie n'existe pas dj
                dico[chemin[0]] = [[], {}]

            if len(chemin) == 1:
                # il n'y a pas de sous-catgorie
                for val in value:
                    # on construit la liste des noms d'actions pour cette catgorie
                    dico[chemin[0]][0].append(val[0])
            else:
                # il y a des sous-catgories
                if chemin[0] not in tmp:
                    # c'est une nouvelle sous-catgorie
                    tmp[chemin[0]] = {}
                reste = '/'.join(chemin[1:])

                if reste not in tmp[chemin[0]]:
                    # c'est une nouvelle sous-catgorie
                    tmp[chemin[0]][reste] = []
                for val in value:
                    # construit la liste des actions pour cette sous-catgorie
                    tmp[chemin[0]][reste].append(val)
        for cle, value in list(tmp.items()):
            dico[cle][1] = self._recurs_menu(value)
        return dico


    def _deferred_error(self, err):
        """
            renvoie si une erreur est survenue à l'éxécution de l'action
        """
        _drop_privileges()
        log.err("Erreur d'exécution :")
        error = err.getBriefTraceback()
        log.err(error)
        brieferror = error.split('\n')[0]
        return 1, uni(brieferror)

    def _deferred_drop_and_return(self, ret):
        """
            Renvoie le résultat de l'action demandée
        """
        _drop_privileges()
        return ret

    def _get_action_class(self, action_name):
        """
            Renvoie la classe de l'action à éxécuter (avec un système de cache)
        """
        try:
            # on récupère la classe de l'action (dans REGISTERED_ACTIONS)
            action_class = get_action(action_name)
            action = action_class()
        except:
            log.err()
            raise Exception(message.UNKNOWN_ACTION % (action_name,))
        return action, action_class

# XMLRPC

    def xmlrpc_local_authentification(self, client_ip, key, username, password, keep, language):
        """
        fournit un magic number a un front end si username/password correspond
        a un couple name/password local
        cette fonction est appelée dans le cas où
        le serveur d'authentification est injoignable
        """
        if debug:
            log.msg("#   Authentification locale    #")
        if username not in self.allowed_users:
            if debug:
                log.msg(message.NOROLE_FOR_USER % username)
            return ''
        if not self.frontend.auth(client_ip.decode(), key):
            if debug:
                log.msg(message.FRONTEND_KEY_ERROR % client_ip.decode())
            return ''
        _gain_privileges(self.uid, self.gid)
        ok = self.local_auth.authenticate(username, password)
        _drop_privileges()
        if not ok:
            if debug:
                log.msg(message.LOCAL_AUTH_NOTOK % username)
            return ''
        #ok, authentification locale reussie
        magic_number = self.magic_number.create(password,
                                                username)
        if debug:
            log.msg(message.LOCAL_AUTH_OK % username)
        #si on garde le magic number
        if keep:
            self.magic_number.store(magic_number,
                                    ({'pam':username}, username),
                                    language, None)
        return magic_number

    def xmlrpc_invalidate_magic_number(self, client_ip, key, app_ticket):
        """
            invalide un magic number lié à un app_ticket depuis un frontend.
            permet un logout sans accès à la session utilisateur
            utilisé en cas de logout centralisé initié par le serveur SSO
        """
        key_ok = False
        # vérification de la clé du frontend
        if not self.frontend.auth(client_ip, key):
            if debug:
                log.msg(message.FRONTEND_KEY_ERROR % client_ip)
            return False
        try:
            if self.auth_server is None and ticket:
                if debug:
                    log.msg(" !!! Le serveur EAD n'est pas configuré en mode SSO !!!")
            # invalidation de la session correspondant à ce ticket
            return self.magic_number.remove_ticket(app_ticket)
        except Exception as e:
            import traceback
            traceback.print_exc()
            log.err("!!! Erreur lors de l'invalidation du ticket %s : %s !!!" % (str(app_ticket), str(e)))
            return False

    def xmlrpc_get_magic_number(self, client_ip, key, app_ticket, app_path, language):
        """
            fournit un magic number a un front end
            si son ticket d'application est valide
        """
        #on est paranoiaque, on verifie que le ticket d'application que l'on a envoye est
        #bien celui que l'on nous retourne. C'est une petite verification de securite supplementaire
        key_ok = False
        # vérification de la clé du frontend
        client_ip = client_ip.decode()
        if not self.frontend.auth(client_ip, key):
            if debug:
                log.msg(message.FRONTEND_KEY_ERROR % client_ip)
            return ''
        try:
            if debug:
                log.msg("#   Authentification auprès du serveur SSO   #")
            if self.auth_server is None:
                if debug:
                    log.msg(" !!! Le serveur EAD n'est pas configuré en mode SSO !!!")
                result = {}
            else:
                if LEMON_SSO == "oui" : 
                    req = requests.get("{}/serviceValidate?ticket={}&service={}".format(AUTH_SERVER, app_ticket, quote(str(app_path), safe="")), verify=False)
                    if req.status_code == 200:
                        result = True
                        infos = {}
                        resp = etree.fromstring(req.text)
                        for status in resp:
                            for user in status : 
                                if "attrib" in user.tag :
                                    for attributes in user :
                                        attributes.tag =  etree.QName(attributes).localname
                                        infos[attributes.tag] = [attributes.text]
                    else:
                        if debug:
                            log.msg(" !!! Le serveur SSO renvoi une page d'erreur !!!")
                        result = False
                else:
                    nb = 0
                    while True:
                        try:
                            result, infos = self.auth_server.get_user_details(app_ticket,
                                                                            app_path)
                            break
                        except SSLEOFError as err:
                            log.msg(f"Une erreur SSL est survenue ({err}), relancer get_user_details")
                            if nb == 1:
                                raise err from err
                            nb += 1
        except:
            log.err()
            return ''
        #si ticket d'application valide
        if result:
            if debug:
                log.msg(message.SSO_AUTH_OK % (get_username(infos),))
            magic_number = self.magic_number.create(app_ticket, app_path)
            #met en place le timeout d'expiration du magic number
            self.magic_number.store(magic_number, (infos, app_path), language, app_ticket)
            #on met en place une association magic_number/app_ticket pour pouvoir retrouver facilement
            #le magic_number correspondant en cas de demande de déconnexion par le serveur SSO
            return magic_number
        else:
            return ''

    def xmlrpc_execute_action(self, client_ip, action_name, params, magic_number):
        """
            Exécute l'action
        """
        validated, ret_message = self._validate_session(magic_number)
        if not validated:
            return validated, ret_message
        language = self.magic_number.lang(magic_number)
        try:
            action, action_class = self._get_action_class(action_name)
        except Exception as err:
            log.err()
            return 1, uni(_(str(err), language))

        avatar = self.magic_number.get_user(magic_number)
        #log de l'execution
        self._log_action_execution(action_name, magic_number, client_ip)
        # calcul des roles
        roles = self.role_manager.get_roles(avatar.get_datas())
        if action_class not in self.perm_manager.authorized_actions_for(roles):
            log.err(message.UNAUTHORIZED_ACTION_FOR_USER % (roles,
                                                            action_name))
            return 1, uni(_(message.UNAUTHORIZED_ACTION_FOR_USER % (
                                                 action_name, roles),
                                                            language))
        else:
            #on reprend nos privileges
            _gain_privileges(self.uid, self.gid)
            #on execute la fontion
            #return_code, result = action.execute(params)
        ## passage d'infos utilisateurs à l'action
        ## (permet la gestion de droits à l'intérieur de l'action)
            client_url = params.get('url', client_ip)
            user = {'user_description':self._get_user_description(client_url,
                                                                  avatar,
                                                                  roles)}
            params.update(user)
            ## lancement de l'éxécution de l'action dans un callback
            ## pour éviter la perte de droit pour les actions différées
            _deferred = defer.Deferred()
            _deferred.callback(params)
            _deferred.addCallback(action.execute)
            _deferred.addCallback(self._deferred_drop_and_return)
            _deferred.addErrback(self._deferred_error)
            return _deferred

    def xmlrpc_get_menu(self, client_ip, magic_number):
        """Renvoie la liste des actions que <username> peut effectuer"""
        validated, ret_message = self._validate_session(magic_number)
        if not validated:
            return validated, ret_message
        language = self.magic_number.lang(magic_number)
        #log de l'execution
        self._log_action_execution("get_menu", magic_number, client_ip)
        avatar = self.magic_number.get_user(magic_number)
        #authentification distante, on a des infos
        roles = self.role_manager.get_roles(avatar.get_datas())
        try:
            actions_menu = {}
            actions = {}
            # dictionnaire catégorie/actions
            for act in self.perm_manager.authorized_actions_for(roles):
                act_infos = (act.name, act.description,
                             act.get_params(), act.libelle)
          #     les actions sans categorie ne doivent pas apparaitre dans le menu
                if act.category is not None:
                    actions_menu.setdefault(act.category, []).append(act_infos)
                actions.setdefault(act.category, []).append(act_infos)
            if actions == {}:
                log.err("===> Erreur ---> get_menu le menu est vide")
        except Exception as e:
            log.err("erreur :", e)
            actions = {}
        menu = {'menu':self._recurs_menu(actions_menu),
                'actions': self._format_action(actions, language),
                'role':self._get_user_description(client_ip, avatar, roles),
               }
        return 0, menu

    def xmlrpc_is_alive(self, client_ip):
        """
            pour dire a un client que l'on est vivant
        """
        return ''

    def xmlrpc_get_module_name(self, client_ip, magic_number):
        """
            renvoie le nom du module du backend
        """
        validated, ret_message = self._validate_session(magic_number)
        if not validated:
            return validated, ret_message
        return 1, EOLE_MODULE

    def xmlrpc_get_template(self, client_ip, template_name, magic_number):
        """
            regarde sur le disque si le template existe
            et retourne une string
        """
        validated, ret_message = self._validate_session(magic_number)
        if not validated:
            return validated, ret_message
        try:
            tplstr = _get_template_from_disk(template_name)
            return 0, uni(str(tplstr))
        except AssertionError as e:
            return False, uni(_(message.NO_TEMPLATE % template_name))

    def xmlrpc_register_frontend(self, client_ip, username, password, language):
        """
        crée une clé pour permettre à une adresse ip d'accéder au backend
        le parametre keep indique si on doit garder la clé
        on peut alors se servir de cette fonction pour verifier un couple login/mdp
        sans garder de nouvelle cle
        """
        if debug:
            log.msg("#     Enregistrement d'un frontend    #")
            log.msg("  ++ client_ip ---> %s, username ---> %s" % (client_ip,
                                                                username))
        _gain_privileges(self.uid, self.gid)
        auth = self.local_auth.authenticate(str(username), str(password))
        if auth:
            #ok, authentification locale reussie
            try:
                key = self.frontend.register(client_ip.decode())
            except:
                log.err()
                _drop_privileges()
                return 1, uni(_("error when saving the key", language))
            if debug:
                log.msg(message.FRONTEND_REGISTERED % client_ip)
            _drop_privileges()
            return 0, key
        else:
            if debug:
                log.msg(message.LOCAL_AUTH_NOTOK % username)
            return 1, uni(_("authentification error", language))

    def xmlrpc_unregister_frontend(self, client_ip, key, username, password):
        """
        supprime une clé permettant a un frontend de se connecter au backend
        on doit fournir key qui est la clé que le backend
                                       avait précédement fournit au frontend
        on verifiera qu'on a bien client_ip :
                                         key dans le dico avant de supprimer
        """
        _gain_privileges(self.uid, self.gid)
        client_ip=client_ip.decode()
        local_auth = self.local_auth.authenticate(username, password)
        if local_auth:
            try:
                self.frontend.unregister(client_ip, key)
            except error.UnknownFrontend as err:
                log.err()
                _drop_privileges()
                return str(err)
            _drop_privileges()
            return 0
        else:
            if debug:
                log.msg(message.LOCAL_AUTH_NOTOK % (username,))
            return uni(_("Authentification failure with user : '%s'" % (
                                                             username,)))

    def _log_action_execution(self, action_name, magic_nb, ip):
        """
            loggue l'éxécution d'une action
        """
        try:
            username = self.magic_number.get_username(magic_nb)
        except:
            username = _("Unknown")
        if debug:
            log.msg(_("Execution of : <%s> by <%s> from frontend <%s> (%s)") % (
                                                        action_name,
                                                        username,
                                                        socket.getfqdn(ip),
                                                        ip))

def run():
    #on est pas root
    if getuid() != 0:
        import sys
        sys.exit(_("You have to launch it as root"))
    reactor.listenSSL(BACKEND_LISTEN_PORT,
                      server.Site(Backend()),
                      ServerContextFactory())
    if debug:
        log.msg("\n"+_("launching server on port %s") % str(
                                    BACKEND_LISTEN_PORT))
    reactor.run()

if __name__ == '__main__':
    run()
