from flask import Blueprint, request, current_app, session
from xmlrpclib import ServerProxy, Binary
from flask.views import MethodView
import json
from os.path import join, isdir, isfile
from os import makedirs, listdir, remove
from eoleauthlib.authclient import login_required
from eop.utils import secure_arg

filesapi = Blueprint('filesapi', __name__, url_prefix = '')

class FilesAPIError(Exception):
    pass

class FilesAPI(MethodView):
    decorators = [ login_required]

    def __init__(self):
        self.username = session["username"]
        self.proxy = ServerProxy(current_app.config['CONTROLEVNC_URL'])
        self.secret = current_app.config['SECRET_KEY']

        self.directories = {"devoir":join(current_app.config['UPLOAD_FOLDER'], current_app.config["CHEMIN_DEV"]),
                            "data":join(current_app.config['UPLOAD_FOLDER'], current_app.config["CHEMIN_DATA"])}

    def _processing_folder(self, secret, username, devname, typedoc):
        """Process the typedoc folder and copy from temp folder to /home/workgroups/devoirs/<username>/<devname>
           Parameters needed :
            in this object : self.directories[typedoc]
        """
        username = secure_arg(username)
        devname = secure_arg(devname)

        destdir = join(self.directories[typedoc], username)
        if not isdir(destdir):
            return "No folder"

        filelist = listdir(destdir)

        if filelist == []:
            if typedoc == "devoir":
                raise FilesAPIError('Error : there must be at least one file in devoir folder')
            else:
                return 'Nothing to copy'

        for afile in filelist:
            filepath = join(destdir, afile)
            if isfile(filepath):
                with open(filepath, "rb") as handle:
                    filecontent = Binary(handle.read())
                self.proxy.remote_copie_fichier(secret, username, devname,
                        typedoc, afile, filecontent)
        return 'Done'

    def get(self, typedoc):
        """Get the list of temp folder typedoc of username
           Parameters needed :
            in this object : self.username, self.directories[typedoc]
        """
        if self.username is None:
            return json.dumps('Error: You need a username to talk to backend')
        if typedoc is None:
            return json.dumps('Error: You need a typedoc to talk to backend')

        username = secure_arg(self.username)
        destdir = join(self.directories[typedoc],username)
        if not isdir(destdir):
            return json.dumps({"listedoc":None})
        else:
            return json.dumps({"listedoc":sorted(listdir(destdir))})

    def post(self):
        """Upload a new file to the typedoc temp folder for username
           Parameters needed :
            in this object : self.username, self.directories[typedoc]
            in the post request json :
                {"typedoc":"devoir"} or {"typedoc":"data"}
            in the post request files : the file content to upload
        """
        response = request.form
        username = secure_arg(self.username)
        typedoc = response["typedoc"]
        destdir = join(self.directories[typedoc],username)
        if not isdir(destdir):
            makedirs(destdir)

        answer = {}
        for fich in request.files.values():
            if fich:
                filename = secure_arg(fich.filename)
                ret = fich.save(join(destdir, filename))
                if ret is None:
                    ret = "OK"
                answer[filename]=ret
        return json.dumps(answer)

    def delete(self, typedoc, filename=None):
        """Remove filename from the typedoc temp folder if filename is specified else empty the typedoc temp folder
           Parameters needed :
            in this object : self.username, self.directories[typedoc]
        """
        if self.username is None:
            return json.dumps('Error: You need a username to talk to backend')
        if typedoc is None:
            return json.dumps('Error: You need a typedoc to talk to backend')

        username = secure_arg(self.username)
        destdir = join(self.directories[typedoc],username)
        if not isdir(destdir):
            return json.dumps('Error: Directory not found')

        if filename is not None:
            # Delete only this file
            filepath = join(destdir, filename)
            if isfile(filepath):
                ret = remove(filepath)
                if ret is None:
                    ret = "OK : File deleted"
                else:
                    ret = "KO : File not deleted"
            return json.dumps(ret)
        else:
            # Empty the entire folder
            for afile in listdir(destdir):
                filepath = join(destdir, afile)
                if isfile(filepath):
                    remove(filepath)

            if listdir(destdir):
                ret = "KO : Directory not empty"
            else:
                ret = "OK : Directory empty"
            return json.dumps(ret)

    def put(self, devname):
        """Send all devname files of username to the backend
           => Copy from temp folder to /home/workgroups/devoirs/<username>/<devname>
           Parameters needed :
            in this object : self.secret, self.username
        """
        if self.secret is None:
            return json.dumps('Error: You need a secret key to talk to backend')
        if self.username is None:
            return json.dumps('Error: You need a username to talk to backend')
        if devname is None:
            return json.dumps('Error: You need a devname to talk to backend')

        try:
            self._processing_folder(self.secret, self.username, devname, "devoir")
            self._processing_folder(self.secret, self.username, devname, "data")
            return "Done"
        except FilesAPIError, e :
            return json.dumps(str(e))

    @classmethod
    def register(cls, mod):
        f = cls.as_view("files_api")
        mod.add_url_rule(mod.url_prefix, view_func=f, methods=["GET", "DELETE", "PUT"],
                defaults={"typedoc": None, "filename": None, "devname": None})
        mod.add_url_rule(mod.url_prefix, view_func=f, methods=["POST"])
        mod.add_url_rule('%s<typedoc>' % mod.url_prefix, view_func=f,
                        methods=['DELETE'])
        mod.add_url_rule('%s<typedoc>' % mod.url_prefix, view_func=f,
                        methods=['GET'])
        mod.add_url_rule('%s<devname>' % mod.url_prefix, view_func=f,
                        methods=['PUT'])
        mod.add_url_rule('%s<typedoc>/<filename>' % mod.url_prefix, view_func=f,
                        methods=['DELETE'])

FilesAPI.register(filesapi)
