from flask import Blueprint, current_app, session, json
from xmlrpclib import ServerProxy
from flask.views import MethodView
from eoleauthlib.authclient import login_required
from eop.utils import secure_arg

vncapi = Blueprint('vncapi', __name__, url_prefix = '/api/vnc/')

class VncAPI(MethodView):
    decorators = [ login_required]

    def __init__(self):
        self.username = session["username"]
        self.proxy = ServerProxy(current_app.config['CONTROLEVNC_URL'])
        self.secret = current_app.config['SECRET_KEY']

    def get(self):
        raise NotImplementedError

    def post(self):
        raise NotImplementedError

    def put(self, login, id , ip):
        """Create a config file named "id" for websockify containing a token for noVNC and the ip address for login
        Returns connection informations for noVNC or false if eleve 'login' is not connected
        Return example : {'token': 'ab45g7jfi3ejk347i4i9', 'password': 'eole', 'login': c31e1}
        """
        if self.secret is None:
            return json.dumps('Error: You need a secret key to talk to backend')
        if self.username is None:
            return json.dumps('Error: You need a username to talk to backend')
        if (login is None) or (id is None) or (ip is None):
            return json.dumps('Error: Not enough informations provided to talk to backend')
        username = secure_arg(self.username)
        login = secure_arg(login)

        # Deny observation of a prof!
        if self.proxy.remote_est_prof(self.secret, login):
            current_app.logger.warning('%s tried to bypass the frontend filtering to observe the session of user %s without being allowed to!' %(self.username, login))
            return json.dumps('Error: access denied.')

        return json.dumps(self.proxy.remote_create_vnc_config(self.secret, username, login, id, ip))

    def delete(self, id):
        """Delete the websockify config file containing token and IP address corresponding to id (wich is of the form login-machine)
        """
        if self.secret is None:
            return json.dumps('Error: You need a secret key to talk to backend')
        if self.username is None:
            return json.dumps('Error: You need a username to talk to backend')
        if id is None:
            return json.dumps('Error: You need an id to talk to backend')
        username = secure_arg(self.username)
        return json.dumps(self.proxy.remote_delete_vnc_config(self.secret, username, id))

    @classmethod
    def register(cls, mod):
        f = cls.as_view("vnc_api")
        mod.add_url_rule('<login>/<id>/<ip>', view_func=f, methods=['PUT'])
        mod.add_url_rule('<id>', view_func=f, methods=['DELETE'])

VncAPI.register(vncapi)
