# -*- coding: UTF-8 -*-
from flask import Blueprint, session, request, current_app, json
from flask.views import MethodView
from eop.utils import check_password, generate_random_password, formate_date, PasswordCheckError
from xmlrpc.client import ServerProxy
from eoleauthlib.authclient import login_required

ldappasswordsapi = Blueprint('ldappasswordsapi', __name__, url_prefix = '/api/ldappasswords/')

class LdappasswordsAPI(MethodView):
    decorators = [ login_required]

    def __init__(self):
        self.username = session["username"]
        self.proxy = ServerProxy(current_app.config['CONTROLEVNC_URL'])
        self.secret = current_app.config['SECRET_KEY']

    def get(self):
        """Returns either the list of class that are administred by <username> if PROFS_CAN_CHANGE_ALL_PASSWORDS is set to "non"
        or the list of all classes if it set to "oui"
           When current user is admin, returns all classes
        """
        if self.username is None:
            current_app.logger.warning('Unbound use of the API, no session username defined!')
            return json.dumps('Error: You need a username to talk to backend')

        if self.username == "admin":
            usergroups = self.proxy.remote_groupes("classe")
        elif current_app.config['PROFS_CAN_CHANGE_ALL_PASSWORDS'] == "oui":
            usergroups = self.proxy.remote_groupes("classe", self.username)
        else:
            usergroups = self.proxy.remote_get_attr(self.secret, self.username, self.username, 'Divcod')

        return json.dumps(usergroups)

    def post(self):
        """Allows the current user to change another's user password
        in the post request :
        {"eleves":'["c31e1", "c31e2"]', "passwordformat":"birthdate", "newpassword":"", "forcechange":"true"}
        passwordformat can be : birthdate, random, name, common
        newpassword is only used if passwordformat is set to "common"
        """
        if self.username is None:
            current_app.logger.warning('Unbound use of the API, no session username defined!')
            return json.dumps('Error: You need a username to talk to backend')

        if None in request.form:
            current_app.logger.warning('Unbound use of the API by user %s.' %self.username)
            return json.dumps('Error in args')
        response = request.form

        eleves = json.loads(response['eleves'])

        if len(eleves) == 0:
            current_app.logger.warning('Unbound use of the API by user %s (empty list of users received).' %self.username)
            return json.dumps('Error in args')

        # checking if the current user is allowed to change the password of all users in "eleves"
        # only if PROFS_CAN_CHANGE_ALL_PASSWORDS is set to "non"
        # admin user is always allowed to change all passwords
        if current_app.config['PROFS_CAN_CHANGE_ALL_PASSWORDS'] == "non" and self.username != "admin":
            for usertochange in eleves:
                if not self.proxy.remote_is_eleve_from_prof(self.secret, usertochange, self.username):
                    current_app.logger.warning('%s  tried to bypass the frontend filtering to modify the password for user %s without being allowed to!' %(self.username, usertochange))
                    return json.dumps('Error: access denied.')

        # preparing the new password for each user in "eleves" according to the current user's choice for the password format
        modified = {}
        passwordformat = response['passwordformat']
        for usertochange in eleves:
            if(passwordformat == "birthdate"):
                date_naissance = self.proxy.remote_get_attr(self.secret, self.username, usertochange, 'dateNaissance')
                if date_naissance == []:
                    current_app.logger.info('Password modification failed because the field \'dateNaissance\' is not set for user %s.' %usertochange)
                    return json.dumps("Erreur : la date de naissance de %s n'est pas renseignée." % usertochange)
                newpassword = formate_date(date_naissance[0], sep='')
            elif(passwordformat == "random"):
                newpassword = generate_random_password()
            elif(passwordformat == "name"):
                nom = self.proxy.remote_get_attr(self.secret, self.username, usertochange, 'sn')[0].lower()
                prenom = self.proxy.remote_get_attr(self.secret, self.username, usertochange, 'givenName')[0].lower()
                newpassword = nom + '.' + prenom
            elif(passwordformat == "common"):
                try:
                    newpassword = check_password(self.username, response['newpassword'])
                except PasswordCheckError as e:
                    current_app.logger.warning('Attempt to bypass the frontend filtering! %s tried to set a new password not matching password policy for user %s.' %(self.username, usertochange))
                    return json.dumps(e.value)
            else:
                current_app.logger.warning('Unbound use of the API. User %s tried to bypass the frontend filtering!' %self.username)
                return json.dumps('Error: access denied')

            modified[usertochange] = newpassword

        # perfoming actual password modification after bad requests filtering
        forcechange = (response['forcechange'] == "true")
        for user, password in list(modified.items()):
            self.proxy.remote_change_mdp_eleve(self.secret, self.username, user, password, forcechange)
            current_app.logger.debug('%s set a new password for user %s.' %(self.username, usertochange))

            # if forcechange:
            #     self.proxy.remote_password_must_change(self.secret, self.username, user)

        file_name = self.proxy.remote_save_password_csv(self.secret, self.username, modified)
        result = {'eleves': modified, 'filename' : file_name}
        return json.dumps(result)

    def delete(self):
        raise NotImplementedError

    def put(self, oldpassword, newpassword):
        raise NotImplementedError

    @classmethod
    def register(cls, mod):
        f = cls.as_view("ldappasswords_api")
        mod.add_url_rule('', view_func=f, methods=["GET", "POST"])

LdappasswordsAPI.register(ldappasswordsapi)
