#!/usr/bin/perl -w
# vim: nu
# vim: set encoding=utf-8
#
# (c) 2006-2013   Stéphane URBANOVSKI <s.urbanovski@ac-nancy-metz.fr>
# (c) 2006        Jean-Christophe TOUSSAINT <jean-christophe.toussaint@ac-nancy-metz.fr>
#                               Pôle supervision - Academie de Nancy-Metz
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty
# of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# you should have received a copy of the GNU General Public License
# along with this program (or with Netsaint);  if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA
#
# $Id: eqosd.pl,v 1.8.0 2013-01-25  $
#
# commande de récupération de l'adresse mac (passage en paramètre)
# ifconfig eth0 | grep HWaddr | sed "s/^.*HWaddr \([^ ]*\).*$/\\1/"


use strict;
use warnings;

use Data::Dumper;
use File::Basename;
#use lib dirname($0).'/lib';
#use libEqosAgent;
use Term::ANSIColor qw(:constants);
use SOAP::Lite on_fault => sub {
	my $soap = shift;
	my $res = shift;
	ref $res ? &make_log(0, BOLD RED . "SOAP fault : " . $res->faultstring): &make_log(0, BOLD RED . "SOAP transport error : " . $soap->transport->status);
	return new SOAP::SOM;
};
use Unicode::String;
use Getopt::Long;
use IO::File;
use POSIX qw(ceil floor);
use POSIX ':sys_wait_h';
use IPC::Open3;
use DBI;
#use Carp;

##################################################################
# Global vars (rewritable by getconf)
##################################################################

my %conf;
$conf{'debuglevel'} = 2; # debug level (verbosity)
# 0 => normal : only low messages, big function calls (task forced), non-managed errors
# 1 => debug  : add message shown, scheduled task
# 2 => dev    : show all debug messages, dump vars, function parameters
$conf{'reloadconf'} = 5;
$conf{'ntpserver'} = "";
$conf{'smtpserver'} = "";
$conf{'feedbacktime'} = 3600;
$conf{'reloadservice'} = -1;
$conf{'plugintimeout'} = 30;


##################################################################
# Global vars
##################################################################

my $apprevision = '$Revision: 1.8.0 $';
my $mac = '';
my $halt = 0; # halt flag (stop service)
my $sleeptime = 5; # time wait between each iteration of main loop
my $proxyserver = ""; # http proxy used by SOAP
my $ceqos_version = '1.0'; # collector version
my $ceqos_server = 'eqos.orion.education.fr';
my $soapuri = "http://$ceqos_server/ceqos"; # SOAP URI ( != url http !!)
my $soapproxy = "http://$ceqos_server/eqos-collector/cgi-bin/ceqos-$ceqos_version.cgi"; # SOAP PROXY
my $getdefaultproxy = "/etc/http_proxy"; # if can't locate proxy env vars, this file is used
my $cmd_getmac = '/sbin/ifconfig eth0 | /bin/grep HWaddr | /bin/sed "s/^.*HWaddr \([^ ]*\).*$/\\1/"'; # get mac address with ifconfig method

my $cmd_setdate = '/bin/date -u -s "1970-01-01 ? sec" > /dev/null 2>&1'; # command to modify the date (? replaced by timestamp)

my $conf_loaded = 0;  # "configuration loaded" flag
my $date_synchro = 1; # synchronisation flag, no action executed if not synchronized

my $max_shift_time_alert = 60; # shift time between current local timestamp and server timestamp, if shift is too big => force synchronisation

my $debuglevel = 2;

# global name
my $globalname = "";



my $pid_checker = -1;
my $exec_checker = "./checker.pl"; # checker exe file
my $kill_checker = 15; # loop waited before killing checker

my $pid_feedback = -1;
my $kill_feedback = 15; # loop waited before killing feedback

# SQLite
my $sqlite_datafile = "/var/lib/sqlite/eqos/eqosd.sqlite";
my $sqlite_structdumpfile = "sql/eqos_struct.sql";

my $feedback_max = 50; # max block sent to each SOAP packet

# SQL query
my $query_pragma_sync = "PRAGMA synchronous = OFF";
my $query_config_delete_all = "DELETE FROM eqos_config WHERE 1;";
my $query_config_insert = "INSERT INTO eqos_config VALUES (NULL, ?, ?)";
my $query_states_select_all = "SELECT id, timestamp, idService, state, message FROM eqos_states ORDER BY timestamp ASC";
my $query_states_delete_id = "DELETE FROM eqos_states WHERE id=?";
my $query_services_delete_all = "DELETE FROM eqos_services WHERE 1;";
my $query_services_insert = "INSERT INTO eqos_services VALUES (?, ?, ?, ?)";
my $query_alerts_insert = "INSERT INTO eqos_alerts VALUES (NULL, ?, ?, ?)";
my $query_alerts_select_all = "SELECT id, timestamp, module, message FROM eqos_alerts;";
my $query_alerts_delete_id = "DELETE FROM eqos_alerts WHERE id=?";

$0 = "eqosd"; # pretty name ^^
$globalname = "eqosd"; # pretty name (for make_log)


##################
# file d'attente #
##################

# wait list for scheduled task
# HASH {
#    'timestamp' => integer
#    'command' => string
#    'param' => string
#    'autoadd' => integer (0 <= 1 exec, n <= add n to timestamp)
# }
my @task_queue = ();


my $soap; # SOAP object
#my $soap_result; # output of soap call function
my $sqlite; # db (SQLite)


##################################################################
# Functions
##################################################################


# Write a dated and formated log message
sub make_log {
	my ($lvl, $str) = @_;
	if ($lvl <= $debuglevel) {
		my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday) = gmtime();
		print WHITE . sprintf("%02d/%02d %02d:%02d:%02d", $mday, $mon, $hour, $min, $sec) . " [$globalname] $str" . RESET . "\n";
	}
}

sub read_text($) {
	my($filename) = @_;
	my $output = "";

	my $fh = new IO::File;
	if ($fh->open("< $filename")) {
		while (<$fh>) {
			$output .= $_;
		}
		$fh->close;
	}
	else {
		&make_log (1, BOLD RED . "can't read $filename");
		$output = undef;
	}
	return $output;
}

sub write_text($;$) {
	my($filename, $data) = @_;
	my $fh = new IO::File;
	if ($fh->open("> $filename")) {
		print $fh $data;
		$fh->close;
		return 1;
	}
	else {
		&make_log (1, BOLD RED . "can't write $filename");
		return 0;
	}
}

# Timestamp to date
sub ts2date($) {
	my ($ts) = @_;
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday) = gmtime($ts);
	return sprintf("%02d/%02d %02d:%02d:%02d", $mday, $mon, $hour, $min, $sec);
}

# Set end loop flag :
sub signal_halt {
	&make_log(0, BOLD RED . "HALT : stop signal received !");
	$halt = 1;
}

# Return eth0 hardware address used for agent identification
sub get_mac_address() {
	my $sysfile = '/sys/class/net/eth0/address';
	my $mac = '';
	if ( -e $sysfile ) {
		open(SYSFILE,"<$sysfile");
		$mac = <SYSFILE>;
		close(SYSFILE);
	} else {
		&make_log(0, BOLD RED . "Enable to read mac address in $sysfile.")
	}
	chomp $mac;
	return uc($mac);
}

sub show_config() {
	&make_log(0, YELLOW . "CONFIG");
	if ($proxyserver ne "") {
		&make_log(0, YELLOW . "  proxy server : " . BOLD MAGENTA . "$proxyserver");
	}
	&make_log(0, YELLOW . "  MAC : " . BOLD MAGENTA . "\"$mac\"");
	&make_log(0, YELLOW . "  soapuri : " . BOLD MAGENTA . "\"$soapuri\"");
	&make_log(0, YELLOW . "  soapproxy : " . BOLD MAGENTA . "\"$soapproxy\"");
	&make_log(0, YELLOW . "  reloadconf : " . BOLD MAGENTA . $conf{'reloadconf'});
	&make_log(0, YELLOW . "  debuglevel : " . BOLD MAGENTA . $conf{'debuglevel'});
	&make_log(0, YELLOW . "  ntpserver : " . BOLD MAGENTA . "\"" . $conf{'ntpserver'} . "\"");
	&make_log(0, YELLOW . "  smtpserver : " . BOLD MAGENTA . "\"" . $conf{'smtpserver'} . "\"");
	&make_log(0, YELLOW . "  feedbacktime : " . BOLD MAGENTA . $conf{'feedbacktime'});
	&make_log(0, YELLOW . "  reloadservice : " . BOLD MAGENTA . $conf{'reloadservice'});
	&make_log(0, YELLOW . "  plugintimeout : " . BOLD MAGENTA . $conf{'plugintimeout'});
	&make_log(0, YELLOW . "CONFIG END");
}

# force time synchronisation
sub sync_time($) {
	my ($new_time) = @_;
	my $cmd = $cmd_setdate;
	$cmd =~ s/\?/$new_time/;
	&make_log(2, YELLOW . "exec shell : " . RESET . "\"" . BOLD WHITE . "$cmd" . RESET . "\"");
	if (system($cmd) == 0) {
		return 1;
	}
	else {
		return 0;
	}
}

# get configuration function by soap
sub soap_load_config() {
	my $results;
	my $valueTask = "";
	my $tempString = "";

	&make_log(1, YELLOW . "SOAP QUERY " . BOLD MAGENTA . " getconf" . RESET . "(\"". BOLD MAGENTA . $mac . RESET . "\")");
	my $soap_result = $soap->eqosd_getconf("$mac");
	if (! $soap_result->fault) { # powered by Stéphane ^^
		$results = $soap_result->result();

		if (defined $results->{'message'}) {
			&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " message : " . BOLD MAGENTA . $results->{'message'});
			if ($results->{'message'} eq "OK") {

				# getconf task
				if (defined $results->{'task'}) {
					foreach $valueTask (@{$results->{'task'}}) {
						&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " task : " . BOLD MAGENTA . $valueTask->{'command'} . "(" . $valueTask->{'param'} . ") at \"" . ts2date($valueTask->{'timestamp'}) . "\"");
						add_scheduled_task($valueTask->{'timestamp'}, $valueTask->{'command'}, $valueTask->{'param'});
					}
				}

				# getconf debuglevel
				if (defined $results->{'debuglevel'}) {
					&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " debuglevel : " . BOLD MAGENTA . $results->{'debuglevel'});
					if ($results->{'debuglevel'} >= 0 && $results->{'debuglevel'} != $conf{'debuglevel'}) {
						&make_log(1, BOLD CYAN . "new debuglevel (" . MAGENTA . $results->{'debuglevel'} . CYAN . ")");
						$conf{'debuglevel'} = $results->{'debuglevel'};
						$debuglevel = $results->{'debuglevel'};
					}
					else {
						&make_log(2, "info : debuglevel isn't >= 0 or same as precedent");
					}
				}


				# getconf timestamp
				if (defined $results->{'timestamp'}) {
					&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " timestamp : " . BOLD MAGENTA . $results->{'timestamp'});

					# check shift time => if shift is too large, synchro + feedback
					my $shifted = ceil($max_shift_time_alert / 2);
					my $timed = time();

					if (($results->{'timestamp'} - $shifted > $timed) || ($timed > $results->{'timestamp'} + $shifted)) {
						$date_synchro = 0;
						make_alert("unsynchronized, force synchronisation");
						&make_log(0, BOLD RED . "info : too big shift between local time and server time ! force synchronisation");
					} else {
						&make_log(2, "info : shift time ok");
					}

					if ($date_synchro == 0) {
						$tempString = gmtime();
#						if (sync_time($results->{'timestamp'})) { # force synchro
#							&make_log(1, BOLD CYAN . "time synchronized (" . MAGENTA . $tempString . CYAN . " => " . MAGENTA . gmtime() . CYAN . ")");
#							$date_synchro = 1;
#						} else {
							&make_log(0, BOLD RED . "error : can't sync time");
#						}
#					} else {
#						&make_log(1, "info : synchronisation useless, ntpserver configured") if (length($conf{'ntpserver'}) > 0);
					}
				}

				# getconf reloadconf
				if (defined $results->{'reloadconf'}) {
					&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " reloadconf : " . BOLD MAGENTA . $results->{'reloadconf'});

					if (($results->{'reloadconf'} >= 1) && ($results->{'reloadconf'} != $conf{'reloadconf'})) {
						$conf{'reloadconf'} = $results->{'reloadconf'};
						&make_log(1, BOLD CYAN . "new reload conf (" . MAGENTA . $conf{'reloadconf'} . BOLD CYAN . ")");
					}
					else {
						&make_log(2, "info : reloadconf is not set or same as precedent");
					}
				}

				# getconf smtpserver
				if (defined $results->{'smtpserver'}) {
					&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " smtpserver : " . RESET . "\"" . BOLD MAGENTA . $results->{'smtpserver'} . RESET . "\"");

					if ((length($results->{'smtpserver'}) > 0) && ($results->{'smtpserver'} ne $conf{'smtpserver'})) {
						$conf{'smtpserver'} = $results->{'smtpserver'};
						&make_log(1, BOLD CYAN . "new smtpserver (\"" . MAGENTA . $conf{'smtpserver'} . BOLD CYAN . "\")");
					}
					else {
						&make_log(2, "info : smtpserver is not set or same as precedent");
					}
				}

				# getconf feedbacktime
				if (defined $results->{'feedbacktime'}) {
					&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " feedbacktime : " . BOLD MAGENTA . $results->{'feedbacktime'} . RESET);
					if ((length($results->{'feedbacktime'}) > 0) && ($results->{'feedbacktime'} ne $conf{'feedbacktime'})) {
						$conf{'feedbacktime'} = $results->{'feedbacktime'};
						&make_log(1, BOLD CYAN . "new feedbacktime (\"" . MAGENTA . $conf{'feedbacktime'} . BOLD CYAN . "\")");
					}
					else {
						&make_log(2, "info : feedbacktime is not set or same as precedent");
					}
				}

				# getconf reloadservice
				if (defined $results->{'reloadservice'}) {
					&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " reloadservice : " . BOLD MAGENTA . $results->{'reloadservice'} . RESET);
					if ((length($results->{'reloadservice'}) > 0) && ($results->{'reloadservice'} ne $conf{'reloadservice'})) {
						$conf{'reloadservice'} = $results->{'reloadservice'};
						add_scheduled_task((time() + 5), 'reloadservice', '');
						&make_log(1, BOLD CYAN . "new reloadservice (\"" . MAGENTA . $conf{'reloadservice'} . BOLD CYAN . "\")");
						&make_log(0, WHITE . "info : services modified, add task to reload services from collector");
					}
					else {
						&make_log(2, "info : reloadservice is not set or same as precedent");
					}
				}


				# getconf reloadservice
				if (defined $results->{'plugintimeout'}) {
					&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " plugintimeout : " . BOLD MAGENTA . $results->{'plugintimeout'} . RESET);
					if ((length($results->{'plugintimeout'}) > 0) && ($results->{'plugintimeout'} ne $conf{'plugintimeout'})) {
						$conf{'plugintimeout'} = $results->{'plugintimeout'};
						&make_log(1, BOLD CYAN . "new plugintimeout (\"" . MAGENTA . $conf{'plugintimeout'} . BOLD CYAN . "\")");
					}
					else {
						&make_log(2, "info : plugintimeout is not set or same as precedent");
					}
				}


				# load into db
				&make_log(2, BOLD YELLOW . "sqlite info : deleting all data in config table (preparing insertion)");
				if (my $delete_all_result = $sqlite->do("$query_config_delete_all")) {
					my $error_spotted = 0;
					#my $pq_config_insert = $sqlite->prepare("$query_config_insert");
					foreach my $keyconf (keys %conf) {
						&make_log(2, BOLD YELLOW . "sqlite info : insert conf $keyconf " . " <= \"" . $conf{$keyconf} . "\"");
						$sqlite->do("$query_config_insert", undef, ($keyconf, $conf{$keyconf})) or $error_spotted++;
					}
#					$pq_config_insert->finish;
					if ($error_spotted == 0) {
						&make_log(0, "info : config written into db");
					}
					else {
						&make_alert("error spotted during loading of config into db");
						&make_log(0, BOLD RED . "error : error spotted during loading of config into db !");
					}
				}

				$conf_loaded = 1;
			}
			else {
				&make_log(0, BOLD RED . " error : message is not OK (" . $results->{'message'} . "), aborting configuration");

			}
		}
		else {
			&make_log(1, BOLD RED . "error : message not defined, aborting config");
		}
	}
	else {
		&make_log(0, BOLD RED . "error : " . join(', ', $soap_result->faultcode, $soap_result->faultstring, $soap_result->faultdetail));
	}
}

# make an alert message to send to collector
sub make_alert($) {
	my ($message) = @_;
	$sqlite->do("$query_alerts_insert", undef, (time(), $globalname, $message)) or &make_log(0, BOLD RED . "sqlite error : can't insert alert into table");
	return 1;
}


#  send alert if it is possible (connection ok)
sub send_alert() {
	my $resultAlerts = $sqlite->selectall_arrayref("$query_alerts_select_all");
	foreach my $row (@{$resultAlerts}) {
		my ($id, $timestamp, $module, $message) = @$row;
		my $cleanedstring = $message;
		$cleanedstring =~ s/\n/ /g;
		if (length($cleanedstring) > 40) {
			$cleanedstring = substr($cleanedstring, 0, 40) . "..."; # cut string
		}
		&make_log(2, "try to send alert message, event from \"" . ts2date($timestamp) . "\" : " . $cleanedstring);
		&make_log(1, YELLOW . 'SOAP QUERY ' . BOLD MAGENTA . ' sendalert' . RESET . '("' . BOLD MAGENTA . $mac . RESET . '", ' . BOLD MAGENTA . $timestamp . RESET .  ', "' . BOLD MAGENTA . $module . RESET . '", "' . BOLD MAGENTA . "$cleanedstring" . RESET . "\")");
		my $soap_result = $soap->sendalert("$mac", $timestamp, $module, $message);
		if (! $soap_result->fault) {
			my $results = $soap_result->result();
			&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " message : " . BOLD MAGENTA . $results->{'message'});
			if ($results->{'message'} eq "OK") {
				if (defined $sqlite->do("$query_alerts_delete_id", undef, ($id))) {
					&make_log(2, BOLD YELLOW . "sqlite info : alert id $id deleted");
				}
				else {
					&make_log(0, BOLD RED . "sqlite error : can't delete alert id $id, stopping and retrying later");
					last;
				}
			}
			else {
				&make_log(0, BOLD RED . "error : an error has been detected during commit message, aborting");
				last;
			}
		}
	}
	return 1;
}

# send an instant alert
sub send_instant_alert($) {
	my ($message) = @_;
	my $cleanedstring = $message;
	$cleanedstring =~ s/\n/ /g;
	if (length($cleanedstring) > 40) {
		$cleanedstring = substr($cleanedstring, 0, 40) . "..."; # cut string
	}

	&make_log(2, "try to send alert message, event from \"" . ts2date(time()) . "\" : " . $cleanedstring);
	&make_log(1, YELLOW . 'SOAP QUERY ' . BOLD MAGENTA . ' sendalert' . RESET . '("' . BOLD MAGENTA . $mac . RESET . '", ' . BOLD MAGENTA . time() . RESET .  ', "' . BOLD MAGENTA . $globalname . RESET . '", "' . BOLD MAGENTA . "$cleanedstring" . RESET . "\")");
	my $soap_result = $soap->sendalert("$mac", time(), $globalname, $message);
	if (! $soap_result->fault) {
		my $results = $soap_result->result();

		&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " message : " . BOLD MAGENTA . $results->{'message'});
		if ($results->{'message'} eq "OK") {
			&make_log(2, "info : message sent");
			return 1;
		}
		else {
			&make_log(0, BOLD RED . "error : an error has been detected during commit message, aborting");
			return 0;
		}
	}
	else {
		# add to end on error and leave function
		&make_log(0, BOLD RED . "error : can't connect to collector, aborting. [" . join(', ', $soap_result->faultcode, $soap_result->faultstring, $soap_result->faultdetail) . "]");
		return 0;
	}
}


# add task at the end of list
sub add_task($;$;$;$) {
	my ($timestamp, $command, $param, $autoadd) = @_;
	my %data =	(
				'timestamp' => $timestamp,
				'command' => $command,
				'param' => $param,
				'autoadd' => $autoadd,
			);
	push @task_queue, \%data;
	&make_log(2, "info : $command(\"" . WHITE . $param . RESET . "\") at \"" . ts2date($timestamp) . "\" (autoadd $autoadd) added to task list") if ($autoadd == 0); # no useless flood message
	return 1;
}

# add task at the end of list, if it isn't in the list (loop task)
sub add_scheduled_task($;$;$) {
	my ($timestamp, $command, $param) = @_;
	my $spot = 0;
	my $value;

	foreach $value (@task_queue) {
		if (($value->{'timestamp'} == $timestamp) && ($value->{'command'} eq $command) && ($value->{'param'} eq $param)) { # check not in the list
			&make_log( 2, "info : $command(\"" . WHITE . $param . RESET . "\") at \"" . ts2date($timestamp) . "\" already scheduled");
			$spot = 1;
		}
	}
	if ($spot == 0) {
		my %data =	(
					'timestamp' => $timestamp,
					'command' => $command,
					'param' => $param,
					'autoadd' => 0,
				);
		push @task_queue, \%data;
		&make_log(2, "info : $command(\"" . WHITE . $param . RESET . "\") at \"" . ts2date($timestamp) . "\" (autoadd 0) added to task list");
	}
	return 1;
}

# process task
sub check_task() {
	my $value;
	my $exec_ok = 0;
	my @copy_task_queue = @task_queue; # copy all task
	@task_queue = (); # empty task list

	if (($#copy_task_queue + 1) > 0) {
		&make_log(2, ($#copy_task_queue + 1) . " task(s) waiting");
		reset @copy_task_queue;
		while (($value = shift @copy_task_queue) && ($halt == 0)) {
			if (time() >= $value->{'timestamp'}) { # exectimeout ?
				&make_log(2, "try to exec task : " . $value->{'command'} . "(" . $value->{'param'} . ") from \"" . ts2date($value->{'timestamp'}) . "\" (autoadd " . $value->{'autoadd'} . ")");
				$exec_ok = 0;
				if ($value->{'command'} eq 'getconf') {
					&make_log(2, "task getconf spotted, starting \"get configuration function\"");
					soap_load_config();
					add_task((time() + $conf{'reloadconf'}), $value->{'command'}, '', $conf{'reloadconf'});
					&make_log(2, "end of \"get configuration function\"");
				}
				elsif ($value->{'command'} eq 'execCmdAlert') {
#					&make_log(2, "task execCmdAlert spotted, exec shell and send result by make_alert");
					&make_log(2, $value->{'command'}." disabled on EOLE servers");
#					execCmdAlert($value->{'param'});
#					&make_log(2, "end of task execCmdAlert");
				}
				elsif ($value->{'command'} eq 'simUnsync') {
					&make_log(2, $value->{'command'}." disabled on EOLE servers");
#					&make_log(2, "task simUnsync spotted, force date_synchro <= 0");
#					$date_synchro = 0;
#					&make_log(2, "end of task simUnsync");
				}
				elsif ($value->{'command'} eq 'reloadservice') {
					&make_log(2, "task reloadservice spotted, reloading services");
					load_services();
					&make_log(2, "end of task reloadservice");
				}
				elsif ($value->{'command'} eq 'feedback') {
					&make_log(2, "task feedback spotted, feedbacking");
					start_feedback();
					add_task((time() + $conf{'feedbacktime'}), $value->{'command'}, '', $conf{'feedbacktime'});
					&make_log(2, "end of task feedback");
				}
				else {
					&make_log(1, "info : " . $value->{'command'} . ", unknown task, ignoring ...");
#					add_task($value->{'timestamp'}, $value->{'command'}, $value->{'param'}, $value->{'autoadd'});
				}
			}
			else { # time not expired, so add at end
				&make_log(2, "info : task \"" . $value->{'command'} . "\", time not expired (" . ($value->{'timestamp'} - time()) . "s)");
				add_task($value->{'timestamp'}, $value->{'command'}, $value->{'param'}, $value->{'autoadd'});
			}
		}
	}
	return 1;
}

# execute a shell command and send result by "alert" (soap)
sub execCmdAlert($) {
	my ($cmd) = @_;
	my %return;
	my @stdout = ();
	my @stderr = ();
	my $cmdstdin = IO::Handle->new();
	my $cmdstdout = IO::Handle->new();
	my $timeout_alarm = 0;
	my $timeout = $conf{'reloadconf'} - 1;

	if ($timeout < 5) {
		$timeout = 5;
	}

	$return{"pid"} = 0;

	$SIG{'ALRM'} = sub {
		&make_log(1, "info : sigalrm caught ! try to kill process");
		$cmdstdin->close();
		$cmdstdout->close();
		$timeout_alarm = 1;
		if ( $return{"pid"} > 1) {
			my $killed = kill 9, $return{"pid"};
			&make_log(2, $killed . " process killed");
		}
	};


	&make_log(1, "info : shell exec of \"" . BOLD WHITE . $cmd . RESET . "\"");

	alarm($timeout);
	$return{"pid"} = open3($cmdstdin, $cmdstdout, $cmdstdout, $cmd) ;

	if ($return{"pid"} == -1) {
		&make_log(0, BOLD RED . "error : can't execute \"" . BOLD WHITE . $cmd . RESET . "\" !");
		make_alert("execCmdAlert at " . ts2date(time()) . " : $cmd\nerror : can't execute \"$cmd\" !");
	}
	else {
		if ($cmdstdout->opened()) {
			@stdout = $cmdstdout->getlines();
		}
		my $waitpid = waitpid($return{"pid"}, 0);
		$return{"return"} = $?;

		if ($timeout_alarm) {
			&make_log(0, BOLD RED . "error : execution of \"" . BOLD WHITE . $cmd . RESET . "\" timeout ! killed...");
			make_alert("execCmdAlert at " . ts2date(time()) . " : $cmd\nerror :  \"$cmd\" timeout ! force kill !");
		}
		else {
			$return{"return"} = $return{"return"} >> 8;
			chomp @stdout;
		}
	}

	$cmdstdin->close();
	$cmdstdout->close();
	alarm(0);

	&make_log(0, "info : execution of \"" . BOLD WHITE . $cmd . RESET . "\" OK ! message added to sendqueue");
	make_alert("execCmdAlert at \"" . ts2date(time()) . "\" : \`$cmd\`\nreturn value : " . $return{"return"} . "\noutput :\n" . join("\n", @stdout));

	return 1;
}

sub start_checker() {
	my $temp_pid = -1;

	select(STDOUT); $|=1;

	local *NULL;
	open (NULL, '/dev/null') or &make_log(0, "can't open /dev/null");
	$temp_pid = open3("<&NULL", ">&STDOUT", ">&STDOUT", $exec_checker);
	if ($temp_pid > 0) {
		$pid_checker = $temp_pid;
		return 1;
	}
	else {
		$pid_checker = -1;
		return 0;
	}
}

sub stop_checker() {
	if ($pid_checker != -1) {
		&make_log(2, "info : try to kill checker gracefully");
		kill 15, $pid_checker; # try to exec a pretty halt
		my $cpt = 1;
		while (($cpt <= $kill_checker) && (waitpid($pid_checker, WNOHANG) != -1)) { # wait for halt
			&make_log(2, "info : ok, checker needs a second... ($cpt/$kill_checker before kill -9)");
			sleep 1; # waiting...
			$cpt++;
		}
		if (waitpid($pid_checker, WNOHANG) != -1) { # halted ?
			&make_log(0, BOLD RED . "error : checker don't want to halt gracefully <= kill -9 sent !");
			make_alert("checker can't halt gracefully !");
			kill 9, $pid_checker; # ok, kill -9 in`da`head !
		}
		$pid_checker = -1;
		return 1;
	}
	else {
		&make_log(1, "info : checker already killed");
		return 0;
	}
}

sub reload_checker() {
	if ($pid_checker != -1) {
		kill 1, $pid_checker;
		&make_log(2, "info : SIGHUP send to checker");
		return 1;
	}
	else {
		&make_log(1, "info : checker not started");
		return 0;
	}
}

# fork to // execution
sub start_feedback() {
	&make_log(1, "info : preparing to fork process for feedback");

	if ($pid_feedback != -1) {
		&make_log(0, "info : feedback already started, so stopping before");
		stop_feedback();
	}

	my $pid = fork();
	if ($pid > 0) {
		&make_log(1, "info : fork OK (pid $pid)");
		$pid_feedback = $pid;
		return 1;
	}
	elsif ($pid == 0) {
		$0 = 'feedback';
		$globalname = 'feedback';
		sleep 2;
		feedback();
		exit 0;
	}
	else {
		&make_log(BOLD RED . "error : can't fork !");
	}
	return 0;

}

sub stop_feedback() {
	if ($pid_feedback != -1) {
		&make_log(2, "info : try to kill feedback gracefully");
		kill 15, $pid_feedback; # try to exec a pretty halt
		my $cpt = 1;
		while (($cpt <= $kill_feedback) && (waitpid($pid_feedback, WNOHANG) != -1)) { # wait for halt
			&make_log(2, "info : ok, feedback needs a second... ($cpt/$kill_feedback before kill -9)");
			sleep 1; # waiting...
			$cpt++;
		}
		if (waitpid($pid_feedback, WNOHANG) != -1) { # halted ?
			&make_log(0, BOLD RED . "error : feedback don't want to halt gracefully <= kill -9 sent !");
			make_alert("feedback can't halt gracefully !");
			kill 9, $pid_feedback; # ok, kill -9 in`da`head !
		}
		$pid_feedback = -1;
		return 1;
	}
	else {
		&make_log(1, "info : feedback already killed");
		return 0;
	}
}

sub feedback() {
	my $rows_cpt = 0;
	my $error = 0;
	my @send_block = ();
	my $current_cpt = 0;

	&make_log(0, GREEN . "starting feedback");

	&make_log(1, "info : init new sqlite connection");
	my $sqlite_feedback = db_connect();
	$sqlite_feedback->{'AutoCommit'} = 0;

	if (! defined $sqlite_feedback) {
		&make_log(0, BOLD RED . "error : can't connect to db");
		exit 1;
	}

	my $resultStates = $sqlite_feedback->selectall_arrayref("$query_states_select_all");

	my @tempArray = @{$resultStates};

	foreach my $row (@{$resultStates}) {
		if ($#send_block < 0) {
			&make_log(2, "info : beginning a new block for feedback sending (max $feedback_max elements)");
			$current_cpt = 0;
		}

		my ($id, $timestamp, $idService, $state, $message) = @$row;

		my %data_add =	(
					'id' => $id,
					'timestamp' => $timestamp,
					'idService' => $idService,
					'state' => $state,
					'message' => $message,
				);
		push @send_block, \%data_add;
		$rows_cpt++;
		$current_cpt++;

		if (($current_cpt >= $feedback_max) || (($#tempArray + 1) == $rows_cpt)) {
			&make_log(2, "info : block ready");

			reset @send_block;
			foreach my $block (@send_block) {
				&make_log(2, BOLD YELLOW . "sqlite info : deleting state, id " . $block->{'id'});
				$sqlite_feedback->do("$query_states_delete_id", undef, ($block->{'id'})) or $error++;
			}

			if ($error == 0) { # no error ? (deleting)
				&make_log(2, "info : preparing to send a feedback block (" . ($#send_block + 1) . " feedbacks)");
				my $soap_result = $soap->feedback("$mac", @send_block);
				if (! $soap_result->fault) {
					my $results = $soap_result->result();
					if (defined $results->{'message'}) {
						if ($results->{'message'} eq "OK") {
							$sqlite_feedback->commit;
							&make_log(2, "info : block sent (" . ($#send_block + 1) . " element(s))");
							@send_block = ();
						}
						else {
							&make_log(0, BOLD RED . "error spotted, rollbacking and halting ! " . $results->{'message'});
							send_instant_alert("error spotted, rollbacking and halting ! " . $results->{'message'});
							$sqlite_feedback->rollback;
							last;
						}
					}
					else {
						&make_log(0, BOLD RED . "error : message undefined, rollbacking and halting");
						send_instant_alert("error : message undefined, rollbacking and halting");
						$sqlite_feedback->rollback;
						last;
					}
				}
				else {
					&make_log(0, BOLD RED . "error : (rollbacking and halting) " . join(', ', $soap_result->faultcode, $soap_result->faultstring, $soap_result->faultdetail));
					send_instant_alert("error : (rollbacking and halting) " . join(', ', $soap_result->faultcode, $soap_result->faultstring, $soap_result->faultdetail));
					$sqlite_feedback->rollback;
					last;
				}
			}
			else {
				&make_log(0, BOLD RED . "error : can't delete states, halting feedback and rollbacking");
				send_instant_alert("error : can't delete states, halting feedback and rollbacking");
				$sqlite_feedback->rollback;
				last;
			}
		}
	} # end foreach

	if ($error == 0) {
		&make_log(1, "info : feedback successfully executed, $rows_cpt row(s) sent");
	}
	else {
		&make_log(0, BOLD RED . "error : an error has been spotted during feedback, stopping feedback");
	}
	&make_log(0, GREEN . "end of feedback");
	$sqlite_feedback->disconnect;
	exit 0;
}


# load services fron controller (by SOAP) and copy into SQLite database
sub load_services() {
	&make_log(2, "info : preparing to load services");

	&make_log(1, YELLOW . "SOAP QUERY " . BOLD MAGENTA . " getservices" . RESET . "(\"". BOLD MAGENTA . $mac . RESET . "\")");
	my $soap_result = $soap->eqosd_getservices("$mac");

	if (! $soap_result->fault) {
		my $results = $soap_result->result();
		if (defined $results->{'message'}) {
			&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " message : " . BOLD MAGENTA . $results->{'message'});
			if ($results->{'message'} eq "OK") {
				if (defined $results->{'services'}) {
					if (defined $sqlite->do("$query_services_delete_all")) {
						&make_log(2, BOLD YELLOW . "sqlite info : all services deleted");
						my $error_spotted = 0;
						foreach my $valueService (@{$results->{'services'}}) {
							&make_log(1, YELLOW . "SOAP RESULT" . CYAN . " service : " . BOLD MAGENTA . $valueService->{'id'} . " \"" . $valueService->{'plugin'} . "\" \"" . $valueService->{'args'} . "\" " . $valueService->{'period'});
							if (defined $sqlite->do("$query_services_insert", undef, ($valueService->{'id'}, $valueService->{'plugin'}, $valueService->{'args'}, $valueService->{'period'}))) {
								&make_log(2, BOLD YELLOW . "sqlite info : insert " . $valueService->{'id'} . " " . $valueService->{'plugin'} . " " . $valueService->{'args'} . " " . $valueService->{'period'} . " => db");
							}
							else {
								&make_log(0, BOLD RED . "sqlite error : can't insert into db, aborting");
								make_alert("error : can't insert into services db");
								$error_spotted++;
								last;
							}
						}
						if ($error_spotted == 0) {
							if (!reload_checker()) {
								&make_log(0, BOLD RED . "error : can't send reload signal to checker, maybe not started");
							}
							else {
								$conf{'reloadservice'} = 0;
							}
						}
					}
					else {
						&make_log(0, BOLD RED . "error : can't flush table eqos_services ! send alert, and scheduling a new task");
						make_alert("can't flush table eqos_services (delete all data), retrying.");
						add_scheduled_task((time() + 5), 'reloadservice', '');
					}
				}
			}
			else {
				&make_log(0, BOLD RED . "error spotted ! " . $results->{'message'});
			}
		}
		else {
			&make_log(0, BOLD RED . "error : message undefined");
		}
	}
	else {
		&make_log(0, BOLD RED . "error : " . join(', ', $soap_result->faultcode, $soap_result->faultstring, $soap_result->faultdetail));
	}
}

sub db_connect() {
	return DBI->connect("dbi:SQLite:dbname=$sqlite_datafile", "", "", { PrintWarn => 1, PrintError => 1, AutoCommit => 1 });
}



##################################################################
# MAIN
##################################################################

# disable buffers
$| = 1;

# Get eth0 hardware address used for agent identification
$mac = &get_mac_address();
if ($mac eq "" || !($mac =~ /[0-9a-z]{2}\:[0-9a-z]{2}\:[0-9a-z]{2}\:[0-9a-z]{2}\:[0-9a-z]{2}\:[0-9a-z]{2}/i)) {
	&make_log(0, BOLD RED . "invalid MAC address ($mac)");
}

# search proxy server, and check
$proxyserver = (defined $ENV{'http_proxy'}) ? $ENV{'http_proxy'} : "";
if ($proxyserver eq "") {
	if (-e $getdefaultproxy && -r $getdefaultproxy && -s $getdefaultproxy) {
		$proxyserver = read_text($getdefaultproxy);
		chomp $proxyserver;
		$ENV{'http_proxy'} = $proxyserver;
	}
}
#$ENV{'https_proxy'} = $ENV{'http_proxy'};

&make_log(0, "starting eqosd-controller $apprevision");

# signal manager
$SIG{'INT'} = 'signal_halt';
$SIG{'QUIT'} = 'signal_halt';
$SIG{'TERM'} = 'signal_halt';
#$SIG{'CHLD'} = 'IGNORE'; # feedback SIGCHLD management

show_config();

$soap = SOAP::Lite
	-> uri("$soapuri")
	-> proxy("$soapproxy", proxy => ['http' => "$proxyserver", 'https' => "$proxyserver"]);
send_instant_alert("eqosd start $apprevision");

&make_log(0, BOLD YELLOW . "sqlite info : loading database");
$sqlite = db_connect();

if (! -s $sqlite_datafile) {
	&make_log(0, BOLD YELLOW . "sqlite info : empty dbfile, creating database and table");
	my $error_spotted = 0;
	open (SQLFILE,"<$sqlite_structdumpfile");
	while (my $line = <SQLFILE>) {
		chomp $line;
		$sqlite->do($line) or $error_spotted++;
	}
	close (SQLFILE);

	if ($error_spotted == 0) {
		&make_log(0, BOLD YELLOW . "sqlite info : database OK");
	}
	else {
		&make_log(0, BOLD RED . "sqlite error : SQLite, unstable state ! can't create table => stopping");
		send_instant_alert("SQLite, unstable state ! can't create table, halting !");
		exit 1;
	}
}
else {
	&make_log(0, BOLD YELLOW . "sqlite info : database OK");
}

my $prama_return = $sqlite->do("$query_pragma_sync");
if (defined $prama_return) {
	&make_log(1, BOLD YELLOW . "sqlite info : pragma sync configured");
}
else {
	&make_log(0, BOLD RED . "sqlite error : can't configure pragma sync");
}


add_task((time() + 1), 'getconf', '', 1); # add getconf task, instant mode (executed on each loop)
add_task((time() + 60), 'feedback', '', 1); # add feedback task, instant mode (executed on each loop)

my $check_pid_child = -1;

if ($halt == 0) {
	&make_log(1, GREEN . "Starting main loop");
}

while ($halt == 0) {
	if (($#task_queue + 1) > 0) {
		check_task() if ($halt == 0);
	}

	if ($conf_loaded == 1 && $date_synchro == 1) {
		if ($pid_checker > 0) {
			$check_pid_child = waitpid($pid_checker, WNOHANG);
			if ($check_pid_child == -1) {
				&make_log(0, BOLD RED . "error : hohooo.. checker dead ! restarting immediately !");
				start_checker();
			}
			else {
				&make_log(2, "info : checker is ok, continuing...");
			}
		}
		else {
			&make_log(0, "info : environment is ok, so trying to start ckecker");
			if (start_checker()) {
				&make_log(0, "info : checker started !");
			}
			else {
				&make_log(0, "info : checker start failed, retrying next loop !");
			}
		}

		if ($pid_feedback > 0) {
			$check_pid_child = waitpid($pid_feedback, WNOHANG);
			if ($check_pid_child == -1) {
				&make_log(0, "info : feedback process finalized");
				$pid_feedback = -1;
			}
		}
	}
	else { # if date is not synchronized, stop checker
		if ($pid_checker > 0) {
			&make_log(0, BOLD RED . "info : stopping checker because unsynchronized !");
			stop_checker();
		}
	}

	send_alert();

	sleep $sleeptime;
}

stop_checker();

make_alert("eqosd halt $apprevision");
send_alert();

&make_log(0, "disconnecting from database (SQLite)");
$sqlite->disconnect;

&make_log(0, GREEN . "clean halt");

exit(0);

