#!/usr/bin/perl -w
#
# Copyright (c) 2002-2013 Stéphane Urbanovski <stephane.urbanovski@ac-nancy-metz.fr>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty
# of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# you should have received a copy of the GNU General Public License
# along with this program (or with Nagios);  if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA
#

use strict;					# should never be differently :-)
use warnings;


use Locale::gettext;
use File::Basename;			# get basename()

use POSIX qw(setlocale);
use Unicode::String qw(latin1 utf8); # Convert between iso-8859-1 and utf-8
use Time::HiRes qw(time);			# get microtime
use POSIX qw(mktime);

use Nagios::Plugin ;

use LWP::UserAgent;			# http client
use HTTP::Request;			# used by LWP::UserAgent
use HTTP::Status;			# to get http err msg

use XML::Parser::Expat;

use Data::Dumper;


my %RETURN_CODE_VALUES = (
	'UNKNOWN'	=> UNKNOWN,
	'OK' 		=> OK,
	'WARN'		=> WARNING,
	'CRIT'		=> CRITICAL,
	'MAINT'		=> UNKNOWN
	);


my $PROGNAME = basename($0);
'$Revision: 2.0 $' =~ /^.*(\d+\.\d+) \$$/;  # Use The Revision from RCS/CVS/SVN
my $VERSION = $1;

my $DEBUG = 0;
my $TIMEOUT = 9;

# i18n :
setlocale(LC_MESSAGES, '');
textdomain('nagios-plugins-perl');


my $np = Nagios::Plugin->new(
	version => $VERSION,
	blurb => _gt('Plugin to check Racvision web url'),
	usage => "Usage: %s [ -v|--verbose ]  -u <url> [-t <timeout>] [ -c|--critical=<threshold> ] [ -w|--warning=<threshold> ]",
	timeout => $TIMEOUT+1
);
$np->add_arg (
	spec => 'debug|d',
	help => _gt('Debug level'),
	default => 0,
);
$np->add_arg (
	spec => 'w=f',
	help => _gt('Warning request time threshold (in seconds)'),
	default => 2,
	label => 'FLOAT'
);
$np->add_arg (
	spec => 'c=f',
	help => _gt('Critical request time threshold (in seconds)'),
	default => 10,
	label => 'FLOAT'
);
$np->add_arg (
	spec => 'url|u=s',
	help => _gt('URL of the racvision xml page.'),
	required => 1,
);
$np->add_arg (
	spec => 'encoding|e:s',
	help => _gt('Enforce caracters encoding (\'utf-8\',\'iso-8859-1\', ...) when parsing XML DATA'),
	default => '',
);
$np->add_arg (
	spec => 'ignore|i=s@',
	help => _gt('Ignore listed tests'),
);

$np->getopts;

$DEBUG = $np->opts->get('debug');
my $verbose = $np->opts->verbose;

# Thresholds :
# time
my $warn_t = $np->opts->get('w');
my $crit_t = $np->opts->get('c');

my $url = $np->opts->get('url');
my $encoding = $np->opts->get('encoding');
my $ignore = $np->opts->get('ignore');


# Create a LWP user agent object:
my $ua = new LWP::UserAgent(
	'env_proxy' => 0,
	'timeout' => $TIMEOUT,
	);
$ua->agent(basename($0));

# Workaround for LWP bug :
$ua->parse_head(0);


# Handle specific availability computing
my $availability = OK;
my $availability_mode = 'NAGIOS'; # Default to Nagios mode

if ( defined($ENV{'http_proxy'}) ) {
	# Normal http proxy :
	$ua->proxy(['http'], $ENV{'http_proxy'});
	# Https must use Crypt::SSLeay https proxy (to use CONNECT method instead of GET)
	$ENV{'HTTPS_PROXY'} = $ENV{'http_proxy'};
}

# Build and submit an http request :
my $request = HTTP::Request->new('GET', $url);
my $timer = time();
my $http_response = $ua->request( $request );
$timer = time()-$timer;


my $connection_status = $np->check_threshold(
	'check' => $timer,
	'warning' => $warn_t,
	'critical' => $crit_t,
);

$np->add_perfdata(
	'label' => 't',
	'value' => sprintf('%.6f',$timer),
	'min' => 0,
	'uom' => 's',
	'threshold' => $np->threshold()
);

if ( $connection_status > OK ) {
	$np->add_message($connection_status, sprintf(_gt("Response time degraded: %.6fs !"),$timer) );
}


if ( $http_response->is_error() ) {
	my $err = $http_response->code." ".status_message($http_response->code)." (".$http_response->message.")";
	$np->nagios_exit(CRITICAL, _gt("HTTP error: ").$err );
}

if ( ! $http_response->is_success() ) {
	my $err = $http_response->code." ".status_message($http_response->code)." (".$http_response->message.")";
	$np->nagios_exit(CRITICAL, _gt("Internal error: ").$err );
}


# ($status, $message) = $np->check_messages();
my ($status, $message) = (OK, '');

# if ( $http_response->is_success() ) {

	# Get xml content ... 
	my $xml = $http_response->content;
	if ($DEBUG) {
		print "------------------===http output===------------------\n$xml\n-----------------------------------------------------\n";
		print "t=".$timer."s\n";
	};
	
	# Remove DOCTYPE and new lines:
	$xml =~ s/\<\!DOCTYPE .*?\>// ;
	my $fxml = '';
	foreach ( split("[\r\n]+",$xml) ) {
		s/^\s+//;
		$fxml .= $_;
	}
	$xml = $fxml ;

	if ($DEBUG) {
		print "------------------===sanitized===------------------\n$xml\n-----------------------------------------------------\n";
	};

	# Get XML tools:
	my $parser;
	
	# Manage encoding :
	if ( $encoding eq 'iso-8859-1' ||  $encoding eq 'iso-8859-15') {
		# transcode iso-8859-1 to utf8 :
		my $tmp = latin1($xml);
		$xml = $tmp->utf8;
	}
	
	#Force xml output to utf-8
	utf8::upgrade($xml);
	
	
	my %d = (
		'_ERROR' => '',
		'DATE' => '',
		'APPLICATION' => {
			'NAME' => '',
			'VERSION' => '',
			'DESCRIPTION' => '',
			'AVAILABILITY' => '',
			'TEST' => [],
			'PERIOD' => {
				'START' => '',
				'END' => ''
			}
		},
	);
	
	if ( !&xmlParse($xml,\%d) ) {
		$np->nagios_exit(CRITICAL, _gt("Xml parse error: ").$d{'_ERROR'} );
	}
	
	# Now that we have something parsable, check if we are in application driven availability computing mode (aka EQOS)
	if ( defined($ENV{'AVAILABILITY_MODE'}) && $ENV{'AVAILABILITY_MODE'} eq 'EQOS') {
		$availability_mode = 'EQOS';
		logD ('Switching to AVAILABILITY_MODE='.$availability_mode.' status='.$connection_status);
	}
	$availability = $connection_status;

	logD ('name['.$d{'APPLICATION'}{'NAME'}.'] - version['.$d{'APPLICATION'}{'VERSION'}.']');
	
	$np->shortname($d{'APPLICATION'}{'NAME'}.' ('.$d{'APPLICATION'}{'VERSION'}.')');
	

	#
	# Handle racvision tests
	################################
	
	my $nb_test = scalar(@{$d{'APPLICATION'}{'TEST'}});
	my $nb_test_done = 0;
	my $nb_test_ignored = 0;
	my $tmp_status = OK;
	
	for ( my $i = 1 ; $i <= $nb_test ; $i++ ) {
		my $test = $d{'APPLICATION'}{'TEST'}[$i-1];
		
		$nb_test_done++;
		
		logD ($i.'/'.$nb_test.') id=['.$test->{'ID'}.'] - description=['.$test->{'DESCRIPTION'}.'] - state=['.$test->{'STATE_VAL'}.'] - msg=['.$test->{'STATE'}.']');
		
		if ( defined($RETURN_CODE_VALUES{$test->{'STATE_VAL'}}) ) {
			$tmp_status = $RETURN_CODE_VALUES{$test->{'STATE_VAL'}};
		} else {
			$np->add_message(WARNING, sprintf(_gt('Bad status for test n°%d'),$i) );
			$tmp_status = OK;
		}
		
		if ( $test->{'VALUE'} =~ /^\d+\.?\d*$/ ) {
			if ( $test->{'ID'} =~ /^[\w_][\w\d\-_\.]*$/ ) {
				$np->add_perfdata(
					'label' => $test->{'ID'},
					'value' => sprintf('%.6f',$test->{'VALUE'}),
					'uom' => $test->{'VALUE_UNIT'},
				);
				
				logD ('DEBUG:  value['.$test->{'VALUE'}.'] - unit['.$test->{'VALUE_UNIT'}.']');
			} elsif ($DEBUG) {
				logW ('Bad Id for test n°'.$i.": '".$test->{'ID'});
			}
		}
		
		if ( defined($ignore) ) {
			if ( grep ($_ eq $test->{'ID'}, @{$ignore}) ) {
				logD ('Ignoring id='.$test->{'ID'});
				$nb_test_ignored++;
				$tmp_status = OK;
			}
		}
		$np->add_message($tmp_status, sprintf(_gt('%s - %s : %s'),$test->{'STATE_VAL'},$test->{'DESCRIPTION'},$test->{'STATE'}) );
		
		if ( ($availability_mode eq 'EQOS') && ($tmp_status != OK) ) {
			if ( $test->{'AVAILABILITY'} ne 'ignore' ) {
				$availability = CRITICAL;
			} else {
				logD ("Ignoring this test (".$test->{'ID'}.") for AVAILABILITY report !");
			}
		}
	}
	
	if ( $verbose ) {
		($status, $message) = $np->check_messages('join' => '<br/>','join_all' => '<br/>');
	} else {
		($status, $message) = $np->check_messages('join' => '<br/>');
	}
	
	if ( $status == OK ) {
		$message = sprintf(_gt("%i test(s) done"),$nb_test_done);
		if ($nb_test_ignored) {
			$message .= sprintf(_gt(" and %i test(s) ignored"),$nb_test_ignored);
		}
		$np->add_message(OK, $message);
	}
	
	
	#
	# Handle check periods
	################################
	
	my $time_start = 0;
	my $time_end = 0;
	if ( $d{'APPLICATION'}{'PERIOD'}{'START'} ne '' ) {
		if ( $d{'APPLICATION'}{'PERIOD'}{'START'} =~ /^(\d{4})\-(\d{2})\-(\d{2})\s(\d{2}):(\d{2}):(\d{2})/) { #2005-09-15 08:00:00
			$time_start = mktime($6, $5, $4, $3, $2-1, $1-1900)||0;
			logD ('Start period: '.$d{'APPLICATION'}{'PERIOD'}{'START'});
		} else {
			logW ('Invalid value for start period: '.$d{'APPLICATION'}{'PERIOD'}{'START'});
		}
	}
	if ( $d{'APPLICATION'}{'PERIOD'}{'END'} ne '' ) {
		if ( $d{'APPLICATION'}{'PERIOD'}{'END'} =~ /^(\d{4})\-(\d{2})\-(\d{2})\s(\d{2}):(\d{2}):(\d{2})/) { #2005-09-15 08:00:00
			$time_end = mktime($6, $5, $4, $3, $2-1, $1-1900)||0;
			logD ('End period: '.$d{'APPLICATION'}{'PERIOD'}{'END'});
		} else {
			logW ('Invalid value for end period: '.$d{'APPLICATION'}{'PERIOD'}{'END'});
		}
	}
	my $time_now = time();
	my $out_of_period = 0;
	logD("period : actual=$time_now start=$time_start end=$time_end");
	if ( $time_start && ($time_now < $time_start) ) {
		$message .= ' '.sprintf(_gt("(before campaign starting at %s)"),$d{'APPLICATION'}{'PERIOD'}{'START'});
		$out_of_period = 1;
	} elsif ( $time_end && ($time_now > $time_end) ) {
		$message .= ' '.sprintf(_gt("(after campaign ending at %s)"),$d{'APPLICATION'}{'PERIOD'}{'END'});
		$out_of_period = 1;
	}
	if ( $out_of_period && ($status == CRITICAL) ) {
		# Lower the status to WARNING if out of period
		$status = WARNING;
	}
	
	logD ("message=".$message);
	
	#
	# Handle EQOS mode
	################################
	
	if ( $availability_mode eq 'EQOS' ) {
		if ( $d{'APPLICATION'}{'AVAILABILITY'} eq 'UP' ) {
			logD ("Force AVAILABILITY to OK due to application requirement");
			$availability = $connection_status;
			
		} elsif ( $d{'APPLICATION'}{'AVAILABILITY'} eq 'DOWN' ) {
			logD ("Force AVAILABILITY to DOWN due to application requirement");
			$availability = CRITICAL;
		}
		# Bypass normal Nagios exit status if we are in EQOS mode.
		$status = $availability;
	}
	
# }


$np->nagios_exit($status, $message );


sub xmlParse {
	our ( $xml, $racvision ) = @_;
	
	our @xpath = ();
	our $currentTag = '';
	
	our $curDataRef = undef;
	
	my $parser = new XML::Parser::Expat('ProtocolEncoding' => 'utf-8');
	
	$parser->setHandlers(
		'Start'	=> \&start_handler,
		'End'	=> \&end_handler,
		'Char'	=> \&char_handler,
	);
	
	eval { $parser->parse($xml) };

	if ( $@ ) {
		$racvision->{'_ERROR'} = "Could not parse XML : $@";
		$racvision->{'_ERROR'} =~ s/\n//g;
		$racvision->{'_ERROR'} =~ s/at $0 line \d+//g;
		return 0;
	}

	return 1;
	
	
	# start element callback
	sub start_handler {
		my ($p, $el, %attr) = @_;
		$el = uc($el);
		my $unknownTag = '';
		
#  		logD ("start_handler for <$el>");
		
		if ($currentTag eq '' && $el ne 'APPTEST') {
			$unknownTag = $el;
			
		} elsif ($currentTag eq 'APPTEST' ) {
			if ( defined($racvision->{$el}) ) {
				$curDataRef = \$racvision->{$el};
			} else {
				$unknownTag = $el;
				$curDataRef = undef;
			}
		} elsif ($currentTag eq 'APPLICATION' ) {
			if ( defined($racvision->{'APPLICATION'}{$el}) ) {
				$curDataRef = \$racvision->{'APPLICATION'}{$el};
# 				print Dumper(\%attr);
				if ($el eq 'TEST' ) {
					push(@{$racvision->{'APPLICATION'}{'TEST'}}, {'DESCRIPTION' => '' , 'STATE' => '', 'STATE_VAL' => '', 'VALUE' => '', 'VALUE_UNIT' => '', 'AVAILABILITY' => ''});
	
					if ( defined($attr{'id'}) ) {
						$racvision->{'APPLICATION'}{'TEST'}[-1]{'ID'} = $attr{'id'};
					} else {
						logW ("No id attibute define for the current test");
					}
						
					if ( defined($attr{'availability'}) ) {
						$racvision->{'APPLICATION'}{'TEST'}[-1]{'AVAILABILITY'} = $attr{'availability'};
					}
				} elsif ($el eq 'AVAILABILITY' ) {
					if ( defined($attr{'available'}) ) {
						$racvision->{'APPLICATION'}{'AVAILABILITY'} = $attr{'available'};
					}
				}
			} else {
				$unknownTag = $el;
				$curDataRef = undef;
			}
		} elsif ($currentTag eq 'TEST' ) {
			if ( $el eq 'DESCRIPTION' ) {
				$curDataRef = \$racvision->{'APPLICATION'}{'TEST'}[-1]{$el};
			} elsif ($el eq 'STATE' ) {
				$curDataRef = \$racvision->{'APPLICATION'}{'TEST'}[-1]{$el};
				if ( defined($attr{'val'}) ) {
					$racvision->{'APPLICATION'}{'TEST'}[-1]{'STATE_VAL'} = $attr{'val'};
				} else {
					logW ("No state val attibute define for the current test");
				}
			} elsif ($el eq 'VALUE' ) {
				$curDataRef = \$racvision->{'APPLICATION'}{'TEST'}[-1]{$el};
				if ( defined($attr{'unit'}) ) {
					$racvision->{'APPLICATION'}{'TEST'}[-1]{'VALUE_UNIT'} = $attr{'unit'};
				}
			} else {
				$unknownTag = $el;
				$curDataRef = undef;
			}
		} elsif ($currentTag eq 'PERIOD' ) {
			if ( defined($racvision->{'APPLICATION'}{'PERIOD'}{$el}) ) {
				$curDataRef = \$racvision->{'APPLICATION'}{'PERIOD'}{$el};
			} else {
				$unknownTag = $el;
				$curDataRef = undef;
			}
		}
		
#		logD ("ref (curDataRef)=" .ref($curDataRef));
		if ( $unknownTag ne '' ) {
			logW ("WARNING: Unknown tag in /".join('/',@xpath).": $el");
		}
#		print STDERR 'PATH /'.join('/',@xpath)." >> $el\n";
		
		$currentTag = $el;
		push (@xpath, $el);
	}
	
	# end element callback
	sub end_handler {
		my ($p, $el) = @_;
		$el = uc($el);
#		logD ("end_handler for <$el>");
		
		if ( $currentTag eq $el ) {
			pop(@xpath);
			$currentTag = $xpath[-1];
		} else {
 			logW ("BUG: el=$el currentTag=$currentTag");
		}
		$curDataRef = undef;
	}
	
	# characters beetwen elements callback
	sub char_handler {
		my ($p, $data) = @_;
		if ( defined($curDataRef) && ref($curDataRef) eq 'SCALAR') {
#			logD ("Set $currentTag=$data ref=".ref($curDataRef));
			$$curDataRef = $data;
		} else {
			logD ("Can't set value '$data' for currentTag=$currentTag ref=".ref($curDataRef));
		}
	}

}
sub logD {
	print STDERR 'DEBUG:   '.$_[0]."\n" if ($DEBUG);
}
sub logW {
	print STDERR 'WARNING: '.$_[0]."\n" if ($DEBUG);
}
# Gettext wrapper
sub _gt {
	return gettext($_[0]);
}


__END__

=head1 NAME

This Nagios plugins check a specified Racvision url (xml document) returned by an application and parse its content.


=head1 NAGIOS CONGIGURATIONS

In F<checkcommands.cfg> you have to add :

	define command {
	  command_name	check_racvision
	  command_line	$USER1$/check_racvision.pl -u $ARG1$
	}


In F<services.cfg> you just have to add something like :

	define service {
	  host_name             www.exemple.org
	  normal_check_interval 10
	  retry_check_interval  5
	  contact_groups        linux-admins
	  service_description	My great web application
	  check_command			check_racvision!http://www.exemple.org/myApplication/racvision
	}

=head1 AUTHOR

Stéphane Urbanovski <stephane.urbanovski@ac-nancy-metz.fr>

=cut
